/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: chnode.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * chnode.c -- available channel info list
 * Robin Kirkham, June 1996
 */

#include <stdio.h>

#include "internal.h"


int iio_chnode_cmp(IIO_SLL *s1, IIO_SLL *s2) {
    /*
     * Used by the ordered singly-linked list insert. Returns a positive,
     * zero or negative result depending on whether s1 is greater, equal
     * or less that s2. For the module list the basis is the ident code,
     * then the global generic sequence number
     */
    int cmp;

    /* compare the channel type names, return lexographic result */
    if ((cmp = iio_string_cmp(
        iio_chtype_string[((IIO_CHNODE *)s1)->type],
        iio_chtype_string[((IIO_CHNODE *)s2)->type]
    )))
        return cmp;
    else
	/* GG sequence number difference should never be zero */
	return (
	    ((IIO_CHNODE *)s1)->seqno[iio_nform_gg] -
	    ((IIO_CHNODE *)s2)->seqno[iio_nform_gg]
	);
}


HIDDEN IIO_STATUS iio_chnode_new(
    IIO_MODULE *module, IIO_CHTYPE type, unsigned int width,
    unsigned int number, IIO_OPFN operate, IIO_CHNODE **chn
) {
    /*
     * Create a new channel info block and link it to the given module,
     * and link it into the list of available channels. This function is
     * called by iio_chnode() only
     */
    IIO_CHNODE *new, *chnode;
    int chan;

    /* allocate a new chnode block */
    iio_eret( iio_mem_alloc(sizeof(IIO_CHNODE), (void **)&new) );

    /* fill it out */
    new->magic = iio_magic_chnode;
    new->type = type;
    new->width = width;
    new->number = number;
    new->module = module;
    new->operate = operate;

    /* initialise the extra bitfield things as well */
    new->rseqno = 0;
    new->rchnode = NULL;

    /* allocate an array of chinfo structures */
    iio_eret( iio_mem_alloc(sizeof(IIO_CHINFO)*number, (void **)&new->chinfo) );

    /* initialise the chinfos to sensible defaults */
    for (chan = 0; chan < number; ++chan) {
	new->chinfo[chan].magic = iio_magic_chinfo;

	iio_chnode_linear(new, chan, 1.0, 0.0, "");
	iio_chnode_limits(new, chan, 0, 0, -1);
	iio_chnode_log(new, chan, iio_bool_false);
    }

    /*
     * Search the current channel list to get the sequence numbers.
     * This is actually done by adding up the seqnos for each matching
     * kind of seqno. There are four seqnos for each channel group
     * (i.e. chnode structure): global generic (number in the sequence of
     * all channels of this type in the system), global specific (number
     * in the sequence of all channels of this type AND width in the
     * system), local generic (seqno in all channels of type on this module)
     * and local specific (seqno in all of this type AND width on module).
     *
     * The seqnos refer to the FIRST (zeroth) channel in the group; seqnos
     * of other channels in the group are all respectively relative it.
     *
     * Since iio_mem_alloc() returns zeroed memory, we know that 
     * unassigned seqno's will be zero
     */
    for (chnode = iio_state->chnode; chnode; chnode = chnode->next) 
	if (chnode->type == new->type) {
	    new->seqno[iio_nform_gg] += chnode->number;

	    if (chnode->module == new->module) 
		new->seqno[iio_nform_lg] += chnode->number;

	    if (chnode->width == new->width) {
		new->seqno[iio_nform_gs] += chnode->number;

		if (chnode->module == new->module) 
		    new->seqno[iio_nform_ls] += chnode->number;
	    }	
	}

    /*
     * Link it in to the main chnode list. The list is ordered by
     * aphabetic channel type, width, then seqno_gg, according
     * to iio_chnode_cmp()
     */
    iio_fret(
	iio_sll_insert(
	    (IIO_SLL **)&iio_state->chnode,
	    (IIO_SLL *)new,
	    iio_chnode_cmp
	)
    );

    /* return the new chnode */
    *chn = new;

    return iio_status_ok;
}


IIO_STATUS iio_chnode(
    IIO_MODULE *module, IIO_CHTYPE type, unsigned int width,
    unsigned int number, IIO_OPFN operate, IIO_CHNODE **chn
) {
    /*
     * Create a new channel info block and link it to the given module,
     * and link it into the list of available channels. This function is
     * called by the module install functions, in turn called by iio_module().
     *
     * For most channel types, it calls iio_chnode_new() to insert the new
     * chnode block. For digital IO channels, however, it calls
     * iio_chnode_new() further to insert the corresponding bitfield chnodes
     */
    IIO_CHNODE *chnode, *bchnode;
    IIO_CHTYPE btype;
    unsigned seqno;

    /* check passed arguments as far as possible */
    if (!module || module->magic != iio_magic_module)
        return iio_fatal("NULL or bad module pointer");
    if (!operate)
        return iio_fatal("NULL operate function pointer");
    if (width > channelwidth(int))
        return iio_fatal("Channel width too large");
    if (number > 1000)
        return iio_fatal("Unreasonable number of channels");
    if (number < 1)
        return iio_fatal("Must have at least one channel");

    /* bitfield channels cannot be registered directly */
    if ((
	(type == iio_chtype_bi)   ||
	(type == iio_chtype_bo)   ||
	(type == iio_chtype_bio)  ||
	(type == iio_chtype_boco) ||
	(type == iio_chtype_bocio)
    ))
	return iio_error("Bitfield channels cannot be registered directly");


    /* for all channel types */
    iio_eret( iio_chnode_new(module, type, width, number, operate, &chnode) );

    /* return chnode pointer (of the main channels) if required */
    if (chn)
	*chn = chnode;

    /*
     * For di/do/dio channels, insert corresponding bi/bo/bio
     * channels. We register the bit-channels in chnode-groups which align
     * with individual di/do/dio channels, i.e. the original channel
     * *width* becomes the *number* of bit-channels in the group. This
     * allows us to use the IIO channel-range split-up mechanism to do
     * the split-up for bit-channels as well.
     *
     * This is not done for drio ports. It does not make sense, because
     * setting an individual bit in a drio port will make bits around it
     * possible change state, which is not what you would expect from a
     * bitfield.
     */

    /* decide the equivalent channel type */
    switch (type) {
    case iio_chtype_di:   btype = iio_chtype_bi;    break;
    case iio_chtype_do:   btype = iio_chtype_bo;    break;
    case iio_chtype_dio:  btype = iio_chtype_bio;   break;
    case iio_chtype_oco:  btype = iio_chtype_boco;  break;
    case iio_chtype_ocio: btype = iio_chtype_bocio; break;
    default:
        return iio_status_ok;
    }

    /* register equivalent number of bit-channels */
    for (seqno = 0; seqno < number; ++seqno) {
	iio_eret(
	    iio_chnode_new(
		module,
		btype,
		1, width,
		operate, &bchnode
	    )
	);

        /*
         * Link this new chnode, which represents a group of <width> bits,
         * with the chnode of the underlying real channel and the seqno
         */
        bchnode->rchnode = chnode;
        bchnode->rseqno = seqno;
    }

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_chnode_check(IIO_CHNODE *chnode, unsigned seqno) {
    /*
     * A quick sanity check of the arguments
     */
    if (! chnode || chnode->magic != iio_magic_chnode)
	return iio_fatal("NULL or bad chnode pointer");
    if (seqno >= chnode->number)
	return iio_fatal("Local sequence number out of range");
    if (chnode->chinfo[seqno].magic != iio_magic_chinfo)
	return iio_fatal("Bad chinfo structure");
    return iio_status_ok;
}


IIO_STATUS iio_chnode_linear(
    IIO_CHNODE *chnode, unsigned seqno, double scale, double offset, char *unit
) {
    /*
     * Sets the factors used for the linear scaling of channel values 
     * for the given simple channel, identified by its channel node and
     * local sequence number. The unit string may also be useful
     */
    iio_eret( iio_chnode_check(chnode, seqno) );

    /* also check that scale is not zero */
    if (scale == 0.0)
	return iio_error("Zero scale factor not allowed");

    chnode->chinfo[seqno].scale = scale;
    chnode->chinfo[seqno].offset = offset;
    if (unit && *unit) {
	iio_eret( iio_string_dup(unit, &chnode->chinfo[seqno].unit) );
    } else
	chnode->chinfo[seqno].unit = "";

    return iio_status_ok;
}


IIO_STATUS iio_chnode_limits(
    IIO_CHNODE *chnode, unsigned seqno, int initial, int lower, int upper
) {
    /*
     * Set the limits to which the channel can be set (not read). Limiting
     * is (later) ignored if the lower limit exceeds the upper. The initial
     * value is also specified here (later limited, if necessary)
     */
    iio_eret( iio_chnode_check(chnode, seqno) );

    chnode->chinfo[seqno].lower = lower;
    chnode->chinfo[seqno].upper = upper;
    chnode->chinfo[seqno].initial = initial;

    return iio_status_ok;
}


IIO_STATUS iio_chnode_log(IIO_CHNODE *chnode, unsigned seqno, IIO_BOOL log) {
    /*
     * Sets logging status of the given local channel seqno to log
     */
    iio_eret( iio_chnode_check(chnode, seqno) );

    chnode->chinfo[seqno].log = log;
    return iio_status_ok;
}


IIO_STATUS iio_chnode_show(IIO_BOOL longform) {
    /*
     * TEMPORARY FUNCTION
     * Print the channel list. If longform is true, print all the 
     * channels and their equivalents
     */
    IIO_CHNODE *chnode;
    int count;

    printf(
	"%3s %14s %14s %22s %22s\n",
	"num",
	"name(gg)  ",
	"name(gs)  ",
	"name(lg)  ",
	"name(ls)  "
    );

    for (chnode = iio_state->chnode; chnode; chnode = chnode->next) {

	/* the channel names */
	char gg[100], gs[100];
	char lg[100], ls[100];
	char *type = iio_chtype_string[chnode->type];

	if (longform)
	    for (count = 0; count < chnode->number; ++count) {

		/* write the channel's four names */
		iio_slog(
		    gg,
		    "%s.%-3d",
		    type,
		    chnode->seqno[iio_nform_gg] + count
		);
		iio_slog(
		    gs,
		    "%s%d.%-3d",
		    type,
		    chnode->width,
		    chnode->seqno[iio_nform_gs] + count
		);
		iio_slog(
		    lg,
		    "%s.%d:%s.%-3d",
		    chnode->module->minfo->ident,
		    chnode->module->seq,
		    type,
		    chnode->seqno[iio_nform_lg] + count
		);
		iio_slog(
		    ls,
		    "%s.%d:%s%d.%-3d",
		    chnode->module->minfo->ident,
		    chnode->module->seq,
		    type,
		    chnode->width,
		    chnode->seqno[iio_nform_ls] + count
		);

		printf(
		    "    %14s %14s %22s %22s\n",
		    gg, gs, lg, ls
		);
	    }

	else {
	    /* do it in range format */

	    /* write the channel's four names */
	    iio_slog(
		gg,
		"%s.%d-%d",
		type,
		chnode->seqno[iio_nform_gg],
		chnode->seqno[iio_nform_gg] + chnode->number - 1
	    );
	    iio_slog(
		gs,
		"%s%d.%d-%d",
		type,
		chnode->width,
		chnode->seqno[iio_nform_gs],
		chnode->seqno[iio_nform_gs] + chnode->number - 1
	    );
	    iio_slog(
		lg,
		"%s.%d:%s.%d-%d",
		chnode->module->minfo->ident,
		chnode->module->seq,
		type,
		chnode->seqno[iio_nform_lg],
		chnode->seqno[iio_nform_lg] + chnode->number - 1
	    );
	    iio_slog(
		ls,
		"%s.%d:%s%d.%d-%d",
		chnode->module->minfo->ident,
		chnode->module->seq,
		type,
		chnode->width,
		chnode->seqno[iio_nform_ls],
		chnode->seqno[iio_nform_ls] + chnode->number - 1
	    );

	    printf(
		"%3d %14s %14s %22s %22s\n",
		chnode->number,
		gg, gs, lg, ls
	    );
	}
    }

    return iio_status_ok;
}
