/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: mc68230.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * mc68230.c -- driver for Motorola MC68230 PI/T
 * Robin Kirkham, October 1996, from PIRAT original
 *
 * The MC68230 is a very complicated and difficult to understand multi-modal
 * double-buffered all-things-to-all-people IO chip. This module falls far
 * short of being a proper driver for it: all it does is initialise a 
 * register pointer for the chip, sets the chip to a "reset" state, or prints
 * out the chip status.
 */
#include "../internal.h"
#include "mc68230.h"


IIO_STATUS iio_mc68230_install(
    IIO_MC68230 *dd,
    IIO channel, IIO_OP space, unsigned base, unsigned stretch
) {
    /*
     * Allocate and initialise an IIO_MC68230 register pointer structure for
     * the given base address and space, and stretch factor (0 is minimum)
     */

    /* fill out register pointers */
    iio_eret(
	iio_resolve_list(
	    channel, space,
	    iio_size_8, base + (IIO_MC68230_PGCR << stretch), &dd->pgcr,
	    iio_size_8, base + (IIO_MC68230_PSR << stretch), &dd->psr,
	    iio_size_8, base + (IIO_MC68230_PSRR << stretch), &dd->psrr,
	    iio_size_8, base + (IIO_MC68230_PIVR << stretch), &dd->pivr,
	    iio_size_8, base + (IIO_MC68230_PACR << stretch), &dd->pacr,
	    iio_size_8, base + (IIO_MC68230_PADDR << stretch), &dd->paddr,
	    iio_size_8, base + (IIO_MC68230_PADR << stretch), &dd->padr, 
	    iio_size_8, base + (IIO_MC68230_PAAR << stretch), &dd->paar, 
	    iio_size_8, base + (IIO_MC68230_PBCR << stretch), &dd->pbcr, 
	    iio_size_8, base + (IIO_MC68230_PBDDR << stretch), &dd->pbddr,
	    iio_size_8, base + (IIO_MC68230_PBDR << stretch), &dd->pbdr, 
	    iio_size_8, base + (IIO_MC68230_PBAR << stretch), &dd->pbar, 
	    iio_size_8, base + (IIO_MC68230_PCDR << stretch), &dd->pcdr, 
	    iio_size_8, base + (IIO_MC68230_PCDDR << stretch), &dd->pcddr,
	    iio_size_8, base + (IIO_MC68230_TCR << stretch), &dd->tcr,  
	    iio_size_8, base + (IIO_MC68230_TSR << stretch), &dd->tsr,  
	    iio_size_8, base + (IIO_MC68230_TIVR << stretch), &dd->tivr, 
	    iio_size_8, base + (IIO_MC68230_CPRH << stretch), &dd->cprh, 
	    iio_size_8, base + (IIO_MC68230_CPRM << stretch), &dd->cprm, 
	    iio_size_8, base + (IIO_MC68230_CPRL << stretch), &dd->cprl, 
	    iio_size_8, base + (IIO_MC68230_CRH << stretch), &dd->crh,  
	    iio_size_8, base + (IIO_MC68230_CRM << stretch), &dd->crm,  
	    iio_size_8, base + (IIO_MC68230_CRL << stretch), &dd->crl,  
	    0
	)
    );

    return iio_status_ok;
}


IIO_STATUS iio_mc68230_init(IIO_MC68230 *dd) {
    /*
     * "Reset" the 68230 by setting all registers to their default
     * power-up values
     */
    *dd->pgcr  = 0x00; *dd->psrr  = 0x00; *dd->paddr = 0x00; *dd->pbddr = 0x00;
    *dd->pcddr = 0x00; *dd->pivr  = 0x0f; *dd->pacr  = 0x00; *dd->pbcr  = 0x00;
    *dd->padr  = 0x00; *dd->pbdr  = 0x00; *dd->paar  = 0x00; *dd->pbar  = 0x00;
    *dd->pcdr  = 0x00; *dd->psr   = 0x00; *dd->tcr   = 0x00; *dd->tivr  = 0x0f;
    *dd->cprh  = 0x00; *dd->cprm  = 0x00; *dd->cprl  = 0x00; *dd->crh   = 0x00;
    *dd->crm   = 0x00; *dd->crl   = 0x00; *dd->tsr   = 0x00;

    return iio_status_ok;
}


IIO_STATUS iio_mc68230_show(IIO_MC68230 *dd) {
    /*
     * Print out current 68230 register values in semi-readable form
     */
    uint32_t cpr, cr;

    iio_log(
	"    PGCR: mode = %d, h34e = %d, h12e = %d, "
	"h4s = %d, h3s = %d, h2s = %d, h1s = %d\n",
	(*dd->pgcr & IIO_MC68230_PGCR_PMC_mask)
	    >> IIO_MC68230_PGCR_PMC_roll,
	(*dd->pgcr & IIO_MC68230_PGCR_H34_ENABLE) ? 1 : 0,
	(*dd->pgcr & IIO_MC68230_PGCR_H12_ENABLE) ? 1 : 0,
	(*dd->pgcr & IIO_MC68230_PGCR_H4_SENSE) ? 1 : 0,
	(*dd->pgcr & IIO_MC68230_PGCR_H3_SENSE) ? 1 : 0,
	(*dd->pgcr & IIO_MC68230_PGCR_H2_SENSE) ? 1 : 0,
	(*dd->pgcr & IIO_MC68230_PGCR_H1_SENSE) ? 1 : 0
    );
    iio_log(
	"    PSRR: svcrqs = %d, ipfs = %d, pipc = %d\n",
	(*dd->psrr & IIO_MC68230_PSRR_SVCRQS_mask)
	    >> IIO_MC68230_PSRR_SVCRQS_roll,
	(*dd->psrr & IIO_MC68230_PSRR_IPFS_mask)
	    >> IIO_MC68230_PSRR_IPFS_roll,
	(*dd->psrr & IIO_MC68230_PSRR_PIPC_mask)
	    >> IIO_MC68230_PSRR_PIPC_roll
    );
    iio_log(
	"    PIVR: ivn = %d\n",
	(*dd->pivr & IIO_MC68230_PIVR_IVN_mask)
	    >> IIO_MC68230_PIVR_IVN_roll
    );
    iio_log(
	"    PSR: h4l = %d, h3l = %d, h2l = %d, h1l = %d, "
	"h4s = %d, h3s = %d, h2s = %d, h1s = %d\n",
	(*dd->psr & IIO_MC68230_PSR_H4_LEVEL) ? 1 : 0,
	(*dd->psr & IIO_MC68230_PSR_H3_LEVEL) ? 1 : 0,
	(*dd->psr & IIO_MC68230_PSR_H2_LEVEL) ? 1 : 0,
	(*dd->psr & IIO_MC68230_PSR_H1_LEVEL) ? 1 : 0,
	(*dd->psr & IIO_MC68230_PSR_H4S) ? 1 : 0,
	(*dd->psr & IIO_MC68230_PSR_H3S) ? 1 : 0,
	(*dd->psr & IIO_MC68230_PSR_H2S) ? 1 : 0,
	(*dd->psr & IIO_MC68230_PSR_H1S) ? 1 : 0
    );
	
    iio_log(
	"    PORT A: ddr = 0x%02x, dr = 0x%02x, alt = 0x%02x, "
	"sm = %d, h2c = %d, h2ie = %d, h1se = %d, h1s = %d\n",
	*dd->paddr,
	*dd->padr,
	*dd->paar,
	(*dd->pacr & IIO_MC68230_PACR_SUBMODE_mask)
	    >> IIO_MC68230_PACR_SUBMODE_roll,
	(*dd->pacr & IIO_MC68230_PACR_H2_CTL_mask)
	    >> IIO_MC68230_PACR_H2_CTL_roll,
	(*dd->pacr & IIO_MC68230_PACR_H2_INT_ENABLE) ? 1 : 0,
	(*dd->pacr & IIO_MC68230_PACR_H1_SVCRQ_ENABLE) ? 1 : 0,
	(*dd->pacr & IIO_MC68230_PACR_H1_STAT_CONTROL) ? 1 : 0
    );
    iio_log(
	"    PORT B: ddr = 0x%02x, dr = 0x%02x, alt = 0x%02x, "
	"sm = %d, h4c = %d, h4ie = %d, h3se = %d, h3s = %d\n",
	*dd->pbddr,
	*dd->pbdr,
	*dd->pbar,
	(*dd->pbcr & IIO_MC68230_PBCR_SUBMODE_mask)
	    >> IIO_MC68230_PBCR_SUBMODE_roll,
	(*dd->pbcr & IIO_MC68230_PBCR_H4_CTL_mask)
	    >> IIO_MC68230_PBCR_H4_CTL_roll,
	(*dd->pbcr & IIO_MC68230_PBCR_H4_INT_ENABLE) ? 1 : 0,
	(*dd->pbcr & IIO_MC68230_PBCR_H3_SVCRQ_ENABLE) ? 1 : 0,
	(*dd->pbcr & IIO_MC68230_PBCR_H3_STAT_CONTROL) ? 1 : 0
    );
    iio_log(
	"    PORT C: ddr = 0x%02x, dr = 0x%02x\n",
	*dd->pcddr,
	*dd->pcdr
    );

    iio_log(
	"    TCR: tctl = %d, zdc = %d, cctl = %d, te = %d, "
	"TSR: zds = %d TIVR: tivr = 0x%02x\n",
	(*dd->tcr & IIO_MC68230_TCR_TCTL_mask)
	    >> IIO_MC68230_TCR_TCTL_roll,
	(*dd->tcr & IIO_MC68230_TCR_ZD_CONTROL) ? 1 : 0,
	(*dd->tcr & IIO_MC68230_TCR_CCTL_mask)
	    >> IIO_MC68230_TCR_CCTL_roll,
	(*dd->tcr & IIO_MC68230_TCR_TIMER_ENABLE) ? 1 : 0,
	(*dd->tsr & IIO_MC68230_TSR_ZD_STATUS) ? 1 : 0,
	*dd->tivr
    );

    cpr = (((*dd->cprh << 8) | *dd->cprm) << 8) | *dd->cprl;
    cr = (((*dd->crh << 8) | *dd->crm) << 8) | *dd->crl;
    iio_log(
	"    COUNTER: cpr = 0x%06x = %d, cr = 0x%06x = %d\n",
	cpr, cpr, cr, cr
    );

    return iio_status_ok;
}
