/* This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: lm628.h 2222 2007-12-04 11:46:42Z roy029 $
 *
 * lm628.h -- National LM628 servo chip driver
 * Robin Kirkham, October 1996, from Mr. Plod
 */

    /* register structure */
typedef struct {
    volatile uint8_t *csr;		/* control/status register address */
    volatile uint8_t *dr;		/* data register address */

    IIO_BOOL bit12;			/* true for 12-bit output DAC */
    IIO_BOOL debug;			/* produce chip-level debug output */
    uint8_t ds;			/* derivative sample multiplier */

    /*
     * Scale factors between user values and LM628 register values.
     * Always <reg> = <user value> * factor
     */
    double
	pgfactor,			/* proportional gain factor */
	dgfactor,			/* derivative gain factor */
	igfactor,			/* integral gain factor */
	ilfactor,			/* integral limit factor */
	dtfactor,			/* dt target factor */
	ddtfactor;			/* ddt target factor */

    /*
     * Initial loop factors from the config file, not fully scaled.
     * These are used by the init function (of the first IIO run) only
     */
    double
	ipgain,				/* initial proportional gain */
	idgain,				/* initial derivative gain */
	iigain,				/* initial integral gain */
	iimax;				/* initial integral limit */

} IIO_LM628_MREG;


    /* state structure */
typedef struct {
    int32_t offset;			/* current target offset */
    IIO_BOOL udf;			/* true if filter needs an update */
    IIO_BOOL accf;			/* true if new accln was given */
    uint32_t acc;			/* new accelleration value to load */

    uint16_t kp, ki, kd, il;	/* shadows of WO feedback registers */
} IIO_LM628_MSTATE;

    /* functions used by module drivers */
IIO_STATUS
    iio_lm628_sc(
	IIO_LM628_MSTATE *state, IIO_LM628_MREG *reg,
	IIO_OPNODE *opnode, IIO_OP op, unsigned seqno
    ),
    iio_lm628_init(
	IIO_LM628_MSTATE *state, IIO_LM628_MREG *reg
    ),
    iio_lm628_install(
	IIO_LM628_MREG *reg, volatile uint8_t *csr,
	volatile uint8_t *dr,
	uint32_t clock, IIO_BOOL bits12, double orange, char *argv[],
	IIO_CHNODE *chnode, unsigned seqno
    );


#ifdef IIO_LM628_DEFINES

    /* LM628 clock pulses per servo interval */
#define IIO_LM628_PERIOD	(2048)

    /* LM628 command uint8_ts */
#define IIO_LM628_C_RESET	0x00	/* Reset LM628 */
#define IIO_LM628_C_PORT8	0x05	/* Select 8-bit Output */
#define IIO_LM628_C_PORT12	0x06	/* Select 12-bit Output */
#define IIO_LM628_C_DFH		0x02	/* Define Home */
#define IIO_LM628_C_SIP		0x03	/* Set Index Position */
#define IIO_LM628_C_LPEI	0x1b	/* Interrupt on Error */
#define IIO_LM628_C_LPES	0x1a	/* Stop on Error */
#define IIO_LM628_C_SBPA	0x20	/* Set Breakpoint, Absolute */
#define IIO_LM628_C_SBPR	0x21	/* Set Breakpoint, Relative */
#define IIO_LM628_C_MSKI	0x1c	/* Mask Interrupts */
#define IIO_LM628_C_RSTI	0x1d	/* Reset Interrupts */
#define IIO_LM628_C_LFIL	0x1e	/* Load Filter Parameters */
#define IIO_LM628_C_UDF		0x04	/* Update Filter Parameters */
#define IIO_LM628_C_LTRJ	0x1f	/* Load Trajectory */
#define IIO_LM628_C_STT		0x01	/* Start Motion */
#define IIO_LM628_C_RDSTAT	0xff	/* Read Status Byte */
#define IIO_LM628_C_RDSIGS	0x0c	/* Read Signals Register */
#define IIO_LM628_C_RDIP	0x09	/* Read Index Position */
#define IIO_LM628_C_RDDP	0x08	/* Read Desired Position */
#define IIO_LM628_C_RDRP	0x0a	/* Read Real Position */
#define IIO_LM628_C_RDDV	0x07	/* Read Desired Velocity */
#define IIO_LM628_C_RDRV	0x0b	/* Read Real Velocity */
#define IIO_LM628_C_RDSUM	0x0d	/* Read Integration Sum */

    /* Filter Control Word: for LFIL command */
#define IIO_LM628_F_DMASK	(0xff << 8)	/* Derivative Sample Mask */
#define IIO_LM628_F_DROLL	8	/* Derivative Sample Roll */
#define IIO_LM628_F_KP		(1<< 3)	/* Loading Kp data */
#define IIO_LM628_F_KI		(1<< 2)	/* Loading Ki data */
#define IIO_LM628_F_KD		(1<< 1)	/* Loading Kd data */
#define IIO_LM628_F_IL		(1<< 0)	/* Loading Il data */

    /* Trajectory Control Word: for LTRJ command */
#define IIO_LM628_T_FORW	(1<<12)	/* Forward Direction (Vel Mode) */
#define IIO_LM628_T_VMODE	(1<<11)	/* Velocity Mode */
#define IIO_LM628_T_SSTOP	(1<<10)	/* Stop Smoothly */
#define IIO_LM628_T_STOP	(1<< 9)	/* Stop Rapidly */
#define IIO_LM628_T_MOFF	(1<< 8)	/* Stop (Motor Off) */
#define IIO_LM628_T_AL		(1<< 5)	/* Acceleration will be Loaded */
#define IIO_LM628_T_ADR		(1<< 4)	/* Acceleration Data is Relative */
#define IIO_LM628_T_VL		(1<< 3)	/* Velocity will be Loaded */
#define IIO_LM628_T_VDR		(1<< 2)	/* Velocity Data is Relative */
#define IIO_LM628_T_PL		(1<< 1)	/* Position will be Loaded */
#define IIO_LM628_T_PDR		(1<< 0)	/* Position Data is Relative */

    /* Mask Byte: for MSKI/RSTI commands */
#define IIO_LM628_I_BDONE	(1<< 6)	/* Breakpoint Reached */
#define IIO_LM628_I_PERR	(1<< 5)	/* Excessive Position Error */
#define IIO_LM628_I_WRAP	(1<< 4)	/* Wraparound Observed */
#define IIO_LM628_I_INDEX	(1<< 3)	/* Index Pulse Observed */
#define IIO_LM628_I_TDONE	(1<< 2)	/* Trajectory Complete */
#define IIO_LM628_I_CERR	(1<< 1)	/* Command Error */

    /* Status Byte: from RDSTAT command */
#define IIO_LM628_S_MOFF	(1<< 7)	/* Motor Off */
#define IIO_LM628_S_BDONE	(1<< 6)	/* Breakpoint Reached */
#define IIO_LM628_S_PERR	(1<< 5)	/* Excessive Position Error */
#define IIO_LM628_S_WRAP	(1<< 4)	/* Wraparound Observed */
#define IIO_LM628_S_INDEX	(1<< 3)	/* Index Pulse Observed */
#define IIO_LM628_S_TDONE	(1<< 2)	/* Trajectory Complete */
#define IIO_LM628_S_CERR	(1<< 1)	/* Command Error */
#define IIO_LM628_S_BUSY	(1<< 0)	/* Busy Bit */

    /* Signals Register */
#define IIO_LM628_G_HINT	(1<<15)	/* Host Interrupt */
#define IIO_LM628_G_ALNU	(1<<14)	/* Acceleration Loaded, not updated */
#define IIO_LM628_G_FLNU	(1<<13)	/* Filter Loaded, not updated */
#define IIO_LM628_G_FORW	(1<<12)	/* Forward direction */
#define IIO_LM628_G_VMODE	(1<<11)	/* Velocity Mode */
#define IIO_LM628_G_TARGET	(1<<10)	/* On Target */
#define IIO_LM628_G_PERROFF	(1<< 9)	/* Turn Off on Excessive Pos Error */
#define IIO_LM628_G_PORT8	(1<< 8)	/* 8-bit Output Mode */
#define IIO_LM628_G_MOFF	(1<< 7)	/* Motor Off */
#define IIO_LM628_G_BDONE	(1<< 6)	/* Breakpoint Reached */
#define IIO_LM628_G_PERR	(1<< 5)	/* Excessive Position Error */
#define IIO_LM628_G_WRAP	(1<< 4)	/* Wraparound Observed */
#define IIO_LM628_G_INDEX	(1<< 3)	/* Index Pulse Observed */
#define IIO_LM628_G_TDONE	(1<< 2)	/* Trajectory Complete */
#define IIO_LM628_G_CERR	(1<< 1)	/* Command Error */
#define IIO_LM628_G_ANI		(1<< 0)	/* Acquire Next Index */

#endif

/* end */
