/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: lm628.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * lm628.c -- National LM628 servo chip driver
 * Robin Kirkham, October 1996, from Mr. Plod
 *
 * This chip driver is really a mini-module driver, "hosted" by a real
 * module driver (such as for the IP-Servo). The main external functions
 * and data structrues match those of the module driver, except that only
 * one channel is dealt with each time. The driver implements all of the
 * IIO server channel operation codes
 *
 * The driver is istelf two-level: there is a chip-access level for
 * communicating with the LM628, and the IIO-style driver level using it
 */
#define IIO_LM628_DEFINES

#include "../internal.h"
#include "lm628.h"

    /* delay writing accelleration flag */
#undef LATEACC

/*
 * ----------------------------------------------------------- LM628 CHIP LEVEL
 */

HIDDEN char *iio_lm628_name[] = {
    "RESET", "STT", "DFH", "SIP", "UDF", "PORT8", "PORT12", "RDDV",
    "RDDP", "RDIP", "RDRP", "RDRV", "RDSIGS", "RDSUM", "", "",
    "", "", "", "", "", "", "", "",
    "", "", "LPES", "LPEI", "MSKI", "RSTI", "LFIL", "LTRJ",
    "SBPA", "SBPR"
};


HIDDEN __inline__ uint8_t iio_lm628_poll(IIO_LM628_MREG *reg) {
    /*
     * Wait for the BUSY bit of the status register of the LM628 to go low.
     * In theory BUSY should never be high more than 100us, and is generally
     * only high for 15 to 25us
     */
    volatile uint8_t status = 0;
    while ((status = *reg->csr) & IIO_LM628_S_BUSY) ;
    return status;
}

HIDDEN IIO_STATUS iio_lm628_command(
    IIO_LM628_MREG *reg, uint8_t command, ...
) {
    /*
     * Send the given command to the LM628 chip selected by the reg structure,
     * then read or write data according to the list passed by the caller.
     * The list is a sequence of tags and integer values, or (for reads)
     * integer pointers, with a zero tag to end the list. It is up to the user
     * to supply the correct list for a given command
     */
    volatile uint8_t status = 0;
    uint16_t u16;
    uint32_t u32;
    IIO_ARG tag;
    va_list ap;
    va_start(ap, command);

    /* wait for busy bit to clear */
    iio_lm628_poll(reg);

    /* send the command code */
    if (reg->debug)
	iio_log("%02x/%-6s ", command, iio_lm628_name[command]);
    *reg->csr = command;

    /* process the zero-terminated user list */
    if (command & 0xf0) {
	/*
	 * WRITE type commands
	 */
	if (reg->debug)
	    iio_log("W ");

	while ((tag = va_arg(ap, IIO_ARG))) {

	    /*
	     * Always read the user data as an int; no matter what the
	     * user passes (or casts to) it will be an int
	     */
	    u32 = (uint32_t)(va_arg(ap, int));

	    /* wait for busy bit to clear */
	    iio_lm628_poll(reg);

	    switch (tag) {
	    case iio_arg_int32:
	    case iio_arg_uint32:

		/* write the two high uint8_t's of the uint32_t */
		*reg->dr = (uint8_t)((u32 >> 24) & 0xff);
		*reg->dr = (uint8_t)((u32 >> 16) & 0xff);
		if (reg->debug)
		    iio_log("%04x ", (u32 >> 16) & 0xffff);

		/* wait for busy bit to clear */
		iio_lm628_poll(reg);

		/* NO BREAK */

	    case iio_arg_int16:
	    case iio_arg_uint16:

		/* write the two low uint8_ts of the uint32_t */
		*reg->dr = (uint8_t)((u32 >> 8) & 0xff);
		*reg->dr = (uint8_t)((u32 >> 0) & 0xff);
		if (reg->debug)
		    iio_log("%04x ", (u32 >> 0) & 0xffff);
		break;

	    default:
		return iio_fatal("Bad write argument tag");
	    }
	}
    } else {
	/*
	 * READ type commands
	 */
	if (reg->debug)
	    iio_log("R ");

	while ((tag = va_arg(ap, IIO_ARG))) {

	    /* wait for busy bit to clear */
	    iio_lm628_poll(reg);

	    switch (tag) {
	    case iio_arg_int16:
	    case iio_arg_uint16:

		/* read the two uint8_ts of the uint16_t */
		u16  = ((uint16_t)(*reg->dr) << 8);
		u16 |= ((uint16_t)(*reg->dr) << 0);
		if (reg->debug)
		    iio_log("%04x ", u16);

		/* assign to user pointer value */
		*(va_arg(ap, uint16_t*)) = u16;
		break;

	    case iio_arg_int32:
	    case iio_arg_uint32:
		/* read the four uint8_ts of the uint32_t */
		u32  = ((uint32_t)(*reg->dr) << 24);
		u32 |= ((uint32_t)(*reg->dr) << 16);
		if (reg->debug)
		    iio_log("%04x ", (u32 >> 16) & 0xffff);

		/* wait for busy bit to clear */
		iio_lm628_poll(reg);

		u32 |= ((uint32_t)(*reg->dr) << 8);
		u32 |= ((uint32_t)(*reg->dr) << 0);
		if (reg->debug)
		    iio_log("%04x ", (u32 >> 0) & 0xffff);

		/* assign to user pointer value */
		*(va_arg(ap, uint32_t*)) = u32;
		break;

	    default:
		return iio_fatal("Bad read argument tag");
	    }
	}
    }

    /* wait for busy bit to clear */
    status = iio_lm628_poll(reg);
    if (reg->debug)
	iio_log("%02x\n", status);

    /* check status byte */
    if (status & IIO_LM628_S_PERR) 
	return iio_error("LM628 Excessive position error");
    if (status & IIO_LM628_S_WRAP)
	return iio_error("LM628 Position wraparound");
    if (status & IIO_LM628_S_CERR) {
	return iio_error("LM628 Command error");
    }
    return iio_status_ok;
}

#define INLINE __inline__ static IIO_STATUS

    /*
     * The chip access functions, which call iio_lm628_command(). There are
     * many of these, but they are simple and are inlined
     */
INLINE iio_lm628_reset(IIO_LM628_MREG *reg)
    { return iio_lm628_command(reg, IIO_LM628_C_RESET, 0); }
INLINE iio_lm628_port8(IIO_LM628_MREG *reg)
    { return iio_lm628_command(reg, IIO_LM628_C_PORT8, 0); }
INLINE iio_lm628_port12(IIO_LM628_MREG *reg)
    { return iio_lm628_command(reg, IIO_LM628_C_PORT12, 0); }
INLINE iio_lm628_dfh(IIO_LM628_MREG *reg)
    { return iio_lm628_command(reg, IIO_LM628_C_DFH, 0); }
INLINE iio_lm628_sip(IIO_LM628_MREG *reg)
    { return iio_lm628_command(reg, IIO_LM628_C_SIP, 0); }
INLINE iio_lm628_stt(IIO_LM628_MREG *reg)
    { return iio_lm628_command(reg, IIO_LM628_C_STT, 0); }
INLINE iio_lm628_udf(IIO_LM628_MREG *reg)
    { return iio_lm628_command(reg, IIO_LM628_C_UDF, 0); }
INLINE iio_lm628_lpei(IIO_LM628_MREG *reg, uint16_t rng)
    { return iio_lm628_command(reg, IIO_LM628_C_LPEI, iio_arg_uint16, rng, 0); }
INLINE iio_lm628_lpes(IIO_LM628_MREG *reg, uint16_t rng)
    { return iio_lm628_command(reg, IIO_LM628_C_LPES, iio_arg_uint16, rng, 0); }
INLINE iio_lm628_sbpa(IIO_LM628_MREG *reg, int32_t pos)
    { return iio_lm628_command(reg, IIO_LM628_C_SBPA, iio_arg_uint32, pos, 0); }
INLINE iio_lm628_sbpr(IIO_LM628_MREG *reg, int32_t pos)
    { return iio_lm628_command(reg, IIO_LM628_C_SBPR, iio_arg_uint32, pos, 0); }
INLINE iio_lm628_mski(IIO_LM628_MREG *reg, uint16_t msk)
    { return iio_lm628_command(reg, IIO_LM628_C_MSKI, iio_arg_uint16, msk, 0); }
INLINE iio_lm628_rsti(IIO_LM628_MREG *reg, uint16_t msk)
    { return iio_lm628_command(reg, IIO_LM628_C_RSTI, iio_arg_uint16, msk, 0); }
INLINE iio_lm628_rdsigs(IIO_LM628_MREG *reg, uint16_t *s)
    { return iio_lm628_command(reg, IIO_LM628_C_RDSIGS, iio_arg_uint16, s, 0);}
INLINE iio_lm628_rdip(IIO_LM628_MREG *reg, int32_t *pos)
    { return iio_lm628_command(reg, IIO_LM628_C_RDIP, iio_arg_int32, pos, 0); }
INLINE iio_lm628_rddp(IIO_LM628_MREG *reg, int32_t *pos)
    { return iio_lm628_command(reg, IIO_LM628_C_RDDP, iio_arg_int32, pos, 0); }
INLINE iio_lm628_rdrp(IIO_LM628_MREG *reg, int32_t *pos)
    { return iio_lm628_command(reg, IIO_LM628_C_RDRP, iio_arg_int32, pos, 0); }
INLINE iio_lm628_rddv(IIO_LM628_MREG *reg, int32_t *vel)
    { return iio_lm628_command(reg, IIO_LM628_C_RDDV, iio_arg_int32, vel, 0); }
INLINE iio_lm628_rdrv(IIO_LM628_MREG *reg, int16_t *vel)
    { return iio_lm628_command(reg, IIO_LM628_C_RDRV, iio_arg_int16, vel, 0); }
INLINE iio_lm628_rdsum(IIO_LM628_MREG *reg, int16_t *sum)
    { return iio_lm628_command(reg, IIO_LM628_C_RDSUM, iio_arg_int16, sum, 0); }


HIDDEN __inline__ uint8_t iio_lm628_rdstat(IIO_LM628_MREG *reg) {
    /*
     * Read LM628 status (by reading the command/status register)
     */
    return *reg->csr;
}


IIO_STATUS iio_lm628_lfil(
    IIO_LM628_MREG *reg, uint8_t fcw, uint8_t ds,
    ... /* kp, ki, kd, il */
) {
    /*
     * LFIL and LTRJ are a bit more complicated than the other LM628 control
     * functions, as the parameters are optional, determined by the fcw.
     * We have to build an argument list piecewise to pass to the command
     * function in one hit
     */
    int data[11], *dp = data;
    va_list ap;
    va_start(ap, ds);

    /* build the 16-bit filter control word */
    *dp++ = iio_arg_uint16;
    *dp++ = ((uint16_t)ds << IIO_LM628_F_DROLL) | (0x0f & fcw);

    /* the four possible filter parameters */
    if (fcw & IIO_LM628_F_KP) {
	*dp++ = iio_arg_uint16;
	*dp++ = va_arg(ap, unsigned);
    }
    if (fcw & IIO_LM628_F_KI) {
	*dp++ = iio_arg_uint16;
	*dp++ = va_arg(ap, unsigned);
    }
    if (fcw & IIO_LM628_F_KD) {
	*dp++ = iio_arg_uint16;
	*dp++ = va_arg(ap, unsigned);
    }
    if (fcw & IIO_LM628_F_IL) {
	*dp++ = iio_arg_uint16;
	*dp++ = va_arg(ap, unsigned);
    }
    *dp = 0;

    /* pass list to command function */
    iio_eret(
	iio_lm628_command(
	    reg, IIO_LM628_C_LFIL,
	    data[0], data[1], data[2], data[3],
	    data[4], data[5], data[6], data[7],
	    data[8], data[9], data[10]
	)
    );
    return iio_status_ok;
}

IIO_STATUS iio_lm628_ltrj(
    IIO_LM628_MREG *reg, uint16_t tcw,
    ... /* int a, int v, int p */ 
) {
    /*
     * LFIL and LTRJ are a bit more complicated than the other LM628 control
     * functions, as the parameters are optional, determined by the tcw.
     * We have to build an argument list piecewise to pass to the command
     * function in one hit
     */
    int data[7], *dp = data;
    va_list ap;
    va_start(ap, tcw);

    /* the four possible filter parameters */
    if (tcw & IIO_LM628_T_AL) {
	*dp++ = iio_arg_uint32;
	*dp++ = va_arg(ap, uint32_t);
    }
    if (tcw & IIO_LM628_T_VL) {
	*dp++ = iio_arg_uint32;
	*dp++ = va_arg(ap, uint32_t);
    }
    if (tcw & IIO_LM628_T_PL) {
	*dp++ = iio_arg_int32;
	*dp++ = va_arg(ap, int32_t);
    }
    *dp = 0;

    /* pass list to command function */
    iio_eret(
	iio_lm628_command(
	    reg, IIO_LM628_C_LTRJ,
	    iio_arg_uint16, tcw,
	    data[0], data[1], data[2], data[3],
	    data[4], data[5], data[6]
	)
    );
    return iio_status_ok;
}

HIDDEN IIO_BOOL iio_lm682_banner = iio_bool_false;

IIO_STATUS iio_lm628_show(IIO_LM628_MREG *reg) {
    /*
     * Print out a line indicating (most) of the status of the given chip.
     * If reg is null, print a header line only
     */
    int32_t position = 0, index = 0, desired = 0, error = 0, veldesired = 0;
    int16_t velocity = 0;
    uint16_t sigs, sigbit;
    static char *sigbitchar = "haufvot8mbewitca";
    char sigsstring[17];

    /* get chip info */
    iio_lm628_rdsigs(reg, &sigs);
    iio_lm628_rddp(reg, &desired);
    iio_lm628_rdip(reg, &index);
    iio_lm628_rdrp(reg, &position);
    iio_lm628_rddv(reg, &veldesired);
    iio_lm628_rdrv(reg, &velocity);
    error = desired - position;

    /* work out signals string */
    for (sigbit = 0; sigbit < 16; ++sigbit)
	sigsstring[sigbit] =
	    (sigs & (1 << (15 - sigbit)))
		? sigbitchar[sigbit]
		: '-';
    sigsstring[16] = '\0';

    /* print header */
    if (! iio_lm682_banner) {
	iio_log("\n%8s  %16s %8s %8s %8s %8s %8s %4s\n",
	    "csr",
	    "sigs",
	    "index",
	    "desired",
	    "actual",
	    "error",
	    "vdes",
	    "vel"
	);
	iio_lm682_banner = iio_bool_true;
    }

    /* print chip info to stdout */
    iio_log("%08x  %16s %08x %08x %08x %08x %08x %04x\n",
	(unsigned)reg->csr,
	sigsstring,
	(unsigned)index,
	(unsigned)desired,
	(unsigned)position,
	(unsigned)error,
	(unsigned)veldesired,
	(unsigned)velocity & 0xffff
    );

    return iio_status_ok;
}

static __inline__ int32_t clip(
    int32_t value, int32_t lower, int32_t upper
) {

    if (value > upper) return upper;
    if (value < lower) return lower;
    return value;
}


/*
 * ----------------------------------------------------------- IIO DRIVER LEVEL
 */

HIDDEN IIO_STATUS iio_lm628_move(
    IIO_LM628_MSTATE *state, IIO_LM628_MREG *reg, uint16_t tcw
) {
    /*
     * Update LM628 and start/stop motion. This is called by iio_lm628_sc()
     * for the sc-start/stop/free operations. First, if the filter coeffs are
     * updated, then the trajectory parameters (if accelleration is changed,
     * the MOFF-nonsense is done). An LTRJ using the given tcw is done, if
     * it isn't zero (for sc-stop and sc-free), then an STT
     */

    /* promulgate new filter settings, if changed */
    if (state->udf) {
	iio_eret( iio_lm628_udf(reg) );
	state->udf = iio_bool_false;
    }

    /*
     * Promulgate new accelleration, if changed. The LM628 datasheet gives
     * somewhat conflicting information about accelleration changing; it says
     * you have to do a MOFF at some state, but it isn't clear if this
     * has to be done before or after
     */
    if (state->accf) {
	iio_eret( iio_lm628_ltrj(reg, IIO_LM628_T_MOFF) );
	iio_eret( iio_lm628_stt(reg) );
#ifdef LATEACC
	iio_eret( iio_lm628_ltrj(reg, IIO_LM628_T_AL, state->acc) );
	iio_eret( iio_lm628_stt(reg) );
#endif
	state->accf = iio_bool_false;
    }

    /* do the special LTRJ if required */
    tcw &= (IIO_LM628_T_SSTOP | IIO_LM628_T_STOP | IIO_LM628_T_MOFF);
    iio_eret( iio_lm628_ltrj(reg, tcw) );

    /* now start the actual motion */
    iio_eret( iio_lm628_stt(reg) );
    return iio_status_ok;
}


IIO_STATUS iio_lm628_sc(
    IIO_LM628_MSTATE *state, IIO_LM628_MREG *reg,
    IIO_OPNODE *opnode, IIO_OP op, unsigned seqno
) {
    /*
     * Operate function. This can only operate on one chip/channel at a time,
     * effectively pre-selected by the register and state structures; but
     * we still need opnode and seqno to access the user data
     */
    int32_t value;

    switch (op) {
    case iio_sc_start:
	iio_eret( iio_lm628_sip(reg) );
	iio_eret( iio_lm628_move(state, reg, 0) );
	break;

    case iio_sc_stop:
    case iio_op_clear:
	iio_eret( iio_lm628_move(state, reg, IIO_LM628_T_STOP) );
	break;

    case iio_sc_free:
	/* disable servo */
	iio_eret( iio_lm628_move(state, reg, IIO_LM628_T_MOFF) );
	break;


    case iio_sc_read_current:
    case iio_op_read:
	/* read current servo value */
	iio_eret( iio_lm628_rdrp(reg, &value) );
	iio_data_set(opnode, seqno, value + state->offset);
	break;

    case iio_sc_read_target:
    case iio_op_readback:
	/* read target servo value */
	iio_eret( iio_lm628_rddp(reg, &value) );
	iio_data_set(opnode, seqno, value + state->offset);
	break;

    case iio_sc_read_index:
	/* read last index value */
	iio_eret( iio_lm628_rdip(reg, &value) );
	iio_data_set(opnode, seqno, value + state->offset);
	break;


    case iio_sc_write_current:
        /* write (a.k.a. calibrate) current position */
	value = iio_data_get(opnode, seqno);
	iio_eret( iio_lm628_dfh(reg) );
	state->offset = value;
	break;

    case iio_sc_write_target:
    case iio_op_write:
        /* write target position */
	value = clip(
	    iio_data_get(opnode, seqno) - state->offset, 
	    0xc0000000, 0x3fffffff
	);
	iio_eret( iio_lm628_ltrj(reg, IIO_LM628_T_PL, value) );
	break;

    case iio_sc_write_target_dt:
        /* write trapezoidal dt (e.g. velocity) */
	value = clip(
	    iio_data_get_real(opnode, seqno) * reg->dtfactor, 
	    0, 0x3fffffff
	);
	iio_eret( iio_lm628_ltrj(reg, IIO_LM628_T_VL, value) );
	break;

    case iio_sc_write_target_ddt:
        /*
	 * write trapezoidal ddt (e.g. acceleration), by computing it and
	 * putting it into the state structure; it gets loaded into the
	 * chip when the sc-start operation is done
	 */
	value = clip(
	    iio_data_get_real(opnode, seqno) * reg->ddtfactor, 
	    0, 0x3fffffff
	);
#ifndef LATEACC
	iio_eret( iio_lm628_ltrj(reg, IIO_LM628_T_AL, value) );
#endif
	if (value != state->acc) {
	    state->acc = value;
	    state->accf = iio_bool_true;
	}
	break;


    case iio_sc_read_gain_p:
        /* read loop proportional gain (from shadows) */
	iio_data_set_real(opnode, seqno, state->kp/reg->pgfactor);
	break;

    case iio_sc_read_gain_d:
        /* read loop derivative gain (from shadows) */
	iio_data_set_real(opnode, seqno, state->kd/reg->dgfactor);
	break;

    case iio_sc_read_gain_i:
        /* read loop integral gain (from shadows) */
	iio_data_set_real(opnode, seqno, state->ki/reg->igfactor);
	break;


    case iio_sc_write_gain_p:
        /* write loop proportional gain */
	iio_eret(
	    iio_lm628_lfil(
		reg, IIO_LM628_F_KP, reg->ds,
		state->kp = clip(
		    iio_round(
			iio_data_get_real(opnode, seqno) * reg->pgfactor
		    ), 0, 0x7fff
		)
	    )
	);
	state->udf = iio_bool_true;
	break;

    case iio_sc_write_gain_d:
        /* write loop derivative gain */
	iio_eret(
	    iio_lm628_lfil(
		reg, IIO_LM628_F_KD, reg->ds,
		state->kd = clip(
		    iio_round(
			iio_data_get_real(opnode, seqno) * reg->dgfactor
		    ), 0, 0x7fff
		)
	    )
	);
	state->udf = iio_bool_true;
	break;

    case iio_sc_write_gain_i:
        /* write loop integral gain */
	iio_eret(
	    iio_lm628_lfil(
		reg, IIO_LM628_F_KI, reg->ds,
		state->ki = clip(
		    iio_round(
			iio_data_get_real(opnode, seqno) * reg->igfactor
		    ), 0, 0x7fff
		)
	    )
	);
	state->udf = iio_bool_true;
	break;

    case iio_op_show:
	iio_eret( iio_lm628_show(reg) );
	break;

    default:
	return iio_error("Operation code not supported by channel");
    }

    /* cancel the show banner */
    if (op != iio_op_show)
	iio_lm682_banner = iio_bool_false;

    return iio_status_ok;
}


IIO_STATUS iio_lm628_init(IIO_LM628_MSTATE *state, IIO_LM628_MREG *reg) {
    /*
     * Initialise the LM628 chip, based on the information in the register
     * structure, which comes from the config file arguments
     */

    /* reset the chip twice, wait 2ms, set output port width */
/*
 * Modified by jmr under rjk's instruction. Removed the eret( ) around
 * the first iio_lm628_reset call
 */
/* 	
    iio_eret( iio_lm628_reset(reg) );
*/
    iio_lm628_reset(reg);
    iio_roughdelay(2);
    iio_eret( iio_lm628_reset(reg) );
    iio_roughdelay(2);
    if (reg->bit12)
	iio_eret( iio_lm628_port12(reg) );

    /* reset and mask all interrupt flags */
    iio_eret( iio_lm628_mski(reg, 0x00) );
    iio_eret( iio_lm628_rsti(reg, 0x00) );

    /* set a huge breakpoint */
    iio_eret( iio_lm628_sbpa(reg, 0x3fffffff) );

    /* load the filter coefficients from the initial (config file) values */
    iio_eret(
	iio_lm628_lfil(
	    reg,
	    IIO_LM628_F_KP | IIO_LM628_F_KI | IIO_LM628_F_KD | IIO_LM628_F_IL,
	    reg->ds,
	    (state->kp = clip(iio_round(reg->ipgain*reg->pgfactor), 0, 0x7fff)),
	    (state->ki = clip(iio_round(reg->iigain*reg->igfactor), 0, 0x7fff)),
	    (state->kd = clip(iio_round(reg->idgain*reg->dgfactor), 0, 0x7fff)),
	    (state->il = clip(iio_round(reg->iimax *reg->ilfactor), 0, 0x7fff))
	)
    );

    /* promulgate initial filter coefficients */ 
    iio_eret( iio_lm628_udf(reg) );

    /* clear the new accelleration flags */
    state->accf = iio_bool_false;
    state->acc = (unsigned)-1;

    /* close the position loop */
    iio_eret( iio_lm628_stt(reg) );

    return iio_status_ok;
}


IIO_STATUS iio_lm628_install(
    IIO_LM628_MREG *reg, volatile uint8_t *csr, volatile uint8_t *dr,
    uint32_t clock, IIO_BOOL bit12, double orange, char *argv[],
    IIO_CHNODE *chnode, unsigned seqno
) {
    /*
     * Install the chip driver. This function has to (a) fill out the
     * register structure, (b) parse the arguments for this channel only,
     * and (c) set the correct channel information
     */
    double period = (double)IIO_LM628_PERIOD/(double)clock;
    double dperiod = period;
    double ofactor = orange/(double)(2 << (bit12 ? 12 : 8));
    double counts = 1.0;
    IIO_BOOL fake = iio_bool_false;
    char *unit = "";
    int ds;

    /* the register structure is pre-allocated */
    reg->csr = csr;
    reg->dr = dr;
    reg->bit12 = bit12;

    /* get the the debugging flags */
    reg->debug = 0;
    iio_eret(
	iio_arg_index_list(
	    argv, seqno,
	    "fake", iio_arg_bool, &fake,
	    "debug", iio_arg_bool, &reg->debug,
	    0
	)
    );

    /* the fake flags points the registers at a safe variable */ 
    if (fake) {
	static uint8_t safe;
	reg->csr = reg->dr = &safe;
    }

    /*
     * get the -counts argument next. -counts is in effect the
     * reciprocal of the channel gain, (and we set it so here), and so it
     * could be specified by a channel directive later in the config file.
     * But if we want to specify the servo feedback values in terms of real
     * units (e.g. radians or revs) we need to know the scale factor here.
     *
     * The number of encoder counts per radian/rev is a more natural way of
     * specifying it. Note that the counts per user unit will be four times
     * the encoder lines per unit.
     *
     * Maybe we should compare counts to a "small" value, not zero,
     * but what "small" value?
     */
    iio_eret( iio_arg_index(argv, "counts", seqno, iio_arg_double, &counts) );
    iio_eret( iio_arg_index(argv, "unit", seqno, iio_arg_string, &unit) );
    if (counts == 0.0) 
	return iio_error("Can't have zero counts factor");
    if (counts < 0.0) 
	return iio_error("Can't have negative counts factor");
    iio_eret( iio_chnode_linear(chnode, seqno, 1.0/counts, 0.0, unit) );

    /*
     * get the remaining arguments, which are the servo loop feedback
     * constants, derivative period multiplier, and integral limit.
     * We don't do much with the initial values, except store them in the 
     * register structure; they are used only once in the init function.
     * Note the default integral limit is the *maximum*, not minimum
     */
    reg->ipgain = 0.0;
    reg->idgain = 0.0;
    reg->iigain = 0.0;
    reg->iimax = (double)0x7fff * 1.0/counts * period/(bit12 ? 16.0 : 256.0);

    /* get arguments */
    iio_eret(
	iio_arg_index_list(
	    argv, seqno,
	    "pgain", iio_arg_double, &reg->ipgain,
	    "dgain", iio_arg_double, &reg->idgain,
	    "igain", iio_arg_double, &reg->iigain,
	    "imax", iio_arg_double, &reg->iimax,
	    "dperiod", iio_arg_double, &dperiod,
	    0
	)
    );

    /* pre-scale initial arguments with counts factor */
    reg->ipgain *= counts;
    reg->idgain *= counts;
    reg->iigain *= counts;
    reg->iimax  *= counts;

    /* compute the derivative sample factor, correct dperiod */
    ds = iio_round(dperiod/period) - 1;
    if ((ds < 0) || (ds > 0xff))
	return iio_error("-dperiod out of range");
    dperiod = period * ((reg->ds = ds) + 1); 

    /*
     * Compute the factors used to compute the LM628 chip register values
     * from the "real unit" ones passed to us from the config file, or
     * from operations that affect these quantities
     *
     *     <reg> = <user value> * factor
     *
     * Note that the values to be scaled will have already been scaled for
     * user units/encoder count (counts), so we don't include that here
     */

    /* compensation filter (see LM628 User Guide, pp 19-20) */
    if (bit12) {
	reg->pgfactor = 16.0/ofactor;
	reg->dgfactor = 16.0/ofactor/dperiod;
	reg->igfactor = 4096.0*period/ofactor;
	reg->ilfactor = 16.0/period;
    } else {
	reg->pgfactor = 256.0/ofactor;
	reg->dgfactor = 256.0/ofactor/dperiod;
	reg->igfactor = 65536.0*period/ofactor;
	reg->ilfactor = 265.0/period;
    }

    /* trajectory generator */
    reg->dtfactor  = 0x10000 * period;
    reg->ddtfactor = 0x10000 * period * period;

    /* check feedback constant ranges here */
    if (
	(iio_round(reg->ipgain * reg->pgfactor) > 0x7fff) ||
	(reg->ipgain < 0.0)
    )
	return iio_error("-pgain factor out of range");
    if (
	(iio_round(reg->idgain * reg->dgfactor) > 0x7fff) ||
	(reg->idgain < 0.0)
    )
	return iio_error("-dgain factor out of range");
    if (
	(iio_round(reg->iigain * reg->igfactor) > 0x7fff) ||
	(reg->iigain < 0.0)
    )
	return iio_error("-igain factor out of range");
    if (
	(iio_round(reg->iimax * reg->ilfactor) > 0x7fff) ||
	(reg->iimax < 0.0)
    )
	return iio_error("-imax out of range");

    return iio_status_ok;
}

/* end */
