/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: ds1620.c 365 2003-05-12 02:31:19Z sik057 $
 *
 * ds1620.c -- driver for the DALLAS DS1620 Digital Thermometer/Thermostat chip
 *
 * The DS1620 is an 8-pin chip that provides 9-bit temperature readings, as 
 * well as three digital outputs which  may be used to control a heating/
 * cooling device. The outputs are controlled by the result of the comparison
 * between the ambient temperature and two pre-settable, non-volatile high/low
 * limits.
 *
 * The chip is programmed and read through a 3-wire serial interface.
 * Because the details of accessing that interface can't be built into this
 * driver, a user-supplied access function (conforming to type IIO_DS1620_ACFN)
 * is called back to actually send or receive each bit. The function is passed
 * the user parameter directly, plus an operation code (IIO_DS1620_OP) and a
 * data pointer, of which only the LSB is valid. The function should return
 * iio_status_ok unless there is an error.
 */
#include "../internal.h"
#include "ds1620.h"


HIDDEN IIO_STATUS iio_ds1620_read(
    IIO_DS1620_ACFN fn, void *user, int number, int *data, IIO_BOOL sex
) {
    /*
     * Clock in <number> bits from the DS1620 using the user-supplied 
     * access function <fn>, and return the result. The bits come LSB-first
     */
    int result = 0, bit, dbit;

    if (! data)
	return iio_error("NULL data return pointer");

    /* read bits one by one and assemble result */
    for (bit = 0; bit < number; ++bit) {
	iio_eret( fn(user, iio_ds1620_getbit, &dbit) );
	result |= ((dbit & 1) << bit);
    }

    /* sign-extend the result if the top bit is set */
    if (sex && (dbit & 1))
	*data = (-1) & result;
    else
	*data = result;
    return iio_status_ok;
}

HIDDEN IIO_STATUS iio_ds1620_write(
    IIO_DS1620_ACFN fn, void *user, int number, int data
) {
    /*
     * Clock out <number> bits of the given <data> out to the DS1620,
     * using the user access function <fn>
     */
    for (; number > 0; --number) {
	iio_eret( fn(user, iio_ds1620_putbit, &data) );
	data = data >> 1;
    }
    return iio_status_ok;
}

HIDDEN __inline__ IIO_STATUS iio_ds1620_begin(IIO_DS1620_ACFN fn, void *user) {
    /*
     * Calls the user access function to set the /reset bit high so that
     * communications on the three-wire interface can occur. This function
     * is called at the begining of each transaction
     */
    if (! fn)
	return iio_error("NULL DS1620 access function");

    iio_eret( fn(user, iio_ds1620_setrst, NULL) );
    return iio_status_ok;
}

HIDDEN __inline__ IIO_STATUS iio_ds1620_end(IIO_DS1620_ACFN fn, void *user) {
    /*
     * Calls the user access function to set the /reset bit low to end
     * communications on the three-wire interface. This function
     * is called at the end of each transaction
     */
    iio_eret( fn(user, iio_ds1620_clrrst, NULL) );
    return iio_status_ok;
}


IIO_STATUS iio_ds1620_conv_start(IIO_DS1620_ACFN fn, void *user) {
    /*
     * Start a temperature conversion (in one-shot mode)
     */
    iio_eret( iio_ds1620_begin(fn, user) );
    iio_eret( iio_ds1620_write(fn, user, 8, IIO_DS1620_CONV_START) );
    iio_eret( iio_ds1620_end(fn, user) );
    return iio_status_ok;
}

IIO_STATUS iio_ds1620_conv_stop(IIO_DS1620_ACFN fn, void *user) {
    /*
     * Start a temperature conversion (in one-shot mode)
     */
    iio_eret( iio_ds1620_begin(fn, user) );
    iio_eret( iio_ds1620_write(fn, user, 8, IIO_DS1620_CONV_STOP) );
    iio_eret( iio_ds1620_end(fn, user) );
    return iio_status_ok;
}

IIO_STATUS iio_ds1620_temp_read(IIO_DS1620_ACFN fn, void *user, int *temp) {
    /*
     * Return the most recent temperature reading (0.5 C/bit)
     */
    iio_eret( iio_ds1620_begin(fn, user) );
    iio_eret( iio_ds1620_write(fn, user, 8, IIO_DS1620_TEMP_READ) );
    iio_eret( iio_ds1620_read(fn, user, 9, temp, iio_bool_true) );
    iio_eret( iio_ds1620_end(fn, user) );
    return iio_status_ok;
}

IIO_STATUS iio_ds1620_th_read(IIO_DS1620_ACFN fn, void *user, int *temp) {
    /*
     * Return the current high temperature limit value
     * (0.5 C/bit)
     */
    iio_eret( iio_ds1620_begin(fn, user) );
    iio_eret( iio_ds1620_write(fn, user, 8, IIO_DS1620_TH_READ) );
    iio_eret( iio_ds1620_read(fn, user, 9, temp, iio_bool_true) );
    iio_eret( iio_ds1620_end(fn, user) );
    return iio_status_ok;
}

IIO_STATUS iio_ds1620_th_write(IIO_DS1620_ACFN fn, void *user, int temp) {
    /*
     * Set the high temperature limit value
     * (0.5 C/bit)
     */
    iio_eret( iio_ds1620_begin(fn, user) );
    iio_eret( iio_ds1620_write(fn, user, 8, IIO_DS1620_TH_WRITE) );
    iio_eret( iio_ds1620_write(fn, user, 9, temp) );
    iio_eret( iio_ds1620_end(fn, user) );
    return iio_status_ok;
}

IIO_STATUS iio_ds1620_tl_read(IIO_DS1620_ACFN fn, void *user, int *temp) {
    /*
     * Return the current low temperature limit value
     * (0.5 C/bit)
     */
    iio_eret( iio_ds1620_begin(fn, user) );
    iio_eret( iio_ds1620_write(fn, user, 8, IIO_DS1620_TL_READ) );
    iio_eret( iio_ds1620_read(fn, user, 9, temp, iio_bool_true) );
    iio_eret( iio_ds1620_end(fn, user) );
    return iio_status_ok;
}

IIO_STATUS iio_ds1620_tl_write(IIO_DS1620_ACFN fn, void *user, int temp) {
    /*
     * Set the low temperature limit value
     * (0.5 C/bit)
     */
    iio_eret( iio_ds1620_begin(fn, user) );
    iio_eret( iio_ds1620_write(fn, user, 8, IIO_DS1620_TL_WRITE) );
    iio_eret( iio_ds1620_write(fn, user, 9, temp) );
    iio_eret( iio_ds1620_end(fn, user) );
    return iio_status_ok;
}

IIO_STATUS iio_ds1620_cr_read(IIO_DS1620_ACFN fn, void *user, int *cr) {
    /*
     * Return the current status register
     */
    iio_eret( iio_ds1620_begin(fn, user) );
    iio_eret( iio_ds1620_write(fn, user, 8, IIO_DS1620_CR_READ) );
    iio_eret( iio_ds1620_read(fn, user, 8, cr, iio_bool_false) );
    iio_eret( iio_ds1620_end(fn, user) );
    return iio_status_ok;
}

IIO_STATUS iio_ds1620_cr_write(IIO_DS1620_ACFN fn, void *user, int cr) {
    /*
     * Write to the status register
     */
    iio_eret( iio_ds1620_begin(fn, user) );
    iio_eret( iio_ds1620_write(fn, user, 8, IIO_DS1620_CR_WRITE)  );
    iio_eret( iio_ds1620_write(fn, user, 8, cr) );
    iio_eret( iio_ds1620_end(fn, user) );
    return iio_status_ok;
}

IIO_STATUS iio_ds1620_show(IIO_DS1620_ACFN fn, void *user) {
    /*
     * Print out DS1620 registers
     */
    int temp, th, tl, cr;

    if (! fn) {
	iio_log(
	    "    %10s  %10s  %4s  %11s  %11s  %11s\n",
	    "ACCESS FN", 
	    "USER DP",
	    "CSR",
	    "TEMP",
	    "THIGH",
	    "TLOW"
	);
	return iio_status_ok;
    }

    iio_eret( iio_ds1620_cr_read(fn, user, &cr) );
    iio_eret( iio_ds1620_temp_read(fn, user, &temp) );
    iio_eret( iio_ds1620_th_read(fn, user, &th) );
    iio_eret( iio_ds1620_tl_read(fn, user, &tl) );

    iio_log(
	"    0x%08x  0x%08x  0x%02x  "
	"0x%03x = %-3d  0x%03x = %-3d  0x%03x = %-3d\n",
	(unsigned)fn, 
	(unsigned)user,
	cr,
	temp, temp,
	th, th,
	tl, tl
    );
    return iio_status_ok;
}
