/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: channel.c 365 2003-05-12 02:31:19Z sik057 $
 *
 * channel.c -- configuration file "channel" directive
 * Robin Kirkham, November 1996
 */
#include "internal.h"


IIO_STATUS iio_channel(char *argv[]) {
    /*
     * The channel directive is used to set some of the miscellaneous 
     * values associated with the channel node/channel info structures.
     * We know argv[0] is the line reference, argv[1] is empty, argv[2]
     * is the name of the channel, and options in the standard manner 
     * follow
     */
    char *alias = NULL, *unit = NULL, *name = argv[2];
    double scale = 1.0;
    double offset = 0.0;
    IIO_BOOL log = iio_bool_false;
    IIO_STATUS stat;
    IIO channel = NULL;
    IIO_OPNODE *opnode;
    unsigned seqno;

    /* first, open the channel */
    iio_fret( stat = iio_open(argv[2], iio_iflag_none, &channel) );
    if (stat) {
	iio_log("%s: %s: %s\n", argv[0], argv[2], iio_emessage_get());
	return stat;
    }
    argv[2] = "";
    
    /* parse the arguments */
    iio_eret(
	iio_arg_list(
	    argv,
	    "alias", iio_arg_string, &alias,
	    "scale", iio_arg_double, &scale,
	    "offset", iio_arg_double, &offset,
	    "unit", iio_arg_string, &unit,
	    "log", iio_arg_bool, &log,
	    0
	)
    );

    /* register any aliases */
    if (alias) {
	iio_eret( iio_alias_insert(iio_atype_global, alias, name) );
    }

    /* for each opnode/chnode/chinfo that has been opened */
    for (opnode = channel->opnode; opnode; opnode = opnode->next) 
	for (
	    seqno = opnode->first;
	    seqno < opnode->first + opnode->number;
	    ++seqno
	) {
	    /* multiply the scale/offset/unit values */
	    if (scale == 0.0) {
		iio_log(
		    "%s: %s: Can't have zero scale factor\n", argv[0], name
		);
		return iio_error("Can't have zero scale factor");
	    }
	    opnode->chnode->chinfo[seqno].scale *= scale;
	    opnode->chnode->chinfo[seqno].offset =
		scale * opnode->chnode->chinfo[seqno].offset + offset;

	    /* change the units */
	    if (unit)
		opnode->chnode->chinfo[seqno].unit = unit;

	    /* change the logging flag */
	    opnode->chnode->chinfo[seqno].log = log;


	    /* leave the limits/initial values until later... */
	}

    /* close the channel */
    iio_close(channel);

    /* check for an unparsed arguments */
    iio_arg_remnants(argv);

    return iio_status_ok;
}
