/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: arg.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * arg.c -- option argument parser
 * Robin Kirkham, June 1996
 */

#include <stdint.h>

#include "internal.h"

    /* macro to get a pointer from stdargs, and assign val to it */
#define TSET(ap, typ, value) { \
        typ *p; \
        p = va_arg(ap, typ*); \
        *p = (value); \
    }

    /* must match IIO_ARG */
HIDDEN char *iio_arg_tstring[] = {
    "boolean",
    "8-bit signed",
    "16-bit signed",
    "32-bit signed",
    "64-bit signed",
    "8-bit unsigned ",
    "16-bit unsigned",
    "32-bit unsigned",
    "64-bit unsigned",
    "float",
    "double",
    "address",
    "channel",
    "file",
    NULL
};

    /* a blank, for blank arguments */
char *iio_arg_blank = "";


HIDDEN IIO_STATUS iio_arg_decode(
    char *argv[], int argk, char *option, IIO_ARG type, va_list ap)
{
    /*
     * argv[argk] should point to a string, which is decoded according to type.
     * If read sucessfully, the result is written to the result union;
     * otherwise an is returned
     */
    long a_long;
    double a_double;
    IIO_BOOL overrange = iio_bool_false;
    IIO_STATUS stat;

    switch (type) {

    case iio_arg_bool:
	/* this is actually not used, but never mind */
	TSET(
	    ap, IIO_BOOL, ( 
		(iio_string_cmp(argv[argk], "yes") == 0) ||
		(iio_string_cmp(argv[argk], "YES") == 0) ||
		(iio_string_cmp(argv[argk], "on") == 0) ||
		(iio_string_cmp(argv[argk], "ON") == 0) ||
		(iio_string_cmp(argv[argk], "true") == 0) ||
		(iio_string_cmp(argv[argk], "TRUE") == 0) ||
		(iio_string_cmp(argv[argk], "1") == 0)
	    ) ? iio_bool_true : iio_bool_false
	);
	return iio_status_ok;

    case iio_arg_int8:
	if (iio_decode_long(argv[argk], &a_long))
	    break;
	if (a_long > 127 || a_long < -128) {
	    overrange = iio_bool_true;
	    break;
	}
	TSET(ap, int8_t, (int8_t)a_long);
	return iio_status_ok;

    case iio_arg_int16:
	if (iio_decode_long(argv[argk], &a_long))
	    break;
	if (a_long > 32767 || a_long < -32768) {
	    overrange = iio_bool_true;
	    break;
	}
	TSET(ap, int16_t, (int16_t)a_long);
	return iio_status_ok;

    case iio_arg_int32:
	if (iio_decode_long(argv[argk], &a_long))
	    break;
	TSET(ap, int32_t, (int32_t)a_long);
	return iio_status_ok;

    case iio_arg_int64:
	return iio_fatal("64-bit object not yet supported");

    case iio_arg_uint8:
	if (iio_decode_long(argv[argk], &a_long))
	    break;
	if ((unsigned)a_long > 255) {
	    overrange = iio_bool_true;
	    break;
	}
	TSET(ap, uint8_t, (uint8_t)a_long);
	return iio_status_ok;

    case iio_arg_uint16:
	if (iio_decode_long(argv[argk], &a_long))
	    break;
	if ((unsigned)a_long > 65535) {
	    overrange = iio_bool_true;
	    break;
	}
	TSET(ap, uint16_t, (uint16_t)a_long);
	return iio_status_ok;

    case iio_arg_uint32:
	if (iio_decode_long(argv[argk], &a_long))
	    break;
	TSET(ap, uint32_t, (uint32_t)a_long);
	return iio_status_ok;

    case iio_arg_uint64:
	return iio_fatal("64-bit object not yet supported");

    case iio_arg_float:
	if (iio_decode_double(argv[argk], &a_double))
	    break;
	TSET(ap, float, (float)a_double);
	return iio_status_ok;

    case iio_arg_double:
	if (iio_decode_double(argv[argk], &a_double))
	    break;
	TSET(ap, double, (double)a_double);
	return iio_status_ok;

    case iio_arg_addr:
	if (iio_decode_long(argv[argk], &a_long))
	    break;
	TSET(ap, void *, (void *)a_long);
	return iio_status_ok;

    case iio_arg_string:
	{
	    /* string -- get a duplicate and return pointer */
	    char *string;
	    iio_eret( iio_string_dup(argv[argk], &string) );
	    TSET(ap, char *, string);
	    return iio_status_ok;
	}

    case iio_arg_channel:
	{
	    /* channel -- open the channel and return descriptor */
	    IIO_OPEN *cd;
	    if ((stat = iio_open(argv[argk], iio_oflag_none, &cd))) {
		iio_log(
		    "%s: Argument to `-%s' could not be opened (%s)\n",
		    argv[0],
		    option, 
		    argv[argk]
		);
		return stat;
	    }
	    TSET(ap, IIO_OPEN *, cd);
	    return iio_status_ok;
	}

    case iio_arg_file:
	{
	    /* file -- open the file read-write and return descriptor */
	    IIO_FILE fd;
	    if ((stat = iio_file_open(argv[argk], iio_fattr_rdwr, &fd))) {
		iio_log(
		    "%s: Argument to `-%s' could not be opened (%s)\n",
		    argv[0],
		    option, 
		    argv[argk]
		);
		return stat;
	    }
	    TSET(ap, IIO_FILE, fd);
	    return iio_status_ok;
	}

    default:
        return iio_fatal("Can't happen"); 
    }

    if (overrange) 
	iio_log(
	    "%s: Argument to `-%s' is out of range (%s)\n",
	    argv[0],
	    option, 
	    iio_arg_tstring[type]
	);
    else
	iio_log(
	    "%s: Argument to `-%s' couldn't be decoded (%s)\n",
	    argv[0],
	    option, 
	    iio_arg_tstring[type]
	);
    return iio_error("Argument could not be decoded");
}


IIO_STATUS iio_arg(char *argv[], char *option, IIO_ARG type, ...) {
    /*
     * Search the NULL-terminated argument list argv[] for an -option of the
     * given type, and if found, write the value to the result pointer
     * (argument after type). Blank out the option and argument from the
     * argument list, so after all options have been searched for we can see
     * the unused options.
     *
     *		-option <value>		for most types
     *		-option			for booleans (value = true)
     *		-no-option		for booleans (value = false)
     *
     * Note that argv[0] is line reference, and is ignored.
     * Returns iio_status_ok, or error status an option value is missing.
     */
    int argk;
    va_list ap;
    va_start(ap, type);


    /* search the list. The last arg is always NULL (as opposed to blank) */
    for (argk = 1 ; argv[argk]; ++argk) 

	/* check for an -option */
	if (*argv[argk] == '-') {
	    
	    /* check argument matches "-option" */
	    if (iio_string_cmp((argv[argk] + 1), option) == 0) {
		
		/* blank it out, so we know it's been "seen" */
		argv[argk] = iio_arg_blank;

		/* for boolean arguments, that's all we need to do */
		if (type == iio_arg_bool) {
		    TSET(ap, IIO_BOOL, iio_bool_true);
		    return iio_status_ok;
		}

		/* otherwise, get value from next argument */
		++argk;

		/* make sure the argument is actually there */
		if (!argv[argk] || !*argv[argk]) {
		    iio_log(
			"%s: Argument to `-%s' is missing (%s)\n",
			argv[0],
			option, 
			iio_arg_tstring[type]
		    );
		    return iio_status_ok;
		}

		/* most types are handled by the decoder */
		iio_arg_decode(argv, argk, option, type, ap);

		/* blank out the used argument and return */
		argv[argk] = iio_arg_blank;
		return iio_status_ok;
	    }

	    /* check if boolean option matches "-no-option" */
	    if (
		(type == iio_arg_bool) &&
		(argv[argk][1] == 'n') &&
		(argv[argk][2] == 'o') &&
		(argv[argk][3] == '-') &&
	        (iio_string_cmp((argv[argk] + 4), option) == 0)
	    ) {
		argv[argk] = iio_arg_blank;
		TSET(ap, IIO_BOOL, iio_bool_false);
		return iio_status_ok;
	    }
	}

    return iio_status_ok;
}


IIO_STATUS iio_arg_index(
    char *argv[], char *option, unsigned index, IIO_ARG type, ...
) {
    /*
     * Same as iio_arg(), but looks for options of the form name.<index>
     * where index is an unsigned decimal number. Useful for indexing options
     * specific to several channels
     */
    char name[100];
    va_list ap;
    va_start(ap, type);

    /* make an option name */
    iio_slog(name, "%.60s.%d", option, index);

    /* call iio_arg() with new name but original pointer argument */
    return iio_arg(argv, name, type, va_arg(ap, void*));
}


IIO_STATUS iio_arg_list(char *argv[], ...) {
    /*
     * Convenience form of iio_arg(), where a large number of arguments
     * have to be found in the one argument list. Following the
     * argv argument is a list of option, type, *addr triplets, which are
     * passed to iio_arg(). The list is terminated by a NULL option
     */
    va_list ap;
    va_start(ap, argv);

    while (1) {
	char *option = va_arg(ap, char *);
	IIO_ARG type;
	void *point; 

	/* list is terminated by a NULL or empty option */
	if (! option || !*option)
	    break;

	/* get the argument type, check for sanity */
	type = va_arg(ap, IIO_ARG);
	if (((int)type < 0) || ((int)type >= IIO_NARG))
	    return iio_fatal("Bad argument type specifier");

	/* get the return pointer value */
	point = va_arg(ap, void *);

	/* call iio_arg() */
	iio_eret( iio_arg(argv, option, type, point) );
    }
    return iio_status_ok;
}


IIO_STATUS iio_arg_index_list(char *argv[], unsigned index, ...) {
    /*
     * Convenience form of iio_arg_index(), where a large number of arguments
     * have to be found in the one argument list. Following the argv and index
     * arguments is a list of option, type, *addr triplets, which are
     * passed to iio_arg_index(). The list is terminated by a NULL option
     */
    va_list ap;
    va_start(ap, index);

    while (1) {
	char *option = va_arg(ap, char *);
	IIO_ARG type;
	void *point; 

	/* list is terminated by a NULL or empty option */
	if (! option || !*option)
	    break;

	/* get the argument type, check for sanity */
	type = va_arg(ap, IIO_ARG);
	if (((int)type < 0) || ((int)type >= IIO_NARG))
	    return iio_fatal("Bad argument type specifier");

	/* get the return pointer value */
	point = va_arg(ap, void *);

	/* call iio_arg_index() */
	iio_eret( iio_arg_index(argv, option, index, type, point) );
    }
    return iio_status_ok;
}


IIO_STATUS iio_arg_remnants(char *argv[]) {
    /*
     * As iio_arg() blanks out arguments in argument list that have been 
     * interpreted, the argument list should be entirely blank when argument
     * parsing is finished. If not, the option arguments were wrong. This
     * function simply prints them out, if any, with a warning message
     */
    int argk;

    for (argk = 1 ; argv[argk]; ++argk) 

	/* check for a -option */
	if (*argv[argk] == '-') {
	    /* check for a -option <arg> */
	    if (argv[argk + 1] && *argv[argk + 1] && *argv[argk + 1] != '-') {
		iio_log(
		    "%s: Option/argument `%s %s' ignored\n",
		    argv[0],
		    argv[argk],
		    argv[argk + 1]
		);
		++argk;
	    } else 
		/* just a -option  */
		iio_log(
		    "%s: Option `%s' ignored\n",
		    argv[0],
		    argv[argk]
		);
	} else
	    /* anything else */
	    if (*argv[argk])
		iio_log(
		    "%s: Remnant `%s' ignored\n",
		    argv[0],
		    argv[argk]
		);

    return iio_status_ok;
}
