/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: alias.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * alias.c -- alias line
 * Robin Kirkham, July 1996
 */

#include <stdio.h>

#include "internal.h"


    /* must match IIO_ATYPE */
HIDDEN char *iio_atype_string[] = {
    " ",
    "global",
    "local",
    "module",
    NULL
};

IIO_STATUS iio_alias(char *argv[]) {
    /*
     * This is called from the config file parser in response to "alias"
     * directives. These can have the form
     *
     *		alias <name> <value>		(global (default) alias)
     *		alias -global <name> <value>	(global (default) alias)
     *		alias -local <name> <value>	(local channel alias)
     *		alias -module <name> <value>	(module name alias)
     *
     */
    char *name, *value;
    IIO_ATYPE type;

    /* check for alias types */
    if (*argv[2] == '-') {
	if (iio_string_lookup(argv[2] + 1, iio_atype_string, (int *)&type)) {
	    iio_log(
		"%s: Bad alias type '%s'",
		argv[0],
		argv[2]
	    );
	    return iio_error("Bad alias type");
	}
	name = argv[3];
	value = argv[4];
    } else {
	type = iio_atype_global;
	name = argv[2];
	value = argv[3];
    }

    /* try and put the alias in */
    if (iio_alias_insert(type, name, value)) {
	iio_log(
	    "%s: Bad alias",
	    argv[0]
	);
	return iio_error("Alias already defined");
    }
    return iio_status_ok;
}


HIDDEN int iio_alias_cmp(IIO_SLL *s1, IIO_SLL *s2) {
    /*
     * Used by the ordered singly-linked list insert. Returns a positive,
     * zero or negative result depending on whether s1 is greater, equal
     * or less that s2. For the module list the basis is the ident code,
     * then the global generic sequence number
     */

    /* compare the alias names, return lexographic result */
    return iio_string_cmp(
        ((IIO_ALIAS *)s1)->name,
        ((IIO_ALIAS *)s2)->name
    );
}


IIO_STATUS iio_alias_insert(IIO_ATYPE type, char *name, char *value) {
    /*
     * Create a new alias and link it to the alias list. The list can
     * subsequently be searched on name/type combination. The name and
     * value strings are duplicated
     */
    IIO_ALIAS *new;

    /* check passed arguments as far as possible */
    if (!name || !*name)
        return iio_fatal("NULL or empty name string");
    if (!value || !*value)
        return iio_fatal("NULL or empty value string");
    if (type == iio_atype_none)
        return iio_fatal("Can't use iio_atype_none here");

    /* alias names may not include a . or : character */
    if (iio_string_pbrk(name, ".:"))
	return iio_error("Illegal characters in alias name");

    /* allocate a new alias block */
    iio_eret( iio_mem_alloc(sizeof(IIO_ALIAS), (void **)&new) );

    /* fill it out */
    new->type = type;
    iio_eret( iio_string_dup(name, &new->name) );
    iio_eret( iio_string_dup(value, &new->value) );

    /*
     * Link it in to the main alias list. The list is ordered by
     * aphabetic name, according to iio_alias_cmp()
     */
    if (
	iio_sll_insert(
	    (IIO_SLL **)&iio_state->alias,
	    (IIO_SLL *)new,
	    iio_alias_cmp
	)
    ) 
	return iio_error("Duplicate alias name");

    return iio_status_ok;
}


IIO_STATUS iio_alias_find(char *name, IIO_ATYPE *type, char **value) {
    /*
     * Searches for an alias of the given name; if found, returns a pointer
     * to the value, and the type of alias into type. The caller should not
     * modify the value string. If the lookup fails, the original name pointer
     * is put into value, and the type returned is iio_atype_none.
     *
     * I don't see a lot of value in making alias lookup recursive, as
     * iio_namex() (which is the main user of this function) can do two levels
     * of lookup itself (global, then module:local). We'll see how it goes
     */
    IIO_ALIAS *alias;

    if (!name || !*name)
	return iio_fatal("NULL or empty name string pointer");
    if (!type)
	return iio_fatal("NULL type return pointer");
    if (!value)
	return iio_fatal("NULL string return pointer");

    /* look up the alias in the list */
    for (alias = iio_state->alias; alias; alias = alias->next)
	if (iio_string_cmp(alias->name, name) == 0) {
	    *value = alias->value;
	    *type = alias->type;
	    return iio_status_ok;
	}

    /* not found */
    *value = name;
    *type = iio_atype_none;
    return iio_error("No such alias");
}


IIO_STATUS iio_alias_show(void) {
    /*
     * TEMPORARY FUNCTION
     * Print the channel list. If longform is true, print all the 
     * channels and their equivalents
     */
    IIO_ALIAS *alias;

    printf(
	"%-8s %30.30s   %s\n",
	"type",
	"name",
	"value"
    );

    for (alias = iio_state->alias; alias; alias = alias->next)
        printf(
	    "%-8s %30.30s = %s\n",
	    iio_atype_string[alias->type],
	    alias->name,
	    alias->value
	);

    return iio_status_ok;
}
