#ifndef _internal_h_
#define _internal_h_
/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: headers-internal.h 2222 2007-12-04 11:46:42Z roy029 $
 *
 * internal.h -- IIO internal data structures and functions
 * Robin Kirkham, June 1996
 */

#include "iio.h"
#include "types.h"
#include <stdarg.h>

/*
 * ---------------------------------------------------------- ENUMERATIVE TYPES
 */

    /* module argument parser types */
typedef enum {
    iio_arg_bool,                       /* boolean "-flag" or "-no-flag" */
    iio_arg_int8,                       /* 8-bit signed */
    iio_arg_int16,                      /* 16-bit signed */
    iio_arg_int32,                      /* 32-bit signed */
    iio_arg_int64,                      /* 64-bit signed */
    iio_arg_uint8,                      /* 8-bit unsigned */
    iio_arg_uint16,                     /* 16-bit unsigned */
    iio_arg_uint32,                     /* 32-bit unsigned */
    iio_arg_uint64,                     /* 64-bit unsigned */
    iio_arg_float,                      /* float */
    iio_arg_double,                     /* double */
    iio_arg_addr,                       /* any address */
    iio_arg_string,                     /* a dynamic string */
    iio_arg_channel,                    /* an IIO channel descriptor */
    iio_arg_file,                       /* a file-descriptor */

    IIO_NARG
} IIO_ARG;

    /* type of channel/module alias */
typedef enum {
    iio_atype_none,                     /* not an alias */
    iio_atype_global,                   /* global alias */
    iio_atype_local,                    /* local alias */
    iio_atype_module                    /* module alias */
} IIO_ATYPE;

    /* the four forms of channel name */
    /* boolean flag type */
typedef enum {
    iio_bool_false = 0,
    iio_bool_off = 0,
    iio_bool_no = 0,
    iio_bool_true = 1,
    iio_bool_on = 1,
    iio_bool_yes = 1
} IIO_BOOL;

    /* the channel type */
typedef enum {
    iio_chtype_di,                      /* digital in */
    iio_chtype_do,                      /* digital out */
    iio_chtype_dio,                     /* digital in/out */
    iio_chtype_oco,                     /* open collector out */
    iio_chtype_ocio,                    /* open collector in/out */

    iio_chtype_bi,                      /* bitwise digital in */
    iio_chtype_bo,                      /* bitwise digital out */
    iio_chtype_bio,                     /* bitwise digital in/out */
    iio_chtype_boco,                    /* bitwise open collector out */
    iio_chtype_bocio,                   /* bitwise open collector in/out */

    iio_chtype_adc,                     /* analogue to digital */
    iio_chtype_dac,                     /* digital to analogue */
    iio_chtype_enc,                     /* positional encoder */
    iio_chtype_rdio,                    /* reversible digital in/out */

    iio_chtype_isa,                     /* ISA space*/
    iio_chtype_ip,                      /* IP space */
    iio_chtype_null,                    /* the null device */
    iio_chtype_vme,                     /* VMEbus space */
    iio_chtype_sc,                      /* servo controller */
    iio_chtype_adam,                    /* Adam 4000 address */
    IIO_NCHTYPES                        /* LAST */
} IIO_CHTYPE;

    /* file open attributes */
typedef enum {
    iio_fattr_rdonly = (1 << 0),        /* open for read */
    iio_fattr_wronly = (1 << 1),        /* open for write */
    iio_fattr_rdwr = (1 << 2),          /* open for read and write */
    iio_fattr_append = (1 << 3),        /* start at end of file */
    iio_fattr_creat = (1 << 4),         /* create if not there */ 
    iio_fattr_trunc = (1 << 5),         /* truncate on open */
    iio_fattr_excl = (1 << 6)           /* access exclusively */
} IIO_FATTR;

    /* structure magic numbers */
typedef enum {
    iio_magic_minfo = 0x4d494e46,       /* "MINF" */
    iio_magic_module = 0x4d4f444c,      /* "MODL" */
    iio_magic_chnode = 0x43484e4f,      /* "CHNO" */
    iio_magic_chinfo = 0x4348494e,      /* "CHIN" */
    iio_magic_ipinfo = 0x4d50494e,      /* "IPIN" */
    iio_magic_map = 0x564d4150,         /* "VMAP" */
    iio_magic_namex = 0x4e414d45,       /* "NAME" */
    iio_magic_open = 0x4f50454e,        /* "OPEN" */
    iio_magic_opnode = 0x4f504e4f,      /* "OPNO" */
    iio_magic_sentinel = 0x4d4d4f4c,    /* "IIOL" */
    iio_magic_state = 0x53544154        /* "STAT" */
} IIO_MAGIC;

    /* mapping type */
typedef enum {
    iio_maptype_memory,
    iio_maptype_port
} IIO_MAPTYPE;

    /* how many times a module can be installed */
typedef enum {
    iio_multi_no,
    iio_multi_yes
} IIO_MULTI;

    /* format of a channel name */
typedef enum {
    iio_nform_gg,                       /* global generic form */
    iio_nform_gs,                       /* global specific form */
    iio_nform_lg,                       /* local generic form */
    iio_nform_ls                        /* local specific form */
} IIO_NFORM;

    /* kind of operation code argument */
typedef enum {
    iio_oparg_none = 0x00,              /* no arguments */
    iio_oparg_in = 0x01,                /* inward argument */
    iio_oparg_out = 0x02,               /* outward argument */
    iio_oparg_inout = 0x03,             /* in and out argument */
    iio_oparg_data = 0x10,              /* data argument */
    iio_oparg_addr = 0x20               /* address argument */
} IIO_OPARG;

    /* type of probing */
typedef enum {
    iio_ptype_read = 0x1,               /* try reading */
    iio_ptype_write = 0x2,              /* try writing */
    iio_ptype_rdwr = 0x3                /* try reading and writing */
} IIO_PTYPE;

    /* mask/roll of IIO_SIZE when rolled into an operation code */
#define IIO_SPACE_MASK  ((1 << 8) - 1)
#define IIO_SIZE_MASK   (IIO_SPACE_MASK << 8)

    /* probe/resolve register/bus access size types */
typedef enum {
    iio_size_8  = (1 << 8),             /* byte access */
    iio_size_16 = (2 << 8),             /* word access */
    iio_size_32 = (4 << 8),             /* long word access */
    iio_size_64 = (8 << 8)              /* really long word access */
} IIO_SIZE;

    /* tfile access method */ 
typedef enum {
    iio_tfile_file,                     /* file descriptor */
    iio_tfile_filename,                 /* file name */
    iio_tfile_string                    /* string pointer */
} IIO_TFILE;

    /* TTY control tags */
typedef enum {
    iio_ttypar_none = 0x0,
    iio_ttypar_even = 0x1,
    iio_ttypar_odd = 0x2,
    iio_ttypar_ignore = 0x10,
    iio_ttypar_mark = 0x20
} IIO_TTYPAR;
 
typedef enum {
    iio_ttyflow_none = 0x0,
    iio_ttyflow_rtscts = 0x1,
    iio_ttyflow_xonxoff = 0x2,
    iio_ttyflow_xonany = 0x4,
    iio_ttyflow_hupcl = 0x10
} IIO_TTYFLOW;
 
typedef enum {
    iio_ttymap_instrip = 0x1,
    iio_ttymap_incrnl = 0x2
} IIO_TTYMAP;
 
typedef enum {
    iio_ttyecho_none = 0x0,
    iio_ttyecho_echo = 0x1
} IIO_TTYECHO;

typedef enum {
    iio_udata_int,                      /* integer user data */ 
    iio_udata_real,                     /* real (double) user data */
    iio_udata_addr,                     /* address user data */
    iio_udata_bit                       /* bitfield data */
} IIO_UDATA;

/*
 * ------------------------------------------------------- FORWARD DECLARATIONS
 */

typedef struct IIO_ADAMINFO IIO_ADAMINFO;
typedef struct IIO_ALIAS IIO_ALIAS;
typedef struct IIO_OPEN IIO_OPEN;
typedef struct IIO_CHNODE IIO_CHNODE;
typedef struct IIO_CHINFO IIO_CHINFO;
typedef struct IIO_OPNODE IIO_OPNODE;
typedef struct IIO_IPINFO IIO_IPINFO;
typedef struct IIO_MAP IIO_MAP;
typedef struct IIO_MINFO IIO_MINFO;
typedef struct IIO_MODULE IIO_MODULE;
typedef struct IIO_MREG IIO_MREG;
typedef struct IIO_MSTATE IIO_MSTATE;
typedef struct IIO_NAMEX IIO_NAMEX;
typedef struct IIO_OPINFO IIO_OPINFO;
typedef struct IIO_SENTINEL IIO_SENTINEL;
typedef struct IIO_STATE IIO_STATE;
typedef struct IIO_SLL IIO_SLL;

    /* opaque mutex and file types */
typedef void IIO_MUTEX;
typedef void IIO_SHMUTEX;
typedef int IIO_FILE;


/*
 * ----------------------------------------------------- FUNCTION POINTER TYPES
 */

    /* type of a module install function */
typedef IIO_STATUS (* IIO_INSTALLFN)(
    IIO_MODULE *module, char *argv[]
);
    /* type of a module init function */
typedef IIO_STATUS (* IIO_INITFN)(
    IIO_MREG *reg, IIO_MSTATE *state
);
    /* type of a module operate function */
typedef IIO_STATUS (* IIO_OPFN)(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE*,
    IIO_OP op, unsigned first, unsigned last
);
    /* compare function type */
typedef int (* IIO_CMPFN)(IIO_SLL*, IIO_SLL*);

    /* tfile callback type */
typedef IIO_STATUS (* IIO_TEXEC)(char *argv[]);


/*
 * ------------------------------------------------------ STRUCTURE DEFINITIONS
 */

struct IIO_ADAMINFO {
    /*
     * Information on a particular range setting for an ADAM module.
     * A table of these, indexed by the range code, is used to look up
     * the scaling constants where required
     */
    double
        escale,                 /* scale factor (eng mode) */
        hscale;                 /* scale factor (hex mode) */
    char *unit, *range;         /* user unit, range/type string */
};

struct IIO_ALIAS {
    /*
     * Linked list of aliases of either whole channels or channel ranges,
     * modules (not ranges) or local channels and ranges. An alias is 
     * actually stored as name-value strings, and is resolved only when
     * invoked in a channel open
     */
    IIO_ALIAS *next;

    IIO_ATYPE type;             /* type of alias */
    char *name;                 /* the alias itself */
    char *value;                /* the thing aliased */
};

struct IIO_CHNODE {
    /*
     * List of available channels. Each element identifies a group of
     * channels of the same type and width on a particular module, and
     * contains the pointer to the operation function. This list is built
     * as the modules are installed, and is searched when modules are 
     * opened
     */
    IIO_CHNODE *next;           /* linked list pointer */
    IIO_MAGIC magic;            /* magic number */
    
    unsigned int number;        /* number of these channels in group */
    IIO_CHTYPE type;            /* channel type */
    unsigned int width;         /* channel width in bits */

    /* sequence numbers of first channel in this group */
    unsigned int seqno[4];      /* one for each of the four name forms */

    IIO_MODULE *module;         /* the module that provides them */
    IIO_OPFN operate;           /* the operation function pointer to use */

    IIO_CHINFO *chinfo;         /* array of information on each channel */

    /* extra data for the bitfield pseudo-driver */
    IIO_CHNODE *rchnode;        /* real chnode */
    unsigned int rseqno;        /* real seqno */
};

struct IIO_CHINFO {
    /*
     * Specific information on a single channel. There is an array of these
     * structures in each IIO_CHNODE, one per local channel. There is a
     * default linear scaling, and later perhaps provision for other
     * conversions. The limits and initial values of the channel are also 
     * stored here
     */
    IIO_MAGIC magic;            /* magic number */

    double scale, offset;       /* standard linear conversion */
    char *unit;                 /* units after linear conversion */

    int initial;                /* initial channel value */
    int upper, lower;           /* limits apply if upper>lower */
    IIO_BOOL log;               /* logging flag */
};

struct IIO_IPINFO {
    /*
     * This structure is used by IndustryPack Carrier cards. It contains a 
     * duplicate of the IP's IDPROM (read as 16-bit words), plus reconstructed
     * forms of the manufacturer/product code and the driver ID code.
     */
    IIO_MAGIC magic;            /* magic number */

    uint32_t mid;           /* manufacturer code */
    uint32_t pid;           /* product code */
    uint32_t rev;           /* module revision number */
    uint32_t did;           /* driver ID code */
    uint32_t flg;           /* flags word */

    uint16_t *pprom;        /* pointer to pack data in data[] */ 
    uint16_t *uprom;        /* pointer to user data in data[] */

    /* the original data from the IDPROM */
    uint16_t prom[0x20];
};

struct IIO_MINFO {
    /*
     * The minfo structure contains information about a particular
     * module driver. Minfos are added to the minfo list with iio_minfo(),
     * which is the first (and usually only) act of a module information
     * function
     */
    IIO_MINFO *next;            /* linked list pointer */
    IIO_MAGIC magic;            /* magic number */

    char *ident;                /* the ident code of the module */
    char *model;                /* manufacturer, module and description */
    char *version;              /* module driver RCS version, date etc */
    IIO_MULTI multi;            /* how many times this module can be used */
    IIO_INSTALLFN install;      /* the install function pointer */
    IIO_INITFN init;            /* the init function pointer */
};

struct IIO_MODULE {
    /*
     * The module list contains one element for each installed module.
     * The module list is extended by iio_module()
     */
    IIO_MODULE *next;           /* linked list pointer */
    IIO_MAGIC magic;            /* magic number */

    IIO_MINFO *minfo;           /* module information block */
    unsigned int seq;           /* module sequence number */
    IIO_BOOL log;               /* log operations on this module */

    IIO_MSTATE *state;          /* global shared state */
    IIO_MREG *reg;              /* module register pointers */
    IIO_SHMUTEX *mutex;         /* module/state mutex */
};

struct IIO_NAMEX {
    /*
     * The namex structure is filled out (not created) by iio_namex, and
     * contains the broken-down (expanded) form of a channel name. The
     * type field indicates the type of the original name. The module
     * ident and sequence numbers are only valid for lg and ls forms
     */
    IIO_MAGIC magic;            /* magic number */
    IIO_NFORM form;             /* form of name */

    IIO_MODULE *module;         /* module pointer */
    IIO_CHTYPE type;            /* channel type */
    unsigned int width;         /* channel width */
    unsigned int seq1, seq2;    /* channel sequence number range */
};

struct IIO_OPEN {
    /*
     * Open channel descriptor. There is one of these for each open
     * channel or channel range, linked into a single list. A channel may
     * represent a single simple channel on a module, or a contiguous range
     * of channels (but only of the same type) spread over several modules
     * (one IIO_OPNODE for each)
     */
    IIO_OPEN *next;             /* main linked list pointer */
    IIO_MAGIC magic;            /* magic number */

    unsigned number;            /* number of simple channels in range */
    char *name;                 /* name under which this channel was opened */
    IIO_MUTEX *mutex;           /* channel mutex (if a range) */
    IIO_OPNODE *opnode;         /* list of channel nodes */
};

struct IIO_OPNODE {
    /* 
     * A list of these structures hangs off the channel descriptor blocks.
     * When a channel is opened by iio_open(), one of these gets
     * created for each chnode associated with the open channel
     */
    IIO_OPNODE *next;           /* main linked list pointer */
    IIO_MAGIC magic;            /* magic number */

    IIO_CHNODE *chnode;         /* channel info block */
    unsigned int first;         /* first channel seqno in range (ls form) */
    unsigned int number;        /* number in the range (local) */
    unsigned int index;         /* index of corresponding user data element */
    IIO_BOOL log;               /* log operations on this channel */

    /*
     * This information changes each operation function call,
     * but is protected by the module semaphore
     */
    IIO_UDATA udata;            /* type of user data */
    void *base;                 /* base of user data */
    IIO_OP op;                  /* the current operation code */
};

struct IIO_OPINFO {
    /*
     * Operation info structure. There is an array iio_opinfo[] of these,
     * indexed by the IIO_OP operation code. It is used to tell how many
     * arguments of what type and what direction 
     */
    char *name;                 /* the name of the operation */
    char *symbol;               /* a symbol for the log message */
    IIO_OPARG arg;              /* kind and direction of the argument */
};

struct IIO_SENTINEL {
    /*
     * The sentinel structure is the first IIO strincture in shared memory.
     * It contains a timestamp from the IIO library itself (output from the
     * date command) and from the configuration file. If both of these match 
     * this the remaining shared memory should be consistent between processes
     */
    IIO_MAGIC magic;            /* magic number */

    double library_time;        /* "mtime" of library */
    unsigned long config_time;  /* mtime of config file */
};

struct IIO_SLL {
    /*
     * All the linked list structures (i.e., most of the above) are
     * ordered, and the ordered insert is done using iio_sll_insert().
     * This requires the *next element TO BE FIRST in these structures.
     * struct IIO_SLL is a dummy structure, with a *next element only
     */
    IIO_SLL *next;
};

struct IIO_STATE {
    /*
     * The state structure contains the heads of the main list data 
     * structures, and the protection mutex for them all
     */
    IIO_MAGIC magic;            /* magic number */

    IIO_ALIAS *alias;           /* head of alias list */
    IIO_OPEN *open;             /* head of open channel list */
    IIO_CHNODE *chnode;         /* head of available channel list */
    IIO_MAP *map;               /* head of available map list */
    IIO_MODULE *module;         /* head of installed module list */
    IIO_MINFO *minfo;           /* head of available module drivers list */

    IIO_MUTEX *mutex;           /* per-process core state mutex */
    IIO_MUTEX *omutex;          /* per-process open-channel list mutex */

    IIO_BOOL init;              /* initialise, as well as install modules */
    IIO_BOOL log;               /* generate some messages */
};

struct IIO_MAP {
    /*
     * IIO_MAP structures represent a mapping from physical memory to 
     * virtual (logical, or process) memory, usually to access module
     * registers. The mappings are created through the OS function accessed
     * using iio_shmap_alloc(), but the OS generally won't reverse-lookup
     * physical addresses, so we need to keep a record of the mappings
     */
    IIO_MAP *next;              /* linked list pointer */
    IIO_MAGIC magic;            /* magic number */

    IIO_MAPTYPE type;           /* mapping type (memory or port) */
    void *paddr;                /* physical address of base of mapping */
    void *vaddr;                /* virtual address of base of mapping */
    unsigned size;              /* size of mapping in bytes */
};


/*
 * ---------------------------------------------- IIO LIBRARY FUNCTIONS/GLOBALS
 */

extern IIO_STATUS
    iio_adam_dread(char *dex, int digits, int *result),
    iio_adam_hread(char *hex, int digits, int *result),
    iio_adam_hwrite(int value, int digits, char *string);
extern IIO_ADAMINFO
    iio_adam_info[];

extern IIO_STATUS
    iio_alias(char *argv[]),
    iio_alias_insert(IIO_ATYPE type, char *name, char *value),
    iio_alias_find(char *name, IIO_ATYPE *atype, char **value),
    iio_alias_show(void);

extern IIO_STATUS
    iio_arg(char *argv[], char *option, IIO_ARG type, ... ),
    iio_arg_index(char *argv[], char *option, unsigned ind, IIO_ARG type, ... ),
    iio_arg_list(char *argv[], ...),
    iio_arg_index_list(char *argv[], unsigned index, ...),
    iio_arg_remnants(char *argv[]);
extern char
    *iio_arg_blank;

extern IIO_INFOFN
    iio_builtin[];

extern IIO_STATUS
    iio_channel(char *argv[]);

extern IIO_STATUS
    iio_chnode(
        IIO_MODULE *module, IIO_CHTYPE type, unsigned int width,
        unsigned int number, IIO_OPFN operate, IIO_CHNODE **chnode
    ),
    iio_chnode_linear(IIO_CHNODE *, unsigned, double sc, double of, char *un),
    iio_chnode_limits(IIO_CHNODE *, unsigned, int initial, int low, int high),
    iio_chnode_log(IIO_CHNODE *, unsigned, IIO_BOOL),
    iio_chnode_show(IIO_BOOL longform);
extern int
    iio_chnode_cmp(IIO_SLL *s1, IIO_SLL *s2);

extern IIO_STATUS
    iio_chtype_find(char *string, IIO_CHTYPE *type, unsigned int *width);
extern char
    *iio_chtype_string[];

extern IIO_STATUS
    iio_config_exec(char *argv[]);

extern int
    iio_data_get(IIO_OPNODE *opnode, unsigned seqno);
extern double 
    iio_data_get_real(IIO_OPNODE *opnode, unsigned seqno);
extern void
    *iio_data_get_addr(IIO_OPNODE *opnode, unsigned seqno);
extern IIO_STATUS
    iio_data_set(IIO_OPNODE *opnode, unsigned seqno, int value),
    iio_data_set_real(IIO_OPNODE *opnode, unsigned seqno, double value),
    iio_data_set_addr(IIO_OPNODE *opnode, unsigned seqno, void *value);

extern IIO_STATUS
    iio_done_iio(void);

extern IIO_STATUS
    iio_init_iio(
        IIO_INFOFN list[], IIO_IFLAG flags,
        IIO_BOOL init, IIO_TFILE method, ...
    );

extern IIO_STATUS
    iio_ipinfo_read(IIO slot, IIO_IPINFO *ipinfo),
    iio_ipinfo_ident(IIO slot, uint32_t mid, uint32_t pid),
    iio_ipinfo_show(IIO slot);

extern IIO_STATUS
    iio_map(IIO_OPEN *chan, IIO_OP space, unsigned lbase, unsigned lsize),
    iio_map_new(IIO_OP space, void *paddr, unsigned psize),
    iio_map_ptov(IIO_MAPTYPE type, void *paddr, unsigned size, void **vaddr),
    iio_map_vtop(IIO_MAPTYPE type, void *vaddr, unsigned size, void **paddr),
    iio_map_type(IIO_OP space, IIO_MAPTYPE *type),
    iio_map_done(void),
    iio_map_show(void);


extern IIO_STATUS
    iio_minfo(
        const char *ident, const char *name, const char *version,
        IIO_MULTI multi, IIO_INSTALLFN install, IIO_INITFN init
    ),
    iio_minfo_call(IIO_INFOFN infofn[]),
    iio_minfo_find(char *ident, IIO_MINFO **minfo),
    iio_minfo_show(void);

extern IIO_STATUS
    iio_module(char *argv[]),
    iio_module_state(IIO_MODULE *module, unsigned int size),
    iio_module_reg(IIO_MODULE *module, unsigned int size, IIO_MREG **reg),
    iio_module_find(char *string, IIO_MODULE **module),
    iio_module_show(void);

extern IIO_STATUS
    iio_namex(char *name, IIO_NAMEX *namex);

extern IIO_STATUS
    iio_null(void);

extern IIO_STATUS
    iio_open_show(IIO_OPEN *chan);

extern IIO_OPINFO
    iio_opinfo[];
extern IIO_STATUS
    iio_opinfo_lookup(char *string, IIO_OP *operation),
    iio_opinfo_show(void);

extern IIO_STATUS
    iio_phantom(void);

extern IIO_STATUS
    iio_resolve(
        IIO_OPEN *chan, IIO_OP space, IIO_SIZE size,
        unsigned laddr, void **vaddr
    ),
    iio_resolve_physical(
        IIO_OPEN *chan, IIO_OP space, IIO_SIZE size,
        unsigned laddr, void **paddr
    ),
    iio_resolve_list(IIO_OPEN *chan, IIO_OP space, ...);

extern IIO_STATUS
    iio_return_error(char *emessage, char *file, unsigned int lineno),
    iio_return_fatal(char *emessage, char *file, unsigned int lineno);

extern IIO_STATUS
    iio_sll_insert(IIO_SLL **head, IIO_SLL *new, IIO_CMPFN cmp);

extern IIO_STATUS
    iio_state_init(void);
extern IIO_STATE
    *iio_state;

extern double 
    iio_timestamp;

extern IIO_STATUS
    iio_tfile(IIO_TEXEC texec, IIO_TFILE method, ...),
    iio_tfile_stdarg(IIO_TEXEC texec, IIO_TFILE method, va_list ap);


/*
 * ----------------------------------------------------- FUNCTIONS IN OS MODULE
 *
 * There is one module per operating system which should contain all of the 
 * functions below, which implement, emulate or re-package certain system calls
 * or standard library calls, so that the remainder of the library only
 * accesses the system through this module. The module is generally named
 * after the system, e.g. vxworks.c, lynxos.c, and so on.
 */

extern char
    *iio_configname;

extern IIO_STATUS
    iio_osinit(IIO_INFOFN list[], IIO_IFLAG iflags),
    iio_osdone(void);

extern IIO_STATUS
    iio_emessage_set(char *message);

    /* the old GCC does not understand __attribute__ */
#if (__GNUC__ == 2)
extern IIO_STATUS
    iio_log(char *fmt, ...) __attribute__ ((format(printf, 1, 2)));
#else
extern IIO_STATUS
    iio_log(char *fmt, ...);
#endif

extern IIO_STATUS
    iio_shmap_alloc(
        void *paddr, unsigned int psize,
        void **pactual, void **vaddr, unsigned int *vsize
    ),
    iio_shmap_free(void *vaddr, void *paddr, unsigned size);

extern IIO_STATUS
    iio_shmem_alloc(unsigned int size, void **new),
    iio_shmem_free(void *old);

extern IIO_STATUS
    iio_shmutex_create(IIO_SHMUTEX **mutex),
    iio_shmutex_grab(IIO_SHMUTEX *mutex),
    iio_shmutex_drop(IIO_SHMUTEX *mutex),
    iio_shmutex_free(IIO_SHMUTEX *mutex);

extern IIO_STATUS
    iio_mutex_create(IIO_MUTEX **mutex),
    iio_mutex_grab(IIO_MUTEX *mutex),
    iio_mutex_drop(IIO_MUTEX *mutex),
    iio_mutex_free(IIO_MUTEX *mutex);

extern IIO_STATUS
    iio_sem_wait(IIO_MUTEX *mutex),
    iio_sem_post(IIO_MUTEX *mutex);

extern IIO_STATUS
    iio_port_alloc(
        void *paddr, unsigned int psize,
        void **pactual, void **vaddr, unsigned int *vsize
    );
extern void
    iio_port_set8(volatile uint8_t *addr, uint8_t val),
    iio_port_set16(volatile uint16_t *addr, uint16_t val),
    iio_port_set32(volatile uint32_t *addr, uint32_t val);
extern uint8_t
    iio_port_get8(volatile uint8_t *addr);
extern uint16_t
    iio_port_get16(volatile uint16_t *addr);
extern uint32_t
    iio_port_get32(volatile uint32_t *addr);

extern IIO_STATUS
    iio_probe(volatile void *vaddr, IIO_SIZE size, IIO_PTYPE ptype);

extern IIO_STATUS
    iio_roughdelay(unsigned milliseconds);

extern int
    iio_round(double x);

extern IIO_STATUS
    iio_tty_line(
        IIO_FILE tty,
        unsigned baud, unsigned nchar, unsigned nstop,
        IIO_TTYPAR parity, IIO_TTYFLOW flow, IIO_TTYMAP map,
        IIO_TTYECHO echo
    ),
    iio_tty_raw(
        IIO_FILE tty,
        unsigned baud, unsigned nchar, unsigned nstop,
        IIO_TTYPAR parity, IIO_TTYFLOW flow, IIO_TTYMAP map,
        unsigned char min, unsigned char time
    ),
    iio_tty_send(IIO_FILE tty, char *buffer, unsigned len),
    iio_tty_recv(
        IIO_FILE tty, char *buffer, unsigned maxlen,
        unsigned *len, unsigned ms
    );

/*
 * --------------------------------------------------- FUNCTIONS IN LIBC MODULE
 *
 * The module libc.c contains wrappers for C library (or standard system call)
 * functions that are used within the rest of the library. If the library is
 * only ever used at user level (in Unix/LynxOS), or on vxWorks and RTEMS
 * (which have a standard C library interface too), then we can probably get
 * rid of all these wrappers
 */

extern IIO_STATUS
    iio_mem_alloc(unsigned int size, void **new),
    iio_mem_free(void *old);

extern IIO_STATUS
    iio_string_lookup(char *token, char **table, int *index),
    iio_string_dup(char *string, char **new),
    iio_string_cpy(char *dst, char *src);
extern char
    *iio_string_chr(char *s, char c),
    *iio_string_pbrk(char *s, char *c);
extern int
    iio_string_cmp(char *s1, char *s2),
    iio_string_ncmp(char *s1, char *s2, int len),
    iio_string_len(char *string);

extern IIO_STATUS
    iio_decode_long(char *string, long int *result),
    iio_decode_double(char *string, double *result);

    /* the old GCC does not understand __attribute__ */
#if (__GNUC__ == 2)
extern IIO_STATUS
    iio_slog(char *out, char *fmt, ...)
        __attribute__ ((format(printf, 2, 3)));
#else
extern IIO_STATUS
    iio_slog(char *out, char *fmt, ...);
#endif
extern IIO_STATUS
    iio_vslog(char *out, char *fmt, va_list ap);

#if (__GNUC__ == 2)
extern IIO_STATUS
    iio_exec(char *fmt, ...)
        __attribute__ ((format(printf, 1, 2)));
#else
extern IIO_STATUS
    iio_exec(char *fmt, ...);
#endif


extern IIO_STATUS
    iio_file_open(char *name, IIO_FATTR fattr, IIO_FILE *file),
    iio_file_close(IIO_FILE file);
extern char
    iio_file_getc(IIO_FILE file);

/*
 * ----------------------------------------------------------------------- MISC
 */

    /* use HIDDEN for local objects */
#define HIDDEN static

#ifndef NULL
#define NULL ((void *)0)
#endif


/* --------------------------------------------------------------- ERROR MACROS
 *
 * This is a slightly embellished error-return-collapse arrangement. All
 * functions return IIO_STATUS, defined in iio.h to be an enumerative: 
 *
 *      iio_status_ok           function executed sucessfully
 *      iio_status_error        function found non-fatal error
 *      iio_status_fatal        function found fatal error
 *
 * There are macros to generate the error returns, and macros to call functions
 * and test the result. This makes the code much more readable, as it is not
 * covered in messy if(...) statement (actually it is, but they are done by
 * macros so they aren't seen). It also allows cunning things to be done, like
 * printing file/line numbers and subroutine call traces for fatal errors.
 *
 * To return an error from a function, use iio_error() or iio_fatal():
 *
 *      return iio_error("Bad thingy whatsit");
 * or
 *      return iio_fatal("Really bad thingy whatsit");
 *
 * The string argument should be a string constant (as shown), or a global
 * or static string pointer; it is stored for later printing. It can be
 * retrieved using iio_emessage_get(). If the error is from a system call
 * or operating system function, use iio_error_system(), or iio_fatal_system(),
 * to generate the message from the system errno or equivalent:
 *
 *      if ((fd = open( ... )) < 0)
 *          return iio_error_system;
 *
 *
 * To call functions, you can use iio_eret(), or iio_fret()
 *
 *      iio_eret( function(arg, ...) );
 * or
 *      iio_fret( function(arg, ...) );
 *
 * iio_eret() calls function(), and if it returns fatal error status, logs
 * a message, and returns fatal error status. If it returns non-fatal error
 * status, no message is printed, and it returns error status; if ok status,
 * nothing is done.
 *
 * iio_fret() is the same, but only returns for fatal errors; non-fatals
 * DO NOT cause a return and it is up to the code at this point to deal with
 * the error (using a variable to catch the status).
 *
 */

    /*
     * This macro is used when generating errors, usually used with 
     * a return statement. The S argument should be a static string pointer.
     */
#define iio_error(S) iio_return_error((S), __FILE__, __LINE__)

    /*
     * This macro is for generating FATAL errors; as for the above, but logs
     * a message indicating where is was detected, as well as printing S
     */
#define iio_fatal(S) iio_return_fatal((S), __FILE__, __LINE__)

    /*
     * This macro evaluates S once (usually a function call) and returns from
     * the calling function if S is an error. This is used to pass up errors
     * at lower levels. If S indicates a fatal error, a tracing message is
     * logged as well
     */
#define iio_eret(S) \
    switch ((S)) { \
    case iio_status_ok: \
        break; \
    case iio_status_error: \
        return iio_status_error; \
    case iio_status_fatal: \
        iio_log( \
            "IIO: called from: file %s, line %d\n", \
            __FILE__, __LINE__); \
        return iio_status_fatal; \
    }

#define iio_fret(S) \
    switch ((S)) { \
    case iio_status_ok: \
        break; \
    case iio_status_error: \
        break; \
    case iio_status_fatal: \
        iio_log( \
            "IIO: called from: file %s, line %d\n", \
            __FILE__, __LINE__); \
        return iio_status_fatal; \
    }

#endif
