/*
 ********************************************************************
 *
 * sirf.c - handler for SIRF binary  based communications
 *           for the uBLOX GPS modules
 *
 *     CSIRO Automation
 *     Queensland Centre for Advanced Technologies
 *     PO Box 883, Kenmore, QLD 4069, Australia
 *     www.cat.csiro.au/cmst
 *
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 ********************************************************************
 */
/**
 ********************************************************************
 *
 * \file sirf.c
 * \brief Handler for SIRF binary based communications for the uBLOX GPS
 * \author Craig Worthington
 *
 * This package provides a general communications interface to 
 * binary protocol SIRF for the uBLOX GPS module
 *
 ********************************************************************
 */

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "sirf.h"

typedef enum _comsState
{
  HEADERSTART,
  HEADERCHECK,
  MESGLENGTH,
  MESGLENGTH2,
  MESG,
  CRCSTART,
  CRCCHECK,
  ENDSTART,
  ENDCHECK
} COMM_STATE;



int ublox_crc( unsigned char *data, int dataLength );



/**
 * Wait for and read SIRF binary packets on the configured port.
 * If no packet recieved for 2 seconds an error will
 * be returned.
 *
 * @param port Pointer to configuration comms port
 * @param mesg Pointer to RtxSIRF structure
 *
 * @return 0  if read SIRF packet
 *         -1 if read error occured
 *         -2 if checksum failed
 *
 **/
int rtx_sirf_read( int port, RtxSIRF *mesg )
{
  int mesgCnt, mesgCRC, CRC, mesgLoop = 1;
  unsigned char chBuff;
  COMM_STATE currState;

  currState = HEADERSTART;
 
  /* read full SiRF message from port */
  while( mesgLoop == 1 )
    {
      if( rtx_serial_read_timeout(port,(char *)&chBuff,1,2.0) < 0 )
	return( - 1 );

      switch( currState )
	{
	case HEADERSTART:
	  if( chBuff == 0xA0 )
	    currState = HEADERCHECK;
	  break;
	case HEADERCHECK:
	  if( chBuff == 0xA2 )
	    currState = MESGLENGTH;
	  else
	    currState = HEADERCHECK;
	  break;
	case MESGLENGTH:
	  mesg->sirfLength = 0;
	  mesg->sirfLength = (chBuff << 8);
	  currState = MESGLENGTH2;
	  break;
	case MESGLENGTH2:
	  mesg->sirfLength += chBuff;
	  mesgCnt = 0;
	  currState = MESG;
	  break;
	case MESG:
	  mesg->sirfMesg[mesgCnt] = chBuff;
	  mesgCnt++;
	  if( mesgCnt >= mesg->sirfLength )
	    currState = CRCSTART;
	  break;
	case CRCSTART:
	  CRC = 0;
	  CRC = (chBuff << 8);
	  currState = CRCCHECK;
	  break;
	case CRCCHECK:
	  CRC += chBuff;
	  currState = ENDSTART;
	  break;
	case ENDSTART:
	  if( chBuff == 0xB0 )
	    currState = ENDCHECK;
	  else
	    currState = HEADERSTART;
	  break;
	case ENDCHECK:
	  if( chBuff == 0xB3 )
	    mesgLoop = 0;
	  else
	    currState = HEADERSTART;
	  break;
	}
    }

  /* calc. message checksum */
  mesgCRC = ublox_crc( mesg->sirfMesg,mesg->sirfLength );

  /** compare calc checksum with mesg checksum **/
  if( mesgCRC != CRC )
    return( -2 );

  return( 0 );
}



/****************
 * rtx_sirf_write - format, generate CRC, and send SiRF binary packet for uBLOX GPS
 *
 *
 */
int rtx_sirf_write( int port, ... )
{
  int mesgCnt, intBuff;
  va_list args;
  RtxSIRF mesg;

  va_start( args, port );

  /* read and create packet */
  for( mesgCnt = 0; mesgCnt < 1024; mesgCnt++ )
    {
      intBuff = va_arg( args, int );
      if( intBuff == 0 )
	break;

      mesg.sirfMesg[mesgCnt] = intBuff;
    }

  /* generate CRC */

  /* send packet to uBLOX */
  

  return( 0 );
}



/*************************
 * ublox_crc - calc. CRC for a SiRF binary packet
 *
 */
int ublox_crc( unsigned char *data, int dataLength )
{
  int cnt;
  int crcVal = 0;

  for( cnt = 0; cnt < dataLength; cnt++ )
    crcVal += data[cnt];

  crcVal = crcVal & 0x7FFF;

  return( crcVal );
}
