/*********************************************************************
 *
 * CSIRO Automation
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * www.cat.csiro.au/cmst
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 *********************************************************************/

/**
 * \file reset.c
 * \brief Reset module for hc12 reset board
 * \author Jonathan.Roberts@csiro.au
 */

#include <stdio.h>
#include <string.h>
#include <time.h>

#include <ddx.h>

#include <rtx/error.h>
#include <rtx/thread.h>
#include <rtx/time.h>
#include <rtx/timer.h>
#include <rtx/message.h>
#include <rtx/nmea.h>

#include "gpslog.h"

static RtxThread	*resetThread;
static int		resetDebug = 1;

static int 
reset_thread(
	GPSlog_stat	*devStat
	)
{
	double	age;
	char	resetString[16];

	sprintf(resetString, "%c%c%s", 0xff, 0xfe, "RESET");
	rtx_time_get(&devStat->nmeaTime);
	devStat->resetTimeout = 3;

	while (1) {
		rtx_timer_sleep(1.0);

		/* Check age of last NMEA packet */
		if ((age = rtx_time_get_delta(&devStat->nmeaTime)) > devStat->resetTimeout) {
			/* 
			 * Too old so see if unit is alive by sending the
			 * battery status NMEA packet 
			 */
			if (resetDebug)
				rtx_message("reset_thread: requesting battery level from GPS unit to see if its alive");
			rtx_nmea_write(devStat->nmeafp, "PLEIQ", "BAT", "2", "200", NULL);
			
			/* Wait 1 second for a reply */
			rtx_timer_sleep(1.0);
			if (rtx_time_get_delta(&devStat->nmeaTime) > 2.0) {

				if (resetDebug)
					rtx_message("reset_thread: no reply\n");

				if (resetDebug) /* Unit must be off so send reset to HC12 */
					rtx_message("reset_thread: sending RESET packet to HC12");
				write (devStat->nmeafp, resetString, 7);	

				if (resetDebug) /* Wait for 7 seconds */
					rtx_message("reset_thread: waiting 7 seconds for reset to work");
				rtx_timer_sleep(7.0);
			} else {
				if (resetDebug)
					rtx_message("reset_thread: got a reply, so unit must be on and looking for satellites");
			}
		}
	}
}

int 
init_reset(
	GPSlog_stat	*devStat
	)
{
	if( (devStat->verbose & 1) )
		fprintf( stderr, "Running RESET INIT\n" );

	/* Launch the reset thread */
	if ((resetThread = rtx_thread_create ("reset_thread", 0,
			RTX_THREAD_SCHED_OTHER, 0, 0,
			RTX_THREAD_CANCEL_ASYNCHRONOUS,
			(void *(*) (void *)) reset_thread,
			(void *)devStat,
			NULL, NULL )) == NULL)
		return rtx_error("init_reset: thread_create() failed");

	return (0);
}

int 
done_reset(
	GPSlog_stat *devStat
	)
{
	/* Kill the reset thread */
	if (rtx_thread_destroy(resetThread))
		return rtx_error("done_reset: rtx_thread_destroy() failed");

	return (0);
}
