/*********************************************************************
 *
 * CSIRO Automation
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * www.cat.csiro.au/cmst
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 *********************************************************************/

/**
 * \file llk.c
 * \brief Process GPLLK NMEA messages
 * \author Craig Worthington
 *
 * Process a NMEA GPLLK message
 *
 */

#include <stdio.h>
#include <string.h>
#include <time.h>

#include <ddx.h>

#include <rtx/error.h>

#include "gpslog.h"
#include "convert_eastnorth.h"

#include "llk.h"

#define INIT_LLK_NAME "llk"



char llk_name[256];
DDX_STORE_ITEM *LLK_storeItem;



int init_llk( GPSlog_stat *devStat )
{
  if( (devStat->verbose & 1) )
    fprintf( stderr, "Running GGA INIT\n" );

  if( devStat->ddxOn )
    {
      strcpy( llk_name, INIT_LLK_NAME );

      if( DDX_STORE_REGISTER_TYPE(devStat->storeId,GPS_LLK_MESG) == -1 )
	return( rtx_error("init_llk: ddx_store_regieter_type() failed") );

      if( (LLK_storeItem=ddx_store_lookup_item(devStat->storeId,llk_name,"GPS_LLK_MESG",0)) == NULL )
	return( rtx_error("init_llk: ddx_store_lookup_item() failed") );
    }
  return( 0 );
}



int process_llk( GPSlog_stat *devStat, RtxNMEA *mesg )
{
  GPS_LLK_MESG llkData;
  char charBuff[82];
  double GPStime;

  /* extract time */
  rtx_nmea_extract_double( mesg, 1, &GPStime );
  rtx_nmea_extract( mesg, 1, charBuff );
  sscanf(charBuff,"%2d%2d%lf", &(llkData.time.hours), &(llkData.time.mins), &(llkData.time.secs) );
  llkData.time.hours = ((int)GPStime / 10000);

  /* extract easting and northing */
  rtx_nmea_extract_double( mesg, 3, &llkData.easting );
  rtx_nmea_extract_double( mesg, 5, &llkData.northing );

  /* extract GPS fix indicator */
  rtx_nmea_extract_short( mesg, 7, &llkData.gpsFIX );

  /* extract GPS no. of sats */
  rtx_nmea_extract_short( mesg, 8, &llkData.noSats );

  /* exract GPS horizontal dilution of pression */
  rtx_nmea_extract_float( mesg, 9, &llkData.GDOP );

  /* extract altitude */
  rtx_nmea_extract_double( mesg, 10, &llkData.altitude );

  if( devStat->ddxOn )
    {
      if( ddx_store_write(LLK_storeItem, &llkData, NULL) == -1)
	return( rtx_error("process_llk: ddx_store_write() failed") );
    }

  /* print messages for interactive display */
  if( devStat->interactive )
    {
      if( llkData.gpsFIX )
	fprintf( stderr, "LLK[FIX %d]: Time: %d %d %f easting: %f northing: %f noSats: %d\n", llkData.gpsFIX, llkData.time.hours, llkData.time.mins, llkData.time.secs, llkData.easting, llkData.northing, llkData.noSats );
      else
	fprintf( stderr, "LLK[NOFIX]: Time: %d %d %f easting: %f northing: %f noSats: %d\n", llkData.time.hours, llkData.time.mins, llkData.time.secs, llkData.easting, llkData.northing, llkData.noSats );	
    }

  return( 0 );
}

int done_llk( GPSlog_stat *devStat )
{
  if( devStat->ddxOn )
    {
      if(ddx_store_done_item(LLK_storeItem) == -1 )
	return( rtx_error("init_llk: ddx_store_done_item() failed") );
    }
  return (0);
}
