/*********************************************************************
 *
 * CSIRO Automation
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * www.cat.csiro.au/cmst
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 *********************************************************************/

/**
 * \file gsv.c
 * \brief process NMEA gsv messages
 * \author your name
 *
 * Processes a number of NMEA gsv messages
 *
 */

#include <stdio.h>
#include <string.h>

#include <ddx.h>

#include <rtx/error.h>

#include "gpslog.h"
#include "convert_eastnorth.h"

#include "gsv.h"

#define INIT_GSV_NAME "gsv"
#define MesgOffset 4
#define MesgLength 4


char gsv_name[256];
DDX_STORE_ITEM *GSV_storeItem;

static int noSats; 
static GPS_GSV_MESG satInfo;


int init_gsv( GPSlog_stat *devStat )
{
  if( (devStat->verbose & 1) )
    fprintf( stderr, "Running GSV INIT\n" );

  noSats = 0;

  if( devStat->ddxOn )
    {
      strcpy( gsv_name, INIT_GSV_NAME );

      if( DDX_STORE_REGISTER_TYPE(devStat->storeId,GPS_GSV_MESG) == -1 )
	return( rtx_error("init_gsv: ddx_store_regieter_type() failed") );

      if( (GSV_storeItem=ddx_store_lookup_item(devStat->storeId,gsv_name,"GPS_GSV_MESG",0)) == NULL )
	return( rtx_error("init_gsv: ddx_store_lookup_item() failed") );
    }
  return( 0 );
}



int process_gsv( GPSlog_stat *devStat, RtxNMEA *mesg )
{
  int noMesgs, currMesg;
  int satCnt; 
  int satNo;
  int i;
  char charBuff[82];
  FILE *outFile;

  rtx_nmea_extract_int( mesg, 1, &noMesgs );   /* get the total no. sat mesgs */
  rtx_nmea_extract_int( mesg, 2, &currMesg );  /* get the current mesg no */
  if( noSats == 0 )
    {
      rtx_nmea_extract_int( mesg, 3, &noSats );  /* get no of visable Sats */
      satInfo.noSats = noSats;
    }

  /* process incoming sat messages */
  satCnt = 0;
  while( satCnt < 4 )
    {
      rtx_nmea_extract_int( mesg, 4, &satNo );
      rtx_nmea_extract_short( mesg, MesgOffset+(satCnt*MesgLength), &satInfo.satID[noSats] );
      rtx_nmea_extract_short( mesg, MesgOffset+(satCnt*MesgLength)+1, &satInfo.satElevation[noSats] );
      rtx_nmea_extract_short( mesg, MesgOffset+(satCnt*MesgLength)+2, &satInfo.satAzimuth[noSats] );
      rtx_nmea_extract_short( mesg, MesgOffset+(satCnt*MesgLength)+3, &satInfo.satSignal[noSats] );

      if( devStat->interactive || devStat->gpsStatus )
	fprintf( stderr, "SatID %d elevation %d azimuth %d signal strength %d\n", satInfo.satID[noSats], satInfo.satElevation[noSats], satInfo.satAzimuth[noSats], satInfo.satSignal[noSats] );

      satCnt++;               /* increment current message sat count */

      noSats--;               /* decrement no of sats looked at */
      if( noSats == 0 )       /* check if all sats looked at */
	break;
    }

  /* write sat info if at end of message list */
  if( currMesg == noMesgs )
    {
      /* write to store if required */
      if( devStat->ddxOn )
	{
	  if( ddx_store_write(GSV_storeItem, &satInfo, NULL) == -1)
	    return( rtx_error("process_gsv: ddx_store_write() failed") );
	}

      if( devStat->gpsStatus )
	{
	  if( satInfo.satAzimuth[1] > 0 )
	    fprintf( stderr, "No of Satallites being tracked: %d\n", satInfo.noSats );
	  else
	    fprintf( stderr, "No Satellites being tracked, check antenna\n" );
	}
      
      if( devStat->file )
	{      
	  rtx_nmea_extract_int( mesg, 3, &noSats );  /* get no of visable Sats */
	  sprintf( charBuff, "%s/gps_satellite.txt", devStat->fileDir );
	  outFile = fopen( charBuff, "wt" );
	  for( i = 1; i <= noSats; i++ )
	    fprintf( outFile, "%d %d %d %d\n", satInfo.satID[i], satInfo.satElevation[i], satInfo.satAzimuth[i], satInfo.satSignal[i] );
	  fclose( outFile );
	}
    }
  return( 0 );
}

int done_gsv( GPSlog_stat *devStat )
{
  if( devStat->ddxOn )
    {
      if(ddx_store_done_item(GSV_storeItem) == -1)
	return( rtx_error("init_gsv: ddx_store_done_item() failed") );
    }
  return (0);
}
