/*
 ********************************************************************
 *
 * gpslog.c - program to process data from GPS devices
 *
 *     CSIRO Automation
 *     Queensland Centre for Advanced Technologies
 *     PO Box 883, Kenmore, QLD 4069, Australia
 *     www.cat.csiro.au/cmst
 *
 *      $Id: gpslog.c 2773 2008-03-14 04:27:17Z had067 $
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 ********************************************************************
 */

static char *rcsid = "$Id: gpslog.c 2773 2008-03-14 04:27:17Z had067 $";

/**
 * \file gpslog.c
 * \brief gpslog program will process a GPS input stream and store it in the ddx store
 * \author Craig Worthington
 *
 * GPSlog is an application which will initalise the GPS and the process
 *  all NMEA packets recieved and store them in the ddx store if required
 * Each message will need a seperate store_init routine and a process routine
 *
 */

#include <ctype.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <ddx.h>

#include <rtx/error.h>
#include <rtx/getopt.h>
#include <rtx/main.h>
#include <rtx/message.h>
#include <rtx/nmea.h>
#include <rtx/param.h>
#include <rtx/serial.h>
#include <rtx/getopt.h>

#include "gpslog.h"
#include "gga.h"
#include "vtg.h"
#include "gsv.h"
#include "llk.h"
#include "reset.h"

#ifdef i486_linux
  #define KILLER_PRIO	0
#else
  #define KILLER_PRIO 90
#endif

GPSlog_mesg gps_mesgs[] = 
{
  { "GPGGA", 0, init_gga, process_gga, done_gga },
  { "GPVTG", 0, init_vtg, process_vtg, done_vtg },
  { "GPGSV", 0, init_gsv, process_gsv, done_gsv },
  { "GPLLK", 0, init_llk, process_llk, done_llk },
  { NULL }
};

char *cmdOptions = "gpgga";
char *cmdNMEADev = "/dev/ttyS0";
char *cmdDGPSDev = "/dev/ttyS0";
char *cmdCorrectDev = "244.0.10.12:2210";
char *cmdType = "generic";
char *cmdDir = "";

GPSlog_stat statDev;

RtxThread *process_thread;
RtxThread *dgps_thread;

/**
 * The command-line options
 */
static RtxGetopt gpslogOpts[] = {
  {"-nmea", "GPS serial port device and baud rate",
   { {RTX_GETOPT_STR, &cmdNMEADev, "port:baud" }, RTX_GETOPT_END_ARG } 
     },
  {"-dgps", "DGPS serial port device and baud rate (disabled)",
   { {RTX_GETOPT_STR, &cmdDGPSDev, "port:baud" },
     {RTX_GETOPT_SET, &statDev.dgps, ""}, RTX_GETOPT_END_ARG } 
     },
  {"-file", "Directory for GPS Log file",
   { {RTX_GETOPT_STR, &cmdDir, "Directory"}, 
     {RTX_GETOPT_SET, &statDev.file, ""}, RTX_GETOPT_END_ARG }
  },
  {"-type", "Type of GPS device [generic|ublox|sr530]",
   { {RTX_GETOPT_STR, &cmdType, "type"}, RTX_GETOPT_END_ARG }
  },
  {"-correction", "DGPS Multicast IP (disabled)",
   { {RTX_GETOPT_STR, &cmdCorrectDev, "ip:port", }, RTX_GETOPT_END_ARG }
  },
  {"-message", "GPS Message list [gpggs|gpllk|gpvtg|gpgsv]" ,
   { {RTX_GETOPT_STR, &cmdOptions, "message" }, RTX_GETOPT_END_ARG }
  },
  {"-ascii", "Print GPS messages",
   { {RTX_GETOPT_SET, &statDev.interactive, ""}, RTX_GETOPT_END_ARG }
  },
  {"-check-gps-comms", "Check GPS Communications",
   { {RTX_GETOPT_SET, &statDev.gpsComms, ""}, RTX_GETOPT_END_ARG }
  },
  {"-check-gps-status", "Print GPS Status",
   { {RTX_GETOPT_SET, &statDev.gpsStatus, ""}, RTX_GETOPT_END_ARG }
  },
  {"-check-dgps-status", "Print DGPS status",
   { {RTX_GETOPT_SET, &statDev.dgpsStatus, ""}, RTX_GETOPT_END_ARG }
  },
  RTX_GETOPT_END
};

char * gpslogHelpStr = "CSIRO GPS logging application";

int processNMEA( void *arg );
int processDGPS( void *arg );
int processGPSCOMMS( void *arg );
int processHUP( void *arg );
int process_cmdline( int argc, char *argv[] );
char *get_portBaud( char *devString, int *baud );
char *get_addrPort( char *devString, int *port );
static void gpslog_reconfig( int signo );
int init_ublox( void );
int init_sr530( void );
int init_rtk( void );


int main( int argc, char *argv[] )
{
  int cnt, errs;

  /* initialise some defaults */
  statDev.nmeadev = "/dev/ttyS0";
  statDev.dgpsdev = "/dev/ttyS0";
  statDev.nmeabaud = 4800;
  statDev.dgpsbaud = 4800;
  statDev.dgps = 0;
  statDev.multiAddr = "244.0.10.12";
  statDev.multiPort = 2210;
  
  statDev.ddxOn = 0;
  statDev.file = 0;
  statDev.interactive = 0;
  statDev.gpsStatus = 0;
  statDev.gpsComms = 0;
  statDev.dgpsStatus = 0; 

  rtx_main_init ("gpslog", RTX_ERROR_STDERR | RTX_ERROR_MESSAGE);
    
  /* process command line */
  if ((errs = RTX_GETOPT_CMD (gpslogOpts, argc, argv, rcsid, gpslogHelpStr)) <= 0) {
    if (errs == -1)
      RTX_GETOPT_PRINT (gpslogOpts, argv[0], rcsid, gpslogHelpStr);
    exit (1);
  }    

  /* post process commandline options */
  process_cmdline( argc, argv );
  fprintf( stderr, "GPS %s %d\n", statDev.nmeadev, statDev.nmeabaud );
  if( statDev.dgps )
    {
      fprintf( stderr, "DGPS %s %d\n", statDev.dgpsdev, statDev.dgpsbaud );
      fprintf( stderr, "MultiCast %s %d\n", statDev.multiAddr, statDev.multiPort );
    }

  /* initialise ddx */
  if( statDev.ddxOn )
    {
      if( ddx_client_init(statDev.verbose) == -1 )
	return( rtx_error("gpslog: ddx_client_init() failed") );

      if( (statDev.storeId=ddx_store_open(NULL,0,5)) == NULL )
	return( rtx_error("gpslog: ddx_store_open() failed") );

      if( (DDX_STORE_REGISTER_TYPE(statDev.storeId, GPS_UTC_TIME)) == -1 )
	return( rtx_error("gpslog: ddx_store_register_type() failed") );
    }

  /* open GPS serial port */
  if( (statDev.verbose & 1) )
    fprintf( stderr, "Opening GPS port: %s @ %d baud\n", statDev.nmeadev, statDev.nmeabaud );
  if( (statDev.nmeafp=rtx_serial_open(statDev.nmeadev,
			       statDev.nmeabaud,8,1,
			       RTX_SERIAL_PARITY_NONE,
			       RTX_SERIAL_FLOW_NONE,
			       RTX_SERIAL_MODEM_OFF,
			       0, 20)) < 0 )
    return( rtx_error("gpslog: could NOT open GPS serial port\n") );

  /* initialise GPS device */
  switch( statDev.devType )
    {
    case GPS_DEVTYPE_UBLOX:
      init_ublox();
      break;
    case GPS_DEVTYPE_SR530:
      init_sr530();
      break;
    case GPS_DEVTYPE_RTK:
      init_rtk();
      break;
    case GPS_DEVTYPE_PC104:
    case GPS_DEVTYPE_TRIMBLE:
    case GPS_DEVTYPE_GENERIC:
    default:
      break;
    }

  /* open DGPS serial port */
  if( statDev.dgps )
    {
      if( (statDev.verbose &1) )
	fprintf( stderr, "Opening DGPS port: %s @ %d baud\n", statDev.dgpsdev, statDev.dgpsbaud );
      if( strcmp(statDev.nmeadev,statDev.dgpsdev) != 0 )
	{
	  if( (statDev.dgpsfp=rtx_serial_open(statDev.dgpsdev,
					      statDev.dgpsbaud,8,1,
					      RTX_SERIAL_PARITY_NONE,
					      RTX_SERIAL_FLOW_NONE,
					      RTX_SERIAL_MODEM_OFF,
					      0, 20)) < 0 )
	    return( rtx_error("gpslog: could NOT open DGPS serial port\n") );
	}
      else
	statDev.dgpsfp = statDev.nmeafp;
    }

  /* initialise processing routines */
  for( cnt = 0; cnt < 256; cnt++ )
    {
      if( gps_mesgs[cnt].name == NULL )
	break;
      
      if( (statDev.verbose & 1) )
	fprintf( stderr, "Initialising: %s\n", gps_mesgs[cnt].name );

      if( gps_mesgs[cnt].initProg( &statDev ) < 0 )
	return( rtx_error("gpslog: did not init") );
    }

  /* start main processing routine */
  if( statDev.gpsComms )
    {  
      if( (statDev.verbose & 1) )
	fprintf( stderr, "gpslog: launching GPS diagnostic thread\n" );
      if ((process_thread = rtx_thread_create ("processThread", 0, 
					       RTX_THREAD_SCHED_OTHER, 0, 0, 
					       RTX_THREAD_CANCEL_ASYNCHRONOUS,
					       (void *(*) (void *)) processGPSCOMMS, 
					       (void *)&statDev,
					       NULL, NULL )) == NULL) 
	return( rtx_error("gpslog: thread_create() failed") );
    }
  else
    {  
      if( (statDev.verbose & 1) )
	fprintf( stderr, "gpslog: launching main processing thread\n" );
      if ((process_thread = rtx_thread_create ("processThread", 0, 
					       RTX_THREAD_SCHED_OTHER, 0, 0, 
					       RTX_THREAD_CANCEL_ASYNCHRONOUS,
					       (void *(*) (void *)) processNMEA, 
					       (void *)&statDev,
					       NULL, NULL )) == NULL) 
	return( rtx_error("gpslog: thread_create() failed") );
    }
  
  if( statDev.dgps )
    {
      if( (statDev.verbose & 1) )
	fprintf( stderr, "gpslog: launching DGPS thread\n" );
      if ((dgps_thread = rtx_thread_create ("dgpsThread", 0, 
					    RTX_THREAD_SCHED_OTHER, 0, 0, 
					    RTX_THREAD_CANCEL_ASYNCHRONOUS,
					    (void *(*) (void *)) processDGPS, 
					    (void *)&statDev,
					    NULL, NULL )) == NULL) 
	return( rtx_error("gpslog: thread_create() dgps failed") );
    }
  

  /* initialise -HUP signal processing */
  signal( SIGHUP, gpslog_reconfig );

  /* wait for kill */
  if (rtx_main_wait_shutdown (KILLER_PRIO)) 
    {
      rtx_error_flush( "Here" );
      rtx_message_warning ("gpslog: main: killer_wait() failed");
    }

  /* cleanup reset module if SR530 unit */
  if (statDev.devType == GPS_DEVTYPE_SR530)
    done_reset(&statDev);

  if( statDev.dgps )
    {
      if (rtx_thread_destroy_sync (dgps_thread)) 
	{
	  rtx_message_warning ("gpslog: thread_destroy(dgps) failed");
	}
    }


  if (rtx_thread_destroy_sync (process_thread)) 
    {
      rtx_message_warning ("gpslog: thread_destroy() failed");
    }

  /* shutdown processing routines */
  for( cnt = 0; cnt < 256; cnt++ )
    {
      if( gps_mesgs[cnt].name == NULL )
	break;
      
      if( (statDev.verbose & 1) )
	fprintf( stderr, "Shutting down: %s\n", gps_mesgs[cnt].name );

      if( gps_mesgs[cnt].doneProg( &statDev ) < 0 )
	return( rtx_error("gpslog: did not init") );
    }

  if (statDev.ddxOn) 
    {
      ddx_store_close( statDev.storeId );
      ddx_client_done ();
    }

  if( statDev.file )
    fclose( statDev.fileId );

  rtx_serial_close( statDev.nmeafp );
  rtx_serial_close( statDev.dgpsfp );

  return( 0 );
}



/*****************/
int processNMEA( void *arg )
{
  int cnt;
  char mesgType[10];
  char chBuff[256];
  RtxNMEA currMesg;
  time_t nowTime;
  struct tm *now;
  int yesterday;

  /* open log file if required */
  if( statDev.file )
    {
      /* initialise yesterday date to move log files */
      nowTime = time( NULL );
      now = localtime( &nowTime );
      yesterday = now->tm_mday;

      /* open GPS log file */
      sprintf(chBuff, "%s/%02d%02d%02d.gps", cmdDir, 1900 + now->tm_year, now->tm_mon, now->tm_mday );
      if( (statDev.verbose & 1) )
	fprintf( stderr, "Opening log file: %s\n", chBuff );
      statDev.fileId = fopen( chBuff, "a+" );
      if( statDev.fileId == NULL )
	return( rtx_error("gpslog: Could NOT open GPS log file") );
    }

  while( 1 )
    {
      if( rtx_nmea_read(statDev.nmeafp,&currMesg) >= 0 )
	{
	  /* Get local time */
	  rtx_time_get(&statDev.nmeaTime);

	  /* NMEA message recieved */
	  rtx_nmea_extract( &currMesg, 0, mesgType );

	  /* if verbose == 2, display all GPS messages */
	  if( (statDev.verbose & 2) )
	    fprintf( stderr, "%s\n", currMesg.nmeaMesg );

	  /* process message */
	  for( cnt = 0; cnt < 256; cnt++ )
	    {
	      if( gps_mesgs[cnt].name == NULL )
		break;

	      if( (strcmp(mesgType,gps_mesgs[cnt].name) == 0) && (gps_mesgs[cnt].enable) )
		{
		  gps_mesgs[cnt].processProg( &statDev, &currMesg );
		  break;
		}
	      /*	      else
		{
		  rtx_nmea_extract( &currMesg, 1, nmeaType );
		  rtx_nmea_extract( &currMesg, 2, packType );
		  rtx_nmea_extract_int( &currMesg, 3, &ackType );
		  fprintf( stderr, "Rejected packet: %s %s %s %d\n", mesgType, nmeaType, packType, ackType );
		  } */
	    }

	  /* check if log file need changing */
	  if( statDev.file )
	    {
	      /* check if date has changed */
	      nowTime = time( NULL );
	      now = localtime( &nowTime );
	      if( now->tm_mday != yesterday )
		{
		  /* if day has changed, close and reopen new log file */
		  fclose( statDev.fileId );
		  
		  /* open GPS log file */
		  sprintf(chBuff, "%s/%02d%02d%02d.gps", cmdDir, 1900 + now->tm_year, now->tm_mon, now->tm_mday );
		  if( (statDev.verbose & 1) )
		    fprintf( stderr, "Opening log file: %s\n", chBuff );
		  if( (statDev.fileId=fopen(chBuff,"w+")) == NULL )
		    return( rtx_error("gpslog: Could NOT open GPS log file") );	  

		  /* update time flag */
		  nowTime = time( NULL );
		  now = localtime( &nowTime );
		  yesterday = now->tm_mday;
		}
	    }

	  /* check if SRPC port is needed */
	  if( statDev.srpc )
	    {
	      
	    }
	}
    }
}



/****************************/
int processDGPS( void *arg )
{
  char buf[1024];
  int n = 32, ret;
  RtxInetEndpoint src;

  /* open multicast port */
  if( (statDev.verbose & 1) )
    fprintf( stderr, "Opening multicast address %s port %d...", statDev.multiAddr, statDev.multiPort );
  if( (statDev.inetFd=rtx_inet_init(RTX_INET_UDP_MULTICAST,
				    NULL,0,
				    "239.255.10.0", statDev.multiPort,
				    NULL,NULL,NULL)) == NULL )
    {
      fprintf( stderr, "\n ########### DGPS error #############\n" );
      return( rtx_error("dpgs_server: rtx_inet_init() failed") );
    }
  if( (statDev.verbose & 1) )
    fprintf( stderr, "OK\n" );
  
  while( 1 )
    {
      if( (ret=rtx_inet_read(statDev.inetFd->sock, buf, n, &src)) == -1) 
	fprintf (stderr, "rtx_inet_read");
      else 
	write( statDev.dgpsfp, buf, n );

      /* print DGPS messages if required */
      if ( statDev.dgpsStatus ) 
	fprintf(stderr, "DGPS correction %s\n", buf );
    }
}



/****************************/
int processGPSCOMMS( void *arg )
{  
  RtxNMEA currMesg;
  
  while( 1 )
    {
#ifdef ZZ
      switch(statDev.devType)
	{
	case GPS_DEVTYPE_SR530:
	  /* request status for Lieca SR530 gps */
	  fprintf( stderr, "Sending\n" );
	  rtx_nmea_write( statDev.nmeafp, "PLEIQ", "BAT", "2", "200", NULL );
	  break;
	}
#endif
      if( rtx_nmea_read(statDev.nmeafp,&currMesg) >= 0 )
	{
	  /* Get local time */
	  rtx_time_get(&statDev.nmeaTime);

	  fprintf( stderr, "%s\n", currMesg.nmeaMesg );
	}
    }
}



/*********************/
int process_cmdline( int argc, char *argv[] )
{
  int i, j, def;
  char optCmd[256];

  statDev.verbose = rtx_getopt_get_verbose(0);
  statDev.ddxOn = rtx_getopt_get_store(0);

  if( statDev.verbose & 4 )
    statDev.interactive = 1;

  /************************/
  /* set GPS device types */
  if( strcmp(cmdType,"pc104") == 0 ) 
      statDev.devType = GPS_DEVTYPE_PC104;
  else if( strcmp(cmdType,"sr530") == 0 )
      statDev.devType = GPS_DEVTYPE_SR530;
  else if( strcmp(cmdType,"rtk") == 0 )
      statDev.devType = GPS_DEVTYPE_RTK;
  else if( strcmp(cmdType,"ublox") == 0 )
      statDev.devType = GPS_DEVTYPE_UBLOX;
  else if( strcmp(cmdType,"trimble") == 0 )
      statDev.devType = GPS_DEVTYPE_GENERIC;
  else if( strcmp(cmdType,"generic") == 0 )
      statDev.devType = GPS_DEVTYPE_GENERIC;
  else
    {
      fprintf( stderr, "Unknown option %s\n", cmdType );
      RTX_GETOPT_PRINT( gpslogOpts, argv[0], rcsid, gpslogHelpStr );
      exit( -1 );
    }
  
  /**********************************************/
  /* adjust parameters if diagnostics requested */
  if( statDev.gpsComms || statDev.dgpsStatus || statDev.gpsStatus )
    {
      statDev.interactive = 0;
      cmdOptions = "gpgga,gpgsv";
    }

  if( statDev.file )
    {
      cmdOptions = "gpgga,gpgsv";
      statDev.fileDir = strdup( cmdDir );
    }

  /************************************************/
  /* enable GPS message specified in command line */
  j = 0;
  for( i = 0; i <= strlen(cmdOptions); i++ )
    {
      if( (cmdOptions[i]==',') || (i==strlen(cmdOptions)) )
	{
	  optCmd[j] = '\0';   /* terminate string */

	  /* check if message is 'LIST' */
	  if( strcmp(optCmd,"LIST") == 0 )
	    {
	      fprintf( stderr, "\nGPS Messages defined: " );
	      for( j = 0; j < 256; j++ )
		{
		  if( gps_mesgs[j].name == NULL )
		    break;
		  fprintf( stderr, "%s ", gps_mesgs[j].name );
		}
	      fprintf( stderr, "\n" );
	      exit( 1 );
	    }

	  /* check if GPS message is defined */	  
	  def = 0;
	  for( j = 0; j < 256; j++ )
	    {
	      if( gps_mesgs[j].name == NULL )
		break;

	      if( strcmp(optCmd,gps_mesgs[j].name) == 0 )
		{
		  gps_mesgs[j].enable = 1;
		  def = 1;
		}
	    }
	  if( def == 0 )
	    {
	      exit( 1 );
	    }
	  
	  j = 0;
	}
      else
	{
	  optCmd[j] = toupper(cmdOptions[i]);
	  j++;
	}
    }
  

  /* get device ports and baud rates */
  statDev.nmeadev = get_portBaud( cmdNMEADev, &statDev.nmeabaud );
  statDev.dgpsdev = get_portBaud( cmdDGPSDev, &statDev.dgpsbaud );
  statDev.multiAddr = get_addrPort( cmdCorrectDev, &statDev.multiPort );

  /* add logic for DGPS diagnostics */
  if( (statDev.dgpsStatus == 1) && (statDev.dgps==0) )
    {
      fprintf( stderr, "#\n#DGPS NOT enabled !!!\n#\n" );
      RTX_GETOPT_PRINT (gpslogOpts, argv[0], rcsid, gpslogHelpStr);
      exit (1);
    }

  return( 0 );
}



/********************/
char *get_portBaud( char *devString, int *baud )
{
  int i, j;
  char chBuff[256];
  char *chPnt, *wordPnt, *device;

  chPnt = devString;
  wordPnt = devString;

  /** Get device name **/
  /* insert '/dev/' if not specified in command line */
  if( chPnt[0] != '/' )
    strcpy( chBuff, "/dev/" );
  else
    strcpy( chBuff, "" );
  
  /* search for ':' to indicate end of device name */
  for( i = 0; i < strlen(devString); i++ )
    {
      if( *chPnt == ':' )
	break;

      chPnt++;
    }

  strncat( chBuff, wordPnt, i );
  device = strdup( chBuff );

  /** Get baud rate **/
  if( *chPnt == ':' )
    {
      chPnt++;
      strcpy( chBuff,"");
      for( j = i+1; j < strlen(devString); j++ )
	{
	  strncat(chBuff,chPnt,1);
	  chPnt++;
	}

      *baud = atoi(chBuff);
    }

  return( device );
}


/*****************/
char *get_addrPort( char *devString, int *port )
{
  int i, j;
  char chBuff[256];
  char *chPnt, *wordPnt, *device;

  chPnt = devString;
  wordPnt = devString;

  /** Get device name **/
  strcpy( chBuff, "" );
  
  /* search for ':' to indicate end of device name */
  for( i = 0; i < strlen(devString); i++ )
    {
      if( *chPnt == ':' )
	break;

      chPnt++;
    }

  strncat( chBuff, wordPnt, i );
  device = strdup( chBuff );

  /** Get baud rate **/
  if( *chPnt == ':' )
    {
      chPnt++;
      strcpy( chBuff,"");
      for( j = i+1; j < strlen(devString); j++ )
	{
	  strncat(chBuff,chPnt,1);
	  chPnt++;
	}

      *port = atoi(chBuff);
    }

  return( device );
}



/****************************************/
static void gpslog_reconfig( int signo )
{
  fprintf( stderr, "Processing HUP signal\n" );

  /* initialise GPS device */
  switch( statDev.devType )
    {
    case GPS_DEVTYPE_UBLOX:
      init_ublox();
      break;
    case GPS_DEVTYPE_SR530:
      init_sr530();
      break;
    case GPS_DEVTYPE_RTK:
      init_rtk();
      break;
    case GPS_DEVTYPE_PC104:
    case GPS_DEVTYPE_TRIMBLE:
    case GPS_DEVTYPE_GENERIC:
    default:
      break;
    }
}

/****************/
int init_ublox( void )
{
  if( (statDev.verbose & 1) )
    fprintf( stderr, "Initialise uBLOX\n" );

  /* setup to send GGA and VTG NMEA messages */
  rtx_nmea_write( statDev.nmeafp, "PSRF103", "0", "0", "1", "1", NULL );
  rtx_nmea_write( statDev.nmeafp, "PSRF103", "5", "0", "1", "1", NULL );

  /* setup DGPS port */
  rtx_nmea_write( statDev.nmeafp, "PSRF102", "4800", "8", "1", "0", NULL );

  return( 0 );
}



/******************/
int init_sr530( void )
{
  int i;

  if( (statDev.verbose & 1) )
    fprintf( stderr, "Initialising SR530\n" );

  /* setup to send NMEA messages */
  for( i = 0; i < 256; i++ ) {
    if( gps_mesgs[i].name == NULL )
      break;

    /* GGA */
    if (!strcmp(gps_mesgs[i].name, "GPGGA")) {
      if (gps_mesgs[i].enable)
        rtx_nmea_write( statDev.nmeafp, "PLEIQ", "GGA", "2", "1", NULL );
      else
        rtx_nmea_write( statDev.nmeafp, "PLEIQ", "GGA", "2", "0", NULL );
    }

    /* VTG */
    if (!strcmp(gps_mesgs[i].name, "GPVTG")) {
      if (gps_mesgs[i].enable)
        rtx_nmea_write( statDev.nmeafp, "PLEIQ", "VTG", "2", "1", NULL );
      else
        rtx_nmea_write( statDev.nmeafp, "PLEIQ", "VTG", "2", "0", NULL );
    }

    /* GSV */
    if (!strcmp(gps_mesgs[i].name, "GPGSV")) {
      if (gps_mesgs[i].enable)
        rtx_nmea_write( statDev.nmeafp, "PLEIQ", "GSV", "2", "1", NULL );
      else
        rtx_nmea_write( statDev.nmeafp, "PLEIQ", "GSV", "2", "0", NULL );
    }

    /* LLK */
    if (!strcmp(gps_mesgs[i].name, "GPLLK")) {
      if (gps_mesgs[i].enable)
        rtx_nmea_write( statDev.nmeafp, "PLEIQ", "LLK", "2", "1", NULL );
      else
        rtx_nmea_write( statDev.nmeafp, "PLEIQ", "LLK", "2", "0", NULL );
    }

  }

  /* run reset module */
  init_reset(&statDev);

  return( 0 );
}



/******************/
int init_rtk( void )
{
  if( (statDev.verbose & 1) )
    fprintf( stderr, "Initialising SR530 RTK\n" );

  /* setup to send GGA and VTG NMEA messages */
  rtx_nmea_write( statDev.nmeafp, "PLEIQ", "GGA", "2", "4", NULL );
  rtx_nmea_write( statDev.nmeafp, "PLEIQ", "VTG", "2", "4", NULL );
  rtx_nmea_write( statDev.nmeafp, "PLEIQ", "GSV", "2", "4", NULL );

  return( 0 );
}
