/*********************************************************************
 *
 * CSIRO Automation
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * www.cat.csiro.au/cmst
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 *********************************************************************/

/**
 * \file template.c
 * \brief a short comment
 * \author your name
 *
 * A detailed description of the stuff in the file
 *
 */

#include <stdio.h>
#include <string.h>
#include <time.h>

#include <ddx.h>

#include <rtx/error.h>

#include "gpslog.h"
#include "convert_eastnorth.h"

#include "gga.h"

#define INIT_GGA_NAME "gga"



char gga_name[256];
DDX_STORE_ITEM *GGA_storeItem;



int init_gga( GPSlog_stat *devStat )
{
  if( (devStat->verbose & 1) )
    fprintf( stderr, "Running GGA INIT\n" );

  if( devStat->ddxOn )
    {
      strcpy( gga_name, INIT_GGA_NAME );

      if( DDX_STORE_REGISTER_TYPE(devStat->storeId,GPS_POS) == -1 )
	return( rtx_error("init_gga: ddx_store_regieter_type() failed") );

      if( (GGA_storeItem=ddx_store_lookup_item(devStat->storeId,gga_name,"GPS_POS",0)) == NULL )
	return( rtx_error("init_gga: ddx_store_lookup_item() failed") );
    }
  return( 0 );
}



int process_gga( GPSlog_stat *devStat, RtxNMEA *mesg )
{
  GPS_POS ggaData;
  char charBuff[82];
  char dir;
  double GPStime;
  int deg;
  double minutes;
  time_t currTime;
  FILE *outFile;

  /* extract time */
  rtx_nmea_extract_double( mesg, 1, &GPStime );
  rtx_nmea_extract( mesg, 1, charBuff );
  sscanf(charBuff,"%2d%2d%lf", &(ggaData.time.hours), &(ggaData.time.mins), &(ggaData.time.secs) );
  ggaData.time.hours = ((int)GPStime / 10000);

  /* extract latitude and longitude */
  rtx_nmea_extract( mesg, 2, charBuff );
  sscanf(charBuff,"%2d%lf",&deg,&minutes);
  ggaData.latitude = deg + (minutes / 60.0);
  rtx_nmea_extract( mesg, 3, &dir );
  if( dir == 'S' )
    ggaData.latitude = - ggaData.latitude;
  rtx_nmea_extract( mesg, 4, charBuff );
  sscanf(charBuff,"%3d%lf",&deg,&minutes);
  ggaData.longitude = deg + (minutes / 60.0);
  rtx_nmea_extract( mesg, 5, &dir );
  if( dir == 'W' )
    ggaData.longitude = - ggaData.longitude;
  rtx_nmea_extract_short( mesg, 7, &ggaData.noSats );

  /* extract GPS fix indicator */
  rtx_nmea_extract_short( mesg, 6, &ggaData.gpsFIX );

  /* exract GPS horizontal dilution of pression */
  rtx_nmea_extract_float( mesg, 8, &ggaData.HDOP );

  /* extract altitude */
  rtx_nmea_extract_double( mesg, 9, &ggaData.altitude );

  /* extract DGPS information */
  rtx_nmea_extract_short( mesg, 13, &ggaData.dgpsage );
  rtx_nmea_extract_short( mesg, 14, &ggaData.dgpsref );

  /* convert latitude and longitude to easting northing */
  if( (ggaData.latitude != 0) && (ggaData.longitude != 0) )
    convert_to_eastnorth( ggaData.latitude, ggaData.longitude, &ggaData.easting, &ggaData.northing );

  if( devStat->ddxOn )
    {
      if( ddx_store_write(GGA_storeItem, &ggaData, NULL) == -1)
	return( rtx_error("process_gga: ddx_store_write() failed") );
    }

  if( devStat->file )
    {
      fprintf( devStat->fileId, "%d %f %f %f %d %d\n", (int)time(&currTime), ggaData.easting, ggaData.northing, ggaData.altitude, ggaData.gpsFIX, ggaData.noSats );
      fflush( devStat->fileId );
      
      sprintf( charBuff, "%s/gps_position.txt", devStat->fileDir );
      outFile = fopen( charBuff, "wt" );
      fprintf( outFile, "%f %f", ggaData.latitude, ggaData.longitude );
      fclose( outFile );
    }

  /* print messages for interactive display */
  if( devStat->interactive )
    {
      if( ggaData.gpsFIX )
	fprintf( stderr, "GGA[FIX %d]: Time: %d %d %f lat: %f long: %f easting: %f northing: %f noSats: %d\n", ggaData.gpsFIX, ggaData.time.hours, ggaData.time.mins, ggaData.time.secs, ggaData.latitude, ggaData.longitude, ggaData.easting, ggaData.northing, ggaData.noSats );
      else
	fprintf( stderr, "GGA[NOFIX]: Time: %d %d %f lat: %f long: %f noSats: %d\n", ggaData.time.hours, ggaData.time.mins, ggaData.time.secs, ggaData.latitude, ggaData.longitude, ggaData.noSats );	
    }

  /* print messages for diagnostics */
  if( devStat->dgpsStatus )
    {
      if( ggaData.gpsFIX == 0 )
	fprintf( stderr, "GPS does NOT have a fix\nAll position data invalid\n" );
      else
	{
	  if( ggaData.gpsFIX == 1 )
	    fprintf( stderr, "GPS has a fix (NO DGPS) at Time: %d %d %f lat: %f long: %f\n", ggaData.time.hours, ggaData.time.mins, ggaData.time.secs, ggaData.latitude, ggaData.longitude ); 
	  else if( ggaData.gpsFIX == 2 )
	    fprintf( stderr, "GPS has a fix (WITH DGPS) at Time: %d %d %f lat: %f long: %f DGPS: age %d sec DGPS reference station %d\n", ggaData.time.hours, ggaData.time.mins, ggaData.time.secs, ggaData.latitude, ggaData.longitude, ggaData.dgpsage, ggaData.dgpsref ); 
	}
    }
  
  return( 0 );
}

int done_gga( GPSlog_stat *devStat )
{
  if( devStat->ddxOn )
    {
      if(ddx_store_done_item(GGA_storeItem) == -1 )
	return( rtx_error("init_gga: ddx_store_done_item() failed") );
    }
  return (0);
}
