#include <rtx/error.h>
#include <ddxvideo1394/features.h>
#include <ddxvideo1394/dc1394.h>
#include <ddxvideo1394/params.h>

/**
 * Simply assigns the requested camera features
 */
static int
assign_feature_parameters (dc1394feature_info_t * feat,	/*!< The camera feature */
		int *autoOn,	/*!< Feature automatic */
		double *val	/*!< Feature value */
		)
{
	double feat_diff = feat->max - feat->min;

	if(feat->available && (int)feat_diff != 0)
	{
		/* convert to (0<->1) normalized ddx */
		*val = (feat->value - feat->min) / feat_diff;
		if(*val > 1.0) *val = 1.0;
		else if(*val < 0.0) *val = 0.0;
	} else
	{
		*val = -1;
	}

	if(feat->current_mode == DC1394_FEATURE_MODE_AUTO)
		*autoOn = 1;
	else
		*autoOn = 0;
	return (0);
}


/**
 * Just a switch statement to pick out only those features which are in the ddx
 * camera feature structure (which may be different to the full camera features)
 */
int
features_2_ddx_params (DDX_VIDEO1394_CAMERA * camera,	/*!< The camera object */
		DDX_VIDEO_1394_PARAMS * paramState	/*!< The camera feature set for the store */
		)
{
	int i;

	for (i = 0; i < DC1394_FEATURE_NUM; i++)
	{
		dc1394feature_info_t * feat = &(camera->features.feature[i]);
		switch (feat->id)
		{
			case DC1394_FEATURE_GAIN:
				assign_feature_parameters (feat,
						&(paramState->autoGain),
						&(paramState->gain));
				break;
			case DC1394_FEATURE_EXPOSURE:
				assign_feature_parameters (feat,
						&(paramState->autoExposure),
						&(paramState->exposure));
				break;
			case DC1394_FEATURE_BRIGHTNESS:
				assign_feature_parameters (feat,
						&(paramState->autoBrightness),
						&(paramState->brightness));
				break;
			case DC1394_FEATURE_SHUTTER:
				assign_feature_parameters (feat,
						&(paramState->autoShutter),
						&(paramState->shutter));
				break;
			case DC1394_FEATURE_WHITE_BALANCE:
				if(feat->available)
				{
					if(feat->current_mode == DC1394_FEATURE_MODE_AUTO)
						paramState->autoWhiteBalance = 1;
					else 
						paramState->autoWhiteBalance = 0;
					paramState->whiteBalance[0] = 
						(feat->BU_value - feat->min) / (((double)feat->max - feat->min));
					paramState->whiteBalance[1] =
						(feat->RV_value - feat->min) / (((double)feat->max - feat->min));
				} else
				{
					paramState->autoWhiteBalance = 0;
					paramState->whiteBalance[0] = -1;
					paramState->whiteBalance[1] = -1;
				}
			case DC1394_FEATURE_SHARPNESS:
				assign_feature_parameters (feat,
						&(paramState->autoSharpness),
						&(paramState->sharpness));
				break;
			case DC1394_FEATURE_TEMPERATURE:
				assign_feature_parameters (feat,
						&(paramState->autoTemperature),
						&(paramState->temperature));
				break;
			case DC1394_FEATURE_HUE:
				assign_feature_parameters (feat,
						&(paramState->autoHue),
						&(paramState->hue));
				break;
			case DC1394_FEATURE_SATURATION:
				assign_feature_parameters (feat,
						&(paramState->autoSaturation),
						&(paramState->saturation));
				break;
			case DC1394_FEATURE_GAMMA:
				assign_feature_parameters (feat,
						&(paramState->autoGamma),
						&(paramState->gamma));
				break;
			case DC1394_FEATURE_TRIGGER:
				assign_feature_parameters (feat,
						&(paramState->autoTrigger),
						&(paramState->trigger));
				break;
			case DC1394_FEATURE_TRIGGER_DELAY:
				assign_feature_parameters (feat,
						&(paramState->autoTriggerDelay),
						&(paramState->triggerDelay));
				break;
			case DC1394_FEATURE_WHITE_SHADING:
				assign_feature_parameters (feat,
						&(paramState->autoWhiteShading),
						&(paramState->whiteShading));
				break;
			case DC1394_FEATURE_FRAME_RATE:
				assign_feature_parameters (feat,
						&(paramState->autoFrameRate),
						&(paramState->frameRate));
				break;
			case DC1394_FEATURE_IRIS:
				assign_feature_parameters (feat,
						&(paramState->autoIris),
						&(paramState->iris));
				break;
			case DC1394_FEATURE_ZOOM:
				assign_feature_parameters (feat,
						&(paramState->autoZoom),
						&(paramState->zoom));
				break;
			case DC1394_FEATURE_FOCUS:
				assign_feature_parameters (feat,
						&(paramState->autoFocus),
						&(paramState->focus));
				break;
			case DC1394_FEATURE_PAN:
				assign_feature_parameters (feat,
						&(paramState->autoPan),
						&(paramState->pan));
				break;
			case DC1394_FEATURE_TILT:
				assign_feature_parameters (feat,
						&(paramState->autoTilt),
						&(paramState->tilt));
				break;
			case DC1394_FEATURE_OPTICAL_FILTER:
				assign_feature_parameters (feat,
						&(paramState->autoOpticalFilter),
						&(paramState->opticalFilter));
				break;
			case DC1394_FEATURE_CAPTURE_SIZE:
				assign_feature_parameters (feat,
						&(paramState->autoCaptureSize),
						&(paramState->captureSize));
				break;
			case DC1394_FEATURE_CAPTURE_QUALITY:
				assign_feature_parameters (feat,
						&(paramState->autoCaptureQuality),
						&(paramState->captureQuality));
				break;
			default:
				break;
		}
	}
	return (0);
}


/**
 * Checks for a change in the structure
 * \returns 0 for no change, else 1
 */
int
check_new_params (
		DDX_VIDEO1394_CAMERA * camera,	/*!< The camera object */
		DDX_VIDEO_1394_PARAMS * paramState,	/*!< Present state */
		DDX_VIDEO_1394_PARAMS * paramStatePrev	/*!< State at previous iteration */
		)
{
	int i;

	dc1394feature_info_t * feat;

/* need to scale the values to dc1394 integers */
#define TEST(fname) \
	((int)(paramState->fname * (feat->max - feat->min)) + feat->min != \
	 (int)(paramStatePrev->fname * (feat->max - feat->min)) + feat->min)

	for (i = 0; i < DC1394_FEATURE_NUM; i++)
	{

		feat = &(camera->features.feature[i]);

		switch (feat->id)
		{
			case DC1394_FEATURE_GAIN:
				if(paramState->autoGain != paramStatePrev->autoGain) return 1;
				if(TEST(gain)) return 1;
				break;
			case DC1394_FEATURE_EXPOSURE:
				if(paramState->autoExposure != paramStatePrev->autoExposure) return 1;
				if(TEST(gain)) return 1;
				break;
			case DC1394_FEATURE_BRIGHTNESS:
				if(paramState->autoBrightness != paramStatePrev->autoBrightness) return 1;
				if(TEST(brightness)) return 1;
				break;
			case DC1394_FEATURE_SHUTTER:
				if(paramState->autoShutter != paramStatePrev->autoShutter) return 1;
				if(TEST(shutter)) return 1;
				break;
			case DC1394_FEATURE_WHITE_BALANCE:
				if(paramState->autoWhiteBalance != paramStatePrev->autoWhiteBalance) return 1;
				if(TEST(whiteBalance[0])) return 1;
				if(TEST(whiteBalance[1])) return 1;
			case DC1394_FEATURE_SHARPNESS:
				if(paramState->autoSharpness != paramStatePrev->autoSharpness) return 1;
				if(TEST(sharpness)) return 1;
				break;
			case DC1394_FEATURE_TEMPERATURE:
				if(paramState->autoTemperature != paramStatePrev->autoTemperature) return 1;
				if(TEST(temperature)) return 1;
				break;
			case DC1394_FEATURE_HUE:
				if(paramState->autoHue != paramStatePrev->autoHue) return 1;
				if(TEST(hue)) return 1;
				break;
			case DC1394_FEATURE_SATURATION:
				if(paramState->autoSaturation != paramStatePrev->autoSaturation) return 1;
				if(TEST(saturation)) return 1;
				break;
			case DC1394_FEATURE_GAMMA:
				if(paramState->autoGamma != paramStatePrev->autoGamma) return 1;
				if(TEST(gamma)) return 1;
				break;
			case DC1394_FEATURE_TRIGGER:
				if(paramState->autoTrigger != paramStatePrev->autoTrigger) return 1;
				if(TEST(trigger)) return 1;
				break;
			case DC1394_FEATURE_IRIS:
				if(paramState->autoIris != paramStatePrev->autoIris) return 1;
				if(TEST(iris)) return 1;
				break;
			case DC1394_FEATURE_ZOOM:
				if(paramState->autoZoom != paramStatePrev->autoZoom) return 1;
				if(TEST(zoom)) return 1;
				break;
			case DC1394_FEATURE_FOCUS:
				if(paramState->autoFocus != paramStatePrev->autoFocus) return 1;
				if(TEST(focus)) return 1;
				break;
			case DC1394_FEATURE_PAN:
				if(paramState->autoPan != paramStatePrev->autoPan) return 1;
				if(TEST(pan)) return 1;
				break;
			case DC1394_FEATURE_TILT:
				if(paramState->autoTilt != paramStatePrev->autoTilt) return 1;
				if(TEST(tilt)) return 1;
				break;
			case DC1394_FEATURE_OPTICAL_FILTER:
				if(paramState->autoOpticalFilter != paramStatePrev->autoOpticalFilter) return 1;
				if(TEST(opticalFilter)) return 1;
				break;
			case DC1394_FEATURE_CAPTURE_SIZE:
				if(paramState->autoCaptureSize != paramStatePrev->autoCaptureSize) return 1;
				if(TEST(captureSize)) return 1;
				break;
			case DC1394_FEATURE_CAPTURE_QUALITY:
				if(paramState->autoCaptureQuality != paramStatePrev->autoCaptureQuality) return 1;
				if(TEST(captureQuality)) return 1;
				break;
			default:
				break;
		}
	}

#undef TEST

	return (0);
}

/**
 * The function that performs the camera setting.
 */
static int
set_feature_parameters (DDX_VIDEO1394_CAMERA * camera,
		int feature_index, /*!< the feature */
		int autoOn,	/*!< Requested auto state */
		int val	/*!< Requested feature value */
		)
{
	dc1394feature_info_t * feat = &(camera->features.feature[feature_index]);
	
	int auto_available = 0;
	int manual_available = 0;
	int autoOnNow;
	int i;
		
	dc1394feature_modes_t modes;
	dc1394_feature_get_modes(camera->handle, feat->id, &modes);

	for(i = 0; i < modes.num; i++)
	{
		if(modes.modes[i] == DC1394_FEATURE_MODE_AUTO)
				auto_available = 1;
		if(modes.modes[i] == DC1394_FEATURE_MODE_MANUAL)
				manual_available = 1;
	}

	if(feat->current_mode == DC1394_FEATURE_MODE_AUTO)
		autoOnNow = 1;
	else
		autoOnNow = 0;
	
	if (auto_available)
	{
		if (autoOn != autoOnNow) {
			if (dc1394_feature_set_mode
					(camera->handle, feat->id,
					 autoOn?DC1394_FEATURE_MODE_AUTO:DC1394_FEATURE_MODE_MANUAL) != DC1394_SUCCESS)
			{
				printf ("Camera %d, Couldn't set auto %s to %u\n",
						camera->bus_number,
						string_of_feature (feat->id), autoOn);
				return (-1);
			}
#if DEBUG
			printf("Camera %d, set auto %s to %d\n",
					camera->bus_number,
					string_of_feature (feat->id), autoOn);
#endif
		}
	}

	if (autoOn == 0)
	{
		if (feat->value != val) {
			if (manual_available) {
				if (val < feat->min) {
					val = feat->min;
					printf
						("Camera %d, Requsted value for %s below minimum allowable, "
						 "clipping to %u\n", camera->bus_number,
						 string_of_feature (feat->id), val);
				} else if (val > feat->max) {
					val = feat->max;
					printf
						("Camera %d, Requsted value for %s exceeds maximum allowable,"
						 "clipping to %u\n", camera->bus_number,
						 string_of_feature (feat->id), val);
				}
				if (dc1394_feature_set_value
						(camera->handle, feat->id,
						 val) != DC1394_SUCCESS) {
					printf ("Camera %d, Couldn't set manual %s to %u\n",
							camera->bus_number,
							string_of_feature (feat->id), val);
					return (-1);
				}
			}
#if DEBUG
			printf("Camera %d, set %s to %d\n",
					camera->bus_number,
					string_of_feature (feat->id), val);
#endif
		}
	}

	return (0);
}

/**
 * Basically selects which features are available and only allows operations on
 * those features, only if they have been modified...
 * This function is for DDX store changing of features
 */
int
check_and_set_parameters_from_ddx (
		DDX_VIDEO1394_CAMERA * camera,
		DDX_VIDEO_1394_PARAMS * parPrev,	/*!< The store last demand parameters */
		DDX_VIDEO_1394_PARAMS * par,	/*!< The store demand parameters */
		int verbose
		)
{
	int i,j;
	static int firstCall = 1;
	int auto_available;
	int manual_available;
	int auto_on_now;
	dc1394feature_modes_t modes;
	dc1394feature_info_t * feat;

	/* Print a message showiung availability of store requested features */
	if (firstCall && verbose)
	{
		for (i = 0; i < DC1394_FEATURE_NUM; i++)
		{
			feat = &(camera->features.feature[i]);
			manual_available = 0;
			auto_available = 0;
			dc1394_feature_get_modes(camera->handle, feat->id, &modes);
			for(j = 0; j < modes.num; j++)
			{
				if(modes.modes[j] == DC1394_FEATURE_MODE_AUTO)
						auto_available = 1;
				if(modes.modes[j] == DC1394_FEATURE_MODE_MANUAL)
						manual_available = 1;
			}

			if (!auto_available)
				printf ("\tAuto %s feature not available!\n",
						string_of_feature (i + DC1394_FEATURE_MIN));
			if (!manual_available)
				printf ("\tManual %s feature not available!\n",
						string_of_feature (i + DC1394_FEATURE_MIN));
		}

		firstCall = 0;
		printf (" ****************\n");
	}

	for (i = 0; i < DC1394_FEATURE_NUM; i++)
	{
#define PAR_AUTO(fname) par->fname

/* scale values to DC1394 integer values */
#define PAR_SCALED(p, fname) \
		(feat->min + ((int)(p->fname*((double)(feat->max - feat->min)))))

#define TEST(fname) \
		(PAR_SCALED(par, fname)  != PAR_SCALED(parPrev, fname))

#define TEST_AUTO(autofname) \
		(par->autofname != parPrev->autofname)

#define NEEDSET(fname,autofname) \
		(TEST(fname) || TEST_AUTO(autofname))

#define TEST_AND_SET(fname,autofname) \
		if(feat->available && NEEDSET(fname,autofname)) {\
			if (verbose) \
				printf("\nSet %s to auto %d value %d ",#fname,\
					PAR_AUTO(autofname),PAR_SCALED(par, fname));\
			if (set_feature_parameters(camera,i,\
						PAR_AUTO(autofname), PAR_SCALED(par, fname))) \
			{ return 1; } \
		}

		feat = &(camera->features.feature[i]);
		manual_available = 0;
		auto_available = 0;
		dc1394_feature_get_modes(camera->handle, feat->id, &modes);
		for(j = 0; j < modes.num; j++)
		{
			if(modes.modes[j] == DC1394_FEATURE_MODE_AUTO)
					auto_available = 1;
			if(modes.modes[j] == DC1394_FEATURE_MODE_MANUAL)
					manual_available = 1;
		}

		if(feat->current_mode == DC1394_FEATURE_MODE_AUTO)
			auto_on_now = 1;
		else
			auto_on_now = 0;

		dc1394feature_info_t * feat = 
			&(camera->features.feature[i]);
		dc1394feature_t fid = feat->id; 
		switch (fid)
		{
			case DC1394_FEATURE_BRIGHTNESS:
				TEST_AND_SET(brightness,autoBrightness);
				break;
			case DC1394_FEATURE_EXPOSURE:
				TEST_AND_SET(exposure,autoExposure);
				break;
			case DC1394_FEATURE_GAIN:
				TEST_AND_SET(gain,autoGain);
				break;
			case DC1394_FEATURE_SHUTTER:
				TEST_AND_SET(shutter,autoShutter);
				break;
			case DC1394_FEATURE_WHITE_BALANCE:	/* A special case as it has 2 parameters */
				if (TEST(autoWhiteBalance) || 
						TEST(whiteBalance[0]) || TEST(whiteBalance[1])) {
					if (auto_available)
					{
						if (auto_on_now != PAR_AUTO(autoWhiteBalance))
							if (dc1394_feature_set_mode (camera->handle, 
										fid, PAR_AUTO(autoWhiteBalance)?DC1394_FEATURE_MODE_AUTO:DC1394_FEATURE_MODE_MANUAL) !=
									DC1394_SUCCESS)
							{
								printf ("Couldn't set auto %s to %u\n",
										string_of_feature (fid),
										PAR_AUTO(autoWhiteBalance));
								return (-1);
							}
					}
					if (par->autoWhiteBalance == 0)
					{
						if (manual_available)
						{
							/* clip whitebalance between 0 and 1 */
							if(par->whiteBalance[0] < 0.0)	par->whiteBalance[0] = 0.0;
							if(par->whiteBalance[0] > 1.0)	par->whiteBalance[0] = 1.0;
							if(par->whiteBalance[1] < 0.0)	par->whiteBalance[1] = 0.0;
							if(par->whiteBalance[1] > 1.0)	par->whiteBalance[1] = 1.0;
							
							if ((feat->BU_value != PAR_SCALED(par, whiteBalance[0])) ||
								(feat->RV_value != PAR_SCALED(par, whiteBalance[1])))
							{
								if (dc1394_feature_whitebalance_set_value(camera->handle, 
											PAR_SCALED(par, whiteBalance[0]),
											PAR_SCALED(par, whiteBalance[1])) !=
										DC1394_SUCCESS)
								{
									printf
										("Couldn't set manual %s to bu = %d and br = %d\n",
										 string_of_feature (fid),
										 PAR_SCALED(par, whiteBalance[0]), 
										 PAR_SCALED(par, whiteBalance[1]));
									return (-1);
								}
							}
						}
					}
				}
				break;
			case DC1394_FEATURE_SHARPNESS:
				TEST_AND_SET(sharpness,autoSharpness);
				break;
			case DC1394_FEATURE_TEMPERATURE:
				TEST_AND_SET(temperature,autoTemperature);
				break;
			case DC1394_FEATURE_HUE:
				TEST_AND_SET(hue,autoHue);
				break;
			case DC1394_FEATURE_SATURATION:
				TEST_AND_SET(saturation,autoSaturation);
				break;
			case DC1394_FEATURE_GAMMA:
				TEST_AND_SET(gamma,autoGamma);
				break;
			case DC1394_FEATURE_TRIGGER:
				TEST_AND_SET(trigger,autoTrigger);
				break;
			case DC1394_FEATURE_IRIS:
				TEST_AND_SET(iris,autoIris);
				break;
			case DC1394_FEATURE_ZOOM:
				TEST_AND_SET(zoom,autoZoom);
				break;
			case DC1394_FEATURE_FOCUS:
				TEST_AND_SET(focus,autoFocus);
				break;
			case DC1394_FEATURE_PAN:
				TEST_AND_SET(pan,autoPan);
				break;
			case DC1394_FEATURE_TILT:
				TEST_AND_SET(tilt,autoTilt);
				break;
			case DC1394_FEATURE_OPTICAL_FILTER:
				TEST_AND_SET(opticalFilter,autoOpticalFilter);
				break;
			case DC1394_FEATURE_CAPTURE_SIZE:
				TEST_AND_SET(captureSize,autoCaptureSize);
				break;
			case DC1394_FEATURE_CAPTURE_QUALITY:
				TEST_AND_SET(captureQuality,autoCaptureQuality);
				break;

			default:
				break;
		}
#undef TEST_AND_SET
#undef NEEDSET
#undef TEST_AUTO
#undef TEST
#undef PAR_SCALED
#undef PAR_AUTO
	}
	return 0;
}
