#include <string.h>
#include <stdlib.h>

#include <dc1394/control.h>
#include <ddxvideo1394/features.h>
#include <ddxvideo1394/const.h>

#ifdef USE_MARLIN_TOOLS
#include <ddxvideo1394/marlin.h>
#endif

/**
 * Convert a camera feature enum into a string 
 * debugging/display purpose
 */

const char * string_of_feature (int feature)
{
	feature -= DC1394_FEATURE_MIN;
	if ((feature < 0) || (feature>DC1394_FEATURE_NUM))
		return NULL;
	return feature_name_list[feature];
}

int  feature_of_string (const char * feature)
{
	int i;
	for (i=0;i<DC1394_FEATURE_NUM;i++) {
		if (strcasecmp(feature,feature_name_list[i]))
			return DC1394_FEATURE_MIN + i;
	}
	return -1;
}

/**
 * Return the value of the config variable 
 * associated with a camera feature enum
 */
int auto_option (const DDX_VIDEO1394_CONFIG * prm, int feature)
{
	switch (feature)
	{
		case DC1394_FEATURE_BRIGHTNESS:
			return prm->auto_brightness;
		case DC1394_FEATURE_EXPOSURE:
			return prm->auto_exposure;
		case DC1394_FEATURE_SHARPNESS:
			return prm->auto_sharpness;
		case DC1394_FEATURE_WHITE_BALANCE:
			return prm->auto_white_balance;
		case DC1394_FEATURE_HUE:
			return prm->auto_hue;
		case DC1394_FEATURE_SATURATION:
			return prm->auto_saturation;
		case DC1394_FEATURE_GAMMA:
			return prm->auto_gamma;
		case DC1394_FEATURE_SHUTTER:
			return prm->auto_shutter;
		case DC1394_FEATURE_GAIN:
			return prm->auto_gain;
		case DC1394_FEATURE_IRIS:
			return prm->auto_iris;
		case DC1394_FEATURE_FOCUS:
			return prm->auto_focus;
		case DC1394_FEATURE_TEMPERATURE:
			return prm->auto_temperature;
		case DC1394_FEATURE_TRIGGER:
			return prm->auto_trigger;
		case DC1394_FEATURE_TRIGGER_DELAY:
			return prm->auto_trigger_delay;
		case DC1394_FEATURE_WHITE_SHADING:
			return prm->auto_white_shading;
		case DC1394_FEATURE_FRAME_RATE:
			return prm->auto_frame_rate;
		case DC1394_FEATURE_ZOOM:
			return prm->auto_zoom;
		case DC1394_FEATURE_PAN:
			return prm->auto_pan;
		case DC1394_FEATURE_TILT:
			return prm->auto_tilt;
		case DC1394_FEATURE_OPTICAL_FILTER:
			return prm->auto_optical_filter;
		case DC1394_FEATURE_CAPTURE_SIZE:
			return prm->auto_capture_size;
		case DC1394_FEATURE_CAPTURE_QUALITY:
			return prm->auto_capture_quality;
		default:
			return 0;
	}
	return 0;
}

int set_feature (DDX_VIDEO1394_CAMERA * camera, 
		char *string, int len, int argc, char *argv[])
{
	dc1394feature_t feature = (dc1394feature_t)-1;
	unsigned int min, max;
	unsigned int value;

	if (strcmp (argv[0], "brightness") == 0) {
		feature = DC1394_FEATURE_BRIGHTNESS;
	}
	if (strcmp (argv[0], "exposure") == 0) {
		feature = DC1394_FEATURE_EXPOSURE;
	}
	if (strcmp (argv[0], "saturation") == 0) {
		feature = DC1394_FEATURE_SATURATION;
	}
	if (strcmp (argv[0], "sharpness") == 0) {
		feature = DC1394_FEATURE_SHARPNESS;
		dc1394_feature_get_boundaries (camera->handle, 
				DC1394_FEATURE_SHARPNESS, &min, &max);
	}
	if (feature == -1) return 1;
	
	if (argc > 0)
	{
		if (argv[1][0] == 'a')
			// Should check Has Auto Mode 
			dc1394_feature_set_mode (camera->handle, 
					feature, DC1394_FEATURE_MODE_AUTO);
		else
		{
			// Should check Has Manual Mode 
			dc1394_feature_set_mode (camera->handle, 
					feature, DC1394_FEATURE_MODE_MANUAL);
			dc1394_feature_set_value (camera->handle, 
					feature, atoi (argv[1]));
		}
	}

#if 1
	dc1394_feature_get_boundaries (camera->handle, 
			DC1394_FEATURE_SHARPNESS, &min, &max);
	dc1394_feature_get_value (camera->handle, feature, &value);
	printf("%s (%d,%d) %d\n", argv[0], min,max,value);
#endif
	// snprintf(string, len, "(%d,%d) %d\n", min,max,value);
	return 0;
}

int set_white_balance (DDX_VIDEO1394_CAMERA * camera,
		char *string, int len, int argc, char *argv[])
{
	dc1394error_t r;
	unsigned int u, v;

	switch (argc)
	{
		case 1:
			r = dc1394_feature_set_mode (camera->handle, 
					DC1394_FEATURE_WHITE_BALANCE, DC1394_FEATURE_MODE_AUTO);
			DC1394_WRN(r,"set_white_balance");
			break;

		case 3:
			r = dc1394_feature_set_mode (camera->handle, 
					DC1394_FEATURE_WHITE_BALANCE, DC1394_FEATURE_MODE_MANUAL);
			DC1394_WRN(r,"set_white_balance");
			r = dc1394_feature_whitebalance_set_value (camera->handle, 
					atoi (argv[1]), atoi (argv[2]));
			DC1394_WRN(r,"set_white_balance");
			break;
	}
#if 1
	dc1394_feature_whitebalance_get_value (camera->handle, &u, &v);
	printf ("White Balance %d %d\n", u, v);
#endif
	return 0;
}

/**
 * Use all auto_* variables to set required 
 * camera feature to auto, if supported
 */

void set_camera_features_auto (DDX_VIDEO1394_CAMERA * camera,
		DDX_VIDEO1394_CONFIG * config)
{
	int i;
	int feature;
	int available;
	int something_auto = config->all_features_auto;

	// First test if something is to be set to auto
	for (feature = 0; feature < DC1394_FEATURE_NUM; feature++)
		something_auto = something_auto || 
			auto_option (config,DC1394_FEATURE_MIN + feature);

	if (!something_auto)
		return;

	// Get feature set informations
	if (dc1394_feature_get_all(camera->handle, 
				&camera->features) != DC1394_SUCCESS)
	{
		printf ("Could not get camera feature information!\n");
		return;
	}

	for (feature = 0; feature < DC1394_FEATURE_NUM; feature++)
	{
		int value;
		dc1394feature_info_t * feat = 
			&(camera->features.feature[feature]);
		dc1394feature_modes_t * modes;
		dc1394_feature_get_modes(camera->handle, feat->id, modes);

		available = 0;
		for(i = 0; i < modes->num; i++)
		{
			if(	modes->modes[i] == DC1394_FEATURE_MODE_MANUAL||
				modes->modes[i] == DC1394_FEATURE_MODE_AUTO)
					available = 1;
		}

		if(available)
		{
			value = config->all_features_auto || auto_option (config,feature+DC1394_FEATURE_MIN);
			if (dc1394_feature_set_mode (camera->handle, 
					(dc1394feature_t)(DC1394_FEATURE_MIN + feature), value?DC1394_FEATURE_MODE_AUTO:DC1394_FEATURE_MODE_MANUAL) !=
					DC1394_SUCCESS)
			{
				printf ("Could not set auto mode for feature %s",
						feature_name_list[feature]);
				continue;
			}
			printf ("  Feature %s set to %s mode\n",
					feature_name_list[feature],
					value ? "AUTO" : "MANUAL");
		}
	}
}

#ifdef USE_MARLIN_TOOLS

/**
 * Manage MARLIN options if supported
 */

int manage_marlin_options (DDX_VIDEO1394_CAMERA * camera,
		DDX_VIDEO1394_CONFIG * config)
{
	int shdg_res = 0;
	if (LUT_has_functionality(camera) == ROK) {
		if (config->lut_table_str != NULL)
		{
			LUT_load_from_file (camera, config);
		}
		else
		{
			LUT_activate (camera, config, 0, 0);
		}
	}

	if (shading_has_functionality(camera) == ROK) {
		if (config->auto_shading)
		{
			shdg_res = shading_build_image (camera, config, 20, 20);
		}
		else if (config->shading_image_str != NULL)
		{
			shdg_res = shading_load_image (camera, config);
		}
		else
		{
			shading_activate (camera, config, 0);
		}

		if (shdg_res != ROK)
			return 1;
#if 0
		shading_show_image (camera, config, 1);
#else
		shading_activate (camera, config, 1);
#endif
	}
	return 0;
}
#endif

