/** \file     
  Copyright (c) CSIRO ICT Robotics Centre
  \brief       Capture dc1394 video and copy to store using the ddx++ video class and libdc1394-v2
  \author      Elliot Duff, Cedric Pradalier, Stephen Nuske 

  \todo   Write SRPC function to return properties so that tkddxvideo is updated. 
 */

static char *rcsid = "";

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>		// Needed by perror()
#include <string.h>		// Needed strerror()

#include <rtx/message.h>    // Real Time Extensions 
#include <rtx/time.h>
#include <rtx/timer.h>
#include <rtx/error.h>
#include <rtx/thread.h>
#include <rtx/main.h>
#include <rtx/getopt.h>
#include <rtx/srpc.h>

#include <ddx.h>
#include <ddxvideo1394/config.h>
#include <ddxvideo1394/params.h>
#include <ddxvideo1394/camera.h>
#include <ddxvideo1394/srpc.h>
#include <ddxvideo1394/dc1394.h>
#include <ddxvideo1394/features.h>
#include <ddxvideo1394/const.h>
#include <ddxvideo1394/convertion.h>
#include <ddxvideo1394/ddxvideo.h>

#ifndef RTX_THREAD_PRIO_MIN
// for backward compatibility
#define RTX_THREAD_PRIO_MIN 0
#endif

#ifndef DOXYGEN_SHOULD_SKIP_THIS


DDX_STORE_ID *storeId = NULL;	// Pointer to Store 
DDX_STORE_ITEM *videoPtr = NULL;	// Pointer to Video  
DDX_STORE_ITEM *ctrlPtr = NULL;	// Pointer to Control  
DDX_STORE_ITEM *framePtr = NULL;	// Pointer to Frame Number 

DDX_VIDEO_CONTROL ctrl;
DDX_VIDEO *video;

DDX_STORE_ITEM **paramDmdPtr = NULL;	// Pointer to parameter demands
DDX_STORE_ITEM **paramStatePtr = NULL;	// Pointer to parameter current state

int done = 0;			// Flag to stop Capture Thread 

DDX_VIDEO_1394_PARAMS *paramDmd;
DDX_VIDEO_1394_PARAMS *paramState;
DDX_VIDEO_1394_PARAMS *paramStatePrev;
unsigned int paramSize;

DDX_VIDEO1394_CONFIG config;
DDX_VIDEO1394_CAMERA * cameras = NULL;

RtxMutex * camera_capture_mutex = NULL;
RtxMutex * params_waiting_mutex = NULL;

#endif

int frame_handler(int camera_index, unsigned char * buffer, 
		unsigned int width, unsigned int height, void * arg)
{
	video = (DDX_VIDEO*)arg;
	unsigned long size =  width*height;
	unsigned char * yp = (unsigned char*)(video->y) + camera_index*size;
	unsigned char * up = (unsigned char*)(video->u) + camera_index*size/4;
	unsigned char * vp = (unsigned char*)(video->v) + camera_index*size/4;

	if (config.bayermode > 0) {
		bayer_frame_to_yuv420p(buffer, config.coding[camera_index], config.bayermode, yp, up, vp, width, height);
	} else { 
		if(frame_to_yuv420p(buffer, config.coding[camera_index], yp, up, vp, width, height))
		{
			return rtx_error("Problem converting frame to yuv420p");
		}
	}
	return 0;
}


/**
 * Capture thread : loops forever, capturing 
 * video frames, and storing them in the Store
 */
	void *
capture_thread (void *arg)
{
	unsigned int j;
	int count = 0;
	int frame = 0;
	RtxTime ts;
	
	//////////////////////////////////////////
	//
	// Into the loop....
	
	while (!done)
	{
		ctrl.pause=0;
		ddx_store_read(ctrlPtr,&ctrl,NULL,0,0);
		rtx_time_get (&ts);

		if (!ctrl.pause) {

			if (config.dots)
			{
				printf (".");
				fflush (stdout);
			}
			count++;
			
			rtx_mutex_lock(camera_capture_mutex);
			if (capture_1394_cameras(cameras,config.num_captured_cameras,
						&config, frame_handler, video)) {
				fprintf(stderr,"Camera capture failed\n");
				done = 1;
			}
			rtx_mutex_unlock(camera_capture_mutex);
		} else {
			rtx_timer_sleep(0.1);
		}

		if (done) break;

		if (!config.nostore)
		{
			if (!ctrl.pause) {
				if (config.nodirect) {
					if (ddx_store_write(videoPtr,video, NULL) == -1)
						rtx_error ("ddx_store_write: %s", strerror (errno));
				} else {
					if (ddx_store_write_direct (videoPtr, NULL) == -1)
						rtx_error ("ddx_store_write: %s", strerror (errno));
				}
				if (ddx_store_write (framePtr, &frame, NULL) == -1)
					rtx_error ("ddx_store_write: %s", strerror (errno));
			}

			////////////////////////////////////////////
			//
			// Deal with camera parameters...
			rtx_mutex_lock(params_waiting_mutex);
			/* Read the current state of the camera */
			if (update_1394_feature_set(cameras,config.num_captured_cameras))
			{
				printf ("Could not get camera feature information!");
				done = 1;
			}
			rtx_mutex_unlock(params_waiting_mutex);
			
			for (j=0;j<config.num_captured_cameras;j++) {
				/* Assign to the store structure and check against previous state to
				 * determine whether a write to the store is required */
				features_2_ddx_params (cameras+j, paramState+j);
				if (check_new_params (cameras+j,paramState+j, paramStatePrev+j))
				{
					if (ddx_store_write (paramStatePtr[j], paramState+j, NULL))
					{
						rtx_message ("Store write failed (camera parameters)!");
						done = 1;
					}
					if (config.dots)
						printf ("]");
				}
			}
			memcpy(paramStatePrev,paramState,paramSize);

		}
		if (config.verbose)
			fprintf (stderr, "%d %ld %ld\r", frame, ts.seconds, ts.nanoSeconds);
		if (frame == config.limit)
			rtx_main_signal_shutdown ();

	}
	rtx_main_signal_shutdown ();
	done = -1;
	return (NULL);
}


/**
 * Update thread : loops forever, capturing 
 * video parameter requests from the store and executing those requests.
 */
	void *
update_thread (void *arg)
{
	unsigned int j = (unsigned int)arg;
	RtxTime ts;
	DDX_VIDEO_1394_PARAMS paramDmdPrev;
	
	while (!done)
	{
		/* Blocking read on parameter demand */
		switch (ddx_store_read (paramDmdPtr[j], paramDmd+j, &ts, 0, 1)) {
			case -1:
				fprintf (stderr,
						"Camera parameter demand read failed badly, exiting!");
				fflush (stderr);
				done = 1;
				continue;
			case +1:
				continue;
			default:
				break;
		}
		if (config.dots)
			printf ("*");

		rtx_mutex_lock(params_waiting_mutex);
		rtx_mutex_lock(camera_capture_mutex);
		if (config.dots)
			printf ("[");
		if(check_and_set_parameters_from_ddx (cameras+j,
					&paramDmdPrev, paramDmd+j,config.verbose) )
		{
			rtx_error_flush("couldn't check and set camera parameters");
			done = 1;
		}
		rtx_mutex_unlock(camera_capture_mutex);
		rtx_mutex_unlock(params_waiting_mutex);

		/* copy demand structure */
		memcpy(&paramDmdPrev,paramDmd+j,sizeof(DDX_VIDEO_1394_PARAMS));
	}

	rtx_main_signal_shutdown ();
	done = -1;
	return (NULL);
}


int main (int ac, char * av[])
{
	/* Useless but I hate warnings */
		
	unsigned int j;
	RtxThread *th = NULL;
	RtxThread **updateThread = NULL;
	RtxSrpc *srpcServerHandle;

	config_init(&config);
	config_parse(&config,rcsid,ac,(const char**)av);
	if (config.verbose)
		config_print(&config,stdout);

	rtx_main_init (av[0], 0);

	if (config.list)
	{
		// Manage the -list option : have a look to each camera
		// Get the number of ports (cards) 

		list_1394_cameras(&config);
		return 0;
	}

	cameras = (DDX_VIDEO1394_CAMERA*)
		malloc(config.num_captured_cameras*sizeof(DDX_VIDEO1394_CAMERA));
	if (init_1394_cameras(cameras,config.num_captured_cameras, &config)){
		fprintf(stderr,"Camera initialization failed. Aborting\n");
		return -1;
	}

	// Finding the biggest camera image
	unsigned int width=0, height=0;
	for (j=0;j<config.num_captured_cameras;j++) {
		unsigned int cam_width = cameras[j].width;
		unsigned int cam_height = cameras[j].height;

		// MONO16 mode than we need to allocate double the memory
		if(config.coding[j] == DC1394_COLOR_CODING_MONO16)
			cam_height*=2;
		
		if (cameras[j].width>width) width = cam_width;
		if (cameras[j].height>height) height = cam_height;
	}

	init_srpc_server(&srpcServerHandle,&config,
			cameras,config.num_captured_cameras);

	// Setup DDX Store
	char controlName[512];
	sprintf(controlName,"%s_ctrl",config.name);
	rtx_message ("Storing video stream in '%s'", config.name);
	if (ddx_client_init (0) == -1) {
		return (rtx_error ("Unable to initialize library"));
	}
	if ((storeId = ddx_store_open (NULL, 0, 5)) == NULL) {
		return (rtx_error ("Unable to open store"));
	}
	if (DDX_STORE_REGISTER_TYPE (storeId, DDX_VIDEO) == -1) {
		return (rtx_error ("Unable to register type DDX_VIDEO"));
	}
	if (DDX_STORE_REGISTER_TYPE (storeId, DDX_VIDEO_CONTROL) == -1) {
		return (rtx_error ("Unable to register type DDX_VIDEO_CONTROL"));
	}
	if ((videoPtr = ddx_store_lookup_item (storeId, 
					config.name, "DDX_VIDEO", 0)) == NULL) {
		return (rtx_error ("Unable to lookup var"));
	}
	if (config.nodirect) {
		video = (DDX_VIDEO *) malloc(sizeof(DDX_VIDEO));
	} else {
		video = (DDX_VIDEO *) ddx_store_var_pointer (videoPtr);
		rtx_message("Using direct access, possibly asynchronous\n");
	}
	video->fields = config.num_captured_cameras;
	// All cameras assumed to have the same capture size!
	video->width = cameras[0].width;
	video->height = cameras[0].height;

	if ((ctrlPtr = ddx_store_lookup_item (storeId, controlName,
					"DDX_VIDEO_CONTROL", 0)) == NULL) {
		return (rtx_error ("Unable to lookup var '%s'",controlName));
	}
	DDX_VIDEO_CONTROL ctrl;
	ctrl.pause = config.paused;
	ddx_store_write(ctrlPtr,&ctrl,NULL);
	printf("Storing video stream in '%s' %dx%d\n",config.name,width,height);

	char frameName[512];
	sprintf(frameName,"%s_frame",config.name);
	if ((framePtr = ddx_store_lookup_item (storeId, 
					frameName, "int", 0)) == NULL)
		return (rtx_error ("Unable to lookup var"));


	// Setup camera parameter stuff through the store
	char paramDmdName[512];
	char paramStateName[512];
	paramDmdPtr = (DDX_STORE_ITEM**)
		malloc(config.num_captured_cameras*sizeof(DDX_STORE_ITEM*));
	paramStatePtr = (DDX_STORE_ITEM**)
		malloc(config.num_captured_cameras*sizeof(DDX_STORE_ITEM*));
	if (DDX_STORE_REGISTER_TYPE (storeId, DDX_VIDEO_1394_PARAMS) == -1)
		return (rtx_error ("Unable to register type"));
	for (j=0;j<config.num_captured_cameras;j++) {
		if (config.num_captured_cameras > 1) {
			sprintf (paramDmdName, "%s_paramDmd_%d", config.name,j);
			sprintf (paramStateName, "%s_paramState_%d", config.name,j);
		} else {
			sprintf (paramDmdName, "%s_paramDmd", config.name);
			sprintf (paramStateName, "%s_paramState", config.name);
		}
		if ((paramDmdPtr[j] = ddx_store_lookup_item (storeId, paramDmdName,
						"DDX_VIDEO_1394_PARAMS", 0)) == NULL)
			return (rtx_error ("Unable to lookup var"));
		if ((paramStatePtr[j] = ddx_store_lookup_item (storeId, paramStateName,
						"DDX_VIDEO_1394_PARAMS", 0)) == NULL)
			return (rtx_error ("Unable to lookup var"));
	}
	
	paramSize = config.num_captured_cameras*sizeof(DDX_VIDEO_1394_PARAMS);
	paramDmd = (DDX_VIDEO_1394_PARAMS*) (malloc(paramSize));
	paramState = (DDX_VIDEO_1394_PARAMS*) (malloc(paramSize));
	paramStatePrev = (DDX_VIDEO_1394_PARAMS*) (malloc(paramSize));
	
	if (update_1394_feature_set(cameras,config.num_captured_cameras))
	{
		printf ("Could not get camera feature information!");
		done = 1;
	}
	for (j=0;j<config.num_captured_cameras;j++) {
		/* Assign to the store structure and check against previous state to
		 * determine whether a write to the store is required */
		features_2_ddx_params (cameras+j, paramState+j);
	
		memcpy(paramDmd+j,paramState+j,sizeof(DDX_VIDEO_1394_PARAMS));
		memcpy(paramStatePrev+j,paramState+j,sizeof(DDX_VIDEO_1394_PARAMS));

		/* initialize demand in ddx */
		if (ddx_store_write (paramDmdPtr[j], paramDmd+j, NULL))
		{
			rtx_error_flush ("Store write failed (camera demanded parameters)!");
			done = 1;
		}
		/* write state to ddx */
		if (ddx_store_write (paramStatePtr[j], paramState+j, NULL))
		{
			rtx_error_flush ("Store write failed (camera state parameters)!");
			done = 1;
		}
	}

	// Setup capture & parameter thread
	camera_capture_mutex = rtx_mutex_init(NULL,RTX_MUTEX_DEFAULT,0);
	params_waiting_mutex = rtx_mutex_init(NULL,RTX_MUTEX_DEFAULT,0);

	updateThread = (RtxThread**)malloc(config.num_captured_cameras*sizeof(RtxThread*));
	for (j=0;j<config.num_captured_cameras;j++) {
		char thrname[128];
		sprintf(thrname,"Update thread %d",j);
		printf("Start %s\n", thrname);
		if ((updateThread[j] = rtx_thread_create (thrname, 0,
						RTX_THREAD_SCHED_OTHER, RTX_THREAD_PRIO_MIN, 0,
						RTX_THREAD_CANCEL_ASYNCHRONOUS,
						update_thread, (void*)j,
						NULL, NULL)) == NULL)
		{
			rtx_error_flush ("rtx_thread_create() failed");
			exit (-1);
		}
	}

	if ((th = rtx_thread_create ("capture thread", 0,
					RTX_THREAD_SCHED_OTHER, RTX_THREAD_PRIO_MIN, 0,
					RTX_THREAD_CANCEL_ASYNCHRONOUS,
					capture_thread, NULL, NULL, NULL)) == NULL)
	{
		rtx_error_flush ("rtx_thread_create() failed");
		exit (-1);
	}

	if (rtx_main_wait_shutdown (90))
		rtx_error_flush ("rtx_main_wait_shutdown() failed");

	// Terminating threads
	
	done = 1;
	usleep(100000);
	if (done > 0) rtx_thread_destroy_sync (th);
	for (j=0;j<config.num_captured_cameras;j++) {
		rtx_thread_destroy_sync (updateThread[j]);
	}
	free(updateThread);
	//rtx_mutex_destroy(camera_capture_mutex);


	// Terminating the store connection
	
	
	ddx_store_done_item (videoPtr);
	ddx_store_done_item (framePtr);
	ddx_store_done_item (ctrlPtr);
	for (j=0;j<config.num_captured_cameras;j++) {
		ddx_store_done_item (paramStatePtr[j]);
		ddx_store_done_item (paramDmdPtr[j]);
	}
	free(paramStatePtr);
	free(paramState);
	free(paramDmdPtr);
	free(paramDmd);

	ddx_store_close (storeId);
	ddx_client_done ();
	
	terminate_srpc_server(srpcServerHandle);

	terminate_1394_cameras(cameras,config.num_captured_cameras);
	free(cameras);
	
	config_terminate(&config);

	printf("\n%s terminated\n",av[0]);
	
	return 0;
}
