#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <rtx/error.h>
#include <rtx/message.h>
#include <libraw1394/raw1394.h>
#include <dc1394/control.h>
#include <dc1394/capture.h>
#include <dc1394/utils.h>

#include <ddxvideo1394/config.h>
#include <ddxvideo1394/camera.h>
#include <ddxvideo1394/const.h>
#include <ddxvideo1394/dc1394.h>
#include <ddxvideo1394/features.h>


#define MAX_CAMERAS 64
#define NUM_BUFFERS 2 

/* This definition comes from dc1394/linux/capture.h, unfortunately we can't include that header directly as it requires
 * a full build of dc1394. */
extern dc1394error_t dc1394_capture_set_device_filename(dc1394camera_t* camera, char *filename);

int list_1394_cameras(DDX_VIDEO1394_CONFIG * config)
{
	unsigned int i;
	
	dc1394_t *dc1394_context = dc1394_new();
	dc1394camera_list_t *camera_list;
	dc1394camera_t * camera;

	if (dc1394_camera_enumerate(dc1394_context, &camera_list)!=DC1394_SUCCESS) {
		return rtx_error("list_1394_cameras: can't find cameras");
	}

	//print camera info
	for (i = 0; i < camera_list->num; i++)
	{
		printf("\n------ Camera Number %d ----\n", i);
		camera = dc1394_camera_new(dc1394_context, camera_list->ids[i].guid);

		dc1394_camera_print_info(camera, stdout);
		
		printf("\n");
		dc1394_camera_free(camera);
	}

	//print  available modes
	for (i = 0; i < camera_list->num; i++)
	{
		camera = dc1394_camera_new(dc1394_context, camera_list->ids[i].guid);

		dc1394video_modes_t modes;
		dc1394framerates_t frates;
		unsigned int j,k;
		dc1394_video_get_supported_modes(camera,&modes);
		
		printf("------ Camera Number %d Available Modes ----\n", i);
		for (j=0;j<modes.num;j++) {
			if (dc1394_is_video_mode_scalable(modes.modes[j])) 
			{
				printf("\t%d: %s (variable region of interest)\n",
						modes.modes[j] - DC1394_VIDEO_MODE_MIN,
						video_mode_to_str(modes.modes[j]));
			} else {
				printf("\t%d: %s: FPS ",
						modes.modes[j] - DC1394_VIDEO_MODE_MIN, 
						video_mode_to_str(modes.modes[j]));
				dc1394_video_get_supported_framerates(camera,modes.modes[j],&frates);
				for (k=0;k<frates.num;k++) {
					printf("%s ",frame_rate_to_str(frates.framerates[k]));
				}
				printf("\n");
			}
		}
		printf("\n");
		dc1394_camera_free(camera);
	}

	dc1394_camera_free_list(camera_list);
	return 0;
}

/**
 * Use ROI variables to check that ROI request
 * is relevant and send it to the camera
 */

static void check_roi_parameters (DDX_VIDEO1394_CONFIG * config,
		DDX_VIDEO1394_CAMERA * camera)
{
	unsigned int read_top = 0, read_left = 0;
	dc1394video_mode_t mode;
	dc1394_video_get_mode(camera->handle,&mode);
	if (!dc1394_is_video_mode_scalable(mode)) {
		rtx_message ("ROI is only possible in Format7");
		return;
	}

	rtx_message ("ROI Required: pos %dx%d size %dx%d ", 
			config->left, config->top, config->width, config->height);
	dc1394_format7_get_image_position (camera->handle, 
			mode, &read_left, &read_top);

	if (config->top == DC1394_QUERY_FROM_CAMERA)
		config->top = read_top;
	if (config->left == DC1394_QUERY_FROM_CAMERA)
		config->left = read_left;
	if ((0 > config->top) || (config->top >= camera->maxheight))
	{
		config->top = DC1394_QUERY_FROM_CAMERA;
	}
	if ((0 > config->left) || (config->left >= camera->maxwidth))
	{
		config->left = DC1394_QUERY_FROM_CAMERA;
	}
	if ((config->height != DC1394_USE_MAX_AVAIL)
			&& ((0 > config->height) || 
				(config->top + config->height > camera->maxheight)))
	{
		config->height = camera->maxheight;
	}
	if ((config->width != DC1394_USE_MAX_AVAIL)
			&& ((0 > config->width) || 
				(config->left + config->width > camera->maxwidth)))
	{
		config->width = camera->maxwidth;
	}
	rtx_message ("ROI feasible: pos %dx%d size %dx%d", 
			config->left, config->top, config->width, config->height);
}


#define DC1394(arg) if (dc1394_format7_##arg != DC1394_SUCCESS) {\
	char tmp[4096]; rtx_message("dc1394: call to %s%s failed",\
			strncpy(tmp,#arg,64),(strlen(#arg)>63)?"...":" "); }

int init_1394_camera(DDX_VIDEO1394_CAMERA * camera, unsigned int index, 
		DDX_VIDEO1394_CONFIG * config)
{
	unsigned int i;
	int valid_mode = 0;
	dc1394error_t err;
	dc1394_t *dc1394_context = dc1394_new();
	dc1394camera_list_t *camera_list;
	dc1394camera_t * dc1394camera;

	if (dc1394_camera_enumerate(dc1394_context, &camera_list)!=DC1394_SUCCESS) {
		return rtx_error("init_1394_camera: can't find cameras");
	}

	dc1394camera = dc1394_camera_new(dc1394_context, camera_list->ids[index].guid);
	dc1394_camera_free_list(camera_list);
	dc1394_camera_print_info(dc1394camera, stdout);
	camera->handle = dc1394camera;

	camera->framebuffer = NULL;
	dc1394_camera_reset(camera->handle);
	dc1394_video_set_transmission(camera->handle,DC1394_OFF);


    err = dc1394_video_set_iso_speed(camera->handle, DC1394_ISO_SPEED_400);
	if (err != DC1394_SUCCESS) { DC1394_WRN(err,"init_1394_camera"); return -1; }
	dc1394video_modes_t modes;
	dc1394_video_get_supported_modes(camera->handle,&modes);
	
	for(i=0; i<modes.num; i++) {
		if(config->videomode[index] == modes.modes[i]) {
			valid_mode = 1; 
		}
	}
	
	if(!valid_mode) {
		dc1394_camera_free(camera->handle);
		return rtx_error("Camera %d, invalid video mode", camera->bus_number);
	}

    err = dc1394_video_set_mode(camera->handle,config->videomode[index]);
	if (err != DC1394_SUCCESS) { DC1394_WRN(err,"init_1394_camera"); return -1; }
	rtx_message("Selected video mode: %s",video_mode_to_str(config->videomode[index]));

	if (!dc1394_is_video_mode_scalable(config->videomode[index])) {

		dc1394framerates_t frates;
		dc1394_video_get_supported_framerates(camera->handle,config->videomode[index],&frates);
		if (config->fps[index] == 0) {
			/* try to go as fast as possible */
			config->fps[index] = frates.framerates[frates.num - 1];
		} else {
			int i,valid = 0;
			for (i=0;i<frates.num;i++) {
				if (frates.framerates[i] == config->fps[index]) valid = 1;
			}
			if (!valid) {
				dc1394_camera_free(camera->handle);
				return rtx_error("Camera %d, invalid framerate %s", camera->bus_number,
						frame_rate_to_str(config->fps[index]));
			}
		}
	    err = dc1394_video_set_framerate(camera->handle,config->fps[index]);
		if (err != DC1394_SUCCESS) { DC1394_WRN(err,"init_1394_camera"); return -1; }
		rtx_message("Selected frame rate: %s",frame_rate_to_str(config->fps[index]));
	}


	// Set required camera features to auto
	set_camera_features_auto (camera, config);

#ifdef USE_MARLIN_TOOLS
	if (config->use_marlin)
		manage_marlin_options (camera, config);
#endif

	if (dc1394_is_video_mode_scalable(config->videomode[index])) {
		uint64_t image_size;
		DC1394 (get_max_image_size
				(camera->handle, config->videomode[index], 
				 &camera->maxwidth,&camera->maxheight));
		check_roi_parameters (config,camera);
		DC1394 (set_image_position
				(camera->handle, config->videomode[index], 
				 config->left, config->top));
		DC1394 (set_image_size
				(camera->handle, config->videomode[index], 
				 config->width, config->height));

		DC1394 (get_image_position
				(camera->handle, 
				 config->videomode[index], &camera->left, &camera->top));
		DC1394 (get_image_size
				(camera->handle, config->videomode[index], 
				 &camera->width,&camera->height));
		DC1394 (get_total_bytes
				(camera->handle, config->videomode[index], &image_size));
		DC1394 (get_color_coding
				(camera->handle, config->videomode[index], &config->coding[index]));
		rtx_message ("Camera %d, \nMode 7 : max size %dx%d\n"
				"\troi pos %dx%d size %dx%d "
				"color coding %d size %lld b",
			 camera->bus_number, camera->maxwidth, camera->maxheight, 
			 camera->left, camera->top, 
			 camera->width, camera->height, 
			 config->coding, image_size);

	} else {
		camera->left = camera->top = 0;
		dc1394_get_image_size_from_video_mode(camera->handle,
				config->videomode[index], &camera->width,&camera->height);
		rtx_message ("Camera %d, %s %s", camera->bus_number,
				video_mode_to_str(config->videomode[index]),
				frame_rate_to_str(config->fps[index]));
	}

	update_1394_feature_set(camera,1);

	dc1394_get_color_coding_from_video_mode(camera->handle, config->videomode[index], &config->coding[index]);
	
	if (config->strongsync) {
		unsigned int bpp;
		dc1394_video_get_data_depth(camera->handle, &bpp);
		camera->frame_size = (unsigned int)(bpp * camera->maxwidth * camera->maxheight + 1);
		camera->framebuffer = (unsigned char*)(malloc(camera->frame_size));
		if (camera->framebuffer == NULL) {
			dc1394_camera_free(camera->handle);
			return rtx_error("Camera %d, Failed to allocate framebuffer memory\n",camera->bus_number);
		}
		if (config->verbose) {
			rtx_message("Camera %d, allocated %lld bytes (%p)",camera->bus_number,
					camera->frame_size,camera->framebuffer);
		}
	}

	dc1394_capture_set_device_filename(camera->handle, config->device);

	if (dc1394_capture_setup(camera->handle, NUM_BUFFERS, DC1394_CAPTURE_FLAGS_DEFAULT) != DC1394_SUCCESS) {
		dc1394_camera_free(camera->handle);
		return rtx_error("Camera %d, unable to setup camera, check video mode and frame rate",
				camera->bus_number);
	}

	/*have the camera start sending us data*/
	if (dc1394_video_set_transmission(camera->handle,DC1394_ON) !=DC1394_SUCCESS) {
		dc1394_capture_stop(camera->handle);
		dc1394_camera_free(camera->handle);
		return rtx_error_errno("Camera %d, unable to start camera iso transmission\n",
				camera->bus_number);
	}


	return 0;

}

int get_used_bandwidth(DDX_VIDEO1394_CAMERA * camera)
{
	unsigned int bandwidth = 0;
	dc1394_video_get_bandwidth_usage (camera->handle, &bandwidth);
	return bandwidth;
}

int init_1394_cameras(DDX_VIDEO1394_CAMERA * camera,
		unsigned int num_cameras,
		DDX_VIDEO1394_CONFIG * config)
{
	int total_bandwidth;
	int i;
	for (i=0;i<config->num_captured_cameras;i++) {
		if (i >= num_cameras) return 1;
		if (init_1394_camera(&camera[i],i, config))
			return 1;
	}

	total_bandwidth = 0;
	for (i=0;i<config->num_captured_cameras;i++) {
		total_bandwidth += get_used_bandwidth(&camera[i]);
	}
	/** Max from libdc1394 source **/
	if (total_bandwidth > 4915) {
		rtx_message("Required bandwidth (%d) is greater than available bandwidth (%d)\n"
				"\tTry adjusting fps, image size or or format/mode",total_bandwidth, 4915);
		return 1;
	}
	rtx_message("Total bandwidth required: %d",total_bandwidth);
	return 0;
}

int update_1394_feature_set(DDX_VIDEO1394_CAMERA * cameras,
		unsigned int num_cameras)
{
	int cam;
	unsigned int i, j;
	dc1394error_t err=DC1394_SUCCESS;

	for (cam=0;cam<num_cameras;cam++) {
		/* Read the current state of the camera */
		for (i= DC1394_FEATURE_MIN, j= 0; i <= DC1394_FEATURE_GAIN; i++, j++)  
		{
			cameras[cam].features.feature[j].id = i;
			if((err=dc1394_feature_get(cameras[cam].handle, &(cameras[cam].features.feature[j])))!=DC1394_SUCCESS)
			{
				rtx_message ("Camera %d, Could not get camera feature information!", cam);
				return 1;
			}
		}
  	}

	
	return 0;
}

int terminate_1394_cameras(DDX_VIDEO1394_CAMERA * camera,
		unsigned int num_cameras)
{
	int i;
	for (i=0;i<num_cameras;i++) {
		free(camera[i].framebuffer);
		camera[i].framebuffer = NULL;
		dc1394_video_set_transmission(camera[i].handle,DC1394_OFF);
		dc1394_capture_stop(camera[i].handle);
		dc1394_camera_free(camera[i].handle);
	}
	return 0;
}

int capture_1394_cameras(DDX_VIDEO1394_CAMERA * cameras,
		unsigned int num_cameras, 
		DDX_VIDEO1394_CONFIG * config,
		FrameHandler fh, void * fharg)
{
	int i;
	dc1394video_frame_t *frame[MAX_CAMERAS];

	for (i=0;i<num_cameras;i++) 
	{
		// #define POLL_IMAGE 
#if POLL_IMAGE
		switch (dc1394_capture_dequeue (cameras[i].handle,DC1394_CAPTURE_POLICY_POLL, &frame[i])) {
			case DC1394_SUCCESS:
				break;
			case DC1394_NO_FRAME: 
#endif
				if (dc1394_capture_dequeue (cameras[i].handle,DC1394_CAPTURE_POLICY_WAIT, &frame[i]) != DC1394_SUCCESS)
				{
					rtx_message ("Unable to capture a frame");
					//exit (-1);
					return 1;
				}
#if POLL_IMAGE
				break;
			default:
				return rtx_error("Failed to capture next frame");
		}
#endif
	}

	if (config->strongsync) {
		/* We need to do this copy to be sure to preserve multi-camera
		 * synchronisation. When using DMA, the capture buffer might be modified 
		 * during the processing. */
		for (i=0;i<num_cameras;i++) {
			memcpy(cameras[i].framebuffer,frame[i]->image,
					cameras[i].frame_size);
		}
	}
	
	for (i=0;i<num_cameras;i++) {
		int ret;
		if (config->strongsync) {
			ret = fh(i,cameras[i].framebuffer,
					cameras[i].width, cameras[i].height, fharg);
		} else {
			ret = fh(i,frame[i]->image,
					cameras[i].width, cameras[i].height, fharg);
		}
		if (ret) return ret;
	}

	for (i=0;i<num_cameras;i++) {
		dc1394_capture_enqueue (cameras[i].handle, frame[i]);
	}
	
	return 0;
}

