
#include <ddxvideo1394/convertion.h>
#include <ddxvideo1394/ddxvideo.h>
#include <rtx/error.h>

int frame_to_yuv420p(unsigned char *ip, dc1394color_coding_t coding, unsigned char *y,
		unsigned char *u, unsigned char *v, int w, int h) 
{
	switch (coding) {
		case DC1394_COLOR_CODING_MONO8:
			mono_to_yuv420p(ip, y,u,v, w,h);
			break;
		case DC1394_COLOR_CODING_MONO16:
			mono16_to_yuv420p(ip, y,u,v, w,h);
			break;
		case DC1394_COLOR_CODING_YUV411:
			yuv411_to_yuv420p(ip, y,u,v, w,h);
			break;
		case DC1394_COLOR_CODING_YUV422:
			yuv422_to_yuv420p(ip, y,u,v, w,h);
			break;
		case DC1394_COLOR_CODING_YUV444:
			yuv444_to_yuv420p(ip, y,u,v, w,h);
			break;
		case DC1394_COLOR_CODING_RGB8:
			rgb_to_yuv420p(ip, y,u,v, w,h);
			break;
		default:
			return rtx_error("Conversion not implemented for this packing (%d)", coding);
	}
	return 0;
}

int bayer_frame_to_yuv420p(unsigned char *ip, dc1394color_coding_t coding, dc1394color_filter_t bayer, unsigned char *y,
		unsigned char *u, unsigned char *v, int w, int h) 
{
	switch (bayer) {
		case DC1394_COLOR_FILTER_RGGB:
			if(coding == DC1394_COLOR_CODING_MONO16)
				return 
					rtx_error("Conversion not implemented for this bayer packing (%d)", bayer);
			else
				rggb_to_yuv420p(ip, y,u,v, w,h);
			break;
		case DC1394_COLOR_FILTER_BGGR:
			if(coding == DC1394_COLOR_CODING_MONO16)
				return 
					rtx_error("Conversion not implemented for this bayer packing (%d)", bayer);
			else
				bggr_to_yuv420p(ip, y,u,v, w,h);
			break;
		default:
			return rtx_error("Conversion not implemented for this bayer packing (%d)", bayer);
	}
	return 0;
}

/**
 * Convert monochrome camera capture buffers into YUV420 Planar format
 * \warning Requires a bit more thought 
 */

void mono_to_yuv420p (unsigned char *ip, unsigned char *y,
		unsigned char *u, unsigned char *v, int w, int h)
{
	int i, p = w * h;
	for (i = 0; i < p; i++)
		*y++ = ip[i];
	if (u && v)
		for (i = 0; i < p / 4; i++) {
			*u++ = 128;
			*v++ = 128;
		}
}

/**
 * Convert 16-bit monochrome camera capture buffers into YUV420 Planar format 
 * \warning Requires a bit more thought 
 */

void mono16_to_yuv420p (unsigned char *ip, unsigned char *y,
		unsigned char *u, unsigned char *v, int w, int h)
{
	int i, p = w * h*2;
	unsigned char * y_upper = y + (w * h);
	for (i = 0; i < p; i+=2)
	{
		unsigned char lower = ip[i];
		unsigned char upper = ip[i+1];
		*y++ = lower;
		*y_upper++ = upper;
	}
	if (u && v)
		for (i = 0; i < p / 4; i++) {
			*u++ = 128;
			*v++ = 128;
		}
}

/**
 * Convert RGB camera capture buffers into YUV420 Planar format
 * Date is packed as rgbrgb. 
 * \todo Map u and v 
 */

void rgb_to_yuv420p (unsigned char *ip, unsigned char *yp,
		unsigned char *up, unsigned char *vp, int w, int h)
{
	int i,j;
	for (j = 0; j < h; j += 1) {
		for (i = 0; i < w; i += 1) {
			unsigned char * p = ip + (j*w+i)*3;
			int r,g,b,y,u,v;
			r = p[0]; g = p[1]; b = p[2];
			DDXVIDEO_RGB2YUV (r, g, b, y, u, v);
			*yp++ = y;
			if (((i%2)==0) && ((j%2)==0)) {
				*up++ = v;
				*vp++ = u;
			}
		}
	}
}

/**
 * Convert YUV444 camera capture buffers into YUV420 Planar format
 * Date is packed as u0 y0 v0 u1 y1 v1 (24bits per pixel) 
 */

void yuv444_to_yuv420p (unsigned char *ip, unsigned char *yp,
		unsigned char *up, unsigned char *vp, int w, int h)
{
	int i,j;
	for (j = 0; j < h; j += 1) {
		for (i = 0; i < w; i += 1) {
			unsigned char * p = ip + (j*w+i)*3;
			*yp++ = p[1];
			if (((i%2)==0) && ((j%2)==0)) {
				*vp++ = p[0];
				*up++ = p[2];
			}
		}
	}
}

/**
 * Convert YUV422 camera capture buffers into YUV420 Planar format
 * Data is packed as u0 y0 v0 y1 u2 y2 v2 y3 (16bits per pixel)
 * Ignore colour information from every second line 
 */

void yuv422_to_yuv420p (unsigned char *ip, unsigned char *y,
		unsigned char *u, unsigned char *v, int w, int h)
{
	int i, j, l;

	for (j = 0; j < h; j++)
	{
		l = j * w * 2;
		for (i = 0; i < w * 2;)
		{
			*v++ = ip[l + i++];
			*y++ = ip[l + i++];
			*u++ = ip[l + i++];
			*y++ = ip[l + i++];
		}
		j++;			// drop even lines
		for (i = 0; i < w * 2;)
		{
			i++;
			*y++ = ip[l + i++];
			i++;
			*y++ = ip[l + i++];
		}
	}
}

/**
 * Convert YUV411 camera capture buffers into YUV420 Planar format
 * Data is packed as u0 y0 y1 v0 y2 y3 u4 y4 y5 v4 y6 y7 (12bits/pixel)
 * \warning In this mode - oversample in Horiz and subsample in Vert
 */

void yuv411_to_yuv420p (unsigned char *ip, unsigned char *y,
		unsigned char *u, unsigned char *v, int w, int h)
{
	int i, j;
	unsigned char c;

	for (j = 0; j < h; j += 2)
	{
		for (i = 0; i < w * 3 / 2; i += 6)
		{
			c = *ip++;
			*v++ = c;
			*v++ = c;
			*y++ = *ip++;
			*y++ = *ip++;
			c = *ip++;
			*u++ = c;
			*u++ = c;
			*y++ = *ip++;
			*y++ = *ip++;
		}
		for (i = 0; i < w * 3 / 2; i += 6)
		{
			ip++;
			*y++ = *ip++;
			*y++ = *ip++;
			ip++;
			*y++ = *ip++;
			*y++ = *ip++;
		}
	}
}

/**
 * Convert BGGR camera capture buffers into YUV420 Planar format
 * \todo Improve Y estimation 
 */

void bggr_to_yuv420p (unsigned char *ip, unsigned char *Y,
		unsigned char *U, unsigned char *V, int w, int h)
{
	int i, j, l;
	int r, g, b, y, u, v;

	for (i = 0; i < h; i += 2)
		for (j = 0; j < w; j += 2)
		{
			l = i * w + j;
			r = ip[l];
			g = (ip[l + 1] + ip[l + w]) / 2;
			b = ip[l + w + 1];
			DDXVIDEO_RGB2YUV (r, g, b, y, u, v);
			Y[l] = y;
			Y[l + 1] = y;
			Y[l + w] = y;
			Y[l + w + 1] = y;
			*U++ = u;
			*V++ = v;
		}
}


/**
 * Convert RGGB camera capture buffers into YUV420 Planar format
 * \todo Improve Y estimation 
 */
void rggb_to_yuv420p (unsigned char *ip, unsigned char *Y,
		unsigned char *U, unsigned char *V, int w, int h)
{
	int i, j, l;
	int r, g, b, y, u, v;

	for (i = 0; i < h; i += 2)
		for (j = 0; j < w; j += 2)
		{
			l = i * w + j;
			b = ip[l];
			g = (ip[l + 1] + ip[l + w]) / 2;
			r = ip[l + w + 1];
			DDXVIDEO_RGB2YUV (r, g, b, y, u, v);
			Y[l] = y;
			Y[l + 1] = y;
			Y[l + w] = y;
			Y[l + w + 1] = y;
			*U++ = u;
			*V++ = v;
		}
}


