#include <string.h>
#include <stdlib.h>
#include <rtx/getopt.h>
#include <rtx/message.h>
#include <libraw1394/raw1394.h>
#include <dc1394/control.h>

#include <ddxvideo1394/const.h>
#include <ddxvideo1394/config.h>

static char default_name[] = "video";
static char default_device[] = "/dev/video1394/0";
static char default_device_template[] = "/dev/video1394/%d";

int config_init(DDX_VIDEO1394_CONFIG * config)
{
	config->program = 60;
	config->nostore = 0;
	config->nodma = 0;
	config->strongsync = 0;
	config->paused = 0;
	config->nodirect = 0;
	config->stdvideo = 0;
	config->name = strdup(default_name);
	config->limit = -1;
	config->list = 0;
	config->dots = 0;
	config->port = 0;
	config->device = strdup(default_device);
	config->num_captured_cameras = 1;
	config->numbers[0] = 0;
	config->verbose = 0;
	config->coding[0] = DC1394_COLOR_CODING_YUV411;
	config->bayermode = (dc1394color_filter_t) 0;
	config->videomode[0] = DC1394_VIDEO_MODE_640x480_YUV411;
	config->dfps[0]=0.0;
	config->fps[0] = DC1394_FRAMERATE_15;
	config->top = DC1394_QUERY_FROM_CAMERA;
	config->left = DC1394_QUERY_FROM_CAMERA;
	config->width = DC1394_USE_MAX_AVAIL;
	config->height = DC1394_USE_MAX_AVAIL;
#ifdef USE_MARLIN_TOOLS
	config->use_marlin = 0;
	config->auto_shading = 0;
	config->lut_table_str = NULL;
	config->shading_image_str = NULL;
#endif
	config->all_features_auto = 0;
	config->auto_brightness = 0;
	config->auto_exposure = 0;
	config->auto_sharpness = 0;
	config->auto_white_balance = 0;
	config->auto_hue = 0;
	config->auto_saturation = 0;
	config->auto_gamma = 0;
	config->auto_shutter = 0;
	config->auto_gain = 0;
	config->auto_iris = 0;
	config->auto_focus = 0;
	config->auto_temperature = 0;
	config->auto_trigger = 0;
	config->auto_zoom = 0;
	config->auto_pan = 0;
	config->auto_tilt = 0;
	config->auto_optical_filter = 0;
	config->auto_capture_size = 0;
	config->auto_capture_quality = 0;
	config->param_time = 1.0;
	return 0;
}

int config_print(DDX_VIDEO1394_CONFIG * config,FILE * fp)
{
	unsigned int i;
	fprintf(fp,"Program: %d\n", config->program); 
	fprintf(fp,"Nostore: %d\n", config->nostore); 
	fprintf(fp,"Nodma: %d\n", config->nodma); 
	fprintf(fp,"Sync: %d\n", config->strongsync); 
	fprintf(fp,"Paused: %d\n", config->paused); 
	fprintf(fp,"Nodirect: %d\n", config->nodirect); 
	fprintf(fp,"Stdvideo: %d\n", config->stdvideo); 
	fprintf(fp,"Name: %s\n", config->name); 
	fprintf(fp,"Limit: %d\n", config->limit); 
	fprintf(fp,"List: %d\n", config->list); 
	fprintf(fp,"Dots: %d\n", config->dots); 
	fprintf(fp,"Port: %d\n", config->port); 
	fprintf(fp,"Device: %s\n", config->device); 
	fprintf(fp,"Num_captured_cameras: %d\n", config->num_captured_cameras); 
	fprintf(fp,"Numbers: [ %d ",config->numbers[0]); 
	for (i=1;i<config->num_captured_cameras;i++) 
		fprintf(fp,"%d ",config->numbers[i]);
	fprintf(fp,"]\n");
	fprintf(fp,"Fps: [ %s", frame_rate_to_str(config->fps[0])); 
	for (i=1;i<config->num_captured_cameras;i++) 
		fprintf(fp,"%s ",frame_rate_to_str(config->fps[i]));
	fprintf(fp,"]\n");
	fprintf(fp,"Mode: [ %s", video_mode_to_str(config->videomode[0])); 
	for (i=1;i<config->num_captured_cameras;i++) 
		fprintf(fp,"%s ",video_mode_to_str(config->videomode[i]));
	fprintf(fp,"]\n");
		
	fprintf(fp,"Param time: %.3fs\n", config->param_time); 
	fprintf(fp,"Verbose: %d\n", config->verbose); 
	fprintf(fp,"Bayer_mode: %d\n", config->bayermode); 
	fprintf(fp,"Top: %d\n", config->top); 
	fprintf(fp,"Left: %d\n", config->left); 
	fprintf(fp,"Width: %d\n", config->width); 
	fprintf(fp,"Height: %d\n", config->height); 
#ifdef USE_MARLIN_TOOLS
	fprintf(fp,"Use marlin: %d\n", config->use_marlin); 
	fprintf(fp,"Auto_shading: %d\n", config->auto_shading); 
	fprintf(fp,"Lut_table_str: %s\n", config->lut_table_str); 
	fprintf(fp,"Shading_image_str: %s\n", config->shading_image_str); 
#endif
	fprintf(fp,"All_features_auto: %d\n", config->all_features_auto); 
	fprintf(fp,"Auto_brightness: %d\n", config->auto_brightness); 
	fprintf(fp,"Auto_exposure: %d\n", config->auto_exposure); 
	fprintf(fp,"Auto_sharpness: %d\n", config->auto_sharpness); 
	fprintf(fp,"Auto_white_balance: %d\n", config->auto_white_balance); 
	fprintf(fp,"Auto_hue: %d\n", config->auto_hue); 
	fprintf(fp,"Auto_saturation: %d\n", config->auto_saturation); 
	fprintf(fp,"Auto_gamma: %d\n", config->auto_gamma); 
	fprintf(fp,"Auto_shutter: %d\n", config->auto_shutter); 
	fprintf(fp,"Auto_gain: %d\n", config->auto_gain); 
	fprintf(fp,"Auto_iris: %d\n", config->auto_iris); 
	fprintf(fp,"Auto_focus: %d\n", config->auto_focus); 
	fprintf(fp,"Auto_temperature: %d\n", config->auto_temperature); 
	fprintf(fp,"Auto_trigger: %d\n", config->auto_trigger); 
	fprintf(fp,"Auto_zoom: %d\n", config->auto_zoom); 
	fprintf(fp,"Auto_pan: %d\n", config->auto_pan); 
	fprintf(fp,"Auto_tilt: %d\n", config->auto_tilt); 
	fprintf(fp,"Auto_optical_filter: %d\n", config->auto_optical_filter); 
	fprintf(fp,"Auto_capture_size: %d\n", config->auto_capture_size); 
	fprintf(fp,"Auto_capture_quality: %d\n", config->auto_capture_quality); 
	return 0;
}

/** Parse the -number option **/
static void interprete_number_string(DDX_VIDEO1394_CONFIG * config,
		const char * number_str, const char * vm_str, const char * fps_str )
{
	const char * str = number_str;
	char sep;
	unsigned int i, pos;
	unsigned int num,numv,numf;
	
	numv = numf = 0;
	config->num_captured_cameras = 0;

	if (number_str == NULL) {
		config->num_captured_cameras = 1;
		config->numbers[0] = 0;
	} else
	{
		str = number_str;
		while (1) {
			int r = sscanf(str,"%u %c %n",&num,&sep,&pos);
			if (r < 1) break;
			if (num < 64) 
				config->numbers[config->num_captured_cameras++] = num;
			if (sep != ',') break;
			str = str + pos;
			if (config->num_captured_cameras >= 64) break;
		}
		if (config->num_captured_cameras == 0) {
			rtx_message("Warning, could not interprete camera numbers. Using default");
			config->num_captured_cameras = 1;
			config->numbers[0] = 0;
		}
		if (config->num_captured_cameras > 1) {
			char snum[128],str[256];
			sprintf(str,"%d",config->numbers[0]);
			for (num=1;num<config->num_captured_cameras;num++) {
				sprintf(snum,",%d",config->numbers[num]);
				strcat(str,snum);
			}
			rtx_message("Capturing multiple cameras %s",str);
		} else {
			rtx_message("Capturing single camera %d",config->numbers[0]);
		}
	}


	/*** parsing the video mode string ***/
	if (vm_str != NULL) {
		str = vm_str;
		while (1) {
			int r = sscanf(str,"%u %c %n",&num,&sep,&pos);
			if (r < 1) break;
			config->videomode[numv++] = (dc1394video_mode_t)(num + DC1394_VIDEO_MODE_MIN);
			if (sep != ',') break;
			str = str + pos;
			if (numv >= 64) break;
		}
	}
	switch (numv) {
		case 0 : 
			config->videomode[0] = DC1394_VIDEO_MODE_640x480_YUV411;
			/* slip through */
		case 1:
			for (i=1;i<config->num_captured_cameras;i++) {
				config->videomode[i] = config->videomode[0];
			}
			break;
		default:
			for (i=numv;i<config->num_captured_cameras;i++) {
				config->videomode[i] = DC1394_VIDEO_MODE_640x480_YUV411;
			}
	}

	/*** parsing the fps string ***/
	if (fps_str != NULL) {
		double dnum = 0.0;
		str = fps_str;
		while (1) {
			int r = sscanf(str,"%lf %c %n",&dnum,&sep,&pos);
			if (r < 1) break;
			if (dnum >= 0) 
				config->dfps[numf++] = dnum;
			if (sep != ',') break;
			str = str + pos;
			if (numf >= 64) break;
		}
	}
	switch (numf) {
		case 0 : 
			config->dfps[0] = 0.0;
			/* slip through */
		case 1:
			for (i=1;i<config->num_captured_cameras;i++) {
				config->dfps[i] = config->dfps[0];
			}
			break;
		default:
			for (i=numf;i<config->num_captured_cameras;i++) {
				config->dfps[i] = 0.0;
			}
	}

}


int config_parse(DDX_VIDEO1394_CONFIG * config,
		const char * rcsid, int argc, const char * argv[])
{
	char * vm_str = NULL;
	char * fps_str = NULL;
	char * number_str = NULL;
	char * bayer_str = NULL;

	RtxGetopt myOpts[] = {
		{"nostore", "No Store", {{RTX_GETOPT_SET, &(config->nostore), "nostore"}
									, RTX_GETOPT_END_ARG}
		} ,
		{"port", "Raw1394 port", {{RTX_GETOPT_INT, &(config->port), "port"}
									 , RTX_GETOPT_END_ARG}
		} ,
		{"device", "Select video device", {{RTX_GETOPT_STR, &(config->device), "device"}
											  , RTX_GETOPT_END_ARG}
		} ,
		{"number", "Camera Number(s) in bus ", {{RTX_GETOPT_STR, &number_str, "number(s)"}
												   , RTX_GETOPT_END_ARG}
		} ,
		{"mode", "Camera Mode (from -list), separated by comas", 
			{{RTX_GETOPT_STR, &vm_str, "mode"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"fps", "Frame rate (1.875,3.75,7.5,15,30,60,120,240), separated by comas",
			{{RTX_GETOPT_STR, &fps_str, "framerate"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"name", "Name of video object in store",
			{{RTX_GETOPT_STR, &(config->name), "name"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"limit", "Run until frame equals limit", {{RTX_GETOPT_INT, &(config->limit), "limit"}
													  , RTX_GETOPT_END_ARG}
		} ,
#if 0  // now in the export function
		{"srpc", "SRPC Program number", {{RTX_GETOPT_INT, &(config->program), "program"}
											, RTX_GETOPT_END_ARG}
		} ,
#endif
		{"nodirect", "Don't use direct access to the store", {{RTX_GETOPT_SET, &(config->nodirect), "nodirect"}
																 , RTX_GETOPT_END_ARG}
		} ,
#if 0 // Obsolete in libdc1394 v2
		{"nodma", "Don't use dma 1394 ability", {{RTX_GETOPT_SET, &(config->nodma), "nodma"}
													, RTX_GETOPT_END_ARG}
		} ,
#endif
		{"sync", "Enforce multi-frame synchronisation, and prevent dma overwriting video buffer while processing it", {{RTX_GETOPT_SET, &(config->strongsync), "sync"}
													, RTX_GETOPT_END_ARG}
		} ,
		{"paused", "Start the driver in paused mode", {{RTX_GETOPT_SET, &(config->paused), "paused"}
													, RTX_GETOPT_END_ARG}
		} ,
		{"stdvideo", "Use old standard DDX_VIDEO structure", {{RTX_GETOPT_SET, &(config->stdvideo), "stdvideo"}
													, RTX_GETOPT_END_ARG}
		} ,
#ifdef USE_MARLIN_TOOLS
		{"marlin", "activate marlin-specific functionalities", {{RTX_GETOPT_SET, &(config->use_marlin), "use_marlin"}
													, RTX_GETOPT_END_ARG}
		} ,
		{"lut", "LUT definition file",
			{{RTX_GETOPT_STR, &(config->lut_table_str), "fileName"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"shading", "Shading PGM image",
			{{RTX_GETOPT_STR, &(config->shading_image_str), "fineName.pgm"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"build_shading", "Request camera to build its own shading image",
			{{RTX_GETOPT_SET, &(config->auto_shading), "bool"}
				, RTX_GETOPT_END_ARG}
		} ,
#endif
		{"left", "Left position of ROI window ",
			{{RTX_GETOPT_INT, &(config->left), "left"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"top", "Top position of ROI window ", {{RTX_GETOPT_INT, &(config->top), "top"}
												   , RTX_GETOPT_END_ARG}
		} ,
		{"width", "Width of ROI window ", {{RTX_GETOPT_INT, &(config->width), "width"}
											  , RTX_GETOPT_END_ARG}
		} ,
		{"height", "Height of ROI window ", {{RTX_GETOPT_INT, &(config->height), "height"}
												, RTX_GETOPT_END_ARG}
		} ,
		{"list", "List available cameras", {{RTX_GETOPT_SET, &(config->list), "bool"}
											   , RTX_GETOPT_END_ARG}
		} ,
		{"bayer", "Interprete data as bayer mosaicing",
			{{RTX_GETOPT_STR, &bayer_str, "(BGGR|RGGB)"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"dots", "Output a dot for each captured frame",
			{{RTX_GETOPT_SET, &(config->dots), "bool"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"auto_all", "Try to set every features to auto",
			{{RTX_GETOPT_SET, &(config->all_features_auto), "bool"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"auto_brightness", "Set brightness to auto",
			{{RTX_GETOPT_SET, &(config->auto_brightness), "bool"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"auto_exposure", "Set exposure to auto",
			{{RTX_GETOPT_SET, &(config->auto_exposure), "bool"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"auto_sharpness", "Set sharpness to auto",
			{{RTX_GETOPT_SET, &(config->auto_sharpness), "bool"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"auto_white_balance", "Set white_balance to auto",
			{{RTX_GETOPT_SET, &(config->auto_white_balance), "bool"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"auto_hue", "Set hue to auto", 
			{{RTX_GETOPT_SET, &(config->auto_hue), "hue"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"auto_saturation", "Set saturation to auto",
			{{RTX_GETOPT_SET, &(config->auto_saturation), "bool"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"auto_gamma", "Set gamma to auto",
			{{RTX_GETOPT_SET, &(config->auto_gamma), "bool"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"auto_shutter", "Set shutter to auto",
			{{RTX_GETOPT_SET, &(config->auto_shutter), "bool"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"auto_gain", "Set gain to auto", 
			{{RTX_GETOPT_SET, &(config->auto_gain), "bool"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"auto_iris", "Set iris to auto", 
			{{RTX_GETOPT_SET, &(config->auto_iris), "bool"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"auto_focus", "Set focus to auto",
			{{RTX_GETOPT_SET, &(config->auto_focus), "bool"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"auto_temperature", "Set temperature to auto",
			{{RTX_GETOPT_SET, &(config->auto_temperature), "bool"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"auto_trigger", "Set trigger to auto",
			{{RTX_GETOPT_SET, &(config->auto_trigger), "bool"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"auto_delay_trigger", "Set trigger delay to auto",
			{{RTX_GETOPT_SET, &(config->auto_trigger_delay), "bool"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"auto_frame_rate", "Set frame rate to auto",
			{{RTX_GETOPT_SET, &(config->auto_frame_rate), "bool"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"auto_white_shading", "Set white shading to auto",
			{{RTX_GETOPT_SET, &(config->auto_white_shading), "bool"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"auto_zoom", "Set zoom to auto", 
			{{RTX_GETOPT_SET, &(config->auto_zoom), "bool"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"auto_pan", "Set pan to auto", 
			{{RTX_GETOPT_SET, &(config->auto_pan), "bool"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"auto_tilt", "Set tilt to auto", 
			{{RTX_GETOPT_SET, &(config->auto_tilt), "bool"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"auto_optical_filter", "Set optical_filter to auto",
			{{RTX_GETOPT_SET, &(config->auto_optical_filter), "bool"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"auto_capture_size", "Set capture_size to auto",
			{{RTX_GETOPT_SET, &(config->auto_capture_size), "bool"}
				, RTX_GETOPT_END_ARG}
		} ,
		{"auto_capture_quality", "Set capture_quality to auto",
			{{RTX_GETOPT_SET, &(config->auto_capture_quality), "bool"}
				, RTX_GETOPT_END_ARG}
		} ,
		RTX_GETOPT_END
	};

	char *help =
		"CSIRO Video Server Project\n"
		"Capture dc1394 video and copy to store";

	if (RTX_GETOPT_CMD (myOpts, argc, (char **)argv, (char*)rcsid, help) == -1)
	{
		RTX_GETOPT_PRINT (myOpts, (char*)argv[0], (char*)rcsid, help);
		exit (-1);
		return -1;
	}
	config->program = rtx_getopt_get_export (60);
	config->verbose = rtx_getopt_get_verbose(0);

	interprete_number_string(config,number_str, vm_str, fps_str);

	if (config->list) {
		return 0;
	}

	if (config->device == NULL) {
		char tmp[128];
		sprintf(tmp,default_device_template,config->port);
		config->device = strdup(tmp);
	}

	unsigned int i;
	for (i=0;i<config->num_captured_cameras;i++) {
		if (config->dfps[i] == 0.0) {
			config->fps[i] = DC1394_FRAMERATE_15;
		} else if (config->dfps[i] == 1.875) {
			config->fps[i] = DC1394_FRAMERATE_1_875;
		} else if (config->dfps[i] == 3.75) {
			config->fps[i] = DC1394_FRAMERATE_3_75;
		} else if (config->dfps[i] == 7.5) {
			config->fps[i] = DC1394_FRAMERATE_7_5;
		} else if (config->dfps[i] == 15) {
			config->fps[i] = DC1394_FRAMERATE_15;
		} else if (config->dfps[i] == 30) {
			config->fps[i] = DC1394_FRAMERATE_30;
		} else if (config->dfps[i] == 60) {
			config->fps[i] = DC1394_FRAMERATE_60;
		} else if (config->dfps[i] == 120) {
			config->fps[i] = DC1394_FRAMERATE_120;
		} else if (config->dfps[i] == 240) {
			config->fps[i] = DC1394_FRAMERATE_240;
		} else  {
			rtx_message("For camera %d, frame rate '%f' is not a valid 1394 frame rate",i,config->dfps[i]);
			config->fps[i] = DC1394_FRAMERATE_15;
		}
		//printf("%d : %f %d %s\n",i,config->dfps[i],config->fps[i],
		//		frame_rate_to_str(config->fps[i]));
	}
	// Bayer Mosaicing
	config->bayermode = (dc1394color_filter_t)0;
	if (bayer_str != NULL) {
		if (strcasecmp (bayer_str, "BGGR") == 0)
		{
			config->bayermode = DC1394_COLOR_FILTER_BGGR;
		}
		if (strcasecmp (bayer_str, "RGGB") == 0)
		{
			config->bayermode = DC1394_COLOR_FILTER_RGGB;
		}
	}
	free(number_str);
	free(bayer_str);
	free(vm_str);
	free(fps_str);
	
	return 0;
}

int config_terminate(DDX_VIDEO1394_CONFIG * config)
{
	free(config->name);
	free(config->device);
#ifdef USE_MARLIN_TOOLS
	free(config->lut_table_str);
	free(config->shading_image_str);
#endif
	return 0;
}

int config_set_name(DDX_VIDEO1394_CONFIG * config, const char * name)
{
	free(config->name);
	config->name = strdup(name);
	return 0;
}

int config_set_device(DDX_VIDEO1394_CONFIG * config, const char * dev)
{
	free(config->device);
	config->device = strdup(dev);
	return 0;
}




