
/** \file
Copyright (c) CSIRO
\brief   Decode UDP video stream and display using SDL
\author  Elliot.Duff@csiro.au
\todo	 Launch seperate thread to make it easier to kill.
\todo    Be more aware of keyframe (GOP) for display and recording. 
\todo	 AVI recording does not have correct frame-rate - get from header
\warning Recording will overwrite existing file
\warning Since this program listens for UDP multicast packets, it must be 
run on a seperate machine to the encoder. 

This program decodes UDP video stream and displays the image to the screen.
With conventional video streaming (Quicktime and ASF) there is continual
negotiation between the encoder and decoder. Although this is essential to 
syncronize video and audio components, there is a performance hit.
Decode does makes not attempt to syncronize the video stream, it simply 
waits for the appriopriate header information to establish the correct
size and codec for the display. 

To record the incoming video stream, type r in the window. A red bar along the
top signifies that recording has started. Type r to stop recording. Type r 
to recommence. No attempt is made to correctly handle the key frames. 

If you want to fix up the resulting file with

\code
mencoder -ocv copy -o fixed.avi decode.avi
\endcode

If you want to use the video in a powerpooint presentation you will
need to convert it to an older codec, such as MPEG1.

\code
ffmpeg -an -i fixed.avi fixed.mpg
\endcode

*/

static char    *rcsid =
	"$Header$";

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>

#include <libavcodec/avcodec.h>
#include <libavformat/avformat.h>

#include <rtx/inet.h>
#include <rtx/message.h>
#include <rtx/time.h>
#include <rtx/timer.h>
#include <rtx/error.h>
#include <rtx/main.h>
#include <rtx/getopt.h>

#include <SDL.h>

#define HDSIZE 20

int             port = 8000;
char           *filename = "decode.avi";

int             codecID = 16;
int             codecGOP = 12;
int             codecBitrate = 200;
int             codecTolerance = 100;
int             codecBuffer = 50000;

RtxGetopt       myOpts[] = {
	{"codec", "Set Codec Id",
	 {{RTX_GETOPT_INT, &codecID, "codecID"}, RTX_GETOPT_END_ARG}},
	{"port", "UDP Port to receive data ",
	 {{RTX_GETOPT_INT, &port, "port"}, RTX_GETOPT_END_ARG}},
	{"filename", "Filename of AVI",
	 {{RTX_GETOPT_STR, &filename, "filename"}, RTX_GETOPT_END_ARG}},
	RTX_GETOPT_END
};

char           *help = "CSIRO Video Server Project \nDecode UDP Video Steam\nWARNING: since this program recieves Multicast packets, it must be run on a seperate machine to the encoder";

int
main(int ac, char *av[])
{

	int             i;
	int             done = 0;
	int             record = 0;
	int             verbose = 0;
	int             w, n;

	SDL_Surface    *screen;
	SDL_Overlay    *overlay;
	SDL_Rect        rect;
	SDL_Event       event;

	int             width;
	int             height;
	int             depth = 8;
	int             modes = SDL_SWSURFACE | SDL_ANYFORMAT | SDL_RESIZABLE;

	AVOutputFormat *fmt;	/* AV Codec Stuff */
	AVFormatContext *oc;
	AVStream       *stream;
	AVCodec        *codec;
	AVCodecContext *c = NULL;
	AVFrame        *picture;
	AVPacket       *packet;
	int             frame, nb, got_picture, len;
	uint8_t        *inbuf;
	int             bufSize;
	char            header[1000];

	RtxInet        *inet = NULL;
	RtxInetEndpoint srcAddr;

	/*
	 ** Process Command Line Options 
	 */

	rtx_main_init(av[0], 0);
	if (RTX_GETOPT_CMD(myOpts, ac, av, rcsid, help) == -1) {
		RTX_GETOPT_PRINT(myOpts, av[0], rcsid, help);
		exit(-1);
	}

	/*
	 ** RTX Inet stuff 
	 */

	printf("Listening on Port %d\n", port);
	if ((inet = rtx_inet_init(RTX_INET_UDP_MULTICAST, NULL, 0,
				  "239.255.15.1", port, NULL, NULL,
				  NULL)) == NULL) {
		printf("rtx_inet_init failed\n");
		exit(1);
	}

	fprintf(stderr, "Trying to Sync ");
	while (rtx_inet_read(inet->sock, header, 1000, &srcAddr) != HDSIZE)
	  fprintf(stderr, ".");
	printf("Success (%s)\n", header);

	sscanf(header, "%d%d%d%d", &width, &height, &codecID, &codecGOP);
	printf("width %d height %d ID %d GOP %d\n",
	       width, height, codecID, codecGOP);

	/*
	 ** Initialize AVfile Stuff 
	 ** This should really be done after the record button, 
	 ** but the pointer to the stream is created here.
	 */

	av_register_all();
	fmt = guess_format(NULL, filename, NULL);
	oc = av_mallocz(sizeof(AVFormatContext));
	oc->oformat = fmt;
	strncpy(oc->filename, filename, sizeof(oc->filename));
	stream = av_new_stream(oc, 0);
	av_set_parameters(oc, NULL);
	dump_format(oc, 0, filename, 1);

	/*
	 ** Initialize AVcodec Stuff 
	 */

	c = stream->codec;
	c->gop_size = codecGOP;
	c->bit_rate = codecBitrate * 1000;
	c->bit_rate_tolerance = codecTolerance * 1000;
        c->time_base = (AVRational){1,15};   // frames per second
	c->flags = 0;
	c->sub_id = 0;
	c->width = width;
	c->height = height;
        c->strict_std_compliance =  FF_COMPLIANCE_INOFFICIAL;

	c->pix_fmt = PIX_FMT_YUV420P;
	c->qcompress = 0.5;
	c->qblur = 0.5;
	c->qmin = 3;
	c->qmax = 15;
	c->max_qdiff = 3;
	c->frame_bits = 166920;

	av_new_packet(packet, 0);

	if (verbose) {

		fprintf(stderr, "Codec ID  \t%d\n", c->codec_id);
		fprintf(stderr, "Codec Type\t%d\n", c->codec_type);
		fprintf(stderr, "Bitrate   \t%d\n", c->bit_rate);
		fprintf(stderr, "Flags     \t%d\n", c->flags);
		fprintf(stderr, "Sub_id    \t%d\n", c->sub_id);
		fprintf(stderr, "Width     \t%d\n", c->width);
		fprintf(stderr, "Height    \t%d\n", c->height);
		fprintf(stderr, "Gop Size  \t%d\n", c->gop_size);
		fprintf(stderr, "Pix Fmt   \t%d\n", c->pix_fmt);
	}

	if (!(codec = avcodec_find_decoder(codecID))) {
		fprintf(stderr, "codec not found\n");
		exit(1);
	}

	if (avcodec_open(c, codec) < 0) {
		fprintf(stderr, "could not open codec\n");
		exit(1);
	}

	picture = avcodec_alloc_frame();
	inbuf = (uint8_t *) malloc(codecBuffer);

	fprintf(stderr, "Type R to toggle recording, Q to Quit\n");
	fprintf(stderr, "Warning Record is not working at the moment (new API to FFMPEG) \n");

	if (url_fopen(&oc->pb, filename, URL_WRONLY) < 0) {
		fprintf(stderr, "could not open file\n");
		exit(1);
	}
	av_write_header(oc);

	/*
	 ** SDL stuff
	 */

	rect.x = 0;
	rect.w = width;
	rect.y = 0;
	rect.h = height;

	if (SDL_Init(SDL_INIT_VIDEO) < 0) {
		fprintf(stderr, "Couldn't initialize SDL: %s\n",
			SDL_GetError());
		exit(1);
	}

	if ((screen = SDL_SetVideoMode(width, height, depth, modes)) == NULL) {
		fprintf(stderr, "Couldn't set video mode: %s\n",
			SDL_GetError());
		exit(1);
	}

	overlay =
		SDL_CreateYUVOverlay(width, height, SDL_YV12_OVERLAY, screen);

	/*
	 ** Lets GO
	 */

	frame = 0;
	bufSize = width * height;

	while (!done) {

		while (SDL_PollEvent(&event)) {
			switch (event.type) {
			case SDL_VIDEORESIZE:
				if ((screen =
				     SDL_SetVideoMode(event.resize.w,
						      event.resize.h, depth,
						      modes)) == NULL) {
					printf("cannot resize\n");
				}
				rect.w = event.resize.w;
				rect.h = event.resize.h;
				break;

			case SDL_KEYDOWN:
				if (event.key.keysym.sym == SDLK_q) {
					done = 1;
					continue;
				}
				if (event.key.keysym.sym == SDLK_r) {
					record = !record;
				}
				break;
			}
		}		// Poll Event 

		frame++;
		nb = rtx_inet_read(inet->sock, inbuf, bufSize, &srcAddr);
		if (nb == HDSIZE)
			continue;

		if (verbose)
			fprintf(stderr, "Decode %d Size %d \r", frame, nb);
		len = avcodec_decode_video(c, picture, &got_picture, inbuf,
					   nb);
		if (len < 0) {
			fprintf(stderr, "Frame Error %d\n", frame);
			exit(1);
		}
		if (!got_picture)
			continue;

		/*
		 ** Display on Screen 
		 ** Remember to Swap U and V
		 */

		SDL_LockYUVOverlay(overlay);

		w = width;
		n = picture->linesize[0];
		for (i = 0; i < height; i++)
			memcpy(overlay->pixels[0] + i * w,
			       picture->data[0] + i * n, w);

		w = width / 2;
		n = picture->linesize[2];
		for (i = 0; i < height / 2; i++)
			memcpy(overlay->pixels[1] + i * w,
			       picture->data[2] + i * n, w);

		n = picture->linesize[1];
		for (i = 0; i < height / 2; i++)
			memcpy(overlay->pixels[2] + i * w,
			       picture->data[1] + i * n, w);

		if (record)
			memset(overlay->pixels[1], 255, width * 3);

		SDL_DisplayYUVOverlay(overlay, &rect);
		SDL_UnlockYUVOverlay(overlay);

		packet->stream_index = stream->index;
		packet->data = inbuf;
		packet->size = nb;

		if (record) av_write_frame(oc, packet);
	}

	SDL_FreeYUVOverlay(overlay);
	SDL_Quit();

	av_write_trailer(oc);
	av_freep(&oc->streams[0]);
	url_fclose(oc->pb);

	avcodec_close(c);
	av_free_packet(packet);
	av_free(picture);
	av_free(oc);
	exit(1);
}
