/** \file
*	   Copyright (c) CSIRO
* \brief   View current video frame 
* \author  Elliot.Duff@csiro.au
* \todo    Insert video information in X header
* \warning The colour transform of the second Field is not quite right
* 
* A half size X display of the current video frame.
*
* If the mouse button is pressed in the window, the colour value is printed on the console. 
* The display can be controlled with a number of keyboard commands:
* - Space Display next video frame
* - 0 Display field 0
* - 1 Display field 1 (if possible)
* - 2 Display field 2 (if possible)
* - c Display in colour
* - y Display Y plane
* - u Display U plane
* - v Display V plane
* - r Display R plane
* - g Display G plane
* - b Display B plane
* - q Quit
*/
static char    *rcsid =
	"$Header$";

char           *name = "video";	/*!< Name of video frame in store */
int             field = -1;	/*!< Display numbered field (-1 is all) */
int             plane = -1;	/*!< Display numbered plane (-1 is in colour) */
int             stereo = 0;	/*!< Display in stereo (double width, halve height) */

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <rtx/message.h>
#include <rtx/time.h>
#include <rtx/timer.h>
#include <rtx/error.h>
#include <rtx/main.h>
#include <rtx/getopt.h>
#include <ddx.h>
#include <xid.h>
#include "ddxvideo.h"

DDX_STORE_ID   *storeId = NULL;
DDX_STORE_ITEM *itemPtr = NULL;
DDX_VIDEO      *video;

char           *help =
	"CSIRO Video Server Project \nView current video frame in store";
RtxGetopt       myOpts[] = {
	{"name", "Name of Video object in Store",
	 {{RTX_GETOPT_STR, &name, "name"}, RTX_GETOPT_END_ARG}},
	{"field", "Display selected Field ",
	 {{RTX_GETOPT_INT, &field, "field"}, RTX_GETOPT_END_ARG}},
	{"stereo", "Display stereo side by side ",
	 {{RTX_GETOPT_SET, &stereo, "stereo"}, RTX_GETOPT_END_ARG}},
	{"plane", "Display Colour Plane (0 to 6 -> YUVRGB) -1 is in colour",
	 {{RTX_GETOPT_SET, &plane, "plane"}, RTX_GETOPT_END_ARG}},
	RTX_GETOPT_END
};
#endif


int
main(int ac, char *av[])
{
	XIDwin         *w;
	char            event[20];
	int             nc, nr, nf, np, NR, NC, i, j, k;
	unsigned char  *ip[6];
	int             reload = 1;

	if (RTX_GETOPT_CMD(myOpts, ac, av, rcsid, help) == -1) {
		RTX_GETOPT_PRINT(myOpts, av[0], rcsid, help);
		exit(-1);
	}

	if ((storeId = ddx_store_open(NULL, 0, 5)) == NULL)
		return rtx_error("Unable to open store");

	if ((itemPtr = ddx_store_lookup_item(storeId, name, NULL, 0)) == NULL)
		return rtx_error("Unable to lookup var");

	video = ddx_store_var_pointer(itemPtr);
	if (ddx_store_read_direct(itemPtr, NULL, 10.0, 0) == -1)
		return rtx_error("Unable to read var ");

	printf("Name (%s) Frame %d Width %d Height %d Fields %d\n",
	       name, video->frame, video->width, video->height, video->fields);

	NC = video->width;
	NR = video->height;
	nc = video->width / 2;	// Work at Half size 
	nr = video->height / 2;
	nf = video->fields;

	if (stereo) {
		nc *= 2;
		nr /= 2;
	}

	/*
	 * The actual number of rows is calculated 
	 */

	if (field >= nf)
		return rtx_error("Invalid Field");
	if (field == -1) {
		nr *= nf;
		NR *= nf;
		field = 0;
	}
	if (field >= nf)
		field = 0;

	XID_Open(NULL);
	w = XID_WindowCreate(name, nc, nr, 0, 0, 0, 0);
	np = nc * nr;

	/* Someone moved this to a spot where np was not defined */

	for (i = 0; i < 6; i++) {
		ip[i] = malloc(np);
		if (ip[i] == NULL) {
			fprintf(stderr, "Cannot allocate memory\n");
			exit(-1);
		}
	}

	while (1) {

		printf("Frame %d Plane %d Field %d\n", video->frame, plane, field);

		if (reload) {
			int             r, g, b, u, v, y;

			reload = 0;
			switch (ddx_store_read_direct(itemPtr, NULL, 10.0, 1)) {
				case  -1:
					return rtx_error("main: Unable to read var ");
				case +1:
					continue;
				default :
					break;
			}

			/*
			 * For speed halve the size of the Y image 
			 */

			k = 0;
			for (i = 0; i < NR; i += 2)
				for (j = 0; j < NC; j += 2)
					ip[0][k++] = video->y[i * NC + j];

			memcpy(ip[1], video->u, np);
			memcpy(ip[2], video->v, np);

			for (i = 0; i < np; i++) {
				y = ip[0][i];
				u = ip[2][i];
				v = ip[1][i];
				YUV2RGB(y, u, v, r, g, b);
				ip[3][i] = r;
				ip[4][i] = g;
				ip[5][i] = b;
			}
		}

		if (plane == -1)
			XID_DrawImageRGB(w, ip[3], ip[4], ip[5], nc, nr);
		else {
			if (field == -1)
				XID_DrawImage(w, ip[plane], nc, nr);
			else
				XID_DrawImage(w, ip[plane] + (np * field), nc,
					      nr);
		}
		XID_GetEvent(event, 20);

		switch (event[0]) {

		case ' ':
			reload = 1;
			break;
		case '0':
			field = 0;
			break;
		case '1':
			if (video->fields > 1)
				field = 1;
			break;
		case '2':
			if (video->fields > 2)
				field = 2;
			break;
		case '3':
			if (video->fields > 3)
				field = 3;
			break;

		case 'y':
			plane = 0;
			break;
		case 'u':
			plane = 1;
			break;
		case 'v':
			plane = 2;
			break;
		case 'r':
			plane = 3;
			break;
		case 'g':
			plane = 4;
			break;
		case 'b':
			plane = 5;
			break;
		case 'c':
			plane = -1;
			break;

		case 'B':
			sscanf(event, "%*s%*d%d%d", &i, &j);
			k = j * nc + i;
			if (plane == -1)
				printf("%d:%d (%d,%d,%d)\n", i, j, ip[0][k],
				       ip[1][k], ip[2][k]);
			else
				printf("%d:%d (%d)\n", i, j,
				       ip[plane][k] + np * field);
			break;

		case 'q':
		case 'Q':
			XID_Destroy(w);
			XID_Close();
			ddx_store_done_item(itemPtr);
			ddx_store_close(storeId);
			ddx_client_done();
			return 0;

		default:
			printf("%s Unknown Event (%s) Availiable commands are:\n", event, av[0]);
			printf("Button\t Display values at mouse location\n");
			printf("Space \t Get next video frame\n");
			printf("0\t Display field 0\n");
			printf("1\t Display field 1 (if possible)\n");
			printf("2\t Display field 2 (if possible)\n");
			printf("c\t Display in colour\n");
			printf("y\t Display Y plane\n");
			printf("u\t Display U plane\n");
			printf("v\t Display V plane\n");
			printf("r\t Display R plane\n");
			printf("g\t Display G plane\n");
			printf("b\t Display B plane\n");
			printf("q\t Quit\n");
			break;
		}
	}
}
