/** \file

This program simulates a simple MJPEG webcam. To test, try:

\code 
vlc http://{ip:port}/image 
\endcode

Rather than using RTX_HTTPD, RTX_INET is used, so that a socket connection remains open.

\author  Elliot.Duff@csiro.au
\todo    Seperate thread for compression, so that multiple request don't present extra load
\todo    Use different compression based upon query string
\warning The framerate is fixed at 5Hz in Y4M header
*/

static char *rcsid = "$Id: ddxvideoserver.c 3855 2008-08-25 05:53:18Z roy029 $";

#include <assert.h>
#include <signal.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <strings.h>
#include <pthread.h>
#include <time.h>
#include <math.h>

#include <rtx/main.h>
#include <rtx/message.h>
#include <rtx/time.h>
#include <rtx/error.h>
#include <rtx/command.h>
#include <rtx/parse.h>
#include <rtx/inet.h>
#include <rtx/httpd.h>
#include <rtx/getopt.h>

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#include <ddx.h>
#include "ddxvideo.h"
DDX_STORE_ID   *storeId = NULL;
DDX_STORE_ITEM *itemPtr = NULL;
DDX_VIDEO      *video;
int             done = 0;
#endif

char   *help   = "CSIRO Video Server Project\nServe video to a browser";
int    port    = 8000;
int    quality = 75;			/*!< Quality of Jpeg Image */
char   *name   = "video";		/*|< Name of video in store */

RtxGetopt myOpts[] = {
	{ "port", "Select tcp port", { { RTX_GETOPT_INT, &port, "port"}, RTX_GETOPT_END_ARG } },
	{ "name", "Name of Video in Store", { { RTX_GETOPT_STR, &name, "name"}, RTX_GETOPT_END_ARG } },
	{ "quality", "Select encoding quality, in [50-100]", { { RTX_GETOPT_INT, &quality, "quality"}, RTX_GETOPT_END_ARG } }, RTX_GETOPT_END 
};

#include <libavcodec/avcodec.h>
	
#define BUFSIZE 200000

/**
** Simple request handler that ignores the query 
** \todo - Use query string to modify type of stream
*/

void *
handler(void *arg1, void *arg2) {

	int i, n;

        RtxTime tsOld;
	RtxInetTcpClient *r = (RtxInetTcpClient *) arg1;
	char str[200], query[800], method[10], url[100];

    	AVCodec *codec;
    	AVCodecContext *c= NULL;
        AVFrame *picture;

	int field;
	int outsize;
        uint8_t *outbuf;
	int offset;

	n = rtx_inet_read (r->sock, query, 800, NULL); // Ignore Query 
	query[n] = 0; sscanf(query, "%s %s", method, url); // Decode string 
        rtx_message("Method %s URL %s\n", method, url);

	if( strcmp(url, "/") == 0 ) {

	  sprintf(str, "HTTP/1.0 200 OK\r\n"
		     "Content-Type: text/plain\r\n"
		     "Content-Length: 18\r\n\r\n"
		     "DDX Video Server\r\n");
          if( rtx_inet_write(r->sock, str, strlen(str), NULL) != (int) strlen(str) ) {
		rtx_error("Unable to write to socket"); return 0; }
	  return 0;
	  }

	if( strcmp(url, "/image") == 0 ) field = 0; 
	else { if( strcmp(url, "/image1") == 0 ) field = 1;
	  else {
	
	  sprintf(str, "HTTP/1.0 200 No Content\r\n"
		       "Content-Length: 0\r\n\r\n");
          if( rtx_inet_write(r->sock, str, strlen(str), NULL) != 
		(int) strlen(str) ) {
		rtx_error("Unable to write to socket"); return 0; }
	  return 0;
	  }
	}

	sprintf(str, "HTTP/1.0 200 OK\r\n"
		     "Cache-Control: no-cache\r\n"
		     "Content-Type: multipart/x-mixed-replace;boundary=myboundary\r\n"); 
        if( rtx_inet_write(r->sock, str, strlen(str), NULL) != (int) strlen(str) ) {
		rtx_error("Unable to write to socket"); return 0; }

        codec = avcodec_find_encoder(CODEC_ID_MJPEG);
        if (!codec) { fprintf(stderr, "codec not found\n"); exit(1); }
        c = avcodec_alloc_context();
        picture = avcodec_alloc_frame();
	outbuf  = malloc(BUFSIZE);

	/* MJpegTools is the simplest way to get a JPeg Image but comsumes 35% CPU
	#include <mjpegtools/jpegutils.h>			
  	outsize = encode_jpeg_raw (outbuf, BUFSIZE, quality, -1, 0, 
	    video->width, video->height, video->y, video->u, video->v); 
	*/

        if (ddx_store_read_direct(itemPtr, &tsOld, 10.0, 1) !=0) {
            rtx_error("Unable to read video"); return 0; }

 	c->bit_rate = 400000;
        c->width    = video->width;
    	c->height   = video->height;
        c->time_base = (AVRational){1,15};   // frames per second
        c->pix_fmt   = PIX_FMT_YUV420P;
        c->strict_std_compliance =  FF_COMPLIANCE_INOFFICIAL;
        if( field == 1 ) offset = video->width * video->height;
	else offset = 0;

        if (avcodec_open(c, codec) < 0) {
          fprintf(stderr, "could not open codec\n"); exit(1); }

        picture->linesize[0] = video->width;
        picture->linesize[1] = video->width / 2;
        picture->linesize[2] = video->width / 2;
        picture->data[0] = video->y + offset;
        picture->data[1] = video->v + offset/4;
        picture->data[2] = video->u + offset/4;

	while(1) {

          if (ddx_store_read_direct(itemPtr, &tsOld, 10.0, 1) !=0) {
                rtx_error("Unable to read video"); return 0; }

          outsize = avcodec_encode_video(c, outbuf, BUFSIZE, picture);
	  sprintf(str, "\r\n--myboundary\r\n"
		       "Content-Type: image/jpeg\r\n"
		       "Content-Length: %d\r\n\r\n", outsize);
          if( rtx_inet_write(r->sock, str, strlen(str), NULL) != strlen(str) ) break;
          if( rtx_inet_write(r->sock, outbuf, outsize, NULL) != outsize) break;
	  }

        rtx_inet_write(r->sock, "\r\n--myboundary--\r\n", 18, NULL);	
        avcodec_close(c);
        av_free(c);
        av_free(picture);
	free(outbuf);
    	return 0;
}

int main(int argc, char **argv) 
{
	RtxInet *server;

	if( RTX_GETOPT_CMD( myOpts, argc, argv, NULL, help) == -1 )
	  { RTX_GETOPT_PRINT( myOpts, argv[0], NULL, help); exit(-1); }

        rtx_main_init (argv[0], RTX_MESSAGE_SYSLOG);
        rtx_message("Initialising simple HTTPD server on port %d",port);

        if ((storeId = ddx_store_open(NULL, 0, 5)) == NULL)
                return rtx_error("Unable to open store");
        if ((itemPtr = ddx_store_lookup_item(storeId, name, NULL, 0)) == NULL)
                return rtx_error("Unable to lookup var");

        avcodec_init();
        avcodec_register_all();

	video  = ddx_store_var_pointer(itemPtr);
	server = rtx_inet_init(RTX_INET_TCP_SERVER, 
		NULL, port, NULL, 0, handler, NULL, NULL);
        if (!server) return rtx_error("inet_init() failed.");

    	rtx_main_wait_shutdown(0);
        rtx_message("Shutting down server ");
	rtx_inet_done(server);
	ddx_store_done_item(itemPtr);
	ddx_store_close(storeId); 
	ddx_client_done(); 
        return 0;
}
