
/** \file
* Copyright (c) CSIRO Robotics Centre
* \brief   Compress DDX Video and stream to UDP port 
* \author  Elliot.Duff@csiro.au
* \todo	   Enable multicast
*/
static char    *rcsid =
	"$Header$";

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <sys/types.h>		// Needed by open()
#include <sys/stat.h>		// Needed by NONBLOCK()
#include <sys/ioctl.h>		// Needed by ioctl()
#include <sys/mman.h>		// Needed by PROT and MAP

#include <fcntl.h>		// Needed by O_RDWR
#include <errno.h>		// Needed by perror()
#include <string.h>		// Needed strerror()

#include <libavcodec/avcodec.h>
#include <rtx/inet.h>
#include <rtx/message.h>
#include <rtx/time.h>
#include <rtx/timer.h>
#include <rtx/error.h>
#include <rtx/main.h>
#include <rtx/getopt.h>

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#include <ddx.h>
#include "ddxvideo.h"
DDX_STORE_ID   *storeId = NULL;
DDX_STORE_ITEM *itemPtr = NULL;
DDX_VIDEO      *video;
int             done = 0;
#endif

#define HDSIZE	20

char           *name = "video";	/*!< Name of video in Store */
int             limit = -1;	/*!< Stop program after frame count has reached limit */
int             debug = 0;	/*!< Print debug information */
int             verbose = 0;	/*!< Print verbose information */

/* Inet Stuff */

RtxInet        *inet;
RtxInetEndpoint destAddr;

char           *client = "localhost";	/*!< Name of machine to send UDP video stream to */
int             port = 8000;	/*!< Port number of machine */
int             drop = 0;	/*!< Number of frames to drop - for testing */
char 		*maddr = "239.255.15.1" ;

/* Compression Stuff */

int             codecID = 16;		/*!< Default Codec is MSMPEGV3 */
int             codecGOP = 12;
int             codecBitrate = 200;
int             codecTolerance = 100;
int             codecBuffer = 100000;	/*!< Size of Buffer to do compression */

AVCodec        *codec;
AVCodecContext *c = NULL;
AVFrame        *p;
int             nb;
uint8_t        *outbuf;

void           *
write_thread(void *arg)
{
	int             frame = 0;	// Local frame counter
	RtxTime         ts;
	char            header[HDSIZE];

	// Need to read the first one just to get its size

	if (ddx_store_read_direct(itemPtr, &ts, 10.0, 1) != 0) {
		rtx_error("Unable to read video");
		return NULL;
	}

	if (debug)
		printf("Video %s frame %d width %d height %d fields %d\n",
		       name, video->frame, video->width, video->height,
		       video->fields);

	// Initialize FFmpeg Codec

	avcodec_init();
	avcodec_register_all();
	codec = avcodec_find_encoder(codecID);
	if (!codec) {
		fprintf(stderr, "codec not found\n");
		exit(1);
	}

	c = avcodec_alloc_context();
	p = avcodec_alloc_frame();
	c->bit_rate = codecBitrate * 1000;
	c->bit_rate_tolerance = codecTolerance * 1000;
	c->gop_size = codecGOP;
	c->width = video->width;
	c->height = video->height * video->fields;
        c->time_base = (AVRational){1,15};   // frames per second
	c->qcompress = 0.5;
	c->qblur = 0.5;
	c->qmin = 3;
	c->qmax = 15;
	c->max_qdiff = 3;
	c->frame_bits = 166920;
	c->pix_fmt = PIX_FMT_YUV420P;
        c->strict_std_compliance =  FF_COMPLIANCE_INOFFICIAL;

	sprintf(header, "%d %d %d %d", c->width, c->height, codecID, codecGOP);

	// Attach the codex to the context 

	if (avcodec_open(c, codec) < 0) {
		fprintf(stderr, "could not open codec\n");
		exit(1);
	}

	if (verbose) {
		fprintf(stderr, "Codec ID  \t%d\n", c->codec_id);
		fprintf(stderr, "Codec Type\t%d\n", c->codec_type);
		fprintf(stderr, "Bitrate   \t%d\n", c->bit_rate);
		fprintf(stderr, "Flags     \t%d\n", c->flags);
		fprintf(stderr, "Sub_id    \t%d\n", c->sub_id);
		fprintf(stderr, "Width     \t%d\n", c->width);
		fprintf(stderr, "Height    \t%d\n", c->height);
		fprintf(stderr, "Gop Size  \t%d\n", c->gop_size);
		fprintf(stderr, "Pix Fmt   \t%d\n", c->pix_fmt);
	}

	/*
	 * Alloc output buffer (must be less than the entire image) 
	 */

	outbuf = malloc(codecBuffer);
	p->linesize[0] = video->width;
	p->linesize[1] = video->width / 2;
	p->linesize[2] = video->width / 2;
	p->data[0] = video->y;
	p->data[1] = video->v;
	p->data[2] = video->u;

	// Now open a multicast socket

	printf("Multicast to [%s] on Port %d\n", client, port);

	if ((inet = rtx_inet_init(RTX_INET_UDP_MULTICAST,
		  NULL, 0, maddr, port, NULL, NULL, NULL)) == NULL) {
		rtx_error_flush("Rtx_inet_init() failed");
		exit(1);
	}

	if (rtx_inet_get_net_addr("239.255.15.1", port, &destAddr) == -1) {
		rtx_error_flush("rtx_inet_get_net_addr");
		exit(-1);
	}

	if (rtx_inet_set_loopback(inet, port) == -1) {
		rtx_error_flush("rtx_inet_set_loopback");
		exit(-1);
	}

	while (!done) {
		switch (ddx_store_read_direct(itemPtr, &ts, 1.0, 1)) {
			case -1:
				rtx_error("Unable to read video");
				return NULL;
			case +1:
				continue;
			default:
				break;
		}

		nb = avcodec_encode_video(c, outbuf, codecBuffer, p);
		if (debug)
			fprintf(stderr, "Frame %d Bytes %d \r", frame, nb);

		if (drop)
			if (frame % drop == 0) {
				continue;
			}


		if (frame % 25 == 0) 		// Send the Sync Data 
		  if (rtx_inet_write(inet->sock, header, HDSIZE, &destAddr) == -1)
		    rtx_error_flush("rtx_inet_write");

		if (rtx_inet_write(inet->sock, outbuf, nb, &destAddr) == -1)
			rtx_error_flush("rtx_inet_write %d", nb);

		if (frame++ == limit)
			rtx_main_signal_shutdown();
	}

	free(outbuf);
	avcodec_close(c);
	return NULL;
}

RtxGetopt       myOpts[] = {
	{"name", "Name of Video Stream in Store",
	 {{RTX_GETOPT_STR, &name, "name"}
	  , RTX_GETOPT_END_ARG}
	 }
	,
	{"client", "Client Name ",
	 {{RTX_GETOPT_STR, &client, "client"}
	  , RTX_GETOPT_END_ARG}
	 }
	,
	{"codec", "Use Codec",
	 {{RTX_GETOPT_INT, &codecID, "codecID"}
	  , RTX_GETOPT_END_ARG}
	 }
	,
	{"port", "Sent UDP to Port ",
	 {{RTX_GETOPT_INT, &port, "port"}
	  , RTX_GETOPT_END_ARG}
	 }
	,
	{"limit", "Frame limit ",
	 {{RTX_GETOPT_INT, &limit, "limit"}
	  , RTX_GETOPT_END_ARG}
	 }
	,
	RTX_GETOPT_END
};

char           *help = "CSIRO Video Server Project \nStream Raw Video ";

int
main(int ac, char *av[])
{

	RtxThread      *th = NULL;

	rtx_main_init(av[0], 0);
	if (RTX_GETOPT_CMD(myOpts, ac, av, rcsid, help) == -1) {
		RTX_GETOPT_PRINT(myOpts, av[0], rcsid, help);
		exit(-1);
	}

	debug = strlen(rtx_getopt_get_debug(""));
	verbose = rtx_getopt_get_verbose(0);

	if ((storeId = ddx_store_open(NULL, 0, 5)) == NULL)
		return rtx_error("Unable to open store");

	if ((itemPtr = ddx_store_lookup_item(storeId, name, NULL, 0)) == NULL)
		return rtx_error("Unable to lookup var");

	video = ddx_store_var_pointer(itemPtr);

	if ((th = rtx_thread_create("write thread", 0,
				    RTX_THREAD_SCHED_OTHER, 0, 0,
				    RTX_THREAD_CANCEL_DEFERRED,
				    write_thread, NULL, NULL, NULL)) == NULL) {
		rtx_error_flush("rtx_thread_create() failed");
		exit(-1);
	}

	if (rtx_main_wait_shutdown(90))
		rtx_error_flush("rtx_main_wait_shutdown() failed");

	done = 1;
	rtx_thread_destroy(th);
	ddx_store_done_item(itemPtr);
	ddx_store_close(storeId);
	ddx_client_done();
	return 0;
}
