
/** \file

This program saves the current DDX video sequence to file. The format of 
the file is defined by the \b YUV4MPEG standard, from the \b MPEGTOOLS 
distribution (see \ref y4mplay.c). 
To prevent frames being dropped it is important that you use the
appropriate file system. It appears that EXT3 file system is not 
suitable for large files, since the journal process interrupts the 
writing process after about a minute of writing. 

To establish the potential speed of you file system, try: 
\code hdparam -t -T /dev/{disk} \endcode

Once the files have been written, they can be played back into the store 
with \ref ddxvideoload.c or they can be view with a number of standard 
utilities \c mplayer and \c yuvplay. To have a closer look at the files use 
\ref y4mplay.c and \ref y4mtest.c 

\author  Elliot.Duff@csiro.au
\todo    Add skip, rate (Hz) and duration (s) 
\todo	 Calculate actual Framerate and insert into header
\warning The framerate is fixed at 5Hz in Y4M header
*/

static char    *rcsid =
	"$Header$";

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <sys/types.h>		// Needed by open()
#include <sys/stat.h>		// Needed by NONBLOCK()
#include <sys/ioctl.h>		// Needed by ioctl()
#include <sys/mman.h>		// Needed by PROT and MAP

#include <fcntl.h>		// Needed by O_RDWR
#include <errno.h>		// Needed by perror()
#include <string.h>		// Needed strerror()

#include <rtx/message.h>
#include <rtx/time.h>
#include <rtx/timer.h>
#include <rtx/error.h>
#include <rtx/main.h>
#include <rtx/getopt.h>

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#include <ddx.h>
#include "ddxvideo.h"
DDX_STORE_ID   *storeId = NULL;
DDX_STORE_ITEM *itemPtr = NULL;
DDX_VIDEO       vobj; /* for the non direct case */
DDX_VIDEO      *video;
int             done = 0;
int             fd = 1;		// Default is STDOUT
#endif

char           *name = "video";	/*|< Name of video in store */
int             limit = -1;	/*!< Number of frame until stop, -1 is limitless */
int             field = -1;	/*!< Number of field, -1 is all fields */
int             verbose = 0;
int				nodirect = 0; 

void           *
write_thread(void *arg)
{
	int             np;
	char            buff[80];
	int             frame = 0;	// Local frame counter
	RtxTime         ts, tsOld;

	// Need to read the first one just to get its size

	if (nodirect) {
		if (ddx_store_read(itemPtr, video, &tsOld, 10.0, 1) !=0) {
			rtx_error("Unable to read video");
			return NULL;
		}
	} else { 
		if (ddx_store_read_direct(itemPtr, &tsOld, 10.0, 1) !=0) {
			rtx_error("Unable to read video");
			return NULL;
		}
	}

	np = video->width * video->height * video->fields;
	sprintf(buff, "YUV4MPEG2 W%d H%d F5:1 Ip A1:1\n",
		video->width, video->height * video->fields);
	write(fd, buff, strlen(buff));
	if (verbose)
		fprintf(stderr, "Frame %d Width %d Height %d Fields %d\n",
			video->frame, video->width, video->height,
			video->fields);

	while (!done) {
		int ret;
		if (nodirect) {
			ret = ddx_store_read(itemPtr, video, &ts, 10.0, 1);
		} else {
			ret = ddx_store_read_direct(itemPtr, &ts, 10.0, 1);
		}
		switch (ret) {
			case -1:
				rtx_error("Unable to read video");
				return NULL;
			case +1:
				continue;
			default :
				break;
		}

		sprintf(buff, "FRAME X%d XS=%d XN=%d XD=%.4f\n", video->frame,
			(int) ts.seconds, (int) ts.nanoSeconds,
			rtx_time_subtract_to_double(&ts, &tsOld));

		if (verbose)
			fprintf(stderr, buff);
		tsOld.seconds = ts.seconds;
		tsOld.nanoSeconds = ts.nanoSeconds;

		write(fd, buff, strlen(buff));
		write(fd, video->y, np);
		write(fd, video->v, np / 4);
		write(fd, video->u, np / 4);

		if (frame++ == limit)
			rtx_main_signal_shutdown();
	}

	close(fd);
	return NULL;
}

char           *help = "CSIRO Video Server Project \nStream Raw Video to file";

RtxGetopt       myOpts[] = {
	{"field", "Select the field you want to save (not implemented)",
	 {{RTX_GETOPT_INT, &field, "field"}, RTX_GETOPT_END_ARG}},
	{"name", "Name of Video Stream in Store ",
	 {{RTX_GETOPT_STR, &name, "name"}, RTX_GETOPT_END_ARG}},
	{"limit", "Frame limit ",
	 {{RTX_GETOPT_INT, &limit, "limit"}, RTX_GETOPT_END_ARG}},
	{"nodirect", "Disable direct reading from memory",
	 {{RTX_GETOPT_SET, &nodirect, "nodirect"}, RTX_GETOPT_END_ARG}},
	RTX_GETOPT_END
};

int
main(int ac, char *av[])
{

	int             ret;
	RtxThread      *th = NULL;

	rtx_main_init(av[0], 0);
	if ((ret = RTX_GETOPT_CMD(myOpts, ac, av, rcsid, help)) == -1) {
		RTX_GETOPT_PRINT(myOpts, av[0], rcsid, help);
		exit(-1);
	}

	if (ret < ac)
		if ((fd = open(av[ret], O_RDWR | O_CREAT | O_TRUNC | O_NONBLOCK,
			  S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP)) < 0)
			return rtx_error("Unable to open %s\n", av[ret]);


	if ((storeId = ddx_store_open(NULL, 0, 5)) == NULL)
		return rtx_error("Unable to open store");
	if ((itemPtr = ddx_store_lookup_item(storeId, name, NULL, 0)) == NULL)
		return rtx_error("Unable to lookup var");

	if (nodirect) {
		video = &vobj;
	} else {
		video = ddx_store_var_pointer(itemPtr);
	}

	if ((th = rtx_thread_create("write thread", 0,
				    RTX_THREAD_SCHED_OTHER, 0, 0,
				    RTX_THREAD_CANCEL_DEFERRED,
				    write_thread, NULL, NULL, NULL)) == NULL) {
		rtx_error_flush("rtx_thread_create() failed");
		exit(-1);
	}

	if (rtx_main_wait_shutdown(90))
		rtx_error_flush("rtx_main_wait_shutdown() failed");

	done = 1;
	rtx_thread_destroy(th);
	ddx_store_done_item(itemPtr);
	ddx_store_close(storeId);
	ddx_client_done();
	return 0;
}
