/** 
\file
	Copyright (c) CSIRO ICT Robotics Centre
\brief  View Video Stream using SDL - faster for Colour video 
\author Elliot.Duff@csiro.au
\warning This program requires XVideo support on the X server
\todo   Handle more events 
*/

static char    *rcsid =
	"$Header$";

#include <SDL.h>
#include <SDL_syswm.h>
#include <X11/Xlib.h>

#include <stdlib.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include <ddx.h>
#include "ddxvideo.h"

#include <rtx/main.h>
#include <rtx/getopt.h>
#include <rtx/error.h>
#include <rtx/message.h>

int             field = -1;	/*<! Select field to view (-1 is all) */
int		noframe = 0;
char           *name = "video";
char 	       *geom = NULL;
char           *help = "CSIRO Video Server Project\nDisplay video using SDL (fast)";

RtxGetopt       myOpts[] = {
	{"field", "Select Video Field to Display",
	 {{RTX_GETOPT_INT, &field, "field"}, RTX_GETOPT_END_ARG}},
	{"noframe", "Don't frame the video",
	  {{RTX_GETOPT_SET, &noframe, "noframe"}, RTX_GETOPT_END_ARG}},
	{"geom", "Window geometry wxh+x+y",
	  {{RTX_GETOPT_STR, &geom, "geom"}, RTX_GETOPT_END_ARG}},
	{"name", "Name of Video in Store",
	 {{RTX_GETOPT_STR, &name, "name"}, RTX_GETOPT_END_ARG}},
	RTX_GETOPT_END
};

int
main(int ac, char *av[])
{

	int             quit = 0;
	int             width, height;
	int             offset = 0;
	int             depth = 9;
	int             modes = SDL_SWSURFACE | SDL_ANYFORMAT | SDL_RESIZABLE;
	int 		x, y, w, h;

	int Y_PLANE_INDEX, U_PLANE_INDEX, V_PLANE_INDEX;

	SDL_Surface    *screen;
	SDL_Overlay    *overlay;
	SDL_Rect        rect;
	SDL_Event       event;

	DDX_STORE_ID   *storeId = NULL;
	DDX_STORE_ITEM *itemPtr = NULL;
	DDX_VIDEO      *video;

	if (RTX_GETOPT_CMD(myOpts, ac, av, rcsid, help) == -1) {
		RTX_GETOPT_PRINT(myOpts, av[0], rcsid, help);
		exit(-1);
	}

	if ((storeId = ddx_store_open(NULL, 0, 5)) == NULL)
		return rtx_error("Unable to open store");

	if ((itemPtr = ddx_store_lookup_item(storeId, name, NULL, 0)) == NULL)
		return rtx_error("Unable to lookup var");

	video = ddx_store_var_pointer(itemPtr);
	if (ddx_store_read_direct(itemPtr, NULL, 10.0, 1) != 0)
		return rtx_error("Unable to read var");

	/* 
	 * Get the width and Height from Image 
	 */

	width = video->width;
	height = video->height;

	if (field == -1) height *= video->fields;
	else if (field < video->fields) offset = width * height * field;
	else return rtx_error("Insufficent Field Depth");

	/* 
	 * Use width and height for display rectangle OR use the specified Geometry 
	 */

	if ( geom == NULL )  { w = width; h = height; }
	else if( sscanf(geom, "%dx%d+%d+%d", &w,&h,&x,&y) != 4 )
	    return rtx_error("Invalid geometry string");

	rect.x = 0;
	rect.y = 0;
	rect.w = w;
	rect.h = h;

	/*
	 * Initialize the SDL library 
	 */

	if ( SDL_Init(SDL_INIT_VIDEO) < 0 )
		return rtx_error("Couldn't initialize SDL: %s\n",
				 SDL_GetError());
	/*
	 * Initialize the display 
	 */

        if ( noframe ) modes = SDL_NOFRAME | modes;
	if ((screen = SDL_SetVideoMode(w, h, depth, modes)) == NULL )
		return rtx_error("Couldn't set video mode: %s\n",
			SDL_GetError());

	SDL_WM_SetCaption("ddxvideoplay", "ddxvideoplay");

	/* 
	 * Move the window to the specified coordinates
         */

	if( geom != NULL ) 
	  {
	  SDL_SysWMinfo info;
	  Window root;
	  Window parent;
	  Window *children;
	  /* XClientMessageEvent xev; */
	  unsigned int count;

	  SDL_VERSION(&info.version);
          if (SDL_GetWMInfo(&info) > 0 ) {
            if (info.subsystem == SDL_SYSWM_X11) {
              XQueryTree(info.info.x11.display,
                   info.info.x11.window, &root, &parent, &children, &count);
              info.info.x11.lock_func();
	      XMoveWindow(info.info.x11.display, parent, x, y);
	      info.info.x11.unlock_func();

	      /* 
	      * Raise to front - Not working Yet

	      xev.type = ClientMessage;
	      xev.window = parent;
	      xev.message_type = XInternAtom(info.info.x11.display, "_WIN_LAYER", False);
	      xev.format = 32;
	      xev.data.l[0] = 10; // Layer 10
	      XSendEvent(info.info.x11.display, &root.window, False, SubstructureNotifyMask, (XEvent *) &xev);
	      */

              if( children ) XFree(children);
              }
            }
          }

	/**
	* When in SDL_YV12_OVERLAY mode, SDL_CreateYUVOverlay appears not to like image widths that AREN'T a multiple of 16. 
	* Thus here is a hack around using SDL_IYUV_OVERLAY mode.
	*
	* Note that there is a need to swap U and V chrominance planes around.
	* Also note that SDL_IYUV_OVERLAY does not seem to work with image widths that ARE a multiple of 16.
	*
    */
	if(width % 16 == 0)
	{
		overlay = SDL_CreateYUVOverlay(width, height, SDL_YV12_OVERLAY, screen);
		Y_PLANE_INDEX = 0;
		U_PLANE_INDEX = 1;
		V_PLANE_INDEX = 2;
	}
	else
	{
		overlay = SDL_CreateYUVOverlay(width, height, SDL_IYUV_OVERLAY, screen);
		Y_PLANE_INDEX = 0;
		U_PLANE_INDEX = 2;
		V_PLANE_INDEX = 1;
	}
	


	while (!quit) {

		while (SDL_PollEvent(&event)) {

			switch (event.type) {
			case SDL_VIDEORESIZE:
				screen = SDL_SetVideoMode(event.resize.w,
							  event.resize.h,
							  depth, modes);
				rect.w = event.resize.w;
				rect.h = event.resize.h;
				break;


			case SDL_KEYDOWN:
				if (event.key.keysym.sym == SDLK_q) {
					quit = 1;
					continue;
				}
				break;

				/*
				 * case SDL_MOUSEMOTION:
				 * printf("Mouse moved by %d,%d to (%d,%d)\n", 
				 * event.motion.xrel, event.motion.yrel,
				 * event.motion.x, event.motion.y);
				 * break;
				 */

			case SDL_MOUSEBUTTONDOWN:
				printf("Mouse button %d pressed at (%d,%d) ",
				       event.button.button, event.button.x, event.button.y);
				int ut = (int)(((double)width )/((double)rect.w)*(double)event.button.x);
				int vt = (int)(((double)height)/((double)rect.h)*(double)event.button.y);
				int i = vt*width+ut;
				/* 
				 * Just a bit tricky here, you need to cast as unsigned and then cast to int
				 * PS. Originally the store did not recognize unsigned char!
				 */
				printf("true x y (%d,%d) yuv = (%d %d %d) \n", 
				  ut, vt, (int) (unsigned char) video->y[i], 
				  (int)(unsigned char)  video->u[i/4], (int)(unsigned char)  video->v[i/4]);

				break;

			case SDL_QUIT:
				quit = 1;
				continue;
				break;

			default:
				break;
			}
		}

		switch (ddx_store_read_direct(itemPtr, NULL, 1.0, 1)) {
			case -1:
				return rtx_error("Unable to read var \n");
			case +1:
				continue;
			default:
				break;
		}

		SDL_LockYUVOverlay(overlay);

		memcpy(overlay->pixels[Y_PLANE_INDEX], video->y + offset, width * height);
		memcpy(overlay->pixels[U_PLANE_INDEX], video->u + offset / 4,
		       width * height / 4);
		memcpy(overlay->pixels[V_PLANE_INDEX], video->v + offset / 4,
		       width * height / 4);

		SDL_DisplayYUVOverlay(overlay, &rect);
		SDL_UnlockYUVOverlay(overlay);

	}

	SDL_FreeYUVOverlay(overlay);
	SDL_Quit();
	return 0;
}
