/** \file
Copyright (c) CSIRO Robotics Centre
\brief  Stream STDIN to DDX store 
\author Elliot.Duff@csiro.au
\todo   Support fields 
*/

static char    *rcsid =
	"$Header$";

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <rtx/message.h>
#include <rtx/time.h>
#include <rtx/timer.h>
#include <rtx/error.h>
#include <rtx/main.h>
#include <rtx/getopt.h>

#include <xid.h>

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#include <ddx.h>
#include "ddxvideo.h"
DDX_STORE_ID   *storeId = NULL;
DDX_STORE_ITEM *itemPtr = NULL;
DDX_VIDEO      *video;
#endif

int             verbose = 0;
char           *name = "video";
char           *help =
	"CSIRO Video Server Project \nStream Y4M Video into store\n\n\n"

	"There are three ways to load video files.\n\n"
	"\tDefault : just loop through the video file\n"
	"\tUsing -ddxreplay : which waits for the frame number from ddxreplay \n"
	"\t\t(frame numbers originate from ddxlog)\n"
	"\tUsing -control : which enables the user to control the loading.\n"
	"\t\t(commands are entered via a thumbnail window)\n\n\n"
	
	  "Commands when using -control\n"
	  
	  "\tLeft click -> Set start frame (derived from window x-axis)\n"
      "\tRight click -> Set end frame (derived from window x-axis)\n"
      "\tMiddle click -> Set frame rate (derived from window y-axis 0Hz-20Hz)\n"
      "\tSpace_bar -> Toggle pause\n"
      "\t','\t\t -> if paused go back one frame\n"
			"\t\t\t -> else divide frame rate by 2\n"
	  "\t'.'\t\t -> if paused go forward one frame\n"
		"\t\t\t -> else multiply frame rate by 2\n"
	  "\t'q' -> quit\n"
	  "\t'h' -> print help\n\n";
	
double          rate = 10.0;
int             dt = 0;
int             looping;
int             loop_ok = 1;
int             do_control = 0;
int             do_ddxreplay = 0;

DDX_STORE_ITEM  *framePtr = NULL;
char			frame_name[512];

#define         EVENT_LENGTH 20


#define MAX_FRAMES 100000
long frame_pntrs[MAX_FRAMES];
/*frame numbers specific to DDXVIDEOSAVE and DDXLOG. Needed to perform DDXREPLAY properly.*/
unsigned int frame_numbers[MAX_FRAMES]; 
int num_frames = 0, start_frame, end_frame;

#define MAX_FRAME_RATE 20

int paused = 0;

long startdata = 0;
int  frame;
char buff[100];

int  fd = 0;
FILE *fp;

char buff[100];
char name2[100];

int  width = 768, height = 576;
int  np;

int window_ratio, window_nr, window_nc;
XIDwin *window;


void load_frames();
char load_single_frames(int file_desc);
int load_from_ddxreplay();
void display_window(unsigned char *image, int nc, int nr);
int init_frame_pointers();
void process_events();


void
exit_handler()
{
	rtx_error_traceback();
}


RtxGetopt       myOpts[] = {
	{"name", "Name of Video Stream in Store ",
	 {{RTX_GETOPT_STR, &name, "name"}, RTX_GETOPT_END_ARG}},
	{"loop", "Endless loop",
	 {{RTX_GETOPT_SET, &looping, "endless"}, RTX_GETOPT_END_ARG}},
	{"rate", "Frame rate",
	 {{RTX_GETOPT_DOUBLE, &rate, "fps"}, RTX_GETOPT_END_ARG}},
    {"control", "Control the way the video is loaded",
	 {{RTX_GETOPT_SET, &do_control, "control"}, RTX_GETOPT_END_ARG}},
	{"ddxreplay", "Wait on store for the frame number sent by ddxreplay",
	 {{RTX_GETOPT_SET, &do_ddxreplay, "ddxreplay"}, RTX_GETOPT_END_ARG}},	
	RTX_GETOPT_END
};

/**
*	This function loads a single frame from a y4m video file into the store.
**/
char
load_single_frame()
{
	int i;

	/* Read Frame header */
	for (i = 0; i < 100; i++) {
		if (read(fd, &(buff[i]), 1) <= 0)
			return EOF;
		if (buff[i] == '\n')
			break;
	}
	buff[i] = 0;
	if (sscanf(buff, "%s", name2) != 1)
		return EOF;

	if (read(fd, video->y, np) != np)
		return EOF;
	if (read(fd, video->v, np / 4) != np / 4)
		return EOF;	// Invert for FFMPEG
	if (read(fd, video->u, np / 4) != np / 4)
		return EOF;

    video->width = width;
	video->height = height;
	video->fields = 1;
	video->frame = frame;

	if (ddx_store_write_direct(itemPtr, NULL) == -1)
		return rtx_error("Unable to write var ");

   	if (verbose)
		fprintf(stderr, "Frame %d: %s\r", frame, buff);

	return 0;
}

/**
*   This function seeks through the entire movie, storing pointers to each frame
**/
int
init_frame_pointers()
{
    int i;
   
    frame_pntrs[0] = startdata;

    while(num_frames < MAX_FRAMES) /* we define max_frames for the case of a really really long movie */
    {
        num_frames++;

   	    /* Read Frame header */
		for (i = 0; i < 100; i++) {
			if (read(fd, &(buff[i]), 1) <= 0)
				goto init_eof;
			if (buff[i] == '\n')
				break;
		}

		if(do_ddxreplay)
		{
			if(sscanf(buff, "FRAME X%d", &frame_numbers[num_frames]) != 1)
				return rtx_error("Unable to read frame number from video file");
		}

		/* store frame pointer */
        if((frame_pntrs[num_frames] = lseek(fd, np+np/2, SEEK_CUR)) < 0)
            goto init_eof;
     }

init_eof:;
    num_frames--;
    end_frame = num_frames;
    lseek(fd, startdata, SEEK_SET);
    printf("%d frames in movie\n", num_frames);

	return 0;
}

/**
 * 
 **/
void
load_frames()
{
    do {
		/* if we are controlling the loading we have a start frame */
		if(do_control)
			lseek(fd, frame_pntrs[start_frame], SEEK_SET);	/* seek to the start frame */
		/* else we just seek back to the begining of the video */
		else
			lseek(fd, startdata, SEEK_SET);	/* seek to start of data section */

		for (frame = start_frame; frame <= end_frame; frame++) 
		{

            if(paused && do_control) /* if we are paused we want to seek back one frame */
            {
                frame = frame-1;
                lseek(fd, frame_pntrs[frame], SEEK_SET);
            }

			if(load_single_frame() == EOF) goto eof;

            /* this is the call to display the frame. also get the user input */
			if(do_control)
            	display_window(video->y, width, height);

			if (verbose)
				fprintf(stderr, "Frame %d: %s\r", frame, buff);
			if (rate == 0.0) {
				char           *xd;

				/* get interframe delay from frame header */
				xd = strstr(buff, "XD=");
				if (xd)
					dt = atof(xd + 3) * 1e6;
			}
			usleep(dt);
		}
	    eof:;
	} while ((looping || do_control) && loop_ok);
}


/**
*	This function waits for a frame number from the store.
**/
int
load_from_ddxreplay()
{
	int ddxreplay_frame;
	int frame_index = 0;
	
	/* Setup the name of the frame number variable in the store */
	sprintf(frame_name, "%s_frame", name);
	if ((framePtr = ddx_store_lookup_item(storeId, frame_name, "int", 0)) == NULL)
		return rtx_error("Unable to lookup var");

	/* read ddxreplay frame number from store */
	while(! ddx_store_read(framePtr, &ddxreplay_frame, NULL, 30, 1) )
	{
		/* search for the local frame index from the list of frame numbers */
		for(frame_index = 0; frame_index < num_frames; frame_index++)
			if(frame_numbers[frame_index] == ddxreplay_frame) break;

		/* if we found the frame index load it into the store */
		if(frame_index < num_frames)
		{
			lseek(fd, frame_pntrs[frame_index], SEEK_SET);	/* seek to the frame */
			load_single_frame(); /* load the frame into the store */
		}
	}
	return 0;
}

/**
*   This function displays the frame in a thumbnail window.
*   Also processes all the keyboard and mouse events.
**/
void 
display_window(unsigned char *image, int nc, int nr)
{
    static int first_call = 1;

    static unsigned char *window_data;

    static int i, j;
    unsigned char *wip;
    
    static int curr_frame_pos, end_frame_pos, progress_bar_size;

    if(first_call)
    {
        if(nr < nc) window_ratio = nr/100;
        else window_ratio = nc/100;

        window_nr = nr/window_ratio;
        window_nc = nc/window_ratio;

        window_data = (unsigned char *) malloc(window_nc * window_nr);

        XID_Open(NULL);
	    window = XID_WindowCreate("ddxvideoload", window_nc, window_nr, 0, 0, 0, 0);
        first_call = 0;

    }

    progress_bar_size = window_nr/10;
    curr_frame_pos = window_nc*(frame/((double)num_frames));
    end_frame_pos = window_nc*(end_frame/((double)num_frames));

    wip = window_data;

    /* render frame, along with progress bar at the top of the window */
    for(j = 0; j < window_nr; j++)
    for(i = 0; i < window_nc; i++, wip++)
    {
        *wip = image[j*window_ratio*nc + i*window_ratio];
        if(j < progress_bar_size && i < curr_frame_pos)
        {
            *wip = 255 - *wip;
        }
        if(j < progress_bar_size && i > end_frame_pos)
        {
            *wip = 255 - *wip;
        }
    }

    XID_DrawImage(window, window_data, window_nc, window_nr);

	process_events();
}

/** 
 * Take mouse and keyboard events from XID to control the video.
 */
void 
process_events()
{
    static char event[EVENT_LENGTH];
    static int click_x, click_y, which_button;
	static int ret_id;

    /* process mouse and keyboard events */
    while((ret_id = XID_IfEvent(event, EVENT_LENGTH)) != 0 && ret_id == window->id)
    {
       switch(event[0])
	   {
	    case 'B': /* a button click */
            sscanf(event, "B%d %d %d", &which_button, &click_x, &click_y);
            switch(which_button)
            {                
                case 1: /* left click. set start frame */
                	start_frame = num_frames * (click_x/(double)window_nc);
                	frame = num_frames;
                	if(end_frame < start_frame) end_frame = start_frame;
                	printf("set start frame to -> %d\n", start_frame);
                break;
                case 3: /* right click. set end frame */
                	end_frame = num_frames * (click_x/(double)window_nc);
                	if(end_frame <= start_frame) end_frame = start_frame + 1;
                	printf("set end frame to -> %d\n", end_frame);
                break;
                case 2: /* middle click. set frame rate */
                	rate = MAX_FRAME_RATE * (click_y/(double)window_nr);
                	if (rate > 0.2)
    		        	dt = (int) (1e6 / rate);
                	printf("set frame rate to -> %f\n", rate);
                break;
            }
        break;
        case ' ': /* toggle 'pause' */
        	paused = (paused * -1) + 1;
        	if(paused) printf("paused\n");
        	else printf("playing\n");
        break;
        case ',':
        	if(paused) /* go back one frame */
        	{
            	frame = frame-1;
            	if(frame < start_frame) frame = start_frame;
        	}
        	else /* divide fps by 2 */
        	{
            	if(rate > 0.4) rate /= 2;
            	printf("load rate %f fps\n", rate);
            	if (rate > 0)
    		  		dt = (int) (1e6 / rate);
        	}
        break;
        case '.': /* go forward one frame */
        	if(paused) 
                frame = frame+1;
        	else /* multiply fps by 2 */
        	{
            	if(rate <= 30) rate *= 2;
            	printf("load rate %f fps\n", rate);
           		if (rate > 0)
    		  		dt = (int) (1e6 / rate);
        	}
        break;
		case 'h': /* print help */
			printf("%s", help);
		break;
        case 'q': /* exit */
        	loop_ok = 0;
			frame = num_frames;
        break;
       }
    }
}

int
main(int ac, char *av[])
{

	int             i;
	char           *e;
	int             optind;



	rtx_error_init(av[0], RTX_ERROR_STDERR, NULL);
	atexit(exit_handler);

	if ((e = getenv("DDXVIDEO")) != NULL)
		name = e;
	if ((optind = RTX_GETOPT_CMD(myOpts, ac, av, rcsid, help)) == -1) {
		RTX_GETOPT_PRINT(myOpts, av[0], rcsid, help);
		exit(-1);
	}

	if (av[optind] != NULL) {
		if ((fp = fopen(av[optind], "r")) == NULL)
			return rtx_error("cant open file %s for reading",
					 av[optind]);

		fd = fileno(fp);
	}

	verbose = rtx_getopt_get_verbose(verbose);

	/*
	 * Read V4M header 
	 */

	for (i = 0; i < 100; i++) {
		read(fd, &(buff[i]), 1);
		if (buff[i] == '\n')
			break;
	}
	buff[i] = 0;
	printf("%s\n", buff);
	sscanf(buff, "YUV4MPEG2 W%d H%d", &width, &height);
	printf("%d %d\n", width, height);
	np = width * height;

	if ((storeId = ddx_store_open(NULL, 0, 5)) == NULL)
		return rtx_error("Unable to open store");

	if (DDX_STORE_REGISTER_TYPE(storeId, DDX_VIDEO) == -1)
		return rtx_error("Unable to register type");

	if (rate > 0)
		dt = (int) (1e6 / rate);

	if ((itemPtr =
	     ddx_store_lookup_item(storeId, name, "DDX_VIDEO", 0)) == NULL)
		return rtx_error("Unable to lookup var");

	video = ddx_store_var_pointer(itemPtr);

	end_frame = MAX_FRAMES;
	start_frame = 0;
	startdata = lseek(fd, 0, SEEK_CUR);

	if(do_control || do_ddxreplay)
    {
        init_frame_pointers();
	}

	if(do_ddxreplay)
	{
    	/* the user has selected -ddxreplay*/
		load_from_ddxreplay();
	} else
	{
		load_frames();
	}

	if(do_ddxreplay) ddx_store_done_item(framePtr);
	ddx_store_done_item(itemPtr);
	ddx_store_close(storeId);
	ddx_client_done();

	return 0;
}
