
/** \file
Copyright (c) CSIRO ICT Robotics Centre 

This programs grabs the current video image in the store and saves it as PNM. 
By default the program only grabs the Y plane and saves it as a raw PGM.
There are a number of options:

- -colour Convert YUV image into RGB and save as a PPM
- -stereo Divide the height by two and double the width
- -quad   Divide the height by four and mutilpy the width by four

\section mux Muxing

With our MUX card set to line interlacing, the full PAL image is made from four images
- left odd
- right odd
- left even
- right even
So to display the image we need to use the quad flag.

\image html quad.jpg "PAL Stereo image with Quad flag"
\image latex quad.eps "PAL Stereo image with Quad flag"

At CIf resolution, it appears that the V4L driver drops the even field.

\image html stereo.jpg "CIF Stereo image with Stereo flag"
\image latex stereo.eps "CIF Stereo image with Stereo flag"

\author  Elliot.Duff@csiro.au
\warning The PIP zoom function tries to free up shared memory
*/

static char    *rcsid =
	"$Header$";

#include <stdio.h>
#include <stdlib.h>
#include <rtx/message.h>
#include <rtx/time.h>
#include <rtx/timer.h>
#include <rtx/error.h>
#include <rtx/main.h>
#include <rtx/getopt.h>
#include <ddx.h>
#include <pip.h>

#include "ddxvideo.h"

DDX_STORE_ID   *storeId = NULL;
DDX_STORE_ITEM *itemPtr = NULL;
DDX_VIDEO       video;

char           *name = "video";
int             plane = 0;
int             colour = 0;
int             transform = 1;
int             verbose = 0;
int             overlay = 0;
int             stereo = 0;
int             quad = 0;
int             half = 0;
int             uvMode = 0;

char           *help =
	"CSIRO Video Server Project\nGrab current video frame in store and save as PPM";

RtxGetopt       myOpts[] = {
	{"name", "Name of Video Stream in Store ",
	 {{RTX_GETOPT_STR, &name, "name"}, RTX_GETOPT_END_ARG}},
	{"half", "Save at half size (UV resolution)",
	 {{RTX_GETOPT_SET, &half, "half"}, RTX_GETOPT_END_ARG}},
	{"yuv", "Save in YUV format ",
	 {{RTX_GETOPT_SET, &uvMode, "uvmode"}, RTX_GETOPT_END_ARG}},
	{"colour", "Save in Colour ",
	 {{RTX_GETOPT_SET, &colour, "colour"}, RTX_GETOPT_END_ARG}},
	{"color", "Save in Colour ",
	 {{RTX_GETOPT_SET, &colour, "colour"}, RTX_GETOPT_END_ARG}},
	{"quad", "Modify image size for line Interleaved stereo ",
	 {{RTX_GETOPT_SET, &quad, "quad"}, RTX_GETOPT_END_ARG}},
	{"stereo", "Modify image size for Stereo",
	 {{RTX_GETOPT_SET, &stereo, "stereo"}, RTX_GETOPT_END_ARG}},
	RTX_GETOPT_END
};

int
main(int ac, char *av[])
{

	Pimage         *yuv[5];
	int             nc, nr, ret;
	pip_silent = 1;
	char           *e;
	char           *filename = NULL;

	if ((e = getenv("DDXVIDEO")) != NULL)
		name = e;

	if ((ret = RTX_GETOPT_CMD(myOpts, ac, av, rcsid, help)) == -1) {
		RTX_GETOPT_PRINT(myOpts, av[0], rcsid, help);
		exit(-1);
	}

	if (ret < ac)
		filename = av[ret];

	if ((storeId = ddx_store_open(NULL, 0, 5)) == NULL)
		return rtx_error("Unable to open store");
	if ((itemPtr = ddx_store_lookup_item(storeId, name, NULL, 0)) == NULL)
		return rtx_error("Unable to lookup var");
	if (ddx_store_read(itemPtr, &video, NULL, 10.0, 1) != 0)
		return rtx_error("Unable to read var ");

	if (verbose)
		fprintf(stderr, "Width %d Height %d Frame %d Fields %d\n",
			video.width, video.height, video.frame, video.fields);

	nc = video.width;
	nr = video.height * video.fields;

	yuv[0] = PIP_new();
	PIP_init(yuv[0], video.y, nc, nr, "DDX Y");
	yuv[1] = PIP_new();
	PIP_init(yuv[1], video.u, nc / 2, nr / 2, "DDX U");
	yuv[2] = PIP_new();
	PIP_init(yuv[2], video.v, nc / 2, nr / 2, "DDX V");
	yuv[3] = PIP_new();
	PIP_init(yuv[3], NULL, nc, nr, "Double DDX U");
	yuv[4] = PIP_new();
	PIP_init(yuv[4], NULL, nc, nr, "Double DDX V");

	if (stereo) {
		yuv[0]->nc *= 2;
		yuv[0]->nr /= 2;
	}
	if (quad) {
		yuv[0]->nc *= 4;
		yuv[0]->nr /= 4;
	}

	/* deal with ddx size difference between Y and UV */
	if (half) {
		PIP_halve(yuv[0]);
	} else if(colour || uvMode) {
		PIP_double(yuv[3], yuv[1]);
		PIP_double(yuv[4], yuv[2]);
		yuv[1] = yuv[3];
		yuv[2] = yuv[4];
	}

	/* output in chosen color format */
	if (uvMode) {
		PIP_saveRGB(yuv, filename);
	} else if(colour) {
		PIP_YCrCbtoRGB(yuv);
		PIP_saveRGB(yuv, filename);
	} else {
		PIP_save(yuv[0], filename);
	}

	ddx_store_done_item(itemPtr);
	ddx_store_close(storeId);
	ddx_client_done();
	return 0;
}
