/**
*\file ddxvideocolourseg.c
*\brief Main source file for testing colour segmentation with ddxvideo images.
*
* This program allows for segmentation of pretaught colour objects.
* \code
>../bin/i486-linux/ddxvideocolourseg [options]
\endcode 
*\author Kane Usher
*\date Jan, 2005
*/
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <math.h>
#include <limits.h>

/* Automation includes */
#include <rtx/error.h>
#include <rtx/message.h>
#include <rtx/time.h>
#include <rtx/timer.h>
#include <rtx/main.h>
#include <rtx/getopt.h>
#include <ddx.h> 
#include "ddxvideo.h"
#include <pip.h> 
#include <xid.h> 


/* local defines */
/*! Process thread kill priority (irrelevant for linux?) */
#define KILLER_PRIO  	 0

/*! cycle time */
#define LOOP_TIME 0.05

#define VIDEO_OUT 1

/*! Computes the sign of A */
#define      signof(A) ((A >= 0) ? 1 : -1)

/*! Computes the minimum of A and B*/
#define min(A,B) ((A)<(B) ? (A) : (B))
/*! Computes the maximum of A and B*/
#define max(A,B) ((A)>(B) ? (A) : (B))

/* DDX video stuff */
DDX_STORE_ID    *storeIdVid = NULL;
DDX_STORE_ITEM  *VidItemPtr = NULL;
DDX_VIDEO       *video;
char            *name = "video";

#if VIDEO_OUT
DDX_STORE_ITEM  *VidItemPtrOut;
DDX_VIDEO       *videoOut;
char            *nameOut = "colourSeg";
#endif

/*
* Program switches / initialisation variables
*/
static int verbose = 0;                   /*!< Controls verbosity */
static int ddxOn;                         /*!< Store switch */
char *lufilename1 = "colourLookup.pgm";/*<! Name of the 2D colour lu table */
/*
* PIP and image related variables.
*/
static int nc;               /*!< Number of pixels in x direction */
static int nr;               /*!< Number of pixels in y direction */
static Pimage *yuv[3];       /*!< The image structures */
static Pimage *lookupImage;  /*!< The lookup image(s) (table) */  
static Pimage *workImage;    /*!< The work (processing) image(s) */
int pip_silent = 1;          /*!< Restrict verbosity of the PIP library */

/*
* Other program variables
*/
static int displayOn = 0;

typedef enum _format_table
{
	RG,	/*!< Manual mode, used in the utility program */
	HS,	/*!< Transient (do nothing) used in the utility program */
	UV,	/*!< Used for winch calibration */
} TableFormat; 
TableFormat tableFormat;

char *colformat = "uv";

/*
* Declare threads and thread functions
*/
RtxThread	*csThread;	
void cstest_control_thread(void);

static char *rcsid = "";

RtxGetopt	options[] = {
	{"lutname", "look-up table filename",
		{
			{RTX_GETOPT_STR, &lufilename1, ""},
			RTX_GETOPT_END_ARG 
		}
	},
	{"name", "name of video stream in store",
		{
			{RTX_GETOPT_STR, &name, "name"},
			RTX_GETOPT_END_ARG 
		}
	},
	{"nameOut", "name of output video stream in store",
		{
			{RTX_GETOPT_STR, &nameOut, ""},
			RTX_GETOPT_END_ARG 
		}
	},
	{"displayOn", "show segmented image (XID)",
		{
			{RTX_GETOPT_SET, &displayOn, ""},
			RTX_GETOPT_END_ARG
		}	
	},
	{"format", "format of output lookup table (rg, hs, uv)",
		{
			{RTX_GETOPT_STR, &colformat, "format"},
			RTX_GETOPT_END_ARG 
		}
	},
	RTX_GETOPT_END
};

int luFirstIndex = 1;
int luSecondIndex = 2;
int luThirdIndex = 0;
int dispIndex = 0;

char *help = "ddxvideo LUT based colour segmentation.\n Pushes segmented image stream back into store. \n";


/**
 * Connect to the store, register appropriate variables
 * \returns 0 on success, else -1
 */
int 
store_connect(void)
{
	char *funcName = "store_connect";

	/* connect with store */
	if(ddx_client_init(0))
		return rtx_error("%s: ddx_client_init() failed", funcName);	

	if ((storeIdVid = ddx_store_open(NULL, 0, 2)) == NULL) 
		return rtx_error("%s: store open failed", funcName);	
	
	if (DDX_STORE_REGISTER_TYPE(storeIdVid, DDX_VIDEO)) {	
		return rtx_error("%s:DDX_STORE_REGISTER_TYPE() failed", funcName);
	}
	
	if ((VidItemPtr = ddx_store_lookup_item (storeIdVid, name, "DDX_VIDEO", sizeof(DDX_VIDEO))) == NULL)
		return rtx_error ("%s: Unable to lookup var", funcName);
	video = ddx_store_var_pointer(VidItemPtr);
#if VIDEO_OUT
	if ((VidItemPtrOut = ddx_store_lookup_item (storeIdVid, nameOut, "DDX_VIDEO", sizeof(DDX_VIDEO))) == NULL)
		return rtx_error ("%s: Unable to lookup var", funcName);
	videoOut = (DDX_VIDEO *)ddx_store_var_pointer(VidItemPtrOut);
#endif

	return(0);
}


/**
 * Disconnect from the store.
 * \returns 0 on success, else -1
 */
int 
store_disconnect(void)
{
	char *funcName = "store_disconnect";
	int errcnt = 0;

	if (ddx_store_done_item(VidItemPtr)){
		errcnt++;
		rtx_error_flush("%s: ddx_store_done_item(VidItemPtr) failed", funcName);
	}
#if VIDEO_OUT
	if (ddx_store_done_item(VidItemPtrOut)){
		errcnt++;
		rtx_error_flush("%s: ddx_store_done_item(VidItemPtrOut) failed", funcName);
	}
#endif	

	if(ddx_store_close(storeIdVid)){
		errcnt++;
		rtx_error_flush("%s: ddx_store_close() failed", funcName);
	}

	if (ddx_client_done()) {
		errcnt++;
		rtx_error_flush("store client done failed");
	}

	if(errcnt)
		return(-1);
	else	
		return(0);
}


/** 
* Main for the cstest program.
*
* Sets up all devices etc, launches appropriate threads.
* \return 0 on success, else -1.
*/
int main(int ac, char *av[])
{
	char *prog_name;
	int errcnt = 0;

	prog_name = av[0];
	verbose = 0;
	ddxOn = 0;

	/* Initialise messsage and error protocols */
	rtx_main_init(prog_name, RTX_ERROR_STDERR|RTX_ERROR_STDOUT);

	/* Process command line options (avs custom)*/
	if( RTX_GETOPT_CMD( options, ac, av, rcsid, help) == -1 ) {
		RTX_GETOPT_PRINT( options, av[0], rcsid, help);
		exit(-1);
	}

	/* Process command line options (standard) */
	ddxOn = rtx_getopt_get_store(ddxOn);
	verbose = rtx_getopt_get_verbose(verbose);

	if (ddxOn == 0 ){
		fprintf(stderr, " Turn store switch on, make sure catalog etc are running\n");
		exit(0);
	}

	/* deal with options */
	if (strcmp(colformat, "rg") == 0) {
		tableFormat = RG;
	} else if(strcmp(colformat, "hs") == 0){
		tableFormat = HS;
	}
	else if((strcmp(colformat, "uv") == 0)){
		tableFormat = UV;
	}
	else {
		fprintf(stderr, "Format must be 'rg', 'hs', or 'uv'\n");
		fprintf(stderr, "Exiting\n");
		return(0);
	}	

	if (ddxOn)
		if (store_connect())
			return rtx_error("%s: couldn't connect to store", prog_name);

	/* Start threads */
	if((csThread = rtx_thread_create("cstestcontrol thread", 0, RTX_THREAD_SCHED_OTHER, 21, 0, RTX_THREAD_CANCEL_ASYNCHRONOUS, (void*(*)(void*))cstest_control_thread, NULL, NULL, NULL))==NULL)
		return rtx_error("%s: rtx_thread_create(csThread) failed", prog_name);

	/* Wait for shutdown signal */
	rtx_main_wait_shutdown(KILLER_PRIO);

	/* Destroy threads */
	if (rtx_thread_destroy_sync(csThread) == -1) {
		errcnt++;
		rtx_error_flush("%s: rtx_thread_destroy_sync(csThread) failed", prog_name);
	}

	/* Close store stuff. */
	if (ddxOn)
		if (store_disconnect()){
			errcnt++;
			rtx_error_flush("%s: disconnect_from_store() failed", prog_name);
		}

	if (errcnt)
		return rtx_error("%s: problem in shutdown", prog_name);

	fprintf(stderr, "`%s' terminated.\n", prog_name);	
	
	return(0);		
}


/**
 * Processes the video frame
 */
int
process_frame(void)
{
	char *funcName = "process_frame";

	if (PIP_lookup(workImage, yuv[luFirstIndex], yuv[luSecondIndex], lookupImage) == -1)
		return rtx_error("%s: PIP_lookup() failed", funcName);
	/*
	 * Could add some image clean-up here, eg
	 * PIP_dilate(workImage, 4);
	 * PIP_erode(workimage, 4);
	 * but it is probably best to leave this for the end user
	 * of the image.
	 */

	return(0);	
}	


/**
* Initialise the video structures
* \return 0 on success, else -1 
*/
int
init_video(void)
{	
	char *funcName = "video_init";
	RtxTime t;
	int i;
	
	/* read video frame */
	if( ddx_store_read_direct(VidItemPtr, &t, 10.0, 1) == -1 )
		return rtx_error("%s: Unable to read video", funcName);

	nc = video->width;
	nr = video->height * video->fields; 

	yuv[0] = PIP_new(); PIP_make(yuv[0], nc, nr, 255, "DDX Y"); 
	yuv[1] = PIP_new(); PIP_make(yuv[1], nc/2, nr/2, 255, "DDX U");
	yuv[2] = PIP_new(); PIP_make(yuv[2], nc/2, nr/2, 255, "DDX V");

	lookupImage = PIP_new();
	if(PIP_load(lookupImage, lufilename1) < 0)
		return rtx_error("5s: Unable to load LUT %s", funcName, lufilename1);

	workImage = PIP_new();
	PIP_make(workImage, nc/2, nr/2, 255, "workim");

#if VIDEO_OUT
	for( i = 0; i < nc*nr; i++){
		videoOut->u[i] = 127;
		videoOut->v[i] = 127;
	}
	videoOut->fields = 1;
	videoOut->frame = 0;
#endif
	/* set-up all the indices etc. */
	switch(tableFormat){
		case RG:
			dispIndex = 2;
			luFirstIndex = 0;
			luSecondIndex = 1;
			luThirdIndex = 2;
			fprintf(stderr, "using normalised r versus g\n");
			break;
		case UV:
			dispIndex = 0;
			luFirstIndex = 1;
			luSecondIndex = 2;
			luThirdIndex = 0;
			fprintf(stderr, "using Cr versus Cb\n");
			break;
		case HS:
			dispIndex = 2;
			luFirstIndex = 0;
			luSecondIndex = 1;
			luThirdIndex = 2;
			fprintf(stderr, "using H versus S\n");
			break;
		default:
			fprintf(stderr, "How did you get this far?\n");
			exit(0);
			break;
	}	

	return (0);
}


int
display_images(void)
{
	static int firstCall = 1;
	static XIDwin *t;
	if (firstCall){
		if( XID_Open(NULL) ) exit(-1);
		t = XID_WindowCreate("Thresholded Image",
				yuv[dispIndex]->nc,yuv[dispIndex]->nr,0,0,0,0);
		firstCall = 0;
	}	

	XID_DrawClear(t);
	XID_DrawImageOver(t,yuv[dispIndex]->data,workImage->data, workImage->nc, workImage->nr);

	return(0);
}


/** 
 * The main control thread for the avs tracking program.
 * \return 0 on success, else -1.
 */
void
cstest_control_thread(void)
{
	RtxTimer *timer;        
	char *threadName = "cstest_control_thread";
	int loopOK = 1;
	RtxTime t;
	int timerOn = 0;

	/* Create timer. */
	timer = rtx_timer_create(0.02, LOOP_TIME, NULL, NULL, 0);
	rtx_timer_start(timer);
	if(init_video()<0){
		rtx_error_flush("%s: problem initialising video/images", threadName);
		loopOK = 0;
	}

	while(loopOK){
		/* Ensures loop runs at the defined cycle time (provided calc time doesn't
		 *	exceed available time) */
		if (timerOn)
			if (rtx_timer_wait(timer) == -1){
				rtx_error_flush ("%s: rtx_timer_wait() failed", threadName);
				loopOK = 0;  
				continue;
			}

		yuv[0]->nc = nc;          // Need to reset the Y plane
		yuv[0]->nr = nr;
		/* read video frame */
		if( ddx_store_read_direct(VidItemPtr, &t, 10.0, 1) == -1 ){
			rtx_error_flush("Unable to read video");
			loopOK = 0;  
			continue;
		}
		/* Make a local copy of the image */
		memcpy(yuv[0]->data, video->y, nc*nr);
		memcpy(yuv[1]->data, video->u, nc*nr/4);
		memcpy(yuv[2]->data, video->v, nc*nr/4);

		/* Half the size of the intensity image */
		PIP_halve(yuv[0]);

		switch(tableFormat){
			case RG:
				PIP_YCrCbtoRGB(yuv);		
				PIP_RGBtoRGY(yuv);
				break;
			case UV:
				break;
			case HS:
				PIP_YCrCbtoRGB(yuv);
				PIP_RGBtoHSV(yuv);
				break;
			default:
				break;
		}	

		/* process the image */
		if(process_frame() < 0){
			rtx_error_flush("Problem processing frame");
			loopOK = 0;
		}
		
		/* If requested, display it using XID */
		if(displayOn)
			display_images();
#if VIDEO_OUT
		videoOut->width = workImage->nc;
		videoOut->height = workImage->nr;
		videoOut->frame++;		
		memcpy(videoOut->y, workImage->data, workImage->nc*workImage->nr);
		if(ddx_store_write_direct(VidItemPtrOut, NULL)){
			rtx_error_flush("ddx_store_write_direct(videoItemPtrOut) failed");
			loopOK = 0;
			continue;
		}
#endif	
	}	

	rtx_timer_destroy(timer);
	rtx_main_signal_shutdown();
}

