/**
* \brief Generates lookup tables from a training image.
*
* This program allows for training of a 2d histogram or lookup table for colour
* segmentation.
* \code
>../bin/i486-linux/ddxvideocolourLUT [options]
\endcode
*\author Kane Usher
*\date Jan, 2005
*/
extern char *optarg;
extern int optind;

/* C library includes */
#include	<unistd.h>
#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<math.h>
#include	<malloc.h>
#include	<time.h>
#include	<sys/types.h>

/* Automation includes */
#include <rtx/error.h>
#include <rtx/main.h>
#include <rtx/getopt.h>
#include <rtx/time.h>
#include <pip.h>
#include <xid.h>
#include <ddx.h>
#include "ddxvideo.h"

/* PIP variables*/
Pimage *ip;       /*<! pointer to first plane of the image */
Pimage *map;      /*<! the threshold lookup matrix */
Pimage *tempmap;  /*<! the temporary currently selected lookup matrix */
Pimage *im[3];    /*<! the loaded training image */
Pimage *workim;   /*<! the segmented image */
Pimage *Maskim;   /*<! the mask image, legacy from the tractor */
Pimage *rgb[3];   /*<! copy of the training image used for display purposes */

/* DDX video stuff */
DDX_STORE_ID    *storeIdVid = NULL;
DDX_STORE_ITEM  *VidItemPtr = NULL;
DDX_VIDEO       *video;

/* Segmentation variables - make into config file???*/
int pMaxBlobArea = 500;
int	pMinBlobArea = 1; 
int boxSize = 3;              /*<! Initial selection box size */

char *LTfilename = "colourLookup.pgm";
char *infilename = NULL;
char *maskFilename = "mask.pgm";

int rgbIn = 0;
int maskLoad = 0;
static char *rcsid = "";
int nc, nr;
typedef enum _format_table
{
  RG,          /*!< Manual mode, used in the utility program */
  HS,          /*!< Transient (do nothing) used in the utility program */
  UV,          /*!< Used for winch calibration */
} TableFormat; 

TableFormat tableFormat;
char *outputformat = "uv";
int ddxOn = 0;
char *name = "video";

RtxGetopt	options[] = {
  {"lufilename", "output file name (the lookup table)",
    {
      {RTX_GETOPT_STR, &LTfilename, "lufilename"},
      RTX_GETOPT_END_ARG 
    }
  },
  {"format", "format of output lookup table (rg, hs, uv)",
    {
      {RTX_GETOPT_STR, &outputformat, "format"},
      RTX_GETOPT_END_ARG 
    }
  },
  {"inputname", "input file name (the training image)",
    {
      {RTX_GETOPT_STR, &infilename, "infilename"},
      RTX_GETOPT_END_ARG 
    }
  },
  {"name", "name of video in store, overrides -inputname",
    {
      {RTX_GETOPT_STR, &name, "name"},
      {RTX_GETOPT_SET, &ddxOn, ""},
      RTX_GETOPT_END_ARG 
    }
  },
  {"maskfilename", "mask image file name",
    {
      {RTX_GETOPT_STR, &maskFilename, "maskfilename"},
      {RTX_GETOPT_SET, &maskLoad, ""},
      RTX_GETOPT_END_ARG 
    }
  },
  {"rgb", "specify input image as rgb format",
    {
      {RTX_GETOPT_SET, &rgbIn, ""},
      RTX_GETOPT_END_ARG
    }	
  },
  RTX_GETOPT_END
};

char *help = "Video Utilities: \nCreate a 2-d lookup table.\n"
"Input image is by default presumed to be yuv format, can specify rgb with \n"
"command line option. Format of output table is by default uv, again can \n"
"be specified with a command line option. For help while the application is \n"
"running type '?' on an XID window.";


int luFirstIndex = 1;
int luSecondIndex = 2;
int dispIndex = 0;

/* Forward function definitions */
int blobs_get(Pimage * im[], Pimage *colLookup,	Pimage *Workim, Pimage *Maskim,
int pMaxBlobArea, int pMinBlobArea,  int debug_im);
int print_help(void);
int display_images(int drawMask);


int
grab_ddx_image(void)
{  
	RtxTime t;
	int n1c, n1r;

	/* read video frame */
	if( ddx_store_read_direct(VidItemPtr, &t, 10.0, 1) == -1 ){
		return rtx_error("Unable to read video");
	}

	nc = video->width;
	nr = video->height * video->fields;

	im[0]->nc = nc;          // Need to reset the Y plane
	im[0]->nr = nr;

	if (rgbIn == 0){
		n1c = nc/2;
		n1r = nr/2;
	}
	else{
		n1c = nc;
		n1r = nr;
	}
	if (im[0]->data == NULL){
		PIP_init(im[0], NULL, nc, nr, "DDX Y"); 
		PIP_init(im[1], NULL, n1c, n1r, "DDX U");
		PIP_init(im[2], NULL, n1c, n1r, "DDX V");
	}

	/* Make a local copy of the image */
	memcpy(im[0]->data, video->y, nc*nr);
	memcpy(im[1]->data, video->u, n1c*n1r);
	memcpy(im[2]->data, video->v, n1c*n1r);

	if(rgbIn == 0){
		/* Half the size of the intensity image */
		PIP_halve(im[0]);
	}

	return(0);
}

/**
* Connect to the store, register appropriate variables
* \returns 0 on success, else -1
*/
int 
store_connect(void)
{
	char *funcName = "store_connect";

	/* connect with store */
	if(ddx_client_init(0))
		return rtx_error("%s: ddx_client_init() failed", funcName);	

	if ((storeIdVid = ddx_store_open(NULL, 0, 2)) == NULL) 
		return rtx_error("%s: store open failed", funcName);	

	if ((VidItemPtr = ddx_store_lookup_item (storeIdVid, name, NULL,0)) == NULL)
		return rtx_error ("%s: Unable to lookup var", funcName);
	video = ddx_store_var_pointer(VidItemPtr);

	/* read video frame */
	if(grab_ddx_image() <0)
		return rtx_error("%s: Unable to read video", funcName);

	return(0);
}

int
store_disconnect(void)
{
	int errcnt = 0;
	char *funcName = "store_disconect";
	
	if(ddx_store_done_item(VidItemPtr) < 0){
		errcnt++;
		rtx_error_flush("%s: ddx_store_done_item() failed", funcName);
	}
	if(ddx_store_close(storeIdVid)){
		errcnt++;
		rtx_error_flush("%s: ddx_store_close() failed", funcName);
	}

	if (ddx_client_done()) {
		errcnt++;
		rtx_error_flush("store client done failed");
	}
	
	if (errcnt)
		return(-1);
	else
		return(0);
}


int
format_images(void)
{
	static int firstCall = 1;
	/* Transform the original input image into the space specified by the output
	 * table type */
	switch(tableFormat){
		case RG:
			PIP_YCrCbtoRGB(im);		
			PIP_RGBtoRGY(im);
			dispIndex = 2;
			luFirstIndex = 0;
			luSecondIndex = 1;
			if (firstCall)
				fprintf(stderr, "Output table will be normalised r versus g\n");
			break;
		case UV:
			dispIndex = 0;
			luFirstIndex = 1;
			luSecondIndex = 2;
			if (firstCall)
				fprintf(stderr, "Output table will be Cr versus Cb\n");
			break;
		case HS:
			PIP_YCrCbtoRGB(im);
			PIP_RGBtoHSV(im);
			dispIndex = 2;
			luFirstIndex = 0;
			luSecondIndex = 1;
			if (firstCall)
				fprintf(stderr, "Output table will be H versus S\n");
			break;
		default:
			if (firstCall)
				fprintf(stderr, "How did you get this far?\n");
			exit(0);
			break;
	}
	firstCall = 0;
	return(0);
}



int
main (int ac, char *av[])
{
	char *func = "gentable";
	int b, x = 100, y = 100;
	char c[80], s[80];
	int imNum = 0;
	char saveimfname[128];
	int boxArea;
	int i;

	/* Initialise messsage and error protocols */
	rtx_main_init(func, RTX_ERROR_STDERR|RTX_ERROR_STDOUT);

	/* Process command line options (avs custom)*/
	if( RTX_GETOPT_CMD( options, ac, av, rcsid, help) == -1 ) {
		RTX_GETOPT_PRINT( options, av[0], rcsid, help);
		exit(-1);
	}
	ddxOn = rtx_getopt_get_store(ddxOn);
	if((ddxOn == 0) && (infilename ==NULL)){
		fprintf(stderr, "No input filename specified and store option not used, exiting\n");
		exit(0);
	}  

	if (strcmp(outputformat, "rg") == 0) {
		tableFormat = RG;
	} else if(strcmp(outputformat, "hs") == 0){
		tableFormat = HS;
	}
	else if((strcmp(outputformat, "uv") == 0)){
		tableFormat = UV;
	}
	else {
		fprintf(stderr, "Format for the output table must be 'rg', 'hs', or 'uv'\n");
		fprintf(stderr, "Exiting\n");
		return(0);
	}	

	/* Ensure that the pip and XID libraries execute in a silent manner */
	pip_silent = 1;
	pip_verbose = 0;
	xid_verbose = 0;

	/* Initialise the images for PIP stuff */
	im[0] = PIP_new();
	im[1] = PIP_new();
	im[2] = PIP_new();
	workim = PIP_new();
	Maskim = PIP_new();

	rgb[0] = PIP_new();
	rgb[1] = PIP_new();
	rgb[2] = PIP_new();	

	map = PIP_new();
	tempmap = PIP_new();

	/* make/allocate space for those that aren't loaded */
	if (PIP_make(map, 256, 256, 255, "map") != PIPOK)
		return rtx_error("%s: can't create map image", func);
	if (PIP_make(tempmap, 256, 256, 255, "tempmap") != PIPOK)
		return rtx_error("%s: can't create tempmap image", func);

	if(ddxOn == 0){
		/* Load the ycrcb image, as specified from command line or default */
		if(PIP_loadRGB(im, infilename))
			exit(-1);
	}  
	else{
		/* Set up the video store stuff */
		store_connect();

	}  

	/* If the input image is RGB, convert it to yuv (really yCrCb)*/
	if (rgbIn)
		PIP_RGBtoYCrCb(im); 

	ip = im[0];

	if (PIP_make(workim, im[0]->nc, im[0]->nr, 255, "im3") != PIPOK)
		return rtx_error("%s: can't create workim image", func);
	/*	if (PIP_make(Maskim, im[0]->nc, im[0]->nr, 255, "im4") != PIPOK)
		return rtx_error("%s: can't create Maskim image", func);
	 */
	if (maskLoad){
		if (PIP_load(Maskim, maskFilename) < 0 )
			return rtx_error("%s: couldn't initialize mask paramaters", func);		
		if ((im[0]->nc != Maskim->nc) | (im[0]->nr != Maskim->nr))
			return rtx_error("%s: mask image size does not match input image", func);		

	}
	/* copy the RGB image for display purposes */
	PIP_copy(rgb[0], im[0]);
	PIP_copy(rgb[1], im[1]);
	PIP_copy(rgb[2], im[2]);
	/* Transform the input image into a RGB space for display*/
	PIP_YCrCbtoRGB(rgb);

	format_images();

	/* First call to display_images opens the XID stuff and creates
	 * the relevant windows. */
	display_images(0);

	while ((XID_GetEvent(s, 80) ) ) {
		boxArea = (2*boxSize+1)*(2*boxSize+1);
		switch( s[0] ) {
			case 'B': /* Mouse Event */
				sscanf(s, "%s%d%d%d", c, &b, &x, &y);
				if( !strcmp(c, "BP") ) {
					ip->c1 = x-boxSize; ip->r1 = y-boxSize;
					ip->c2 = x+boxSize; ip->r2 = y+boxSize;
				}

				/* Add to the histogram */
				PIP_roi(im[luFirstIndex], ip->r1, ip->c1, ip->r2, ip->c2);
				PIP_roi_mask(im[luFirstIndex]);
				PIP_roi(im[luSecondIndex], ip->r1, ip->c1, ip->r2, ip->c2);
				PIP_roi_mask(im[luSecondIndex]);
				PIP_bivariant(tempmap, im[luFirstIndex], im[luSecondIndex]);
				PIP_unary(tempmap, '>', 0.7*boxArea);
				PIP_maths(map,'|',tempmap);
				PIP_unary(map, '>', 0);
				display_images(0);
				break;
			case 'u': /* enlargen the selecting box */
				boxSize++;
				printf("box size = %d\n", 2*boxSize+1);	
				break;
			case 'd': /* shrink the selecting box */
				boxSize--;
				if (boxSize<1) boxSize = 1;
				printf("box size = %d\n", 2*boxSize+1);	
				break;	
			case 'r':	/* redraw the training image */
				display_images(0);
				break;
			case 'c': /* perform morphological closing of the histogram  */
				PIP_dilate(map, 1);
				PIP_erode(map, 3);
				display_images(0);
				break;
			case 'o':	/* perform morphological opening of the histogram */
				PIP_erode(map, 1);
				PIP_dilate(map, 3);
				display_images(0);
				break;
			case 'n':	/* renew the mask and histogram images */
				PIP_unary(map, ':', 0);
				PIP_unary(workim, ':', 0);
				display_images(0);
				break;
			case 's':	/* save the lookup image */
				pip_silent = 0;
				PIP_save(map, LTfilename);
				pip_silent = 1;
				break;
			case 'm':	/* draw the mask */
				display_images(1);
				if (Maskim->data != NULL)
					PIP_unary(Maskim, '=',0);
				break;
			case 'w':	/* save segmented image */
				pip_silent = 0;
				sprintf(saveimfname, "im%03d.ppm", imNum++);
				PIP_save(workim, saveimfname);
				pip_silent = 1;
				break;
			case ' ':	/* grab new image from store */
				if(ddxOn){
					if(grab_ddx_image() < 0){
						XID_Close();
						exit(0);
					}
					for(i = 0;i<3; i++)
						memcpy(rgb[i]->data, im[i]->data, im[i]->nr*im[i]->nc);
					/* Transform the input image into a RGB space for display*/
					PIP_YCrCbtoRGB(rgb);
					format_images();
					display_images(0);
				}
				else
					fprintf(stderr, "STORE is not on!!\n");

				break;
			case '?':
				print_help();
				break;	
			case 'q': XID_Close(); exit(0);
			default:
					  break;
		}	
	}
	if(ddxOn)
		store_disconnect();
	return (0);
}


/**
* Does the blob extraction via a 2-d lookup table.
*/
int
blobs_get(Pimage * im[], Pimage *colLookup, Pimage *Workim, Pimage *Maskim, int pMaxBlobArea, int pMinBlobArea,  int debug_im)
{
	char *func_name = "blobs_get";
	Plabel			blobs;
	int maskOn = 0;

	if (PIP_lookup(Workim, im[luFirstIndex], im[luSecondIndex], colLookup) == -1)
		return rtx_error("%s: PIP_lookup() failed", func_name);
	if (maskOn)
		if (PIP_maths(Workim, '&', Maskim) == -1)
			return rtx_error("%s: PIP_maths(&) failed", func_name);

	/* extract the blobs */
	blobs.n = PIP_label(Workim, 4, pMinBlobArea, 0);
	if (blobs.n < 0)
		return rtx_error("%s: PIP_label() failed", func_name);
	if (debug_im)
		fprintf(stderr, "%d blobs found\n", blobs.n);

	return(0);
}



int
display_images(int drawMask)
{
	static int firstCall = 1;
	static XIDwin *t, *h, *w;
	if (firstCall){
		if( XID_Open(NULL) ) exit(-1);
		t = XID_WindowCreate("Thresholded Image",im[dispIndex]->nc,im[dispIndex]->nr,0,0,0,0);
		h = XID_WindowCreate("Bivariant Histogram", 256, 256, 100, 100, 0, 0);
		w = XID_WindowCreate("Training Image",im[dispIndex]->nc,im[dispIndex]->nr,0,0,0,0); 
		firstCall = 0;
	}	
#if 1
	/* draw the input image */
	XID_DrawImageRGB(w, rgb[0]->data, rgb[1]->data, rgb[2]->data,
			rgb[0]->nc, rgb[0]->nr);				
	XID_DrawBox(w, ip->c1, ip->r1, ip->c2-ip->c1, ip->r2-ip->r1, 0);

	/* draw the histogram */
	XID_DrawClear(h);
	XID_DrawImage(h, map->data, map->nc, map->nr);
	/* draw the thresholded stuff */
	blobs_get(im, map, workim, Maskim, pMaxBlobArea, pMinBlobArea, 0);
	XID_DrawClear(t);
	XID_DrawImageOver(t,im[dispIndex]->data,workim->data,workim->nc,workim->nr);
#endif  
	if (drawMask){
		if (Maskim->data == NULL)
			fprintf(stderr, "No mask loaded!!\n");
		else	
			XID_DrawImageOver(t, im[dispIndex]->data, Maskim->data, Maskim->nc, Maskim->nr);
	}	
	return(0);
}	


int
print_help(void)
{
	printf("Training image to load is  `%s'.\n",infilename);
	printf("Output lookup table will be written to `%s' and\n", LTfilename);
	printf("the format of the lookup table will be `%s'\n", outputformat);
	printf("Press `r' to redraw images\n");
	printf("Press `o' to morphologically open the histogram\n");
	printf("Press `c' to morphologically close the histogram\n");
	printf("Press `n' to clear the current histogram and thresholded image\n");
	printf("Press `s' to save the histogram (in %s)\n", LTfilename);
	printf("Press `m' to display the mask used\n");
	printf("Press `w' to save segmented image\n");
	printf("Press `q' to quit\n");
	printf("Initial selection box size is %d (centred on the point selected).\n",2*boxSize+1);
	printf("This can be changed to any odd positive number within the\n");
	printf("limits of the image size.\n");
	printf("Press `u' to increase the selection box size\n");
	printf("Press `d' to decrease the selection box size\n");
	printf("Press `g' to grab new image from the store\n");
	return(0);
}
