
/** \file
Copyright (c) CSIRO Robotics
\brief 		Capture video and copy to store
\author		Elliot Duff 
\todo		Fix up colour bleed from interlaced image
\todo		Correctly deinterlace PAL into 4 fields 
\todo		Get correct timestamp from video frame

This programs capture video from Video4Linux driver and 
copies result into the store.

\section 	sizes Image sizes

The V4L driver support any number of resolutions, however there are a
number of predefined names that can be used.
- PAL - 768x576
- CIF - 352x288 
- 4CIF - 704x576
- QCIF - 176x144

\section 	stereo Stereo
 In line interlacing mode, the resulting image is made from 4 potential sources;
- left odd (white square in top left)
- left even (white square in top left 1/50 latter)
- right odd
- right even

By default these images are interlaced, with half of the data being thrown away.\n
For a CIF image, the even field is also thrown way, so we are left with only two images:
- left odd
- right odd

\image html cif_mux.jpg "CIF line-interlaced stereo image"
\image latex cif_mux.eps "CIF line-interlaced stereo image" width=10cm

A better way to handle a stereo image, is to use the \c deinterlace flag, so that
the image is deinterlaced BEFORE it gets into the store. So rather than having 
a left and right image, we have a top an bottom image. 

\image html double.jpg "CIF de-interlaced image"
\image latex double.eps "CIF de-interlaced image" width=10cm

\warning Interlaced colour is not supported very well under V4L. It should be better under V4L2, but to avoid some confusion, the colour has been removed from the second image
*/

static char    *rcsid =
	"$Header$";

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>		// Needed by open()
#include <sys/ioctl.h>		// Needed by ioctl()
#include <sys/mman.h>		// Needed by PROT and MAP
#include <fcntl.h>		// Needed by O_RDWR
#include <errno.h>		// Needed by perror()
#include <string.h>		// Needed strerror()

#include <linux/types.h>	// Video 4 Linux stuff
#include <linux/videodev.h>

#include <rtx/message.h>	// Real Time Extensions
#include <rtx/time.h>
#include <rtx/error.h>
#include <rtx/thread.h>
#include <rtx/main.h>
#include <rtx/getopt.h>

int             verbose = 0;	/*!< Print debugging information */
int             skip = 1;	/*!< Number of frames to skip */
int             limit = -1;	/*!< Run for ever */
int             deinterlace = 0;	/*!< Deinterlace the video stream */

int             camera = 0;	/*!< Camera Number (0 to 3) */
int             colour = 32768;	/*!< Colour Saturation */
int             contrast = 32768;	/*!< Picture Contrast */
char           *device = "/dev/video0";	/*!< Name of Video Device */
char           *size = "PAL";	/*!< String Describing Video Size */
char           *name = "video";	/*!< Name of Video in store  */

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#include <ddx.h>
#include "ddxvideo.h"
DDX_STORE_ID   *storeId = NULL;	// Pointer to Store 
DDX_STORE_ITEM *itemPtr = NULL;	// Pointer to Store Item 

int             done = 0;	// Flag to stop Capture Thread 
int             width;		/*!< Width of image */
int             height;		/*!< Height of image */

int             videoNorm;
int             videoFrame;
int             videoFormat;
int             videoHandle;	// The Video Device handler 
char           *videoMemory;	// Pointer to Video Memory 

struct video_capability vid_cap;
struct video_channel vid_chn;
struct video_mbuf vid_buf;
struct video_mmap vid_map;
struct video_picture vid_pic;
#endif

void           *
capture_thread(void *arg)
{
	int             i, np;
	int             count = 0;
	int             frame = 0;
	RtxTime         ts;
	void           *p;
	DDX_VIDEO      *v;
	int             w1, w2, h1, h2, h4;

	v = (DDX_VIDEO *) ddx_store_var_pointer(itemPtr);

	np = width * height;
	w1 = width;
	w2 = width / 2;
	h1 = height;
	h2 = height / 2;
	h4 = height / 4;

	if (deinterlace) {
		v->fields = 2;
		v->width = width;
		v->height = height / 2;
	}
	else {
		v->fields = 1;
		v->width = width;
		v->height = height;
	}

	while (!done) {

		count++;
		rtx_time_get(&ts);

		// Start copying to the next frame 

		vid_map.frame = (count + 1) % vid_buf.frames;
		if (ioctl(videoHandle, VIDIOCMCAPTURE, &vid_map) == -1) {
			perror("VIDIOCMCAPTURE");
			exit(-1);
		}

		// Stop writing to the old frame 

		videoFrame = count % vid_buf.frames;
		while (((ioctl(videoHandle, VIDIOCSYNC, &videoFrame) < 0)
			&& ((errno == EAGAIN) || errno == EINTR)));

		// Skip every nth frame, if skip == 1 don''t skip.

		if (count % skip != 0)
			continue;

		// Insert the image into the store

		v->frame = frame++;

		if (deinterlace) {
			p = videoMemory + vid_buf.offsets[videoFrame];

			for (i = 0; i < h2; i++) {
				memcpy(v->y + i * w1, p, w1);
				p += w1;
				memcpy(v->y + (h2 + i) * w1, p, w1);
				p += w1;
			}
			for (i = 0; i < h4; i++) {
				memcpy(v->v + i * w2, p, w2);
				p += w1;
				memset(v->v + (h4 + i) * w2, 128, w2);
				p += w1;
			}
			// memcpy(v->v+(h4+i)*w2,p,w2); p += w1; }
			for (i = 0; i < h4; i++) {
				memcpy(v->u + i * w2, p, w2);
				p += w1;
				memset(v->u + (h4 + i) * w2, 128, w2);
				p += w1;
			}
			// memcpy(v->u+(h4+i)*w2,p,w2); p += w1; }
		}
		else {
			p = videoMemory + vid_buf.offsets[videoFrame];
			memcpy(v->y, p, np);
			p += np;	// Y in FIFO 1
			memcpy(v->v, p, np / 4);
			p += np / 4;	// Cb in FIFO 2
			memcpy(v->u, p, np / 4);	// Cr in FIFO 3
		}

		if (ddx_store_write_direct(itemPtr, NULL) == -1)
			rtx_error("ddx_store_write: %s", strerror(errno));

		if (verbose)
			fprintf(stderr, "%d %d %d\r", frame, (int) ts.seconds,
				(int) ts.nanoSeconds);
		if (frame == limit)
			rtx_main_signal_shutdown();
	}

	return (NULL);
}

char           *help =
	"CSIRO Video Server Project \nCapture video and copy to Store";

RtxGetopt       myOpts[] = {
	{"name", "Name of Video in Store",
	 {{RTX_GETOPT_STR, &name, "name"}, RTX_GETOPT_END_ARG}},
	{"deinterlace", "Deinterlace image",
	 {{RTX_GETOPT_SET, &deinterlace, "deinterlace"}, RTX_GETOPT_END_ARG}},
	{"skip", "Skip frames",
	 {{RTX_GETOPT_INT, &skip, "skip"}, RTX_GETOPT_END_ARG}},
	{"limit", "Run until frame equals limit",
	 {{RTX_GETOPT_INT, &limit, "limit"}, RTX_GETOPT_END_ARG}},
	{"camera", "Select video camera",
	 {{RTX_GETOPT_INT, &camera, "camera"}, RTX_GETOPT_END_ARG}},
	{"colour", "Set colour saturation",
	 {{RTX_GETOPT_INT, &colour, "colour"}, RTX_GETOPT_END_ARG}},
	{"contrast", "Set video contrast",
	 {{RTX_GETOPT_INT, &contrast, "contrast"}, RTX_GETOPT_END_ARG}},
	{"device", "Select video device",
	 {{RTX_GETOPT_STR, &device, "device"}, RTX_GETOPT_END_ARG}},
	{"size", "Select video size",
	 {{RTX_GETOPT_STR, &size, "size"}, RTX_GETOPT_END_ARG}},
	RTX_GETOPT_END
};

int
main(int ac, char *av[])
{
	int             i;
	RtxThread      *th = NULL;

	if (RTX_GETOPT_CMD(myOpts, ac, av, rcsid, help) == -1) {
		RTX_GETOPT_PRINT(myOpts, av[0], rcsid, help);
		exit(-1);
	}

	switch (size[0]) {
	case 'P':
		width = 768;
		height = 576;
		break;
	case '4':
		width = 704;
		height = 576;
		break;
	case 'C':
		width = 352;
		height = 288;
		break;
	case 'Q':
		width = 176;
		height = 144;
		break;
	default:
		sscanf(size, "%dx%d", &width, &height);
	}

	if (deinterlace) {
		videoFrame = 0;
		videoFormat = VIDEO_PALETTE_YUV422P;
	}
	else {
		videoFrame = 0;
		videoFormat = VIDEO_PALETTE_YUV420P;
	}

	// Open Video4Linux Channel

	vid_chn.channel = 1;	/* Need to set a channel before opening device */
	if (verbose)
		printf("Opening [%s] Camera [%d]\n", device, camera);
	for (i = 5; i > 0; i--) {
		videoHandle = open(device, O_RDWR);
		if (videoHandle == -1)
			sleep(1);
		else
			break;
	}
	if (i == 0) {
		fprintf(stderr, "Can't open [%s]\n", device);
		exit(-1);
	}

	if (ioctl(videoHandle, VIDIOCGCAP, &vid_cap) == -1) {
		perror("VIDIOCGCAP");
		exit(-1);
	}
	if (ioctl(videoHandle, VIDIOCGCHAN, &vid_chn) == -1) {
		perror("VIDIOCGCHAN");
		exit(-1);
	}
	if (ioctl(videoHandle, VIDIOCGMBUF, &vid_buf) == -1) {
		perror("VIDIOMBUF");
		exit(-1);
	}
	if (ioctl(videoHandle, VIDIOCGPICT, &vid_pic) == -1) {
		perror("VIDIOCGPICT");
		exit(-1);
	}

	if ((videoMemory =
	     mmap(0, vid_buf.size, PROT_READ | PROT_WRITE, MAP_SHARED,
		  videoHandle, 0)) == ((void *) -1)) {
		perror("mmap()");
		exit(-1);
	}

	if (verbose) {
		fprintf(stderr, "Brightness : %d\n", vid_pic.brightness);
		fprintf(stderr, "Whiteness  : %d\n", vid_pic.whiteness);
		fprintf(stderr, "Hue        : %d\n", vid_pic.hue);
		fprintf(stderr, "Colour     : %d\n", vid_pic.colour);
		fprintf(stderr, "Contrast   : %d\n", vid_pic.contrast);
		fprintf(stderr, "Depth      : %d\n", vid_pic.depth);
		fprintf(stderr, "Palette    : %d\n", vid_pic.palette);
	}

	vid_pic.colour = colour;
	vid_pic.contrast = contrast;
	vid_map.width = width;
	vid_map.height = height;
	vid_map.frame = videoFrame;
	vid_map.format = videoFormat;
	vid_chn.norm = videoNorm;
	vid_chn.channel = camera;

	if (ioctl(videoHandle, VIDIOCSPICT, &vid_pic) == -1) {
		perror("VIDIOCGPICT");
		exit(-1);
	}
	if (ioctl(videoHandle, VIDIOCSCHAN, &vid_chn) == -1) {
		perror("VIDIOCSCHAN");
		exit(-1);
	}

	rtx_main_init(av[0], 0);

	if (ddx_client_init(0) == -1)
		return (rtx_error("Unable to initialize library"));

	if ((storeId = ddx_store_open(NULL, 0, 5)) == NULL)
		return (rtx_error("Unable to open store"));

	if (DDX_STORE_REGISTER_TYPE(storeId, DDX_VIDEO) == -1)
		return (rtx_error("Unable to register type"));

	if ((itemPtr =
	     ddx_store_lookup_item(storeId, name, "DDX_VIDEO", 0)) == NULL)
		return (rtx_error("Unable to lookup var"));

	if ((th = rtx_thread_create("capture thread", 0,
				    RTX_THREAD_SCHED_OTHER, 0, 0,
				    RTX_THREAD_CANCEL_DEFERRED,
				    capture_thread, NULL,
				    NULL, NULL)) == NULL) {
		rtx_error_flush("rtx_thread_create() failed");
		exit(-1);
	}

	if (rtx_main_wait_shutdown(90))
		rtx_error_flush("rtx_main_wait_shutdown() failed");

	done = 1;
	rtx_thread_destroy_sync(th);
	ddx_store_done_item(itemPtr);
	ddx_store_close(storeId);
	ddx_client_done();
	close(videoHandle);
	munmap(videoMemory, vid_buf.size);
	exit(0);
}
