
/** \file
Copyright (c) CSIRO Robotics
\brief 		Capture video and copy to store - Adhoc version for Sub
\author		Elliot Duff 

This programs capture video from Video4Linux driver and 
copies 3 images into the store (2 stereo at CIF resolution and 1 at PAL).

\warning 	V4L can only capture frames (not fields). Thus there is some 
unusual behaviour when we process Muxed images. See \ref ddxvideoV4L.c
*/

static char    *rcsid =
	"$Header$";

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>		// Needed by open()
#include <sys/ioctl.h>		// Needed by ioctl()
#include <sys/mman.h>		// Needed by PROT and MAP
#include <sys/stat.h>
#include <fcntl.h>		// Needed by O_RDWR
#include <errno.h>		// Needed by perror()
#include <string.h>		// Needed strerror()
#include <unistd.h>

#include <linux/types.h>	// Video 4 Linux stuff
#include <linux/videodev.h>

#include <rtx/message.h>	// Real Time Extensions
#include <rtx/time.h>
#include <rtx/error.h>
#include <rtx/thread.h>
#include <rtx/main.h>
#include <rtx/getopt.h>

int             verbose = 0;	/*!< Print debugging information */
int             limit = -1;	/*!< Run for ever */
int             colour = 32768;	/*!< Colour Saturation */
int             contrast = 32768;	/*!< Picture Contrast */
char           *device = "/dev/video0";	/*!< Name of Video Device */
char           *name1 = "video";	/*!< Name of Video in store  */
char           *name2 = "down";	/*!< Name of Video in store  */
char           *name3 = "front";	/*!< Name of Video in store  */

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#include <ddx.h>
#include "ddxvideo.h"
DDX_STORE_ID   *storeId = NULL;	// Pointer to Store 
DDX_STORE_ITEM *itemPtr1 = NULL;	// Pointer to Store Item 
DDX_STORE_ITEM *itemPtr2 = NULL;	// Pointer to Store Item 
DDX_STORE_ITEM *itemPtr3 = NULL;	// Pointer to Store Item 

int             done = 0;	// Flag to stop Capture Thread 
int             videoHandle;	// The Video Device handler 
char           *videoMemory;	// Pointer to Video Memory 
struct video_capability vid_cap;
struct video_channel vid_chn;
struct video_mbuf vid_buf;
struct video_mmap vid_map;
struct video_picture vid_pic;
#endif

void           *
capture_thread(void *arg)
{
	int             i, np, fd;
	int             frame = -1;	// Number of frames read
	void           *p;	// Pointer to video memory 
	int             w1, w2, h1, h2, h4;
	int             width, height, fields;
	int             videoFormat;
	int             videoFrame = 0;
	int             stereo = 0;
	RtxTime         ts;
	DDX_VIDEO      *v, *v1, *v2, *v3;
	DDX_STORE_ITEM *ip;
	const char      MUXmono = 0x00;
	const char      MUXstereo = 0x18;

	v1 = (DDX_VIDEO *) ddx_store_var_pointer(itemPtr1);	// Down PAL 
	v2 = (DDX_VIDEO *) ddx_store_var_pointer(itemPtr2);	// Down Stereo
	v3 = (DDX_VIDEO *) ddx_store_var_pointer(itemPtr3);	// Forward Stereo

	// Switch Downward Camera MUX to stereo

	if ((fd = open("/dev/port", O_RDWR)) == -1) {
		perror("Failed to open parallel port");
		exit(1);
	}
	if (lseek(fd, 0x378, SEEK_SET) == -1) {
		perror("Failed to seek parallel port");
		exit(2);
	}
	if (write(fd, &MUXstereo, 1) != 1) {
		perror("Failed to write to parallel port");
		exit(3);
	}

	// Start on Downward Camera */

	stereo = 1;
	vid_chn.norm = 0;
	vid_chn.channel = 2;
	if (ioctl(videoHandle, VIDIOCSCHAN, &vid_chn) == -1) {
		perror("VIDIOCSCHAN");
		exit(-1);
	}

	while (!done) {

		rtx_time_get(&ts);
		if (++frame == limit)
			rtx_main_signal_shutdown();

		/*
		 ** Assuming 25Hz - consider the next 2 seconds 
		 ** V is a pointer to the current video frame in the store
		 ** By default V is NULL which means that we should Grab a frame 
		 ** but don't write to the store
		 */

		v = NULL;

		switch (frame % 50) {
		case 0:
			v = v2;
			ip = itemPtr2;
			break;
		case 5:
			v = v2;
			ip = itemPtr2;
			break;
		case 10:
			v = v2;
			ip = itemPtr2;
			break;
		case 15:
			v = v2;
			ip = itemPtr2;
			break;

		case 16:
			vid_chn.channel = 3;	// Switch to Forward Camera 
			if (ioctl(videoHandle, VIDIOCSCHAN, &vid_chn) == -1) {
				perror("VIDIOCSCHAN");
				exit(-1);
			}
			break;	// Wait four frames to settle 

		case 20:
			v = v3;
			ip = itemPtr3;
			break;

		case 21:
			vid_chn.channel = 2;	// Switch back to Downward Camera
			if (ioctl(videoHandle, VIDIOCSCHAN, &vid_chn) == -1) {
				perror("VIDIOCSCHAN");
				exit(-1);
			}
			break;	// Wait four frames to settle 

		case 25:
			v = v2;
			ip = itemPtr2;
			break;
		case 30:
			v = v2;
			ip = itemPtr2;
			break;
		case 35:
			v = v2;
			ip = itemPtr2;
			break;
		case 40:
			v = v2;
			ip = itemPtr2;
			break;

		case 41:	// Insert code to switch to single downward camera
			stereo = 0;
			printf("Mono\n");
			if (write(fd, &MUXmono, 1) != 1) {
				perror("write failed");
				exit(3);
			}
			break;

		case 45:
			v = v1;
			ip = itemPtr1;
			break;

		case 46:	// Insert code to switch back to stereo
			stereo = 1;
			printf("Stereo\n");
			if (write(fd, &MUXstereo, 1) != 1) {
				perror("write failed");
				exit(3);
			}
			break;
		}

		if (stereo) {
			// Use Half size image to avoid motion tear
			fields = 2;
			width = 384;
			height = 288;
			videoFormat = VIDEO_PALETTE_YUV422P;
		}
		else {
			fields = 1;
			width = 768;
			height = 576;
			videoFormat = VIDEO_PALETTE_YUV420P;
		}

		// Start copying to the next frame 

		vid_map.width = width;
		vid_map.height = height;
		vid_map.format = videoFormat;
		vid_map.frame = (frame + 1) % vid_buf.frames;
		if (ioctl(videoHandle, VIDIOCMCAPTURE, &vid_map) == -1) {
			perror("VIDIOCMCAPTURE");
			exit(-1);
		}

		// Stop writing to the old frame (ring buffer)

		videoFrame = frame % vid_buf.frames;
		while (((ioctl(videoHandle, VIDIOCSYNC, &videoFrame) < 0)
			&& ((errno == EAGAIN) || errno == EINTR)));

		// Skip this frame if pointer is NULL

		if (v == NULL)
			continue;
		if (verbose)
			printf("%d %d %p %d\n", frame, frame % 50, v, stereo);

		// Insert the image into the store

		v->frame = frame;
		v->fields = fields;
		v->width = width;
		v->height = height / fields;
		p = videoMemory + vid_buf.offsets[videoFrame];

		if (stereo) {
			// Deinterlace stereo image 
			// It appears that the odd lines in UV space are ignored 
			w1 = width;
			w2 = width / 2;
			h1 = height;
			h2 = height / 2;
			h4 = height / 4;
			for (i = 0; i < h2; i++) {
				memcpy(v->y + i * w1, p, w1);
				p += w1;	// Y Top Image
				memcpy(v->y + (h2 + i) * w1, p, w1);
				p += w1;
			}	// Y Bottom Image
			for (i = 0; i < h4; i++) {
				memcpy(v->v + i * w2, p, w2);
				p += w1;	// V Top Image
				memset(v->v + (h4 + i) * w2, 128, w2);
				p += w1;
			}	// V Bottom Image
			// memcpy(v->v+(h4+i)*w2,p,w2); p += w1; }
			for (i = 0; i < h4; i++) {
				memcpy(v->u + i * w2, p, w2);
				p += w1;	// U Top Image
				memset(v->u + (h4 + i) * w2, 128, w2);
				p += w1;
			}	// U Bottom Image
			// memcpy(v->u+(h4+i)*w2,p,w2); p += w1; }
		}
		else {
			np = width * height;
			memcpy(v->y, p, np);
			p += np;	// Y in FIFO 1
			memcpy(v->v, p, np / 4);
			p += np / 4;	// Cb in FIFO 2
			memcpy(v->u, p, np / 4);	// Cr in FIFO 3
		}

		if (ddx_store_write_direct(ip, NULL) == -1)
			rtx_error("ddx_store_write: %s", strerror(errno));

		if (verbose)
			fprintf(stderr, "%d %d %d\r",
				frame, (int) ts.seconds, (int) ts.nanoSeconds);
	}

	return (NULL);
}

char           *help =
	"CSIRO Video Server Project \nCapture video and copy to Store";

RtxGetopt       myOpts[] = {
	{"limit", "Run until frame equals limit",
	 {{RTX_GETOPT_INT, &limit, "limit"}, RTX_GETOPT_END_ARG}},
	{"colour", "Set colour saturation",
	 {{RTX_GETOPT_INT, &colour, "colour"}, RTX_GETOPT_END_ARG}},
	{"contrast", "Set video contrast",
	 {{RTX_GETOPT_INT, &contrast, "contrast"}, RTX_GETOPT_END_ARG}},
	{"device", "Select video device",
	 {{RTX_GETOPT_STR, &device, "device"}, RTX_GETOPT_END_ARG}},
	RTX_GETOPT_END
};

int
main(int ac, char *av[])
{
	int             i;
	RtxThread      *th = NULL;

	if (RTX_GETOPT_CMD(myOpts, ac, av, rcsid, help) == -1) {
		RTX_GETOPT_PRINT(myOpts, av[0], rcsid, help);
		exit(-1);
	}

	// Open Video4Linux Channel

	vid_chn.channel = 1;	/* Need to set a channel before opening device */
	if (verbose)
		printf("Opening [%s]\n", device);
	for (i = 5; i > 0; i--) {
		videoHandle = open(device, O_RDWR);
		if (videoHandle == -1)
			sleep(1);
		else
			break;
	}
	if (i == 0) {
		fprintf(stderr, "Can't open [%s]\n", device);
		exit(-1);
	}

	if (ioctl(videoHandle, VIDIOCGCAP, &vid_cap) == -1) {
		perror("VIDIOCGCAP");
		exit(-1);
	}
	if (ioctl(videoHandle, VIDIOCGCHAN, &vid_chn) == -1) {
		perror("VIDIOCGCHAN");
		exit(-1);
	}
	if (ioctl(videoHandle, VIDIOCGMBUF, &vid_buf) == -1) {
		perror("VIDIOMBUF");
		exit(-1);
	}
	if (ioctl(videoHandle, VIDIOCGPICT, &vid_pic) == -1) {
		perror("VIDIOCGPICT");
		exit(-1);
	}

	if ((videoMemory =
	     mmap(0, vid_buf.size, PROT_READ | PROT_WRITE, MAP_SHARED,
		  videoHandle, 0)) == ((void *) -1)) {
		perror("mmap()");
		exit(-1);
	}

	vid_pic.colour = colour;
	vid_pic.contrast = contrast;
	if (ioctl(videoHandle, VIDIOCSPICT, &vid_pic) == -1) {
		perror("VIDIOCGPICT");
		exit(-1);
	}

	/*
	 ** Store Stuff 
	 */

	rtx_main_init(av[0], 0);

	if (ddx_client_init(0) == -1)
		return (rtx_error("Unable to initialize library"));

	if ((storeId = ddx_store_open(NULL, 0, 5)) == NULL)
		return (rtx_error("Unable to open store"));
	if (DDX_STORE_REGISTER_TYPE(storeId, DDX_VIDEO) == -1)
		return (rtx_error("Unable to register type"));
	if ((itemPtr1 = ddx_store_lookup_item(storeId, name1, "DDX_VIDEO", 0))
	    == NULL)
		return (rtx_error("Unable to lookup var"));
	if ((itemPtr2 = ddx_store_lookup_item(storeId, name2, "DDX_VIDEO", 0))
	    == NULL)
		return (rtx_error("Unable to lookup var"));
	if ((itemPtr3 = ddx_store_lookup_item(storeId, name3, "DDX_VIDEO", 0))
	    == NULL)
		return (rtx_error("Unable to lookup var"));

	if ((th = rtx_thread_create("capture thread", 0,
				    RTX_THREAD_SCHED_OTHER, 0, 0,
				    RTX_THREAD_CANCEL_DEFERRED,
				    capture_thread, NULL,
				    NULL, NULL)) == NULL) {
		rtx_error_flush("rtx_thread_create() failed");
		exit(-1);
	}

	if (rtx_main_wait_shutdown(90))
		rtx_error_flush("rtx_main_wait_shutdown() failed");

	done = 1;
	rtx_thread_destroy_sync(th);
	ddx_store_done_item(itemPtr1);
	ddx_store_done_item(itemPtr2);
	ddx_store_done_item(itemPtr3);
	ddx_store_close(storeId);
	ddx_client_done();
	close(videoHandle);
	munmap(videoMemory, vid_buf.size);
	exit(0);
}
