/*********************************************************************
 *
 * Csiro Automation
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * www.cat.csiro.au/cmst
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 *********************************************************************/

static char *rcsid = "$Id: ddxread-main.c 1360 2006-09-21 03:57:13Z duf048 $";

/**
 * \file 
 * \brief DDX log file reader
 * \author Mike Bosse and Julian Ryde
 *
 */

#undef MATLAB 

/**
 * \page ddxread ddxread
 *
 * The DDX log file reader (ddxread) can be used to read log files
 * generated by \ref ddxlog. The program can output logged data in ASCII
 * tabular format.
 *
 * The program can be used as follows:
 * - ddxread [options] <log-file> <var1> .... <varN>
 * - ddxread [options] [-1|-2|-3|-4|-5] <var1> ... <varN>
 *
 * ddxread supports the following options:
 * - -c\n
 *    extract comments only from the log file (currently not supported)
 * - -r\n
 *    print timestamps relative to the first timestamp in the log file
 * - -l\n
 *    print a list of all variables in the log file (along with the
 *    number of records)
 * - -n\n
 *    print a list of all the variables in the log file
 * - -t\n
 *    print the timestamps for the specified variables (if no variables
 *    are specified, all the varibales are considered)
 * - -v\n
 *    turn debugging on
 * - -q\n
 *    if a specified variable is not in the file, ignore the error and
 *    continue. The default behavior is to exit.
 * - -o <filename>\n
 *    print data in tabular ASCII format to the specified file. If this
 *    option is not used, the program prints to stdout.
 * - -1\n
 * - -2\n
 * - -3\n
 * - -4\n
 * - -5\n
 *    sort all the log files in the directory by timestamp (in descending
 *    order) and then choose the one specified
 * - -h\n
 *    print a help message and exit
 *
 * The options -l and -n ignore the variable names specified on the command
 * line.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <strings.h>
#include <string.h>
#include <unistd.h>
#include <dirent.h>

#include <rtx/logread.h>
#include <rtx/getopt.h>

#ifdef	MATLAB
#include "mat.h"
#endif

#define  dtime(ts)  ((ts).seconds + (ts).nanoSeconds*1e-9)

int             verbose;
int		quiet_mode;
int             listmode = 0;
int             namemode = 0;
int             timemode = 0;
int             relative_timing = 0;
int             first = 0;
int             second = 0;
int             third = 0;
int             fourth = 0;
int             fifth = 0;
char            *pname;
pid_t           pid;
RtxLogread      *readlog_log;

#ifdef	MATLAB
static void     matlab_append_t(RtxLogreadVar * v);
static int      matlab_append(RtxParseVar * v, char *d, int offset);
void            put_structure(MATFile * fp, char *name);
void            new_field(char *name, RtxLogreadMatlabHeader * mh);
void            matlab_write(char *fname);
#endif

void            save_video_frame(FILE *fp, RtxLogreadVar *rv);

void            new_structure(char *name);

char           *tabularfile = NULL, *matfile = NULL;
int             commentMode;

char *videofile = NULL;

static double   t0 = 0.0;

char            logfilename[BUFSIZ];
int             numTopVars;

int             type_size[] = {
	0,
	0,
	sizeof(char),
	sizeof(short),
	sizeof(int),
	sizeof(long),
	sizeof(float),
	sizeof(double),
};

/* command-line options */

RtxGetopt ddxlogreadOpts[] = {
  {"-1", "select first log file in directory", 
   {
     {RTX_GETOPT_SET, &first, ""},
     RTX_GETOPT_END_ARG
   }
  },
  {"-2", "select second log file in directory", 
   {
     {RTX_GETOPT_SET, &second, ""},
     RTX_GETOPT_END_ARG
   }
  },
  {"-3", "select third log file in directory", 
   {
     {RTX_GETOPT_SET, &third, ""},
     RTX_GETOPT_END_ARG
   }
  },
  {"-4", "select fourth log file in directory", 
   {
     {RTX_GETOPT_SET, &fourth, ""},
     RTX_GETOPT_END_ARG
   }
  },
  {"-5", "select fifth log file in directory", 
   {
     {RTX_GETOPT_SET, &fifth, ""},
     RTX_GETOPT_END_ARG
   }
  },
  {"-comments", "extract comments from log file", 
   {
     {RTX_GETOPT_SET, &commentMode, ""},
     RTX_GETOPT_END_ARG
   }
  },
  {"-relative", "compute timestamps relative to the first timestamp in the log file", 
   {
     {RTX_GETOPT_SET, &relative_timing, ""},
     RTX_GETOPT_END_ARG
   }
  },
  {"-list", "list all records in the log file", 
   {
     {RTX_GETOPT_SET, &listmode, ""},
     RTX_GETOPT_END_ARG
   }
  },
  {"-name", "list all variable names in the log file", 
   {
     {RTX_GETOPT_SET, &namemode, ""},
     RTX_GETOPT_END_ARG
   }
  },
  {"-timestamp", "list all timestamps in the log file", 
   {
     {RTX_GETOPT_SET, &timemode, ""},
     RTX_GETOPT_END_ARG
   }
  },
  {"-quiet", "work quietly", 
   {
     {RTX_GETOPT_SET, &quiet_mode, ""},
     RTX_GETOPT_END_ARG
   }
  },
  {"-matlab", "write output in binary MATLAB format", 
   {
     {RTX_GETOPT_STR, &matfile, "file name"},
     RTX_GETOPT_END_ARG
   }
  },
  {"-output", "write output in tabular text format to specified file", 
   {
     {RTX_GETOPT_STR, &tabularfile, "filename"},
     RTX_GETOPT_END_ARG
   }
  },
  {"-videofname", "write output in a raw YUV4MPEG format to specified file", 
   {
     {RTX_GETOPT_STR, &videofile, "filename"},
     RTX_GETOPT_END_ARG
   }
  },
  RTX_GETOPT_END
};

char * ddxlogreadHelpStr = "Notes:\n"
    "1. The log file to be read is specified by the\n"
    "options -1, -2, -3, -4, -5 or\n"
    "by the first non-option argument.\n"
    "The remaining arguments specify the variables\n"
    "to be extracted from the log file.\n";

int
main(int ac, char *av[])
{
	int            count;
	char           **files;
	int            nfiles;
	int            ret;
	FILE           *fp = NULL;
	RtxLogreadVar      *rv;
	int            fileno = -1;

	pname = av[0];

	if ((ret = RTX_GETOPT_CMD (ddxlogreadOpts, ac, av, rcsid, ddxlogreadHelpStr)) == -1) {
	        RTX_GETOPT_PRINT (ddxlogreadOpts, av[0], rcsid, ddxlogreadHelpStr);
		exit (1);
	}
	if (ret == 0)
	        exit (1);

	verbose = rtx_getopt_get_verbose (0);

	if (first)
	        fileno = 0;
	else if (second)
	        fileno = 1;
	else if (third)
	        fileno = 2;
	else if (fourth)
	        fileno = 3;
	else if (fifth)
	        fileno = 4;

	if (fileno >= 0) {
	        files = rtx_logread_get_files(".", &nfiles);
		if (fileno >= nfiles) {
		        fprintf(stderr, "only %d files to choose from\n",
				nfiles);
			exit (1);
		}
		sprintf(logfilename, "%s", files[fileno]);
		fprintf(stderr, "opening logfile %s\n",
			logfilename);
	} else {
		strcpy(logfilename, av[ret++]);
	}

	if ( (tabularfile!=NULL) + (matfile!=NULL) + (videofile!=NULL) > 1) {
		fprintf(stderr, "select one of -m or -o or -v\n");
		exit(1);
	}
	else if (matfile) {
#ifndef	MATLAB
		fprintf(stderr,
			"MATLAB output not supported on this architecture\n");
		exit(1);
#endif
	}
	else if (tabularfile) {
		if ((fp = fopen(tabularfile, "w")) == NULL) {
			fprintf(stderr, "cant open output file %s\n", tabularfile);
			exit(1);
		}
	}
    else if (videofile) {
      if ((fp = fopen(videofile, "w")) == NULL) {
			fprintf(stderr, "cant open video output file %s\n", videofile);
			exit(1);
		}      
    }
	else
		fp = stdout;

	if ((readlog_log = rtx_logread_init(logfilename, verbose)) == NULL) {
		fprintf(stderr, "rtx_logread_init() failed\n");
		exit(1);
	}
	fprintf(stderr, "%d top level variables in file\n", readlog_log->numVars);

	if (namemode) {
	        for (rv=readlog_log->vars; rv!=NULL; rv=rv->next)
		        fprintf (stderr, "%s ", rv->v->name);
		fprintf (stderr, "\n");
		exit (0);
	}

	/*
	 *  mark all variables that are required to be printed.
	 */
	for (count = 0; ret < ac; ret++, count++) {
		if (rtx_logread_tag_var(readlog_log, av[ret]) == NULL) {
			fprintf(stderr, "variable <%s> not found",
				av[ret]);
			if (quiet_mode)
				fprintf(stderr, ", continuing.\n");
			else {	
				fprintf(stderr, "\n");
				exit(1);
			}
		}
	}
	if ((count == 0) && !listmode && !commentMode && !timemode) {
		fprintf(stderr, "%s: no variables specified for retrieval\n",
			pname);
		exit(1);
	}
	if ((matfile == NULL) && (count > 1)) {
		fprintf(stderr,
			"%s: only 1 top-level variable can be specified for tabular or video output mode\n",
			pname);
		exit(1);
	}

	while ((rv = rtx_logread_next(readlog_log)) != NULL) {
		if (relative_timing) {
			t0 = dtime(rv->vh.ts);
			relative_timing = 0;
		}
		if (! listmode) {
#ifdef MATLAB
  		        /*
			 * stash it in memory if required
			 * append to a vector later writing
			 */
		        if ((rv->v->attribs & VAR_ATTR_TAG) && !timemode) {
			        if (matfile != NULL) {
				          matlab_append_t(rv);
					  matlab_append(rv->v, rv->d, 0);
				}
			}
#endif
			/*
			 * write to an ASCII tabular format file
			 * always print the time at start of a row
			 */
			
			if (!timemode) {
			        if (rv->v->attribs & VAR_ATTR_TAG) {
                      if (videofile != NULL) {
                        save_video_frame(fp,rv);
                      } else {
				        if (fp) {
                          fprintf(fp, "%f ", dtime(rv->vh.ts) - t0);
                          rtx_logread_print(fp, rv);
                          fprintf(fp, "\n");
                        }
					}
				}
			} else if (count) {
			        if (rv->v->attribs & VAR_ATTR_TAG) {
				        fprintf (stderr, "%s: %f\n", rv->v->name, 
						 dtime(rv->vh.ts) - t0);
				}
			} else {
			        fprintf (stderr, "%s: %f\n", rv->v->name, 
					 dtime(rv->vh.ts) - t0);
			}
		}
		rv->numRecords++;
	}
	if (listmode) {
                for (rv=readlog_log->vars; rv!=NULL; rv=rv->next)
		        fprintf (stderr, "%s [%d]\n", rv->v->name, rv->numRecords);
		exit (0);
	}

#ifdef	MATLAB
	if (matfile) {
		matlab_write(matfile);
	}
#endif

	exit(0);

}

#ifdef	MATLAB

#define	MAX_FIELDS	100

/*
 * append the variable's value to a vector hanging off the MATHEADER mh
 * structure.
 */

static int
matlab_append(RtxParseVar * v, char *d, int offset)
{
	int             i, m, n, k;
	int             nbytes;
	char           *charp;
	short          *shortp;
	int            *intp;
	long           *longp;
	float          *floatp;
	double         *doublep, *p;
	RtxParseVar            *v1;
	RtxLogreadMatlabHeader *mh;

	if (v == NULL)
		return (0);
	if (!(v->attribs & VAR_ATTR_TAG))
		return (0);

	m = 0;
	n = 1;
	if (v->dim) {
		if (v->attribs & VAR_ATTR_ENTIRE_ARRAY) {
			for (i = 0; i < v->dim; i++) {
				n *= v->arrayDim[i];
			}
		}
		else {
			m = 0;
			k = 1;
			for (i = v->dim - 1; i >= 0; i--) {
				m += (k * v->arrayIndex[i]);
				k *= v->arrayDim[i];
			}
			n = m + 1;
		}
	}
	if (v->matHdr == NULL) {
		/*
		 * allocate a mat header 
		 */
		if ((v->matHdr = calloc(1, sizeof(RtxLogreadMatlabHeader)))
		    == NULL) {
			fprintf(stderr, "matlab_append: no mem\n");
			exit(1);
		}
		mh = (RtxLogreadMatlabHeader *) v->matHdr;
		mh->dim = n;
	}
	mh = (RtxLogreadMatlabHeader *) v->matHdr;
	switch (v->type) {
	case rtx_char_t:
		/*
		 * allocate more storage as required 
		 */
		nbytes = n * type_size[rtx_double_t];
		mh->p0.p = (char *) realloc(mh->p0.p, nbytes + mh->len);
		p = (double *) (mh->p0.p + mh->len);	/*
							 * end of vector 
							 */
		charp = (char *) &(d[v->offset + offset]);
		for (i = m; i < n; i++)
			*p++ = (double) charp[i];
		mh->len += nbytes;	/*
					 * update length of vector 
					 */
		mh->rows++;
		break;
	case rtx_short_t:
		/*
		 * allocate more storage as required 
		 */
		nbytes = n * type_size[rtx_double_t];
		mh->p0.p = (char *) realloc(mh->p0.p, nbytes + mh->len);
		p = (double *) (mh->p0.p + mh->len);	/*
							 * end of vector 
							 */
		shortp = (short *) &(d[v->offset + offset]);
		for (i = m; i < n; i++)
			*p++ = (double) shortp[i];
		mh->len += nbytes;	/*
					 * update length of vector 
					 */
		mh->rows++;
		break;
	case rtx_int_t:
		/*
		 * allocate more storage as required 
		 */
		nbytes = n * type_size[rtx_double_t];
		mh->p0.p = (char *) realloc(mh->p0.p, nbytes + mh->len);
		p = (double *) (mh->p0.p + mh->len);	/*
							 * end of vector 
							 */
		intp = (int *) &(d[v->offset + offset]);
		for (i = m; i < n; i++)
			*p++ = (double) intp[i];
		mh->len += nbytes;	/*
					 * update length of vector 
					 */
		mh->rows++;
		break;
	case rtx_long_t:
		/*
		 * allocate more storage as required 
		 */
		nbytes = n * type_size[rtx_double_t];
		mh->p0.p = (char *) realloc(mh->p0.p, nbytes + mh->len);
		p = (double *) (mh->p0.p + mh->len);	/*
							 * end of vector 
							 */
		longp = (long *) &(d[v->offset + offset]);
		for (i = m; i < n; i++)
			*p++ = (double) longp[i];
		mh->len += nbytes;	/*
					 * update length of vector 
					 */
		mh->rows++;
		break;
	case rtx_float_t:
		/*
		 * allocate more storage as required 
		 */
		nbytes = n * type_size[rtx_double_t];
		mh->p0.p = (char *) realloc(mh->p0.p, nbytes + mh->len);
		p = (double *) (mh->p0.p + mh->len);	/*
							 * end of vector 
							 */
		floatp = (float *) &(d[v->offset + offset]);
		for (i = m; i < n; i++)
			*p++ = (double) floatp[i];
		mh->len += nbytes;	/*
					 * update length of vector 
					 */
		mh->rows++;
		break;
	case rtx_double_t:
		/*
		 * allocate more storage as required 
		 */
		nbytes = n * type_size[rtx_double_t];
		mh->p0.p = (char *) realloc(mh->p0.p, nbytes + mh->len);
		p = (double *) (mh->p0.p + mh->len);	/*
							 * end of vector 
							 */
		doublep = (double *) &(d[v->offset + offset]);
		for (i = m; i < n; i++)
			*p++ = (double) doublep[i];
		mh->len += nbytes;	/*
					 * update length of vector 
					 */
		mh->rows++;
		break;
	case rtx_struct_t:
		for (i = m; i < n; i++)
			for (v1 = v->subVar; v1 != NULL; v1 = v1->next)
				matlab_append(v1, d, v->offset + offset + i * v->elemSize);
		break;
	default:
		return (-1);
	}
	return (0);
}

/*
 * append the variable's time value to a vector hanging off the MATHEADER mht
 * structure. Called only a struct, to create a time vector 
 */

static void
matlab_append_t(RtxLogreadVar * rv)
{
	int             nbytes;
	double         *p;

	if (rv->mht == NULL) {
		/*
		 * allocate a mat header 
		 */
		if ((rv->mht = calloc(1, sizeof(RtxLogreadMatlabHeader)))
		    == NULL) {
			fprintf(stderr, "matlab_append: no mem\n");
			exit(1);
		}
		rv->mht->dim = 1;
	}

	/*
	 * allocate more storage as required 
	 */
	nbytes = type_size[rtx_double_t];
	rv->mht->p0.p = realloc(rv->mht->p0.p, nbytes + rv->mht->len);
	p = (double *) (rv->mht->p0.p + rv->mht->len);	/*
							 * end of vector 
							 */
	*p = dtime(rv->vh.ts) - t0;
	rv->mht->len += nbytes;	/*
				 * update length of vector 
				 */
	rv->mht->rows++;
}

void
matlab_write_var(RtxParseVar * v)
{
	RtxParseVar            *v1;

	if (v->subVar != NULL) {
		for (v1 = v->subVar; v1 != NULL; v1 = v1->next)
			matlab_write_var(v1);
		return;
	}
	new_field(v->name, (RtxLogreadMatlabHeader *) v->matHdr);
}

void
matlab_write(char *fname)
{
	MATFile        *fp;
	RtxLogreadVar        *rv;
	RtxParseVar            *v;

	if ((fp = matOpen(fname, "w")) == NULL) {
		fprintf(stderr, "cant open file %s\n", fname);
		exit(1);
	}

	for (rv=readlog_log->vars; rv!=NULL; rv=rv->next) {
		if ((rv->v->attribs & VAR_ATTR_TAG) == 0)
			continue;
		if (rv->numRecords <= 0)
		        continue;
		v = rv->v;
		new_structure(v->name);
		new_field("t", rv->mht);

		matlab_write_var(v);
		put_structure(fp, v->name);
	}
	matClose(fp);
}

static mxArray *smat;		/*
				 * global structure matrix 
				 */

typedef struct _field FIELD;

struct _field {
	int             nrows;
	int             ncols;
	double         *p;
};

FIELD           field[MAX_FIELDS];
int             nfields;

char           *field_names[MAX_FIELDS];

/*
 * to start a new structure, scrap the old structure and reset index
 */
void
new_structure(char *name)
{
	if (smat) {
	  /* Dont destroy; there may be other refs
		mxDestroyArray(smat);
	  */
		smat = NULL;
	}

	nfields = 0;
}

/*
 * for each new field, stash the name, dimensions and data pointer
 */
void
new_field(char *name, RtxLogreadMatlabHeader * mh)
{
	FIELD          *f = &field[nfields];

	field_names[nfields] = name;

	f->p = (double *) mh->p0.f;
	f->nrows = mh->rows;
	f->ncols = mh->dim;

	if (nfields++ > MAX_FIELDS) {
		fprintf(stderr, "new_field: too many fields\n");
		exit(1);
	}
}


/*
 * all the elements of this structure exist, create a MATLAB matrix for
 * each, and stick them into a structure and save the structure to the file.
 */
void
put_structure(MATFile * fp, char *name)
{
	int             i;
	mxArray        *mat;

	smat =
	    mxCreateStructMatrix(1, 1, nfields,
				 (const char **) &field_names[0]);
	mxSetName(smat, name);

	for (i = 0; i < nfields; i++) {
		double         *d;
		double         *f = field[i].p;
		int             nr = field[i].nrows;
		int             nc = field[i].ncols;
		int             r, c;

		/*
		 * allocate a new matrix and set dimensions 
		 */
		mat = mxCreateDoubleMatrix(nr, nc, mxREAL);

		/*
		 * we're nearly there.  We need to transpose the data into
		 * column major order (for MATLAB) and convert from the
		 * temporary float storage to doubles
		 */
		d = malloc(nr * nc * sizeof(double));	/*
							 * temporary vector 
							 */
		if (d == NULL) {
			fprintf(stderr, "put_structure: no mem\n");
			exit(1);
		}

		/*
		 * transpose 
		 */
#define	EL(a,r,c,nc)	((a)[(r)*(nc)+(c)])

		for (r = 0; r < nr; r++) {
			for (c = 0; c < nc; c++) {
				EL(d, c, r, nr) = EL(f, r, c, nc);
			}
		}
		mxSetData(mat, d);	/*
					 * hang it off the matrix 
					 */
		free(f);	/*
				 * and free our temp storage 
				 */

		/*
		 * hang the matrix off the structure 
		 */
		mxSetFieldByNumber(smat, 0, i, mat);
	}

	/*
	 * and write the structure to the stream 
	 */
	matPutArray(fp, smat);
}

#endif


#include "ddxvideo.h"

void save_video_frame(FILE *fp, RtxLogreadVar *rv)
{
  // cast data to a video structure; 
  // just assume it is the correct type for now
  DDX_VIDEO *video = (DDX_VIDEO *) rv->d;
  
  int np =0;
  char buff[256];

  static int first_time=1;
  static RtxTime  tsOld;
  RtxTime         ts;

  np = video->width * video->height * video->fields;

  // sanity checks
  if (video->width <=0 || video->width > 2000 || 
      video->height <=0 || video->height > 2000 || 
      video->fields <=0 || video->fields > 5) {
    rtx_error("nonsense in video header, %d x %d x %d\n",
              video->width, video->height, video->fields);
    exit(1);
  } 

  first_time = (rv->numRecords == 0);
  
  if (first_time) {
    first_time = 0;
    // dump file header
  	fprintf(fp, "YUV4MPEG2 W%d H%d F5:1 Ip A1:1\n",
            video->width, video->height * video->fields);
	if (verbose)
      fprintf(stderr, "Frame %d Width %d Height %d Fields %d\n",
              video->frame, video->width, video->height,
              video->fields);
    tsOld = rv->vh.ts;    
  }

  ts = rv->vh.ts;

  sprintf(buff, "FRAME X%d XS=%d XN=%d XD=%.4f\n", video->frame,
          (int) ts.seconds, (int) ts.nanoSeconds,
          rtx_time_subtract_to_double(&ts, &tsOld));
  
  if (verbose)
    fprintf(stderr, buff);
  tsOld.seconds = ts.seconds;
  tsOld.nanoSeconds = ts.nanoSeconds;
  
  fwrite(buff,     sizeof(char), strlen(buff), fp);
  fwrite(video->y, sizeof(char), np,     fp);
  fwrite(video->v, sizeof(char), np / 4, fp);
  fwrite(video->u, sizeof(char), np / 4, fp);
  
}
