/**Signature>
* Author       : Cedric Pradalier 
* Organisation : CSIRO
* Email        : cedric.pradalier@csiro.au
* Contexte     : Post-doctoral fellowship
* Date         : 12/2004
<Signature**/
#include <signal.h>
#include "ddxVariable.h"
#ifndef Py_RETURN_NONE
#define Py_RETURN_NONE Py_INCREF(Py_None); \
  return Py_None;
#endif


static void
ddxVariable_dealloc(ddxVariable* self)
{
	//printf("Dealloc for '%s'\n",self->var->name);
	if (self->toplevel) {
		Py_DECREF(self->toplevel);
	}
	if (self->owneditem) {
		if (self->item != NULL) 
			ddx_store_done_item(self->item);
		if (self->var != NULL) 
			rtx_parse_free_var(self->var);
		if (!self->direct && self->toplevel) {
			if (self->buffer != NULL) 
				free(self->buffer);
		}
	}
	self->ob_type->tp_free((PyObject*)self);
}

static PyObject * 
ddxVariable_new(PyTypeObject *type, PyObject *args, PyObject *kwds) {
	ddxVariable * self;
	self = (ddxVariable*)(type->tp_alloc(type,0));
	self->direct = 0;
	self->owneditem = 0;
	self->toplevel = NULL;
	self->item = NULL;
	self->var = NULL;
	self->buffer = NULL;
	self->arrayDim = NULL;
	self->size = 0;
	self->dim = 0;
	self->offset = 0;
	self->timestamp = NULL;
	return (PyObject*)self;
}

static int lookup_var(ddxVariable * self, 
		DDX_STORE_ITEM * item, RtxParseVar * var, int direct) 
{
	self->direct = direct;
	self->item = item;
	if (self->item == NULL) return 0;
	self->var = var;
	if (self->var == NULL) return 0;
	self->owneditem = 1;
	if (direct) {
		self->buffer = ddx_store_var_pointer(self->item);
	} else {
		self->buffer = malloc(var->size);
		memset(self->buffer,0,var->size);
	}
	if (self->buffer == NULL) return 0;
	self->arrayDim = var->arrayDim;
	self->dim = var->dim;
	self->size = var->size;
	self->offset = 0;
	self->timestamp = (double*)malloc(sizeof(double));
	*(self->timestamp) = 0.0;
	return 1;
}

PyObject * ddxVariable_Create(DDX_STORE_ITEM * item, 
		RtxParseVar * pvar, int direct) 
{
	ddxVariable * var = (ddxVariable*)ddxVariable_new(&ddx_VariableType,NULL,NULL);
	if (!lookup_var(var,item,pvar,direct)) {
		Py_DECREF(var);
		rtx_error_flush("Rtx error flush");
		PyErr_SetString(PyExc_NameError,"Cannot find variable\n");
		return NULL;
	}
	return (PyObject*)var;
}

static int update_direct_buffer(ddxVariable * self)
{
	void * pbuffer = NULL;
	if (self->direct) {
		pbuffer = ddx_store_var_pointer(self->item);
		assert(pbuffer);
	} else if (self->toplevel) {
		pbuffer = self->toplevel->buffer;
	} else {
		pbuffer = self->buffer;
	}
	self->buffer = ((char*)(pbuffer)) + self->offset;
#if 0
	printf("Top %p Buffer: %p Offset %d\n",pbuffer,self->buffer,self->offset);
#endif
	return 1;
}

static int fill_as_subvar(ddxVariable * self, 
		ddxVariable * parent, char * name) 
{
	if (parent->var->type != rtx_struct_t) return 0;
	RtxParseVar * v = parent->var->subVar;
	while (v!=NULL) {
		if (strcmp(v->name,name)==0) {
			if (parent->toplevel) 
				self->toplevel = parent->toplevel;
			else
				self->toplevel = parent;
			Py_INCREF(self->toplevel);
			self->direct = parent->direct;
			self->owneditem = 0;
			self->item = parent->item;
			self->var = v;
			self->offset = parent->offset + v->offset;
			self->arrayDim = v->arrayDim;
			self->dim = v->dim;
			self->size = v->size;
			self->timestamp = parent->timestamp;
			update_direct_buffer(self);
			return 1;
		}
		v = v->next;
	}
	return 0;
}

static int fill_as_item(ddxVariable * self, 
		ddxVariable * parent, unsigned int index) 
{
	unsigned int i;
	//printf("Parent dim : %d\n",parent->dim);
	if (parent->dim == 0) {
		rtx_error_flush("Rtx error flush");
		PyErr_SetString(PyExc_IndexError,"Not an array");
		return 0;
	}
	//printf("Parent length : %d -> index %d\n",parent->arrayDim[0],index);
	if (index >= parent->arrayDim[0]) {
		rtx_error_flush("Rtx error flush");
		PyErr_SetString(PyExc_IndexError,"Index out of bound");
		return 0;
	}
	if (parent->toplevel) 
		self->toplevel = parent->toplevel;
	else
		self->toplevel = parent;
	Py_INCREF(self->toplevel);
	self->direct = parent->direct;
	self->owneditem = 0;
	self->item = parent->item;
	self->var = parent->var;
	self->size = self->var->elemSize;
	for (i=1;i<parent->dim;i++)
		self->size *= parent->arrayDim[i];
	self->offset = parent->offset + index * self->size;
	update_direct_buffer(self);
	self->dim = parent->dim - 1;
	if (self->dim == 0)
		self->arrayDim = NULL;
	else
		self->arrayDim = parent->arrayDim + 1;
	self->timestamp = parent->timestamp;
	return 1;
}


static PyObject * 
ddxVariable_read_direct(ddxVariable * self, PyObject *args, PyObject *kwds)
{
	RtxTime ts;
	double timearg = 0.0;
	int skipframe = 0;

    static char *kwlist[] = {"timearg", "skipframe", NULL};
    if (! PyArg_ParseTupleAndKeywords(args, kwds, "|di", kwlist, 
				&timearg, &skipframe)) {
		return NULL;
	}
	
	int res = ddx_store_read_direct(self->item,&ts,
			timearg,skipframe);
	*(self->timestamp) = rtx_time_to_double(&ts);
	return PyInt_FromLong(res);
}

static PyObject * 
ddxVariable_write_direct(ddxVariable * self)
{
	int res;
	res = ddx_store_write_direct(self->item,NULL);
	if (res != 0) {
		rtx_error_flush("Rtx error flush");
		PyErr_SetString(PyExc_IOError,"Error writing variable");
		return NULL;
	}
	Py_RETURN_NONE;
}

static PyObject * 
ddxVariable_read(ddxVariable * self, PyObject *args, PyObject *kwds)
{
	if (self->direct)
		return ddxVariable_read_direct(self,args,kwds);
	else {
		RtxTime ts;
		double timearg = 0.0;
		int skipframe = 0;

		static char *kwlist[] = {"timearg", "skipframe", NULL};


		if (! PyArg_ParseTupleAndKeywords(args, kwds, "|di", kwlist, 
					&timearg, &skipframe)) {
			return NULL;
		}

		int res;
		void * buffer;
		update_direct_buffer(self);
		if (self->toplevel==NULL) 
			buffer = self->buffer;
		else
			buffer = self->toplevel->buffer;
		res = ddx_store_read(self->item,buffer,&ts,
				timearg,skipframe);

#if 0
		int i;
		printf("Buffer %p: ",buffer);
		for (i=0;i<30;i++) {
			printf("%02X ",(int)(*((unsigned char*)(buffer)+i)));
		}
		printf("\n");
#endif
		*(self->timestamp) = rtx_time_to_double(&ts);
		return PyInt_FromLong(res);
	}
}

static PyObject * 
ddxVariable_write(ddxVariable * self)
{
	if (self->direct)
		return ddxVariable_write_direct(self);
	else {
		int res;
#if 0
		if (!self->owneditem) {
			rtx_error_flush("Rtx error flush");
			PyErr_SetString(PyExc_NotImplementedError,"Can only write toplevel variables");
			return NULL;
		}
		res = ddx_store_write(self->item,self->buffer,NULL);
#else
		update_direct_buffer(self);
		if (self->toplevel==NULL)
			res = ddx_store_write(self->item,self->buffer,NULL);
		else
			res = ddx_store_write(self->item,self->toplevel->buffer,NULL);
#endif
		if (res != 0) {
			rtx_error_flush("Rtx error flush");
			PyErr_SetString(PyExc_IOError,"Error writing variable");
			return NULL;
		}
		Py_RETURN_NONE;
	}
}


static PyObject *
ddxVariable_getTimeStamp(ddxVariable *self, void *closure)
{
	return PyFloat_FromDouble(*(self->timestamp));
}

static PyObject *
ddxVariable_getSize(ddxVariable *self, void *closure)
{
	return PyInt_FromLong(self->size);
}

static PyObject *
ddxVariable_getBuffer(ddxVariable *self, void *closure)
{
	update_direct_buffer(self);
	return PyBuffer_FromReadWriteMemory(self->buffer,self->size);
}

static PyMethodDef ddxVariable_methods[];

static PyObject *
ddxVariable_getattr(PyObject *self, char *name)
{
	ddxVariable * vself = (ddxVariable*)self;
	//printf("In variable %s looking for attr %s\n", vself->var->name, name);
	ddxVariable * ddxvar = 
		(ddxVariable*)ddxVariable_new(&ddx_VariableType,NULL,NULL);
	if (!fill_as_subvar(ddxvar,vself,name)) {
		Py_DECREF(ddxvar);
		return Py_FindMethod(ddxVariable_methods,self,name);
	}
	return (PyObject*)ddxvar;
}

static int 
setIntValue(ddxVariable * self, long f)
{
	if (self->dim != 0) {
		rtx_error_flush("Rtx error flush");
		PyErr_SetString(PyExc_TypeError,"Cannot affect a integer to an array");
		return -1;
	}
	if (self->var->type == rtx_struct_t) {
		rtx_error_flush("Rtx error flush");
		PyErr_SetString(PyExc_TypeError,"Cannot affect a integer to a struct");
		return -1;
	}
	if (self->direct) {
		update_direct_buffer(self);
	}
	assert(self->buffer);
	switch (self->var->type) 	{
			case rtx_char_t : 
				*((char *)(self->buffer)) = (char)f;
				break;
			case rtx_short_t : 
				*((short *)(self->buffer)) = (short)f;
				break;
			case rtx_int_t : 
				*((int *)(self->buffer)) = (int)f;
				break;
			case rtx_long_t : 
				*((long *)(self->buffer)) = (long)f;
				break;
			case rtx_float_t : 
				*((float *)(self->buffer)) = (float)f;
				break;
			case rtx_double_t : 
				*((double *)(self->buffer)) = (double)f;
				break;
			default :
				rtx_error_flush("Rtx error flush");
				PyErr_SetString(PyExc_TypeError,"Unkown variable type");
				return -1;
	}
	return 0;
}

static int 
setFloatValue(ddxVariable * self, double f)
{
	if (self->dim != 0) {
		rtx_error_flush("Rtx error flush");
		PyErr_SetString(PyExc_TypeError,"Cannot affect a float to an array");
		return -1;
	}
	if (self->var->type == rtx_struct_t) {
		rtx_error_flush("Rtx error flush");
		PyErr_SetString(PyExc_TypeError,"Cannot affect a float to a struct");
		return -1;
	}
	if (self->direct) {
		update_direct_buffer(self);
	}
	assert(self->buffer);
	switch (self->var->type) 	{
			case rtx_char_t : 
				*((char *)(self->buffer)) = (char)f;
				break;
			case rtx_short_t : 
				*((short *)(self->buffer)) = (short)f;
				break;
			case rtx_int_t : 
				*((int *)(self->buffer)) = (int)f;
				break;
			case rtx_long_t : 
				*((long *)(self->buffer)) = (long)f;
				break;
			case rtx_float_t : 
				*((float *)(self->buffer)) = (float)f;
				break;
			case rtx_double_t : 
				*((double *)(self->buffer)) = (double)f;
				break;
			default :
				rtx_error_flush("Rtx error flush");
				PyErr_SetString(PyExc_TypeError,"Unkown variable type");
				return -1;
	}
	return 0;
}

static int 
setStringValue(ddxVariable * self, char * f)
{
	if ((self->var->type != rtx_char_t) || (self->dim > 1)) {
		rtx_error_flush("Rtx error flush");
		PyErr_SetString(PyExc_TypeError,"Can only affect a string to a 1D char array");
		return -1;
	}
	if (self->var->type == rtx_struct_t) {
		rtx_error_flush("Rtx error flush");
		PyErr_SetString(PyExc_TypeError,"Cannot affect a string to a struct");
		return -1;
	}
	if (self->direct) {
		update_direct_buffer(self);
	}
	assert(self->buffer);
	strncpy((char *)(self->buffer),f,self->size);
	return 0;
}

static int 
ddxVariable_set(ddxVariable * self, PyObject * value)
{
	if (PyFloat_Check(value))
		return setFloatValue(self,PyFloat_AsDouble(value));
	else if (PyString_Check(value))
		return setStringValue(self,PyString_AsString(value));
	else if (PyInt_Check(value) || PyLong_Check(value))
		return setIntValue(self,PyInt_AsLong(value));
	else if (PyObject_TypeCheck(self,&ddx_VariableType)) {
		/* Only called after in place operator */
		if (self->buffer == ((ddxVariable*)value)->buffer)
			return 0;
	}
	rtx_error_flush("Rtx error flush");
	PyErr_SetString(PyExc_ValueError,"Cannot set attribute to this value");
	return -1;
}

static int
ddxVariable_setattr(PyObject *oself, char *name, PyObject * value)
{
	int res = -1;
	//printf("setting attribute %s in %s\n",name,((ddxVariable*)oself)->var->name);
	ddxVariable * self = (ddxVariable*)ddxVariable_getattr(oself,name);
	if (self == NULL) return -1;

	res = ddxVariable_set(self,value);
	Py_DECREF(self);
	return res;
}

#if HAVE_SSIZE_T
#  define PY_INDEX Py_ssize_t
#else
#  define PY_INDEX int
#endif

static PyObject *
ddxVariable_getitem(PyObject *self, PY_INDEX index)
{
	ddxVariable * var = (ddxVariable*)ddxVariable_new(&ddx_VariableType,NULL,NULL);
	if (!fill_as_item(var,(ddxVariable*)self,index)) {
		Py_DECREF(var);
		return NULL;
	}
	return (PyObject*)var;
}

static int
ddxVariable_setitem(PyObject *oself, int index, PyObject * value)
{
	int res = -1;
	ddxVariable * self = (ddxVariable*)ddxVariable_getitem(oself,index);
	if (self == NULL) return -1;

	res = ddxVariable_set(self,value);
	Py_DECREF(self);
	return res;
}

static PyObject * 
ddxVariable_lshift_affect(PyObject * oself, PyObject * value)
{ 
	ddxVariable * self = (ddxVariable*)oself;
	PyObject * type = PyObject_Type(value);
	PyObject * str1 = PyObject_Str(value);
	PyObject * str2 = PyObject_Str(type);
	Py_DECREF(str1); Py_DECREF(str2); Py_DECREF(type);

	ddxVariable_set(self,value);
	Py_INCREF(oself);
	return oself;
}

static PyObject * 
ddxVariable_setValue(PyObject * oself, PyObject * args)
{ 
	ddxVariable * self = (ddxVariable*)oself;
	PyObject * value;
    if (! PyArg_ParseTuple(args, "O", &value)) {
		return NULL;
	}
	ddxVariable_set(self,value);
	Py_RETURN_NONE;
}

#define MIN(A,B) (((A)<(B))?(A):(B))

PyObject * 
buildDefinition(PyObject * s,RtxParseVar * v,unsigned int dim,int * arrayDim,
		unsigned int indentLevel, int pretty, void * buffer)
{
	char tmp[1024];
	unsigned int i;

	if (v == NULL)
		return s;

	if (pretty) {
		for (i=0; i<indentLevel; i++) 
			tmp[i] = '\t';
		tmp[indentLevel] = 0;
		PyString_ConcatAndDel(&s,PyString_FromString(tmp));
	}
	if (v->type == rtx_struct_t) {
		RtxParseVar * vit = v->subVar;
		sprintf(tmp,"struct {%c",pretty?'\n':' ');
		PyString_ConcatAndDel(&s,PyString_FromString(tmp));
		while (vit != NULL) {
			s = buildDefinition (s, vit, vit->dim, vit->arrayDim, indentLevel+1, 
					pretty,(buffer==NULL)?NULL:(buffer + vit->offset));
			vit = vit->next;
		}
		if (pretty) {
			for (i=0; i<indentLevel; i++) 
				tmp[i] = '\t';
			strcpy(tmp+indentLevel,"} ");
			PyString_ConcatAndDel(&s,PyString_FromString(tmp));
		} else {
			PyString_ConcatAndDel(&s,PyString_FromString("} "));
		}
	} else {
		switch (v->type) {
			case rtx_char_t : 
				PyString_ConcatAndDel(&s,PyString_FromString("char "));
				break;
			case rtx_short_t : 
				PyString_ConcatAndDel(&s,PyString_FromString("short "));
				break;
			case rtx_int_t : 
				PyString_ConcatAndDel(&s,PyString_FromString("int "));
				break;
			case rtx_long_t : 
				PyString_ConcatAndDel(&s,PyString_FromString("long "));
				break;
			case rtx_float_t : 
				PyString_ConcatAndDel(&s,PyString_FromString("float "));
				break;
			case rtx_double_t : 
				PyString_ConcatAndDel(&s,PyString_FromString("double "));
				break;
		}
	}
	PyString_ConcatAndDel(&s,PyString_FromString(v->name));
	for (i=0; i<(unsigned int)(dim); i++) {
		sprintf (tmp,"[%d]", arrayDim[i]);
		PyString_ConcatAndDel(&s,PyString_FromString(tmp));
	}
	if (buffer != NULL) {
#if 0
		printf("Building definition for buffer %p\n",buffer);
#endif
		switch (v->type) {
			case rtx_char_t : 
				{
					char *c = (char *)(buffer);
					switch (dim) {
						case 0: 
							sprintf(tmp," = %d 0x%02X '%c'",(int)*c,(int)*c,*c); 
							PyString_ConcatAndDel(&s,PyString_FromString(tmp));
							break;
						case 1: 
							PyString_ConcatAndDel(&s,PyString_FromString(" = [ "));
							for (i=0;i<MIN(arrayDim[0],10);i++) {
								sprintf(tmp,"%02X ",(int)c[i]); 
								PyString_ConcatAndDel(&s,PyString_FromString(tmp));
							}
							if (arrayDim[0] > 10) { 
								PyString_ConcatAndDel(&s,PyString_FromString("...] = '"));
								snprintf(tmp,10,"%s...'",c);
								PyString_ConcatAndDel(&s,PyString_FromString(tmp));
							} else {
								PyString_ConcatAndDel(&s,PyString_FromString("] = '"));
								snprintf(tmp,arrayDim[0],"%s'",c);
								PyString_ConcatAndDel(&s,PyString_FromString(tmp));
							}
							break;
						default : break;
					}
				}
				break;
			case rtx_short_t : 
				{
					short * c = (short *)(buffer);
					switch (dim) {
						case 0: 
							sprintf(tmp," = %d 0x%04X",(int)*c,(int)*c); 
							PyString_ConcatAndDel(&s,PyString_FromString(tmp));
							break;
						case 1: 
							PyString_ConcatAndDel(&s,PyString_FromString(" = [ "));
							for (i=0;i<MIN(arrayDim[0],10);i++) {
								sprintf(tmp,"%d ",(int)c[i]); 
								PyString_ConcatAndDel(&s,PyString_FromString(tmp));
							}
							if (arrayDim[0] > 10) { 
								PyString_ConcatAndDel(&s,PyString_FromString("...]"));
							} else {
								PyString_ConcatAndDel(&s,PyString_FromString("]"));
							}
							break;
						default : break;
					}
				}
				break;
			case rtx_int_t : 
				{
					int * c = (int *)(buffer);
					switch (dim) {
						case 0: 
							sprintf(tmp," = %d",*c); 
							PyString_ConcatAndDel(&s,PyString_FromString(tmp));
							break;
						case 1: 
							PyString_ConcatAndDel(&s,PyString_FromString(" = [ "));
							for (i=0;i<MIN(arrayDim[0],10);i++) {
								sprintf(tmp,"%d ",c[i]); 
								PyString_ConcatAndDel(&s,PyString_FromString(tmp));
							}
							if (arrayDim[0] > 10) { 
								PyString_ConcatAndDel(&s,PyString_FromString("...]"));
							} else {
								PyString_ConcatAndDel(&s,PyString_FromString("]"));
							}
							break;
						default : break;
					}
				}
				break;
			case rtx_long_t : 
				{
					long * c = (long *)(buffer);
					switch (dim) {
						case 0: 
							sprintf(tmp," = %ld 0x%08lX",*c,*c); 
							PyString_ConcatAndDel(&s,PyString_FromString(tmp));
							break;
						case 1: 
							PyString_ConcatAndDel(&s,PyString_FromString(" = [ "));
							for (i=0;i<MIN(arrayDim[0],10);i++) {
								sprintf(tmp,"%ld ",c[i]); 
								PyString_ConcatAndDel(&s,PyString_FromString(tmp));
							}
							if (arrayDim[0] > 10) { 
								PyString_ConcatAndDel(&s,PyString_FromString("...]"));
							} else {
								PyString_ConcatAndDel(&s,PyString_FromString("]"));
							}
							break;
						default : break;
					}
				}
				break;
			case rtx_float_t :
				{
					float * c = (float *)(buffer);
					switch (dim) {
						case 0: 
							sprintf(tmp," = %e",(double)*c); 
							PyString_ConcatAndDel(&s,PyString_FromString(tmp));
							break;
						case 1: 
							PyString_ConcatAndDel(&s,PyString_FromString(" = [ "));
							for (i=0;i<MIN(arrayDim[0],10);i++) {
								sprintf(tmp,"%e ",(double)c[i]); 
								PyString_ConcatAndDel(&s,PyString_FromString(tmp));
							}
							if (arrayDim[0] > 10) { 
								PyString_ConcatAndDel(&s,PyString_FromString("...]"));
							} else {
								PyString_ConcatAndDel(&s,PyString_FromString("]"));
							}
							break;
						default : break;
					}
				}
				break;
			case rtx_double_t :
				{
					double * c = (double *)(buffer);
					switch (dim) {
						case 0: 
							sprintf(tmp," = %e",*c); 
							PyString_ConcatAndDel(&s,PyString_FromString(tmp));
							break;
						case 1: 
							PyString_ConcatAndDel(&s,PyString_FromString(" = [ "));
							for (i=0;i<MIN(arrayDim[0],10);i++) {
								sprintf(tmp,"%e ",c[i]); 
								PyString_ConcatAndDel(&s,PyString_FromString(tmp));
							}
							if (arrayDim[0] > 10) { 
								PyString_ConcatAndDel(&s,PyString_FromString("...]"));
							} else {
								PyString_ConcatAndDel(&s,PyString_FromString("]"));
							}
							break;
						default : break;
					}
				}
				break;
		}
	}
	if (pretty)
		PyString_ConcatAndDel(&s,PyString_FromString(";\n"));
	else
		PyString_ConcatAndDel(&s,PyString_FromString("; "));
	return s;
}

static PyObject * ddxVariable_repr(PyObject * oself) 
{
	ddxVariable * self = (ddxVariable*)oself;
	PyObject * result = PyString_FromString("");
	if (self->direct) {
		update_direct_buffer(self);
	}
	return buildDefinition(result, self->var,self->dim,self->arrayDim,
			0,1,self->buffer);
}

static PyObject * ddxVariable_str(PyObject * oself) 
{
	ddxVariable * self = (ddxVariable*)oself;
	PyObject * result = PyString_FromString("");
	if (self->direct) {
		update_direct_buffer(self);
	}
	return buildDefinition(result, self->var,self->dim,self->arrayDim,
			0,0,self->buffer);
}

static PyObject * ddxVariable_getDef(PyObject * oself)
{
	ddxVariable * self = (ddxVariable *)oself;
	PyObject * result = PyString_FromString("");
	return buildDefinition(result, self->var,self->dim,self->arrayDim,
			0,1,NULL);
}

static PyObject * ddxVariable_isDirect(PyObject * oself)
{
	ddxVariable * self = (ddxVariable*)oself;
	return PyBool_FromLong(self->direct);
}

static PyObject * ddxVariable_int(PyObject * oself) 
{
	ddxVariable * self = (ddxVariable *)oself;
	if (self->dim > 0) {
		rtx_error_flush("Rtx error flush");
		PyErr_SetString(PyExc_TypeError,"Cannot convert an array into an integer");
		return NULL;
	}
	if (self->var->type == rtx_struct_t) {
		rtx_error_flush("Rtx error flush");
		PyErr_SetString(PyExc_TypeError,"Cannot convert a struct into an integer");
		return NULL;
	}
	PyObject * result = NULL;
	if (self->direct) {
		update_direct_buffer(self);
	}
	switch (self->var->type) {
		case rtx_char_t : 
			result = PyInt_FromLong(*(char *)(self->buffer));
			break;
		case rtx_short_t : 
			result = PyInt_FromLong(*(short *)(self->buffer));
			break;
		case rtx_int_t : 
			result = PyInt_FromLong(*(int *)(self->buffer));
			break;
		case rtx_long_t : 
			result = PyInt_FromLong(*(long *)(self->buffer));
			break;
		case rtx_float_t :
			result = PyInt_FromLong((long)(*(float *)(self->buffer)));
			break;
		case rtx_double_t :
			result = PyInt_FromLong((long)(*(double *)(self->buffer)));
			break;
		default :
			rtx_error_flush("Rtx error flush");
			PyErr_SetString(PyExc_NotImplementedError,"Variable type unkonwn");
	}
	return result;
}

static PyObject * ddxVariable_float(PyObject * oself) 
{
	ddxVariable * self = (ddxVariable *)oself;
	if (self->dim > 0) {
		rtx_error_flush("Rtx error flush");
		PyErr_SetString(PyExc_TypeError,"Cannot convert an array into an float");
		return NULL;
	}
	if (self->var->type == rtx_struct_t) {
		rtx_error_flush("Rtx error flush");
		PyErr_SetString(PyExc_TypeError,"Cannot convert a struct into an float");
		return NULL;
	}
	PyObject * result = NULL;
	if (self->direct) {
		update_direct_buffer(self);
	}
	switch (self->var->type) {
		case rtx_char_t : 
			result = PyFloat_FromDouble(*(char *)(self->buffer));
			break;
		case rtx_short_t : 
			result = PyFloat_FromDouble(*(short *)(self->buffer));
			break;
		case rtx_int_t : 
			result = PyFloat_FromDouble(*(int *)(self->buffer));
			break;
		case rtx_long_t : 
			result = PyFloat_FromDouble(*(long *)(self->buffer));
			break;
		case rtx_float_t :
			result = PyFloat_FromDouble((*(float *)(self->buffer)));
			break;
		case rtx_double_t :
			result = PyFloat_FromDouble((*(double *)(self->buffer)));
			break;
		default :
			rtx_error_flush("Rtx error flush");
			PyErr_SetString(PyExc_NotImplementedError,"Variable type unkonwn");
	}
	return result;
}



static int
ddxVariable_arrayLength(ddxVariable *self)
{
	if (self->dim == 0) 
		return 0;
	return self->arrayDim[0];
}

static PyObject *
ddxVariable_getFields(PyObject *oself)
{
	ddxVariable * self = (ddxVariable*)oself;
	PyObject * list = PyList_New(0);
	RtxParseVar * v = self->var->subVar;
	while (v != NULL) {
		PyList_Append(list,PyString_FromString(v->name));
		v = v->next;
	}
	return list;
}

static PyObject *
ddxVariable_getDimensions(PyObject *oself)
{
	ddxVariable * self = (ddxVariable*)oself;
	PyObject * list = PyList_New(0);
	unsigned int i;
	for (i=0;i<self->dim;i++) {
		PyList_Append(list,PyInt_FromLong(self->arrayDim[i]));
	}
	return list;
}

static PyObject *
ddxVariable_getDimension(PyObject *oself)
{
	ddxVariable * self = (ddxVariable*)oself;
	return PyInt_FromLong(self->dim);
}

static PyObject *
ddxVariable_getCount(PyObject *oself)
{
	ddxVariable * self = (ddxVariable*)oself;
	return PyInt_FromLong(self->item->headerP->count);
}

static PyObject *
ddxVariable_getName(PyObject *oself)
{
	ddxVariable * self = (ddxVariable*)oself;
	return PyString_FromString(self->item->varName);
}

static int trigger = 0;
static void sigcatch(int n)
{
	trigger ++;
}

static PyObject *
ddxVariable_watch(PyObject *oself, PyObject *args, PyObject *kwds)
{ 
	PyObject * pystdout;
	ddxVariable * self = (ddxVariable*)oself;
	double delay = 2;
	int detail = 0;
	int raw = 0;
    static char *kwlist[] = {"delay", "detail", "raw", NULL};
    if (! PyArg_ParseTupleAndKeywords(args, kwds, "|dii", kwlist, 
				&delay, &detail, &raw)) {
		return NULL;
	}
	pystdout = PyFile_FromFile(stdout, "stdout", "0644", NULL);
	trigger = 0;
	signal(SIGINT,sigcatch);
	while (trigger==0) {
		RtxTime ts,tsnow;
		double now;
		int res;
		void * buffer;
		update_direct_buffer(self);
		if (self->direct) {
			res = ddx_store_read_direct(self->item,&ts, 1,0);
			buffer = self->buffer;
		} else {
			if (self->toplevel==NULL) 
				buffer = self->buffer;
			else
				buffer = self->toplevel->buffer;

			res = ddx_store_read(self->item,buffer, &ts, 1,0);
		}


		if (res >= 0) {
			*(self->timestamp) = rtx_time_to_double(&ts);
			rtx_time_get(&tsnow);
			now = rtx_time_to_double(&tsnow);
			if (detail) {
				char qualif[64]="in the future";
				double age = now - *(self->timestamp);
				double sec = fabs(age);
				int day, hr, min;
				printf("TimeStamp: %f s", *(self->timestamp));
				if (*(self->timestamp) != 0.0) {
					printf(" \t Age: ");
					if (age >= 0) {
						qualif[0]=0;
					}

					day = (int)floor(sec/(24*3600)); sec -= day*24*3600;
					hr  = (int)floor(sec/3600); sec -= hr*3600;
					min = (int)floor(sec/60); sec -= min*60;

					if (age < 60) {
						printf("%f s",sec);
					} else if (age < 3600) {
						printf("%d min %f s",min,sec);
					} else if (age < 24*3600) {
						printf("%d hr %d min %f s",hr,min,sec);
					} else {
						printf("%d d %d hr %d min %f s",day,hr,min,sec);
					}
					printf(" %s\n",qualif);
				} else {
					printf("\n");
				}
				printf("Size:      %d bytes\n",self->item->varSize);
				printf("Direct:    %s\n",(self->direct?"yes":"no"));
				printf("Count:     %d\n",self->item->headerP->count);
			}
			if (raw) {
				PyFile_WriteObject(oself,pystdout,Py_PRINT_RAW);
				PyFile_WriteString("\n",pystdout);
			} else {
				PyFile_WriteObject(oself,pystdout,0);
			}
		}
		
		usleep((unsigned int)(delay*1e6));
	}
	Py_DECREF(pystdout);
	signal(SIGINT,NULL);
		
	Py_RETURN_NONE;
}


static PyMethodDef ddxVariable_methods[] = {
    {"read", (PyCFunction)ddxVariable_read, METH_VARARGS | METH_KEYWORDS,
     "Fills a variable with the content of the shared memory. Only for toplevel variables."
    },
    {"write", (PyCFunction)ddxVariable_write, METH_NOARGS,
     "Fills shared memory with the content of a variable. Only for toplevel variables."
    },
	{"watch",(PyCFunction)ddxVariable_watch,METH_VARARGS | METH_KEYWORDS,
	 "display the repr of a variable every delay second. 3 args:\n"
		 "'delay' the time in second (float) between update,\n"
		 "'detail' boolean: detailed value\n"
		 "'raw' boolean: use str instead of repr"
	},
    {"set", (PyCFunction)ddxVariable_setValue, METH_VARARGS,
     "Affects, if possible, a value to the variable."
    },
    {"getTimeStamp", (PyCFunction)ddxVariable_getTimeStamp, METH_NOARGS,
     "Returns time stamp associated with last variable read."
    },
    {"getSize", (PyCFunction)ddxVariable_getSize, METH_NOARGS,
     "Returns the size of the variable shared memory."
    },
    {"getBuffer", (PyCFunction)ddxVariable_getBuffer, METH_NOARGS,
     "Returns a PyBuffer object giving raw access to the variable buffer."
    },
    {"getDefinition", (PyCFunction)ddxVariable_getDef, METH_NOARGS,
     "Returns the string definition of the variable."
    },
    {"getFields", (PyCFunction)ddxVariable_getFields, METH_NOARGS,
     "Returns the available fields in the structure (if any)."
    },
    {"getDimension", (PyCFunction)ddxVariable_getDimension, METH_NOARGS,
     "Returns the number of dimensions of the variable."
    },
    {"getDimensions", (PyCFunction)ddxVariable_getDimensions, METH_NOARGS,
     "Returns the dimensions list of the variable."
    },
    {"getCount", (PyCFunction)ddxVariable_getCount, METH_NOARGS,
     "Returns the update count of the variable."
    },
    {"getName", (PyCFunction)ddxVariable_getName, METH_NOARGS,
     "Returns the name of the variable."
    },
    {"isDirect", (PyCFunction)ddxVariable_isDirect, METH_NOARGS,
     "Tests if a variable has been obtained by store.directVariable."
    },
    {NULL}  /* Sentinel */
};


static PySequenceMethods ddxVariable_asSeq = {
	(inquiry)ddxVariable_arrayLength,  /* sq_length */
	0,                                 /* sq_concat */
	0,                                 /* sq_repeat */
	ddxVariable_getitem,               /* sq_item   */
	0,                                 /* sq_slice  */
	(intobjargproc)ddxVariable_setitem,/* sq_ass_item */
	0,                                 /* sq_ass_slice */
	0,                                 /* sq_contains */
	0,                                 /* sq_inplace_concat */
	0,                                 /* sq_inplace_repeat */
};

static PyNumberMethods ddxVariable_asNum = {
	0,                                 /* nb_add */
	0,                                 /* nb_substract */
	0,                                 /* nb_multiply */
	0,                                 /* nb_divide */
	0,                                 /* nb_remainder */
	0,                                 /* nb_divmod */
	0,                                 /* nb_power */
	0,                                 /* nb_negative */
	0,                                 /* nb_positive */
	0,                                 /* nb_absolute */
	0,                                 /* nb_nonzero */
	0,                                 /* nb_invert */
	0,                                 /* nb_lshift */
	0,                                 /* nb_rshift */
	0,                                 /* nb_and */
	0,                                 /* nb_xor */
	0,                                 /* nb_or */
	0,                                 /* nb_coerce */
	ddxVariable_int,                   /* nb_int */
	ddxVariable_int,                   /* nb_long */
	ddxVariable_float,                 /* nb_float */
	0,                                 /* nb_oct */
	0,                                 /* nb_hex */
	0,                                 /* nb_inplace_add */
	0,                                 /* nb_inplace_subtract */
	0,                                 /* nb_inplace_multiply */
	0,                                 /* nb_inplace_divide */
	0,                                 /* nb_inplace_remainder */
	0,                                 /* nb_inplace_power */
	ddxVariable_lshift_affect,         /* nb_inplace_lshift */
};

PyTypeObject ddx_VariableType = {
    PyObject_HEAD_INIT(NULL)
    0,                          /* ob_size*/
    "ddxInterface.ddxVariable", /* tp_name*/
    sizeof(ddxVariable), /* tp_basicsize*/
    0,                         /* tp_itemsize*/
    (destructor)ddxVariable_dealloc, /* tp_dealloc*/
    0,                         /* tp_print*/
    ddxVariable_getattr,       /* tp_getattr*/
    ddxVariable_setattr,       /* tp_setattr*/
    0,                         /* tp_compare*/
    ddxVariable_repr,          /* tp_repr*/
    &ddxVariable_asNum,        /* tp_as_number*/
    &ddxVariable_asSeq,        /* tp_as_sequence*/
    0,                         /* tp_as_mapping*/
    0,                         /* tp_hash */
    0,                         /* tp_call*/
    ddxVariable_str,           /* tp_str*/
    0,                         /* tp_getattro*/
    0,                         /* tp_setattro*/
    0,                         /* tp_as_buffer*/
    Py_TPFLAGS_DEFAULT,        /* tp_flags*/
    "DDX Variable objects",    /* tp_doc */
	0,		                   /* tp_traverse */
    0,		                   /* tp_clear */
    0,		                   /* tp_richcompare */
    0,		                   /* tp_weaklistoffset */
    0,		                   /* tp_iter */
    0,		                   /* tp_iternext */
    0,                         /* tp_methods */
    0,                         /* tp_members */
    0,                         /* tp_getset */
    0,                         /* tp_base */
    0,                         /* tp_dict */
    0,                         /* tp_descr_get */
    0,                         /* tp_descr_set */
    0,                         /* tp_dictoffset */
    0,                         /* tp_init */
    0,                         /* tp_alloc */
    ddxVariable_new,           /* tp_new */
};

