/**Signature>
* Author       : Cedric Pradalier 
* Organisation : CSIRO
* Email        : cedric.pradalier@csiro.au
* Contexte     : Post-doctoral fellowship
* Date         : 12/2004
<Signature**/
#include "ddxStore.h"
#include "ddxVariable.h"
#include <ctype.h>

#include <structmember.h>

#include <rtx/message.h>
#include <rtx/time.h>
#include <rtx/error.h>
#include <rtx/parse.h>
#include <ddx.h>

#ifndef Py_RETURN_NONE
#define Py_RETURN_NONE Py_INCREF(Py_None); \
  return Py_None;
#endif


#define VERBOSE 0
unsigned int nbCreatedStores = 0;



static void
ddxStore_dealloc(ddxStore* self)
{
    self->ob_type->tp_free((PyObject*)self);
	nbCreatedStores -= 1;
	if (nbCreatedStores == 0) {
		//printf("ddxInterface: finalizing ddx client\n"); 
		ddx_client_done();
	}
}

static PyObject *
ddxStore_new(PyTypeObject *type, PyObject *args, PyObject *kwds)
{
	unsigned int oldNbStores = nbCreatedStores;
	ddxStore *self;

	self = (ddxStore *)type->tp_alloc(type, 0);
	if (self == NULL) {
		rtx_error_flush("Rtx error flush");
		PyErr_SetString(PyExc_SystemError,"Cannot allocate store memory");
		return NULL;
	}
	self->id = NULL;

	if (nbCreatedStores == 0) {
		printf("ddxInterface: initializing ddx client\n");
		if (ddx_client_init(VERBOSE) != 0)
			goto error;

	}
	nbCreatedStores += 1;


	return (PyObject *)self;

error :
	rtx_error_flush("Rtx error flush");
	PyErr_SetString(PyExc_SystemError,"Cannot connect to store");
	Py_XDECREF(self);
	nbCreatedStores = oldNbStores;
	return NULL;
}


static int
ddxStore_init(ddxStore *self, PyObject *args, PyObject *kwds)
{
    char *host="localhost";
	int port=0,timeout=5;
    static char *kwlist[] = {"host", "port", "timeout", NULL};

    if (! PyArg_ParseTupleAndKeywords(args, kwds, "|sii", kwlist, 
                                      &host, &port, &timeout))
        return -1; 
	if (self->id != NULL)
		ddx_store_close(self->id);

	self->id = ddx_store_open(host,port,timeout);
	if (self->id == NULL) {
		char tmp[1024];
		sprintf(tmp,"Cannot open store '%s:%d' (timeout %ds)",host,port,timeout);
		rtx_error_flush("Rtx error flush");
		PyErr_SetString(PyExc_SystemError,tmp);
		return -1;
	}
	
    return 0;
}



static PyObject *
ddxStore_getVariable(ddxStore *self, PyObject *args)
{
    char *name=NULL;
	DDX_STORE_ITEM * item;
	RtxParseVar * var;

    if (! PyArg_ParseTuple(args, "s", &name)) {
		return NULL;
	}
	item = ddx_store_lookup_item(self->id,name,NULL,0);
	if (item == NULL) {
		rtx_error_flush("Rtx error flush");
		PyErr_SetString(PyExc_NameError,"Cannot find variable");
		return NULL;
	}
	var = ddx_store_item_get_parsed_var(item);
		
	return ddxVariable_Create(item, var,0);
}

static PyObject *
ddxStore_getDirectVariable(ddxStore *self, PyObject *args)
{
    char *name=NULL;
	DDX_STORE_ITEM * item;
	RtxParseVar * var;

    if (! PyArg_ParseTuple(args, "s", &name)) {
		return NULL;
	}
	item = ddx_store_lookup_item(self->id,name,NULL,0);
	if (item == NULL) {
		rtx_error_flush("Rtx error flush");
		PyErr_SetString(PyExc_NameError,"Cannot find variable");
		return NULL;
	}
	var = ddx_store_item_get_parsed_var(item);
		
	return ddxVariable_Create(item, var,1);
}

static PyObject *
ddxStore_flush(ddxStore * self) {
	rtx_error_flush("Rtx Flush");
	Py_RETURN_NONE;
}

static PyObject * 
ddxStore_getType(ddxStore * self, PyObject *args)
{
	char *name=NULL, *def=NULL;
	PyObject * result = NULL;

	if (! PyArg_ParseTuple(args, "s", &name)) {
		return NULL;
	}

	def = ddx_store_lookup_type(self->id,name);
	if (def == NULL) {
		rtx_error_flush("Rtx error flush");
		PyErr_SetString(PyExc_NameError,"Cannot find type");
		return NULL;
	}
	result = PyString_FromString(def);
	free(def);

	return result;
}

static PyObject * 
ddxStore_getConstant(ddxStore * self, PyObject *args)
{
	char *name=NULL;
	int val = 0;
	PyObject * result = NULL;

	if (! PyArg_ParseTuple(args, "s", &name)) {
		return NULL;
	}

	if (ddx_store_lookup_const(self->id,name,&val) != 0) {
		rtx_error_flush("Rtx error flush");
		PyErr_SetString(PyExc_NameError,"Cannot find constant");
		return NULL;
	}

	result = PyInt_FromLong(val);
	return result;
}

char * checkTypeSyntax(char * type) 
{
	char * ret;
	int len = strlen(type);
	int i;
	ret = (char*)malloc(len+2);
	strcpy(ret,type);
	for (i = 0;i < len; i++) {
		if (isspace(ret[i])) ret[i] = ' ';
	}
	for (i = len-1;i>0;i--) {
		if (isalpha(ret[i]) || ispunct(ret[i]))
			break;
	}
	if (i < 0) return NULL; // no valid char in there
	if (ret[i] != ';') {
		ret[i+1] = ';';
		ret[i+2] = 0;
	}
	return ret;
}

static PyObject *
ddxStore_registerType(PyObject * oself, PyObject * args)
{
	char *name=NULL;
	ddxStore * self = (ddxStore*)oself;

	if (! PyArg_ParseTuple(args, "s", &name)) {
		return NULL;
	}
	name = checkTypeSyntax(name);
	if (name == NULL) {
		rtx_error_flush("Rtx error flush");
		PyErr_SetString(PyExc_SystemError,"No valid character in typedef");
		return NULL;
	}
	//printf("Registering type '%s'\n",name);

	if (ddx_store_register_type(self->id,name) != 0) {
		free(name);
		rtx_error_flush("Rtx error flush");
		PyErr_SetString(PyExc_SystemError,"Could not register type");
		return NULL;
	}
	free(name);
	Py_RETURN_NONE;
}

static PyObject *
ddxStore_registerConst(PyObject * oself, PyObject * args)
{
	char *name=NULL;
	int value = 0;
	ddxStore * self = (ddxStore*)oself;

	if (! PyArg_ParseTuple(args, "si", &name, &value)) {
		return NULL;
	}

	if (ddx_store_register_const(self->id,name,value) != 0) {
		rtx_error_flush("Rtx error flush");
		PyErr_SetString(PyExc_SystemError,"Could not register const");
		return NULL;
	}
	Py_RETURN_NONE;
}

static PyObject *
ddxStore_registerVariable(PyObject * oself, PyObject * args)
{
	DDX_STORE_ITEM * item;
	char *name=NULL;
	char *type=NULL;
	ddxStore * self = (ddxStore*)oself;

	if (! PyArg_ParseTuple(args, "ss", &name, &type)) {
		return NULL;
	}

	
	item = ddx_store_lookup_item(self->id,name,type,0);
	if (item == NULL) {
		rtx_error_flush("Rtx error flush");
		PyErr_SetString(PyExc_SystemError,"Could not create variable");
		return NULL;
	}
	ddx_store_done_item(item);
	Py_RETURN_NONE;
}


static PyMethodDef ddxStore_methods[] = {
    {"variable", (PyCFunction)ddxStore_getVariable, METH_VARARGS,
     "Return a new ddxVariable which can be used to access DDX shared memory.\n"
		 "This variable owns a internal buffer in which shared memory segment\n"
		 "must be copied, with ddxVariable.read() function\n"
		 "Argument: [String]variable name"
    },
    {"directVariable", (PyCFunction)ddxStore_getDirectVariable, METH_VARARGS,
     "Return a new ddxVariable which can be used to access DDX shared memory.\n"
		 "This variable points directly to the shared memory segment. read/write\n"
		 "functions will only be used for timestamp/signals management."
		 "Argument: [String]variable name"
    },
    {"type", (PyCFunction)ddxStore_getType, METH_VARARGS,
     "Return the description of a type from the store.\n"
		 "Argument: [String]type name"
    },
    {"constant", (PyCFunction)ddxStore_getConstant, METH_VARARGS,
     "Return the value of a constant from the store.\n"
		 "Argument: [String]constant name"
    },
    {"registerVariable", (PyCFunction)ddxStore_registerVariable, METH_VARARGS,
     "Register a new variable in the store. It will be accessed with the\n"
		 "variable method. Arguments: [String]variable name, [String]type name"
    },
    {"registerType", (PyCFunction)ddxStore_registerType, METH_VARARGS,
     "Register a new type in the store.\n"
		 "Argument: [String] type definition (containing type name)"
    },
    {"registerConstant", (PyCFunction)ddxStore_registerConst, METH_VARARGS,
     "Register a new constant in the store.\n"
		 "Arguments: [String]constant name, [Int]constant value"
    },
#if 0
	// No longer needed, integrated in getattr
    {"update", (PyCFunction)ddxStore_update, METH_NOARGS,
     "update the list of variables/types/consts available in the store.\n"
		 "No argument"
    },
#endif
    {"flush", (PyCFunction)ddxStore_flush, METH_NOARGS,
     "Flush the rtx error stack.\n"
		 "No argument"
    },
    {NULL}  /* Sentinel */
};

static PyObject *
ddxStore_getattr(PyObject *oself, char *name)
{
	ddxStore * self = (ddxStore*)oself;
	PyObject * result = NULL;
	DDX_STRING_LIST * list = NULL;
	
	if (strcmp(name,"variables")==0) {
		list = ddx_store_get_variable_list(self->id);
		result = PyList_New(0);
	}
	if (strcmp(name,"types")==0) {
		list = ddx_store_get_type_list(self->id);
		result = PyList_New(0);
	}
	if (strcmp(name,"constants")==0) {
		list = ddx_store_get_constant_list(self->id);
		result = PyList_New(0);
	}
	
	// If result is not null, then the attribute was constants, 
	// variables or types.
	if (result != NULL) {
		DDX_STRING_LIST * it = list;
		while (it != NULL) {
			PyList_Append(result,PyString_FromString(it->name));
			it = it->next;
		}
		ddx_stringlist_free(list);
		return result;
	} /* else */
	return Py_FindMethod(ddxStore_methods,oself,name);
}


PyTypeObject ddx_StoreType = {
    PyObject_HEAD_INIT(NULL)
    0,                         /* ob_size*/
    "ddxInterface.ddxStore",   /* tp_name*/
    sizeof(ddxStore),          /* tp_basicsize*/
    0,                         /* tp_itemsize*/
    (destructor)ddxStore_dealloc, /* tp_dealloc*/
    0,                         /* tp_print*/
    ddxStore_getattr,          /* tp_getattr*/
    0,                         /* tp_setattr*/
    0,                         /* tp_compare*/
    0,                         /* tp_repr*/
    0,                         /* tp_as_number*/
    0,                         /* tp_as_sequence*/
    0,                         /* tp_as_mapping*/
    0,                         /* tp_hash */
    0,                         /* tp_call*/
    0,                         /* tp_str*/
    0,                         /* tp_getattro*/
    0,                         /* tp_setattro*/
    0,                         /* tp_as_buffer*/
    Py_TPFLAGS_DEFAULT| Py_TPFLAGS_BASETYPE, /* tp_flags*/
    "DDX Store objects",       /* tp_doc */
	0,		                   /* tp_traverse */
    0,		                   /* tp_clear */
    0,		                   /* tp_richcompare */
    0,		                   /* tp_weaklistoffset */
    0,		                   /* tp_iter */
    0,		                   /* tp_iternext */
    ddxStore_methods,          /* tp_methods */
    0,                         /* tp_members */
    0,                         /* tp_getset */
    0,                         /* tp_base */
    0,                         /* tp_dict */
    0,                         /* tp_descr_get */
    0,                         /* tp_descr_set */
    0,                         /* tp_dictoffset */
    (initproc)ddxStore_init,   /* tp_init */
    0,                         /* tp_alloc */
    ddxStore_new,              /* tp_new */
};

