/*********************************************************************
 *
 * CSIRO Automation
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * www.cat.csiro.au/cmst
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 *********************************************************************/

static char *rcsid = "$Id: store-queue-consumer.c 2200 2007-11-23 05:46:38Z roy029 $";

/**
 * \file store-consumer.c
 * \brief A reader client for the store
 * \author Pavan Sikka
 */

/** \page consumer consumer
 * An application that consumes (reads) data from the store.
 */

/** \example store-consumer.c
 * A store client that reads data from the store.
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <rtx/message.h>
#include <rtx/time.h>
#include <rtx/error.h>
#include <rtx/main.h>
#include <rtx/thread.h>
#include <rtx/getopt.h>

#include "ddx.h"

#define MAX_NUM_POINTS 256

DDX_STORE_TYPE (PLS_T, struct {int numPoints; int range[MAX_NUM_POINTS]; });
DDX_STORE_TYPE (TEST_T, struct {int x; int y; });
DDX_STORE_TYPE (TEST1_T, struct {PLS_T pls[3]; int x; TEST_T test[10]; });

DDX_STORE_ID * storeId = NULL;
DDX_STORE_ITEM * itemP = NULL;
DDX_STORE_ITEM * itemTestP = NULL;
DDX_STORE_ITEM * itemTest1P = NULL;

PLS_T plsRead, pls;
TEST_T test;
TEST1_T cmplx;

int done = 0;
RtxTime tsnow;
int verbose = 0;
RtxThread * th = NULL;
int directRead = 0;

RtxGetopt consumerOpts[] = {
  {"direct", "write to the store using ddx_store_read_direct()", 
   {
     {RTX_GETOPT_SET, &directRead, ""},
     RTX_GETOPT_END_ARG
   }
  },
  RTX_GETOPT_END
};

char * consumerHelpStr = "DDX consumer";

void *
thfunc (void * arg)
{
  PLS_T * plsP;
  int tailCount=0;
  int headCount;

  plsP = (PLS_T *) ddx_store_var_pointer(itemP);
  while (! done) {
    if (directRead) {
      if (ddx_store_read_direct (itemP, &tsnow, 3.0, 1) == -1) {
	rtx_message_warning ("store-consumer: ddx_store_sync_read: %s",
			     strerror (errno));
	sleep (1);
	continue;
      }
      rtx_message_routine ("store-consumer: ts = %d, %d, "
			   "pls.numPoints = %d",
			   tsnow.seconds, tsnow.nanoSeconds,
			   plsP->numPoints);
    } else {
      //int ret = ddx_store_sync_read (itemP, &pls, &tsnow, 3.0);
      
      int ret = ddx_store_queue_read(itemP, &pls, &tsnow, 3.0, &tailCount);
      switch (ret) {
      case -1:
	rtx_message_warning ("store-consumer: ddx_store_sync_read: %s",
			     strerror (errno));
	sleep (1);
	continue;
      case -2:
	rtx_message_warning("store-consumer: ddx_store_sync_read: timeout");
	continue;
      default:
	break;
      }
      headCount = itemP->headerP->count;
      rtx_message_routine ("store-consumer: ts = %d, %d, "
			   "pls.numPoints = %d, tail=%d, head=%d, ret=%d",
			   tsnow.seconds, tsnow.nanoSeconds, 
			   pls.numPoints, tailCount, headCount, ret);
    }
  }
  return (0);
}

int
main (int argc, char * argv[])
{
  char name[256];
  int ret;

  rtx_main_init ("store-consumer",RTX_MESSAGE_MESSAGE | RTX_ERROR_STDERR);

  if ((ret = RTX_GETOPT_CMD (consumerOpts, argc, argv, rcsid, 
			     consumerHelpStr)) == -1) {
    RTX_GETOPT_PRINT (consumerOpts, argv[0], rcsid, consumerHelpStr);
    exit (1);
  }
  verbose = rtx_getopt_get_verbose (0);

  if (verbose > 1) {
    if (ddx_client_init (verbose) == -1)
      return (rtx_error ("main: Unable to initialize library"));
  } else {
    if (ddx_client_init (0) == -1)
      return (rtx_error ("main: Unable to initialize library"));
  }

  if (gethostname (name, 256) == -1)
    return (rtx_error ("main: Unable to get host name"));

  if ((storeId = ddx_store_open (NULL, 0, 5)) == NULL)
    return (rtx_error ("main: Unable to open store"));

  if (DDX_STORE_REGISTER_CONST (storeId, MAX_NUM_POINTS) == -1)
    return (rtx_error ("main: Unable to register constant"));

  if (DDX_STORE_REGISTER_TYPE (storeId, PLS_T) == -1)
    return (rtx_error ("main: Unable to register type"));

  if (DDX_STORE_REGISTER_TYPE (storeId, TEST_T) == -1)
    return (rtx_error ("main: Unable to register type"));

  if (DDX_STORE_REGISTER_TYPE (storeId, TEST1_T) == -1)
    return (rtx_error ("main: Unable to register type"));

  if ((itemP = ddx_store_lookup_item (storeId, "pls", "PLS_T", 0)) == NULL)
    return (rtx_error ("main: Unable to lookup var"));

  if ((itemTestP = ddx_store_lookup_item (storeId, "test", "TEST_T", 0)) == NULL)
    return (rtx_error ("main: Unable to lookup var"));

  if ((itemTest1P = ddx_store_lookup_item (storeId, "cmplx", "TEST1_T", 0)) 
      == NULL)
    return (rtx_error ("main: Unable to lookup var"));


  ddx_store_item_set_queue(itemP, 10);


  if ((th = rtx_thread_create ("thfunc", 0,
			       RTX_THREAD_SCHED_OTHER, RTX_THREAD_PRIO_MIN, 0, 
			       RTX_THREAD_CANCEL_DEFERRED,
			       thfunc, NULL,
			       NULL, NULL)) == NULL)
    return (rtx_error ("main: rtx_thread_create() failed"));

  if (rtx_main_wait_shutdown (0) == -1)
    rtx_error_flush ("main: rtx_main_wait_shutdown() failed");
  else
    rtx_message_routine ("Caught SIGINT/SIGQUIT, exiting ...");
  done = 1;

  rtx_thread_destroy_sync (th);
  if (itemP != NULL)
    ddx_store_done_item (itemP);
  if (itemTestP != NULL)
    ddx_store_done_item (itemTestP);
  if (itemTest1P != NULL)
    ddx_store_done_item (itemTest1P);
  if (storeId != NULL)
    ddx_store_close (storeId);
  ddx_client_done ();

  return (0);
}
