
/*********************************************************************
 *
 * CSIRO Automation
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * www.cat.csiro.au/cmst
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 *********************************************************************/

static char *rcsid = "$Id: store-producer.c 2200 2007-11-23 05:46:38Z roy029 $";

/**
 * \file store-producer.c
 * \brief A writer client to the store
 * \author Pavan Sikka
 */

/** \page producer producer
 * An application that produces (writes) data to the store.
 */

/** \example store-producer.c
 * A store client that writes data to the store.
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <string.h>
#include <errno.h>
#include <time.h>

#include <rtx/message.h>
#include <rtx/time.h>
#include <rtx/timer.h>
#include <rtx/error.h>
#include <rtx/thread.h>
#include <rtx/main.h>
#include <rtx/getopt.h>

#include "ddx.h"

// #include "store-producer-types.h"

#define MAX_NUM_POINTS 256

DDX_STORE_TYPE (PLS_T, struct {int numPoints; int range[MAX_NUM_POINTS]; });
DDX_STORE_TYPE (TEST_T, struct {unsigned int x; int y; });
DDX_STORE_TYPE (TEST1_T, struct {PLS_T pls[3]; int x; TEST_T test[10]; });

DDX_STORE_ID * storeId;
DDX_STORE_ITEM * itemP;
DDX_STORE_ITEM * itemTestP;
DDX_STORE_ITEM * itemTest1P;

PLS_T plsRead, pls;
TEST_T test;
TEST1_T cmplx;

int done = 0;
int directWrite = 0;
int noTypedefs = 0;
int verbose = 0;
RtxTime tsnow;
double loopDelay = 1.0;
RtxThread * th = NULL;

RtxGetopt producerOpts[] = {
  {"direct", "write to the store using ddx_store_write_direct()", 
   {
     {RTX_GETOPT_SET, &directWrite, ""},
     RTX_GETOPT_END_ARG
   }
  },
  {"notypedefs", "do not register typedefs with the store",
   {
     {RTX_GETOPT_SET, &noTypedefs, ""},
     RTX_GETOPT_END_ARG
   }
  },
  RTX_GETOPT_END
};

char * producerHelpStr = "DDX producer";

void *
thfunc (void * arg)
{
    PLS_T * plsP;
	int i,j;
	int count=0;

    plsP = (PLS_T *) ddx_store_var_pointer(itemP);
	while (! done) {
		rtx_time_get (&tsnow);
		//pls.numPoints = rand();
		pls.numPoints = count++;
		for (j=0;j<MAX_NUM_POINTS;j++) 
			pls.range[j] = rand();
		for (i=0;i<3;i++)
			for (j=0;j<MAX_NUM_POINTS;j++) 
				cmplx.pls[i].range[j] = rand();
		if (verbose)
			rtx_message ("store-producer: ts = %d, %d, pls.numPoints = %d",
					tsnow.seconds, tsnow.nanoSeconds, pls.numPoints);
		if (directWrite) {
			plsP->numPoints = pls.numPoints;
			if (ddx_store_write_direct (itemP, NULL) == -1) {
				rtx_error ("main: store-producer: ddx_store_write: %s",
						strerror (errno));
			}
		} else {
			if (ddx_store_write (itemP, &pls, NULL) == -1) {
				rtx_error ("main: store-producer: ddx_store_write: %s",
						strerror (errno));
			}
		}
		if (ddx_store_write (itemTest1P, &cmplx, NULL) == -1) {
			rtx_error ("main: store-producer: ddx_store_write: %s",
					strerror (errno));
		}
		rtx_timer_sleep (loopDelay);
	}
    return (NULL);
}

int
main (int argc, char * argv[])
{
    char name[256];
    int ret;

    if ((ret = RTX_GETOPT_CMD (producerOpts, argc, argv, rcsid, 
			       producerHelpStr)) == -1) {
        RTX_GETOPT_PRINT (producerOpts, argv[0], rcsid, producerHelpStr);
        exit (1);
    }
    verbose = rtx_getopt_get_verbose (0);
    loopDelay = rtx_getopt_get_sample (1.0);

    rtx_main_init ("store-producer", RTX_ERROR_STDERR);

    if (verbose > 1) {
        if (ddx_client_init (verbose) == -1)
	    return (rtx_error ("main: Unable to initialize library"));
    } else {
        if (ddx_client_init (0) == -1)
	    return (rtx_error ("main: Unable to initialize library"));
    }

    if (gethostname (name, 256) == -1)
        return (rtx_error ("main: Unable to get host name"));

    if ((storeId = ddx_store_open (NULL, 0, 5)) == NULL)
        return (rtx_error ("main: Unable to open store"));

    if (noTypedefs == 0) {
        if (DDX_STORE_REGISTER_CONST (storeId, MAX_NUM_POINTS) == -1)
            return (rtx_error ("main: Unable to register constant"));

	if (DDX_STORE_REGISTER_TYPE (storeId, PLS_T) == -1)
	    return (rtx_error ("main: Unable to register type"));

	if (DDX_STORE_REGISTER_TYPE (storeId, TEST_T) == -1)
	    return (rtx_error ("main: Unable to register type"));

	if (DDX_STORE_REGISTER_TYPE (storeId, TEST1_T) == -1)
            return (rtx_error ("main: Unable to register type"));
    }

    if ((itemP = ddx_store_lookup_item (storeId, "pls", "PLS_T", 0)) == NULL)
        return (rtx_error ("main: Unable to lookup var"));

    if ((itemTestP = ddx_store_lookup_item (storeId, "test", "TEST_T", 0)) == NULL)
        return (rtx_error ("main: Unable to lookup var"));

    if ((itemTest1P = ddx_store_lookup_item (storeId, "cmplx", "TEST1_T", 0)) 
	    == NULL)
        return (rtx_error ("main: Unable to lookup var"));


    if ((th = rtx_thread_create ("thfunc", 0,
				 RTX_THREAD_SCHED_OTHER, RTX_THREAD_PRIO_MIN, 0, 
				 RTX_THREAD_CANCEL_DEFERRED,
				 thfunc, NULL,
				 NULL, NULL)) == NULL)
        return (rtx_error ("main: rtx_thread_create() failed"));

    if (rtx_main_wait_shutdown (0) == -1)
        rtx_error_flush ("main: rtx_main_wait_shutdown() failed");
    else
        rtx_message_routine ("Caught SIGINT/SIGQUIT, exiting ...");
    done = 1;

    rtx_thread_destroy_sync (th);
    ddx_store_done_item (itemP);
    ddx_store_done_item (itemTestP);
    ddx_store_done_item (itemTest1P);
    ddx_store_close (storeId);
    ddx_client_done ();
    return (0);
}
 
