
/***********************************************************************
 * store-mem.c - functions for handling shared memory
 *
 * CSIRO MANUFACTURING SCIENCE & TECHNOLOGY
 * HIGH-SPEED SERIAL INTERFACE PROJECT
 * 
 * $Id: store-mem.c 2200 2007-11-23 05:46:38Z roy029 $
 * 
 * Copyright (c) CSIRO Manufacturing Science & Technology.
 *
 ***********************************************************************
 */

#ifdef __GNUC__
#define DDX_STATIC_ATTRIBUTE __attribute__((unused))
#else
#define DDX_STATIC_ATTRIBUTE
#endif
static char *rcsid DDX_STATIC_ATTRIBUTE = "$Id: store-mem.c 2200 2007-11-23 05:46:38Z roy029 $";

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <pthread.h>

typedef struct _free_blk {
    int offset;
    int size;
    struct _free_blk * next;
} FREE_BLK;

static FREE_BLK * freep = NULL;
pthread_mutex_t storeMemMutex;
int storeMemUnitSize = __alignof__ (double);

int
store_init_mem (
	int size
    )
{
    if ((freep = malloc (sizeof (FREE_BLK))) == NULL)
        return (-1);
    
    freep->offset = 0;
    freep->size = size;
    freep->next = NULL;
    if (pthread_mutex_init (&storeMemMutex, NULL) == -1) {
        free (freep);
	return (-1);
    }
    return (0);
}

int
store_alloc_mem (
        int size
    )
{
    FREE_BLK * fbp, * prev = NULL;
    int numUnits;
    int retOffset;

    numUnits = size / storeMemUnitSize;
    if (size % storeMemUnitSize)
        numUnits++;
    size = numUnits * storeMemUnitSize;
    for (fbp = freep; fbp != NULL; fbp = fbp->next) {
        if (fbp->size == size) {
	    if (prev == NULL) {
	        retOffset = fbp->offset;
		freep = fbp->next;
	    } else {
	        retOffset = fbp->offset;
		prev->next = fbp->next;
	    }
	    free (fbp);
#ifdef DEBUG
	    rtx_message("store_alloc_mem(%d) = %d",size,retOffset);
#endif
	    return (retOffset);
	} else if (fbp->size > size) {
	    retOffset = fbp->offset;
	    fbp->offset += size;
	    fbp->size -= size;
#ifdef DEBUG
	    rtx_message("store_alloc_mem(%d) = %d",size,retOffset);
#endif
	    return (retOffset);
	}
        prev = fbp;
    }
    return (-1);
}

int
store_free_mem (
	int offset,
        int size
    )
{
    FREE_BLK * fbp, * prev = NULL, * t;
    int numUnits;

#ifdef DEBUG
    rtx_message("store_free_mem(%d,%d) called",offset,size);
#endif

    numUnits = size / storeMemUnitSize;
    if (size % storeMemUnitSize)
        numUnits++;
    size = numUnits * storeMemUnitSize;
    for (fbp = freep; fbp != NULL; fbp = fbp->next) {
        if (fbp->offset > offset)
	    break;
        prev = fbp;
    }
    /* nothing in the free list */
    if ((fbp == NULL) && (prev == NULL)) {
	if ((t = malloc (sizeof (FREE_BLK))) == NULL)
	    return (-1);
	t->offset = offset;
	t->size = size;
	t->next = fbp;
	freep = t;
	return (0);
    }
    /* beginning of the list */
    if (prev == NULL) {
        if ((offset + size) >= fbp->offset) {
  	    fbp->offset = offset;
	    fbp->size += (fbp->offset - offset);
	    return (0);
	}
	if ((t = malloc (sizeof (FREE_BLK))) == NULL)
	    return (-1);
    
	t->offset = offset;
	t->size = size;
	t->next = fbp;
	freep = t;
	return (0);
    }
    /* end of the list */
    if (fbp == NULL) {
        if ((prev->offset + prev->size) >= offset) {
	    prev->size = size + (offset - prev->offset);
	    return (0);
	}
	if ((t = malloc (sizeof (FREE_BLK))) == NULL)
	    return (-1);
    
	t->offset = offset;
	t->size = size;
	t->next = NULL;
	prev->next = t;
	return (0);
    }
    /* middle of the list */
    if ((prev->offset + prev->size) >= offset) {
        prev->size = size + (offset - prev->offset);
    }
    if ((offset + size) >= fbp->offset) {
  	fbp->offset = offset;
	fbp->size += (fbp->offset - offset);
    }
    if ((prev->offset + prev->size) >= fbp->offset) {
        prev->size = size + (offset - prev->offset);
	prev->next = fbp->next;
	free (fbp->next);
	return (0);
    }
    return (0);
}

