
/*********************************************************************
 *
 * CSIRO Automation
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * www.cat.csiro.au/cmst
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 *********************************************************************/

static char *rcsid = "$Id: store-main.c 1431 2007-10-16 05:38:13Z roy029 $";
#ifdef __GNUC__
#define DDX_STATIC_ATTRIBUTE __attribute__((unused))
#else
#define DDX_STATIC_ATTRIBUTE
#endif
static char *tagname DDX_STATIC_ATTRIBUTE = "$Name$";

/**
 * \file store-main.c
 * \brief DDX store
 * \author Pavan Sikka
 */

/**
 * \page store store
 *
 * The store creates a shared memory segment that is used to store
 * data for clients. It also manages communications with other
 * stores and the catalog. The store is also responsible for 
 * managing the semaphores in shared memory that allow clients
 * to synchronize.
 *
 */


#include <unistd.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <errno.h>
#include <signal.h>
#include <pthread.h>
#include <sys/socket.h>
#include <sys/wait.h>

#include <rtx/message.h>
#include <rtx/timer.h>
#include <rtx/sem.h>
#include <rtx/thread.h>
#include <rtx/main.h>
#include <rtx/error.h>
#include <rtx/mutex.h>
#include <rtx/signal.h>
#include <rtx/parse.h>
#include <rtx/inet.h>
#include <rtx/getopt.h>

#include "store-ipc.h"
#include "store-msg.h"
#include "store-messages.h"
#include "catalog-messages.h"
#include "store-mem.h"
#include "store-main.h"
#include "store-handlers.h"
#include "store-common.h"

/**
 * global state
 */
STORE mainStoreStruct;
STORE * ddxStore = &mainStoreStruct;
int startweb = 0;
char webprogram[] = "ddxweb";
char * memsize_str = "64k";

RtxGetopt storeOpts[] = {
	{"catalog", "catalog hostname/address", 
		{
			{RTX_GETOPT_STR, &mainStoreStruct.catalogHostname, "hostname/address"},
			RTX_GETOPT_END_ARG
		}
	},
	{"number", "catalog port number",
		{
			{RTX_GETOPT_INT, &mainStoreStruct.catalogPortNum,
				"port number"},
			RTX_GETOPT_END_ARG
		}
	},
	{"port", "store port number",
		{
			{RTX_GETOPT_INT, &mainStoreStruct.portNum, "port number"},
			RTX_GETOPT_END_ARG
		}
	},
	{"memsize", "shared memory block size (in bytes)",
		{
			//{RTX_GETOPT_INT, &mainStoreStruct.shmemSize, "size"},
			{RTX_GETOPT_STR, &memsize_str, "size"},
			RTX_GETOPT_END_ARG
		}
	},
	{"-daemon", "fork and run as a daemon", 
		{
			{RTX_GETOPT_SET, &mainStoreStruct.forkFlag, ""},
			RTX_GETOPT_END_ARG
		}
	},
	{"-launchcatalog", "launch the catalog in the background", 
		{
			{RTX_GETOPT_SET, &mainStoreStruct.launchCatalog, ""},
			RTX_GETOPT_END_ARG
		}
	},
	{"include", "include header file with typedefs", 
		{
			{RTX_GETOPT_STR, &mainStoreStruct.incFileName, "filename"},
			RTX_GETOPT_END_ARG
		}
	},
	{"semprio", "priority of the sem Q handler thread",
		{
			{RTX_GETOPT_INT, &mainStoreStruct.semQHandlerThPrio, "priority"},
			RTX_GETOPT_END_ARG
		}
	},
	{"semdebug", "debug the sem Q handler thread",
		{
			{RTX_GETOPT_INT, &mainStoreStruct.semQHandlerDebug, "level"},
			RTX_GETOPT_END_ARG
		}
	},
	{"ttlmulticast", "TTL value for multicast sockets",
		{
			{RTX_GETOPT_INT, &mainStoreStruct.multicastTtl, "TTL"},
			RTX_GETOPT_END_ARG
		}
	},
	{"web", "start the web server (ddxweb)",
		{
			{RTX_GETOPT_SET, &startweb, "web"},
			RTX_GETOPT_END_ARG
		}
	},
	RTX_GETOPT_END
};

char * storeHelpStr = "DDX store";

/**
 * process requests received from store clients
 *
 * @return -1 on error, 0 if OK
 */
int
store_process_request (
		RtxInetConn * inetConn,     /**< Inet endpoint of
									 **  client */
		char * recvBuf,             /**< rx buffer */
		char * sendBuf,             /**< tx buffer */
		STORE * store               /**< global state */
		)
{
	int numChars;
	MSG * mp;
	int ret;

	if ((mp = parse_message (keyTable, recvBuf)) == NULL) {
		strcpy (sendBuf,
				"storeErrorReply response = message parse failed");
		if ((numChars = rtx_inet_write (inetConn, sendBuf,
						strlen (sendBuf), NULL)) == -1) {
			rtx_error ("rtx_inet_write() failed");
		}
		rtx_error ("msg parse failed on msg from %s:%d [%s]",
				rtx_inet_print_net_addr (&inetConn->remote),
				rtx_inet_print_net_portnum (&inetConn->remote),
				recvBuf);
		return (-1);
	}
	if (identify_message (msgTable, mp) == -1) {
		strcpy (sendBuf, "storeErrorReply response = invalid message");
		if ((numChars = rtx_inet_write (inetConn, sendBuf,
						strlen (sendBuf), NULL)) == -1) {
			rtx_error ("rtx_inet_write() failed");
		}
		rtx_error ("msg identify failed on msg from %s:%d [%s]",
				rtx_inet_print_net_addr (&inetConn->remote),
				rtx_inet_print_net_portnum (&inetConn->remote),
				recvBuf);
		return (-1);
	}
	ret = 0;
	switch (mp->id) {
		case MSG_STORE_OPEN_REQUEST :
			ret = store_handle_open_request (mp, recvBuf, sendBuf, 
					inetConn, store);
			break;
		case MSG_STORE_REGISTER_CONST_REQUEST :
			ret = store_handle_register_const_request (mp, recvBuf, 
					sendBuf, inetConn, 
					store);
			break;
		case MSG_STORE_REGISTER_TYPE_REQUEST :
			ret = store_handle_register_type_request (mp, recvBuf, sendBuf, 
					inetConn, store);
			break;
		case MSG_STORE_LOOKUP_VAR_REQUEST :
			ret = store_handle_lookup_var_request (mp, recvBuf, 
					sendBuf, inetConn, store);
			break;
		case MSG_STORE_DONE_VAR_REQUEST :
			ret = store_handle_done_var_request (mp, recvBuf, 
					sendBuf, inetConn, store);
			break;
		case MSG_STORE_CLOSE_REQUEST :
			ret = store_handle_close_request (mp, recvBuf, sendBuf,
					inetConn, store);
			break;
		case MSG_STORE_GET_CATALOG_REQUEST :
			ret = store_handle_get_catalog_request (mp, recvBuf,
					sendBuf, inetConn, store);
			break;
		case MSG_STORE_GET_VAR_LIST_REQUEST :
			ret = store_handle_get_var_list_request (mp, recvBuf,
					sendBuf, inetConn, store);
			break;
		case MSG_STORE_GET_VAR_STATUS_REQUEST :
			ret = store_handle_get_var_status_request (mp, recvBuf, 
					sendBuf, inetConn,
					store);
			break;
		case MSG_STORE_GET_VAR_VALUE_REQUEST :
			ret = store_handle_get_var_value_request (mp, recvBuf, 
					sendBuf, inetConn, 
					store);
			break;
		case MSG_STORE_SET_VAR_VALUE_REQUEST :
			ret = store_handle_set_var_value_request (mp, recvBuf,
					sendBuf, inetConn, 
					store);
			break;
		case MSG_STORE_LOOKUP_FIELD_REQUEST :
			ret = store_handle_lookup_field_request (mp, recvBuf, 
					sendBuf, inetConn, store);
			break;
		case MSG_STORE_SET_VAR_QUEUE_REQUEST :
			ret = store_handle_set_var_queue_request (mp, recvBuf,
					sendBuf, inetConn, 
					store);
			break;
		default :
			ret = -1;
			rtx_error ("unknown msg id (%d) from %s:%d [%s]",
					mp->id, rtx_inet_print_net_addr (&inetConn->remote),
					rtx_inet_print_net_portnum (&inetConn->remote), recvBuf);
			strcpy (sendBuf, "storeErrorReply response = "
					"unknown message id");
			if ((numChars = rtx_inet_write (inetConn, sendBuf,
							strlen (sendBuf), NULL)) == -1) {
				rtx_error ("rtx_inet_write() failed");
			}
			break;
	}
	free_message (keyTable, mp);
	if (ret == -1) {
		return (rtx_error ("error handling request"));
	}
	return (0);

}

/**
 * thread started by the Inet module for each store client
 *
 * @return NULL
 */
void *
store_request_handler_thread (
		void * arg1,    /**< RtxInetTcpClient */
		void * arg2     /**< STORE */
		)
{
	RtxInetTcpClient * client;
	STORE * store;
	char * recvBuf, * sendBuf, ch;
	int done = 0, n = 1, i = 0;

	client = (RtxInetTcpClient *) arg1;
	store = (STORE *) arg2;
	rtx_error_init ("store[client_handler_thread]", 
			RTX_ERROR_STDERR | RTX_ERROR_MESSAGE, NULL);
	if (rtx_signal_block (SIGINT))
		rtx_error ("rtx_signal_block(SIGINT) failed");
	if (rtx_signal_block (SIGQUIT))
		rtx_error ("rtx_signal_block(SIGQUIT) failed");
	if (rtx_signal_block (SIGTSTP))
		rtx_error ("rtx_signal_block(SIGTSTP) failed");

	if ((recvBuf = calloc (1, STORE_BUF_SIZE)) == NULL) {
		rtx_error ("unable to allocate buffer");
		pthread_exit ((void *) 0);
	}
	if ((sendBuf = calloc (1, STORE_BUF_SIZE)) == NULL) {
		rtx_error ("unable to allocate buffer");
		free (recvBuf);
		pthread_exit ((void *) 0);
	}
	while (! done) {
		i = 0; n = 1;
		while ((n > 0) && (i < (STORE_BUF_SIZE-1))) {
			if ((n = read (client->sock->sockfd, &ch, 1)) < 1) {
				done = 1;
				if ((n < 0) && (! store->done) && 
						(errno != ECONNRESET))
					rtx_error_errno ("read failed [%d]", errno);
				done = store->done;
				continue;
			}
			if (ch == '\n') {
				recvBuf[i++] = '\0';
				break;
			}
			recvBuf[i++] = ch;
		}
		if ((n <= 0) || (done)) {
			done = 1;
			continue;
		}
		/* Have a complete request for processing */
		store_debug (("store_request_handler_thread: "
					"from %s %d - %s", 
					rtx_inet_print_net_addr (&client->sock->remote),
					rtx_inet_print_net_portnum (&client->sock->remote),
					recvBuf));
		if (store_process_request (client->sock, recvBuf, sendBuf,
					store) == -1) {
			rtx_error_flush ("store_process_request() failed");
		}
	}
	if ((n == 0) || (errno == ECONNRESET))
		rtx_error_traceback_depth (0);
	store_clean_sync_semaphores(store,client->sock);
	store_debug(("Client terminated %s %d\n",
			rtx_inet_print_net_addr (&client->sock->remote),
			rtx_inet_print_net_portnum (&client->sock->remote)));
	free (recvBuf);
	free (sendBuf);
	return (0);
}

void *
store_catalog_handler_thread (
		STORE * store
		)
{
	char * recvBuf, * sendBuf, ch;
	int done = 0, n = 1, i = 0;
	DDX_STORE_ID * catalog = &(store->catalog);
	RtxInet * fromAddr = store->catalog.server;
	MSG * mp;
	int ret;

	if ((recvBuf = calloc (1, STORE_BUF_SIZE)) == NULL) {
		rtx_error ("unable to allocate buffer");
		pthread_exit ((void *) 0);
	}
	if ((sendBuf = calloc (1, STORE_BUF_SIZE)) == NULL) {
		rtx_error ("unable to allocate buffer");
		free (recvBuf);
		pthread_exit ((void *) 0);
	}
	rtx_thread_cleanup_push(free,recvBuf);
	rtx_thread_cleanup_push(free,sendBuf);
	while (! done) {
		i = 0; n = 1;
		while ((n > 0) && (i < (STORE_BUF_SIZE-1))) {
			if ((n = read (catalog->server->sock->sockfd, &ch, 1)) < 1) {
				done = 1;
				if ((n < 0) && (! store->done) && 
						(errno != ECONNRESET))
					rtx_error_errno ("read failed [%d]", errno);
				done = store->done;
				continue;
			}
			if (ch == '\n') {
				recvBuf[i++] = '\0';
				break;
			}
			recvBuf[i++] = ch;
		}
		if ((n <= 0) || (done)) {
			done = 1;
			continue;
		}
		/* Have a complete request for processing */
		store_debug (("store_catalog_handler_thread: rx [%s]", recvBuf));
		if ((mp = parse_message (catKeyTable, recvBuf)) == NULL) {
			rtx_error_flush ("msg parse failed on msg [%s]", recvBuf);
			continue;
		}
		if (identify_message (catMsgTable, mp) == -1) {
			rtx_error_flush ("msg identify failed on msg [%s]", recvBuf);
			continue;
		}
		store_debug (("store_catalog_handler_thread: msg id [%d]", mp->id));
		ret = 0;
		switch (mp->id) {
			case MSG_CATALOG_START_MULTICAST_CMD_TO_STORE :
				ret = store_handle_start_multicast_request
					(mp, recvBuf, sendBuf, fromAddr->sock, store);
				free_message (catKeyTable, mp);
				break;
			case MSG_CATALOG_STOP_MULTICAST_CMD_TO_STORE :
				ret = store_handle_stop_multicast_request
					(mp, recvBuf, sendBuf, fromAddr->sock, store);
				free_message (catKeyTable, mp);
				break;
			default :
				if (rtx_list_enqueue (store->catalogMsgQ, mp) == -1) {
					rtx_error ("rtx_list_enqueue");
					ret = -1;
				}
				break;
		}
		if (ret == -1) {
			rtx_error_flush ("error handling request");
		}
	}
	if ((n == 0) || (errno == ECONNRESET))
		rtx_error_traceback_depth (0);
	rtx_thread_cleanup_pop(1); /* free recvBuf */
	rtx_thread_cleanup_pop(1); /* free sendBuf */
	return (NULL);
}

/* test if two end points are the same,
 * returns 0 if identic, non zero otherwise */
int endpoint_compare(RtxInetEndpoint *e1, RtxInetEndpoint *e2)
{
	int i;
	if (e1->portNum != e2->portNum) return 1;
	if (e1->addrLen != e2->addrLen) return 2;
	for (i=0;i<e1->addrLen;i++) 
		if (e1->addr[i] != e2->addr[i])
			return 3;
	return 0;
}

void *
store_sem_Q_handler_thread (
		STORE * store
		)
{
	int varId = 0;
	int syncSemId = 0;
	STORE_SEM_Q_ELEMENT req;
	RtxTime ts1, ts2, ts3;
	double pullTime = 0.0, execTime = 0.0;
	double avgPullTime = 0.0, avgExecTime = 0.0;
	double maxPullTime = 0.0, maxExecTime = 0.0;
	unsigned int loopCount = 0;
	char semName[128];
	STORE_SEM_T * semP;
	RtxInetConn * conn;

	while (! store->done) {
		if (rtx_sem_wait_ignore_eintr (&(store->semQ->dataSem)) == -1) {
			rtx_error_flush ("rtx_sem_wait(dataSem) failed");
			rtx_timer_sleep (0.01);
			continue;
		}
		if (store->done)
			continue;
		if (rtx_sem_wait_ignore_eintr (&(store->semQ->lockSem)) == -1) {
			rtx_error_flush ("rtx_sem_wait(lockSem) failed");
			rtx_timer_sleep (0.01);
			continue;
		}
		rtx_time_get (&ts2);
		if (store->semQ->nextVarInQ != store->semQ->nextEmptySlotInQ) {
			/* Get the var id from Q */
			req = store->semQ->Q[store->semQ->nextVarInQ];
			if (req.action == STORE_SEM_Q_ACTION_POST) {
				ts1.seconds = 
					store->semQ->Q[store->semQ->nextVarInQ].ts.tv_sec;
				ts1.nanoSeconds = 
					store->semQ->Q[store->semQ->nextVarInQ].ts.tv_nsec;
			}
			store->semQ->nextVarInQ++;
			if (store->semQ->nextVarInQ >= store->semQ->QSize)
				store->semQ->nextVarInQ = 0;
		}
		if (rtx_sem_post (&(store->semQ->lockSem)) == -1) {
			rtx_error_flush ("rtx_sem_post(lockSem) failed");
			rtx_timer_sleep (0.01);
			continue;
		}
		varId = req.varId;
		switch (req.action) {
			case STORE_SEM_Q_ACTION_POST :
				while ((semP = (STORE_SEM_T *) rtx_list_iterate 
							(store->vars[varId]->syncSemList)) != NULL)
					if (store_post_binary_sem (semP) == -1) {
						rtx_error_flush ("rtx_sem_post(sem) failed");
					}
				break;
			case STORE_SEM_Q_ACTION_ADD :
				sprintf (semName, "%s-%d", 
						(store->vars[varId])->vEntry->name,
						((STORE_SEM_T *) req.arg)->id);
				if (rtx_list_add ((store->vars[varId])->syncSemList,
							semName,
							(STORE_SEM_T *) req.arg) == -1)
					rtx_error_flush ("rtx_list_add() failed");
				break;
			case STORE_SEM_Q_ACTION_DEL :
				syncSemId = (int) req.arg;
				sprintf (semName, "%s-%d", 
						(store->vars[varId])->vEntry->name,
						syncSemId);
				if ((semP = (STORE_SEM_T *) rtx_list_del 
							((store->vars[varId])->syncSemList,
							 semName, 0)) == NULL) {
					rtx_error_flush ("rtx_list_del() failed");
					break;
				}
				if (store_destroy_sem (semP))
					rtx_error_flush ("store_destroy_sem() failed");
				break;
			case STORE_SEM_Q_ACTION_CLEAN:
				conn = (RtxInetConn*)req.arg;
				for (varId=0;varId<SIZE_VAR_INFO_LIST;varId++) {
					RtxList * del;
					if (store->vars[varId] == NULL) continue;
					del = rtx_list_init();
					while ((semP = (STORE_SEM_T *) rtx_list_iterate 
								(store->vars[varId]->syncSemList)) != NULL)
					{
						if (semP->conn == NULL) 
							continue;
						if (endpoint_compare(&(semP->conn->remote), &(conn->remote))) 
							continue; 
						/* let's destroy this semaphore */
						syncSemId = semP->id;
						sprintf (semName, "%s-%d", 
								(store->vars[varId])->vEntry->name,
								syncSemId);
						rtx_list_add(del,semName,semP);
					}
					while ((semP = (STORE_SEM_T *) rtx_list_iterate 
								(del)) != NULL)
					{
						if (rtx_list_del_node ((store->vars[varId])->syncSemList,
									semP, 0) == NULL) {
							rtx_error_flush ("rtx_list_del_node() failed");
							continue;
						}
						if (store_destroy_sem (semP))
							rtx_error_flush ("store_destroy_sem() failed");
					}
					rtx_list_destroy(del,0);
				}
				break;
			default :
				rtx_message_warning ("invalid action (%d)", req.action);
				break;
		}
		rtx_time_get (&ts3);
		if (req.action == STORE_SEM_Q_ACTION_POST) {
			pullTime = rtx_time_subtract_to_double (&ts2, &ts1);
			execTime = rtx_time_subtract_to_double (&ts3, &ts2);
			if (execTime > 1.0)
				rtx_message_warning ("execTime [%f]", execTime);
			if (pullTime > 1.0)
				rtx_message_warning ("pullTime [%f]", pullTime);
			if (store->semQHandlerDebug) {
				loopCount++;
				if (pullTime > maxPullTime)
					maxPullTime = pullTime;
				if (execTime > maxExecTime)
					maxExecTime = execTime;
				avgPullTime += pullTime;
				avgExecTime += execTime;
				if ((loopCount % 100) == 0) {
					rtx_message ("pullTime (avg/max) [%f/%f]",
							avgPullTime / 100.0, maxPullTime);
					rtx_message ("execTime (avg/max) [%f/%f]", 
							avgExecTime / 100.0, maxExecTime);
					avgPullTime = 0.0;
					avgExecTime = 0.0;
				}
			}
		}

	}
	return (NULL);
}

int
store_init_sem_Q (
		STORE * store
		)
{
	char name[32];

	sprintf (name, "/%s-%d", STORE_SEM_Q_NAME, store->pid);
	store_debug (("store_init_sem_Q: creating shmem block %s", name));
	if ((store->semQShmemP = store_create_shmem (name, sizeof (STORE_SEM_Q)))
			== NULL)
		return (rtx_error ("store_init_sem_Q: store_create_shmem() failed"));
	store->semQ = (STORE_SEM_Q *) store->semQShmemP->d;
	store->semQ->QSize = STORE_SEM_Q_SIZE;
	store->semQ->nextEmptySlotInQ = 0;
	store->semQ->nextVarInQ = 0;
	store_debug (("store_init_sem_Q: initializing lock sem"));
	if (rtx_sem_init (&(store->semQ->lockSem), 1, 1) == NULL) {
		store_destroy_shmem (store->semQShmemP);
		return (rtx_error ("store_sem_Q_init: sem_init failed"));
	}
	store_debug (("store_init_sem_Q: initializing data sem"));
	if (rtx_sem_init (&(store->semQ->dataSem), 1, 0) == NULL) {
		store_destroy_shmem (store->semQShmemP);
		return (rtx_error ("store_sem_Q_init: sem_init failed"));
	}
	store_debug (("store_init_sem_Q: creating semQ handler thread"));
	if ((store->semQHandlerThread =
				rtx_thread_create ("store[semQHandler]",
					store->verbose,
					RTX_THREAD_SCHED_OTHER,
					store->semQHandlerThPrio,
					16,
					RTX_THREAD_CANCEL_DEFERRED,
					(void * (*)(void *)) 
					store_sem_Q_handler_thread,
					(void *) store,
					NULL,
					NULL)) == NULL) 
		{
			store_destroy_shmem (store->semQShmemP);
			return (rtx_error ("store_sem_Q_init: rtx_thread_create() failed"));
		}
	return (0);
}

int
store_init_header (
		STORE * store
		)
{
	RtxParseVar * v = NULL;
	int parseIndex;
	RtxParseToken t;
	int i;

	for (i=0; i<STORE_ARCH_MAX_NUM; i++) {
		parseIndex = 0;
		memset (&t, 0, sizeof (t));
		if ((v = calloc (1, sizeof (RtxParseVar))) == NULL)
			return (rtx_error_errno ("store_init_header: calloc() failed"));
		if (rtx_parse_decl (STORE_TIMESPEC_STR, &parseIndex, &t, v, store->symTabs[i]) <= 0) {
			rtx_parse_free_var (v);
			return (rtx_error ("store_init_header: parse_decl "
						"(STORE_TIMESPEC_STR) failed"));
		}
		if (rtx_parse_add_type_symbol (v, 0, store->symTabs[i]->typeTable,
					&(store->symTabs[i]->typeNumber)) == -1) {
			return (rtx_error ("store_init_header: add_type (%s) failed",
						v->name));
		}
		parseIndex = 0;
		memset (&t, 0, sizeof (t));
		if ((v = calloc (1, sizeof (RtxParseVar))) == NULL)
			return (rtx_error_errno ("store_init_header: calloc() failed"));
		if (rtx_parse_decl (STORE_VAR_HEADER_STR, &parseIndex, &t, v, store->symTabs[i]) <= 0) {
			return (rtx_error ("store_init_header: parse_decl "
						"(STORE_VAR_HEADER_STR) failed"));
		}
		if (rtx_parse_add_type_symbol (v, 0, store->symTabs[i]->typeTable,
					&(store->symTabs[i]->typeNumber)) == -1) {
			return (rtx_error ("store_init_header: add_type (%s) failed",
						v->name));
		}
		parseIndex = 0;
		memset (&t, 0, sizeof (t));
		if ((v = calloc (1, sizeof (RtxParseVar))) == NULL)
			return (rtx_error_errno ("store_init_header: calloc() failed"));
		if (rtx_parse_decl (STORE_VAR_HEADER_VAR, &parseIndex, &t, v, store->symTabs[i]) <= 0) {
			return (rtx_error ("store_init_header: parse_decl "
						"(STORE_VAR_HEADER_VAR) failed"));
		}
		if (rtx_parse_expand_var_definition (v, store->symTabs[i]->typeTable) == -1) {
			rtx_parse_free_var (v);
			return (rtx_error ("store_init_header: expand_var_definition "
						"(%s) failed", v->name));
		}
		store->headerVarP[i] = v;
	}
	store->myHeaderVarP = store->headerVarP[store->myArchType];
	return (0);
}

int
store_init_var_list (
		STORE * store
		)
{
	store->curVarId = 0;
	store->sizeVarList = SIZE_VAR_INFO_LIST;
	if ((store->varList = rtx_list_init ()) == NULL)
		return (rtx_error ("store_init_var_list: "
					"rtx_list_init() failed"));
	if ((store->vars = (VAR_INFO_ITEM **) calloc (SIZE_VAR_INFO_LIST,
					sizeof (VAR_INFO_ITEM *))) == NULL) {
		return (rtx_error_errno ("store_init_var_list: calloc () failed"));
	}
	return (0);
}

int
store_init_catalog_connection (
		STORE * store
		)
{
	int alignment[8];
	int dataSize[8];
	int byteOrderInt = 0x01020304;
	char * byteOrder;
	MSG * mp;

	byteOrder = (char *) &byteOrderInt;
	rtx_parse_get_compiler_parms (dataSize, alignment);

	store_debug (("store_init_catalog_connection: connecting to catalog "
				"addr = %s:%d", store->catalogHostname,
				store->catalogPortNum));
	if ((store->catalog.server = rtx_inet_init (RTX_INET_TCP_CLIENT, NULL, 0, 
					store->catalogHostname,
					store->catalogPortNum,
					NULL, NULL, NULL)) == NULL)
		return (rtx_error_errno ("store_init_catalog_connection: "
					"rtx_inet_init(%s:%d) failed",
					store->catalogHostname,
					store->catalogPortNum));
	store_debug (("store_init_catalog_connection: catalog addr = %s:%d",
				rtx_inet_print_net_addr 
				(&(store->catalog.server->sock->remote)),
				rtx_inet_print_net_portnum 
				(&(store->catalog.server->sock->remote))));
	sprintf (store->catalog.buf, "%s %s = %s %s = %d %s = %d %d %d %d %d %d "
			"%s = %d %d %d %d %d %d %s = %d %d %d %d\n",
			catKeyTable[MSG_TOK_CATALOG_REGISTER_STORE_REQUEST].name,
			catKeyTable[MSG_TOK_CATALOG_HOST_NAME].name,
			store->myHostName,
			catKeyTable[MSG_TOK_CATALOG_PORT_NUM].name,
			store->portNum,
			catKeyTable[MSG_TOK_CATALOG_ALIGNMENT].name,
			alignment[rtx_char_t], alignment[rtx_short_t], alignment[rtx_int_t],
			alignment[rtx_long_t], alignment[rtx_float_t], alignment[rtx_double_t],
			catKeyTable[MSG_TOK_CATALOG_DATA_SIZE].name,
			dataSize[rtx_char_t], dataSize[rtx_short_t], dataSize[rtx_int_t],
			dataSize[rtx_long_t], dataSize[rtx_float_t], dataSize[rtx_double_t],
			catKeyTable[MSG_TOK_CATALOG_BYTE_ORDER].name,
			byteOrder[0], byteOrder[1], byteOrder[2], byteOrder[3]);
	if ((mp = store_get_response (store->catalog.server->sock->sockfd, 
					store->catalog.buf1, 
					store->catalog.buf, 
					5, &(store->catalog), catKeyTable,
					catMsgTable, 
					"store_init_catalog_connection",
					store->verbose)) == NULL) {
		return (rtx_error ("store_init_catalog_connection: "
					"no response from catalog (5s)"));
	}
	if (mp->id != MSG_CATALOG_REGISTER_STORE_REPLY) {
		return (rtx_error ("store_init_catalog_connection: "
					"invalid/error reply from catalog"));
	}
	free_message (catKeyTable, mp);
	store_debug (("store_init_catalog_connection: done"));
	return (0);
}

int
store_init (
		STORE * store
		)
{
	char name[32];
	int i;

	store->done = 0;
	store->pid = getpid();
	if ((store->client = rtx_list_init ()) == NULL)
		return (rtx_error ("store_init: rtx_list_init() failed"));
	if ((store->catalogMsgQ = rtx_list_init ()) == NULL)
		return (rtx_error ("store_init: rtx_list_init() failed"));
	if (store_init_shmem_table () == -1) {
		return (rtx_error ("store_init: store_init_shmem_table"
					" () failed"));
	}
	if (rtx_mutex_init (&(store->mutex), RTX_MUTEX_DEFAULT, 0) == NULL)
		return (rtx_error ("store_init: rtx_mutex_init() failed"));
	if (store_init_sem_pool () == -1)
		return (rtx_error ("store_init: store_init_sem_pool "
					"() failed"));
	sprintf (name, "/%s-%d", STORE_SHMEM_NAME, store->pid);
	if ((store->shmemP = store_create_shmem (name, store->shmemSize))
			== NULL) {
		store_destroy_sem_pool ();
		return (rtx_error ("store_init: store_create_shmem "
					"%s, %d) failed", name, store->shmemSize));
	}
	if (store_init_mem (store->shmemSize) == -1) {
		store_destroy_sem_pool ();
		store_destroy_shmem (store->shmemP);
		return (rtx_error ("store_init: store_init_mem () failed"));
	}
	if ((store->server =  rtx_inet_init (RTX_INET_TCP_SERVER,
					NULL, store->portNum, NULL, 0,
					store_request_handler_thread, 
					NULL, store)) == NULL) {

		store_destroy_sem_pool ();
		store_destroy_shmem (store->shmemP);
		return (rtx_error ("store_init: store_init_request_handler"
					" () failed"));
	}
	for (i=0; i<STORE_ARCH_MAX_NUM; i++) {
		if ((ddxStore->symTabs[i] = rtx_parse_init_remote 
					(storeArchs[i].dataAlign, storeArchs[i].dataSize)) == NULL) {
			store_destroy_sem_pool ();
			store_destroy_shmem (store->shmemP);
			return (rtx_error ("store_init: init_type_table () failed"));
		}
	}
	if (store_init_header (store) == -1) {
		store_destroy_sem_pool ();
		store_destroy_shmem (store->shmemP);
		return (rtx_error ("store_init: store_init_header() failed"));
	}
	if (store->verbose > 1)
		rtx_parse_print_decl (store->myHeaderVarP, 0);
	if (store_init_var_list (store) == -1) {
		store_destroy_sem_pool ();
		store_destroy_shmem (store->shmemP);
		return (rtx_error ("store_init: store_init_var_list ()"
					" failed"));
	}
	if (store_init_catalog_connection (store) == -1) {
		store_destroy_sem_pool ();
		store_destroy_shmem (store->shmemP);
		return (rtx_error ("store_init: store_init_catalog_connection"
					"() failed "));
	}
	if (store_init_sem_Q (store) == -1) {
		store_destroy_sem_pool ();
		store_destroy_shmem (store->shmemP);
		return (rtx_error ("store_init: store_init_sem_Q () failed"));
	}
	if ((store->catalogHandlerThread =
				rtx_thread_create ("store[catalogHandler]",
					store->verbose,
					RTX_THREAD_SCHED_OTHER,
					RTX_THREAD_PRIO_MIN,
					0,
					RTX_THREAD_CANCEL_DEFERRED,
					(void * (*)(void *)) 
					store_catalog_handler_thread,
					(void *) store,
					NULL,
					NULL)) == NULL) {
		store_destroy_sem_pool ();
		store_destroy_shmem (store->shmemP);
		return (rtx_error ("store_init: rtx_thread_create () failed"));
	}
	return (0);
}

int
store_close (
		STORE * store
		)
{
	MSG * mp;
	int errs = 0;

	store_debug (("Destroying the catalog handler thread"));
	if (rtx_thread_destroy_sync (store->catalogHandlerThread)) {
		rtx_error ("store_close: rtx_thread_destroy() failed");
		errs++;
	}
	sprintf (store->catalog.buf, "%s %s = %s\n",
			catKeyTable[MSG_TOK_CATALOG_DEREGISTER_STORE_REQUEST].name,
			catKeyTable[MSG_TOK_CATALOG_HOST_NAME].name,
			store->myHostName);
	if ((mp = store_get_response (store->catalog.server->sock->sockfd, 
					store->catalog.buf1, 
					store->catalog.buf, 
					5, &(store->catalog), catKeyTable,
					catMsgTable, 
					"store_close",
					store->verbose)) == NULL) {
		rtx_error ("store_close: no response from catalog (5s)");
		errs++;
	}
	if (mp->id != MSG_CATALOG_DEREGISTER_STORE_REPLY) {
		rtx_error ("store_close: error/invalid response from catalog");
		errs++;
	}
	free_message (catKeyTable, mp);
	store_debug (("store_close: closing socket"));
	if (rtx_inet_done (store->server) == -1) {
		rtx_error ("store_close: rtx_inet_done(server) failed");
		errs++;
	}
	store_debug (("store_close: closing catalog socket"));
	if (rtx_inet_done (store->catalog.server) == -1) {
		rtx_error ("store_close: rtx_inet_done(catalog) failed");
		errs++;
	}
	store_debug (("Waiting for sem Q handler thread"));
	if (rtx_sem_post (&(store->semQ->dataSem)) == -1) {
		rtx_error ("store_close: rtx_sem_post() failed");
		errs++;
	}
	if (rtx_thread_join (store->semQHandlerThread) == -1) {
		rtx_error ("store_close: rtx_thread_destroy_sync() failed");
		errs++;
	}
	store_debug (("Destroying shmem"));
	if (store_destroy_shmem (store->shmemP) == -1) {
		rtx_error ("store_close: failed to destroy shared memory");
		errs++;
	}
	store_debug (("Destroying semQ semaphores"));
	if (rtx_sem_destroy (&(store->semQ->lockSem)) == -1) {
		rtx_error ("store_close: failed to destroy mutex semQ->lockSem");
		errs++;
	}
	if (rtx_sem_destroy (&(store->semQ->dataSem)) == -1) {
		rtx_error ("store_close: failed to destroy mutex semQ->dataSem");
		errs++;
	}
	store_debug (("Destroying semQ"));
	if (store_destroy_shmem (store->semQShmemP) == -1) {
		rtx_error ("store_close: failed to destroy sem Q");
		errs++;
	}
	store_debug (("Destroying sem pool"));
	if (store_destroy_sem_pool () == -1) {
		rtx_error ("store_close: failed to destroy sem pool");
		errs++;
	}
	store_debug (("Destroying shmem table"));
	if (store_destroy_shmem_table () == -1) {
		rtx_error ("store_close: failed to destroy shared memory table");
		errs++;
	}
	store_debug (("Destroying mutex"));
	if (rtx_mutex_destroy (&store->mutex) == -1) {
		rtx_error ("store_close: failed to destroy mutex");
		errs++;
	}
	store_debug (("store_close: done"));
	if (errs)
		return (-1);
	return (0);
}

int
store_add_typedefs (
		char * declBuf,
		STORE * store
		)
{
	RtxParseVar * v = NULL;
	int parseIndex;
	RtxParseToken t;
	int done = 0, ret;

	parseIndex = 0;
	do {
		memset (&t, 0, sizeof (t));
		if ((v = calloc (1, sizeof (RtxParseVar))) == NULL)
			return (rtx_error_errno ("store_add_typedefs: calloc() failed"));
		if ((ret = rtx_parse_decl (declBuf, &parseIndex, &t, v, store->mySymTabs)) == -1) {
			rtx_parse_free_var (v);
			return (rtx_error ("store_add_typedefs: parse_decl() failed"));
		}
		if (ret == 0) {
			done = 1;
			continue;
		}
		if (rtx_parse_add_type_symbol (v, 0, store->mySymTabs->typeTable,
					&(store->mySymTabs->typeNumber)) == -1) {
			return (rtx_error ("store_init_header: add_type (%s) failed",
						v->name));
		}
	} while (! done);
	return (0);
}

int
store_snarf_typedefs (
		char * filename,
		STORE * store
		)
{
	int fd, ret;
	struct stat finfo;
	char * declBuf = NULL;
	FILE * fp = NULL;
	static char cmd[BUFSIZ];
	static char tmpFile[BUFSIZ];

	store_debug (("store_snarf_typedefs: starting"));
	sprintf (tmpFile, "/tmp/store_%d_typedefs", (int) store->pid);
#ifdef sparc_solaris
	sprintf (cmd, "/usr/local/bin/cpp -P %s > %s", filename, tmpFile);
#else
	sprintf (cmd, "/usr/bin/cpp -P %s > %s", filename, tmpFile);
#endif
	if ((fp = popen (cmd, "r")) == NULL)
		return (rtx_error_errno ("store_snarf_typedefs: popen(%s) failed",
					cmd));
	if (pclose (fp))
		return (rtx_error_errno ("store_snarf_typedefs: "
					"pclose(%s) failed", cmd));
	store_debug (("store_snarf_typedefs: pre-processed outout = %s",
				tmpFile));
	if ((fd = open (tmpFile, O_RDONLY)) == -1)
		return (rtx_error_errno ("store_snarf_typedefs: open(%s) failed",
					tmpFile));
	if (fstat (fd, &finfo) == -1)
		return (rtx_error_errno ("store_snarf_typedefs: fstat(%s) failed",
					tmpFile));
	if ((declBuf = (char *) malloc (finfo.st_size + 1)) == NULL)
		return (rtx_error_errno ("store_snarf_typedefs: malloc(%d) failed",
					finfo.st_size + 1));
	if ((ret = read (fd, declBuf, finfo.st_size)) == -1)
		return (rtx_error_errno ("store_snarf_typedefs: read(%s) failed",
					tmpFile));
	close (fd);
	unlink (tmpFile);
	declBuf[finfo.st_size] = '\0';
	store_debug (("store_snarf_typedefs: buffer = %d bytes", 
				strlen (declBuf)));
	if (ret != finfo.st_size) {
		free (declBuf);
		return (rtx_error_errno ("store_snarf_typedefs: read(%s) needed"
					" %d bytes, got %d bytes", tmpFile,
					finfo.st_size, ret));
	}
	store_debug (("store_snarf_typedefs: adding typedefs"));
	if (store_add_typedefs (declBuf, store) == -1) {
		free (declBuf);
		return (rtx_error_errno ("store_snarf_typedefs: store_add_typedefs(%s) failed",
					tmpFile));
	}
	free (declBuf);
	store_debug (("store_snarf_typedefs: done"));
	return (0);
}

long unsigned int parseMemSize(const char * s)
{
	long int res = 0;
	char mul = 0;
	sscanf(s,"%ld %c",&res,&mul);
	if (res < 0) return 0;
	switch (mul) {
		case 'b' :
		case 'B' :
			res *= 1;
			break;
		case 'k' :
		case 'K' :
			res *= (1<<10);
			break;
		case 'm' :
		case 'M' :
			res *= (1<<20);
			break;
		case 'g' :
		case 'G' :
			fprintf(stderr,"If you really want to allocate "
					"gigas of shared memory,\n\t please modify this line (%s,%d)\n",
					__FILE__,__LINE__);
			return 0;
		default :
			break;
	}
	return res;
}

int
main (
		int argc,
		char * argv[]
	 )
{
	int webpid = 0;
	pid_t pid, childPid;
	int statloc = 0;
	int ret, errs = 0;
	STORE * store = ddxStore;
	unsigned int mySwapWord = 0x01020304;

	umask(0002);

	rtx_main_init ("store", RTX_ERROR_STDERR | RTX_ERROR_MESSAGE);
	rtx_message_init ("store", RTX_MESSAGE_STDERR | RTX_MESSAGE_MESSAGE);

	if (gethostname (ddxStore->myHostName, 64) == -1) {
		rtx_error_errno_flush ("gethostname() failed");
		exit (1);
	}

	ddxStore->catalogHostname = ddxStore->myHostName;
	ddxStore->incFileName =  "";
	ddxStore->catalogPortNum = CATALOG_PORT_NUM;
	ddxStore->portNum = STORE_PORT_NUM;
	ddxStore->semQHandlerThPrio = RTX_THREAD_PRIO_MIN;
	ddxStore->semQHandlerDebug = 0;
	ddxStore->shmemSize = STORE_SHMEM_SIZE;
	ddxStore->multicastTtl = 3;
	ddxStore->forkFlag = 0;
	ddxStore->launchCatalog = 0;
	memcpy (&(ddxStore->myArch.swapword), (unsigned char *) &mySwapWord, 
			sizeof (mySwapWord));
	rtx_parse_get_compiler_parms (ddxStore->myArch.dataSize, ddxStore->myArch.dataAlign);
	if ((ddxStore->myArchType = store_get_arch (&(ddxStore->myArch))) == -1) {
		rtx_error_errno_flush ("unable to determine architecture");
		exit (1);
	}

	if ((ret = RTX_GETOPT_CMD (storeOpts, argc, argv, rcsid, 
					storeHelpStr)) == -1) {
		RTX_GETOPT_PRINT (storeOpts, argv[0], rcsid, storeHelpStr);
		exit (1);
	}

	/* Make sure that both -catalog and -launchcatalog have not been 
	   used together */
	if ((ddxStore->launchCatalog != 0) && 
			(strcmp (ddxStore->catalogHostname, ddxStore->myHostName) != 0))
		return (rtx_error ("main: -launchcatalog and -catalog options are "
					"mutually exclusive"));
	/* Launch the catalog if needed */
	if (ddxStore->launchCatalog) {
		if ((childPid = fork ()) == -1)
			return (rtx_error_errno ("main: fork catalog failed"));
		if (childPid == 0) { /* child */
			if (execlp ("catalog", "catalog", "-daemon", (char *) 0) == -1) {
				rtx_error_errno ("main: execlp(catalog) failed, exiting");
				exit (1);
			}
			exit (1);
		} else { /* parent */
			/* wait for child to complete */
			if (wait (&statloc) == -1) {
				rtx_error_errno ("main: wait(catalog) failed, exiting");
				exit (1);
			}
			sleep (1); /* give the catalog some time to startup */
		}
	}

	/*      
	 * Fork the process
	 */     
	if (ddxStore->forkFlag) {
		pid = fork();
		if (pid < 0) {
			rtx_error_errno_flush ("main: fork() failed\n"); 
			exit(1);
		} else if (pid > 0) {
			exit(0);
		}
		setsid();
		chdir("/");
		umask(0002);
	}


	if (startweb) {
		signal(SIGCHLD,SIG_IGN);
		webpid = fork();
		if (webpid == 0) {
			/* child */
			char * args[2] = {webprogram,NULL};
			sigset_t sigset;
			sigfillset(&sigset);
			sleep(2);
			sigprocmask(SIG_UNBLOCK,&sigset,NULL);
			rtx_message("Starting web server '%s'\n",webprogram);
			execvp(webprogram,args);
			exit(0);
		}
	}

	ddxStore->shmemSize = parseMemSize(memsize_str);
	store->verbose = rtx_getopt_get_verbose (0);
	if (ddxStore->shmemSize < STORE_SHMEM_SIZE)
		ddxStore->shmemSize = STORE_SHMEM_SIZE;
	rtx_message("Allocating %d bytes\n",ddxStore->shmemSize);

	if (store_init (ddxStore) == -1) {
		rtx_error_flush ("main: store_init() failed");
		exit (1);
	}
	ddxStore->mySymTabs = ddxStore->symTabs[ddxStore->myArchType];
	if (ddxStore->verbose)
		rtx_list_print (ddxStore->varList);

	if (strcmp (ddxStore->incFileName, "") != 0) {
		store_debug (("main: reading typedefs from %s", 
					ddxStore->incFileName));
		if (store_snarf_typedefs (ddxStore->incFileName, ddxStore) == -1)
			rtx_error_flush ("main: store_snarf_typedefs(%s) failed",
					ddxStore->incFileName);
	}

	store_debug (("store: pid = %d", ddxStore->pid));

	if (ddxStore->verbose)
		rtx_list_print (ddxStore->varList);
	if (rtx_main_wait_shutdown (0) == -1) {
		rtx_error ("main: rtx_init_wait_shutdown() failed");
		errs++;
	}

	ddxStore->done = 1;
	store_debug (("store: exiting ..."));
	if (webpid > 0) {
		rtx_message("Terminating web server\n");
		kill(webpid,SIGINT);
	}
	if (store_close (ddxStore) == -1) {
		rtx_error ("main: store_close() failed");
		errs++;
	}
	if (errs) {
		rtx_error_flush ("main: error during shutdown");
		return (-1);
	}
	return (0);
}
