
/***********************************************************************
 * store-ipc.c - functions for IPC (semaphores and shared memory)
 *
 * CSIRO MANUFACTURING SCIENCE & TECHNOLOGY
 * HIGH-SPEED SERIAL INTERFACE PROJECT
 * 
 * $Id: store-ipc.c 3075 2008-05-15 05:17:08Z roy029 $
 * 
 * Copyright (c) CSIRO Manufacturing Science & Technology.
 *
 ***********************************************************************
 */
#ifdef __GNUC__
#define DDX_STATIC_ATTRIBUTE __attribute__((unused))
#else
#define DDX_STATIC_ATTRIBUTE 
#endif


static char *rcsid DDX_STATIC_ATTRIBUTE = "$Id: store-ipc.c 3075 2008-05-15 05:17:08Z roy029 $";

#ifndef USE_SYSV_API

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <semaphore.h>
#include <pthread.h>
#include <rtx/message.h>
#include <rtx/error.h>
#include <rtx/timer.h>
#include <rtx/thread.h>
#include <rtx/getopt.h> /* for verbose */
#include "store-ipc.h"

/***************************************************************************
 * Shared memory stuff
 ***************************************************************************
 */

/***************************************************************************
 * Typedefs
 */

/* Entry structure for the global shared memory table */

typedef struct _store_shmem_table {
	struct _store_shmem_table        * next;
	STORE_SHMEM_T                    * shmemEntryP;
	int                              created;
} STORE_SHMEM_TABLE_T;

/***************************************************************************
 * global variables
 */

/* Global shared memory table - entries are dynamically created */

static STORE_SHMEM_TABLE_T * storeShmemTable = NULL;
pthread_mutex_t store_shmem_table_mutex;

/***************************************************************************
 * functions
 */


/***************************************************************************
 * store_init_shmem_table - initilaize the global shared memory table
 */
	int
store_init_shmem_table (void)
{
	pthread_mutexattr_t attr;

#if defined (i486_lynxos) && defined (__OSREV_3_0_1)
	if (pthread_mutexattr_create (&attr) == -1) {
		return (-1);
	}
	if (pthread_mutex_init (&store_shmem_table_mutex, attr) == -1) {
		pthread_mutexattr_delete (&attr);
		return (-1);
	}
	if (pthread_mutexattr_delete (&attr) == -1) {
	}
#else
	if (pthread_mutexattr_init (&attr) == -1) {
		return (-1);
	}
	if (pthread_mutex_init (&store_shmem_table_mutex, &attr) == -1) {
		pthread_mutexattr_destroy (&attr);
		return (-1);
	}
	if (pthread_mutexattr_destroy (&attr) == -1) {
	}
#endif
	return (0);
}

/***************************************************************************
 * store_destroy_shmem_table - destroy the global shared memory table
 */
	int 
store_destroy_shmem_table (void)
{
	if (pthread_mutex_destroy (&store_shmem_table_mutex) == -1) {
	}
	return (0);
}

/***************************************************************************
 * store_shmem_table_add_entry - adds an entry to the global shared memory
 *                               table
 */
static int
store_shmem_table_add_entry (
		STORE_SHMEM_T * shmemp,
		int created
		)
{
	STORE_SHMEM_TABLE_T * shmemTabP;
	int res, errs = 0;

	while ((res = pthread_mutex_lock (&store_shmem_table_mutex)) == -1) {
		if (errno == EINTR)
			continue;
		else {
			return (-1);
		}
	}

	if ((shmemTabP = (STORE_SHMEM_TABLE_T *) calloc (1,
					sizeof (STORE_SHMEM_TABLE_T))) == NULL) {
		errs++;
	} else {
		shmemTabP->shmemEntryP = shmemp;
		shmemTabP->created = created;
		shmemTabP->next = storeShmemTable;
		storeShmemTable = shmemTabP;
	}

	while ((res = pthread_mutex_unlock (&store_shmem_table_mutex)) == -1) {
		if (errno == EINTR)
			continue;
		else {
			return (-1);
		}
	}

	if (errs)
		return (-1);
	return (0);
}

/***************************************************************************
 * store_shmem_table_get_entry - finds an entry in the global shared memory
 *                               table; return -1 if not found, 0 on success
 */
static STORE_SHMEM_T *
store_shmem_table_get_entry (
		char * name,
		int size
		)
{
	STORE_SHMEM_TABLE_T * tp = storeShmemTable;
	STORE_SHMEM_T * shmemp = NULL;
	int res;

	while ((res = pthread_mutex_lock (&store_shmem_table_mutex)) == -1) {
		if (errno == EINTR)
			continue;
		else {
			return (NULL);
		}
	}

	while (tp != NULL) {
		if (strcmp (tp->shmemEntryP->name, name) != 0) {
			tp = tp->next;
			continue;
		}
		if (tp->shmemEntryP->size != size) {
			tp = tp->next;
			continue;
		} else {
			shmemp = tp->shmemEntryP;
		}
		tp = tp->next;
	}

	while ((res = pthread_mutex_unlock (&store_shmem_table_mutex)) == -1) {
		if (errno == EINTR)
			continue;
		else {
			return (NULL);
		}
	}

	return (shmemp);
}

/***************************************************************************
 * store_shmem_table_del_entry - delete an entry in the global shared memory
 *                               table; return -1 if not found, 0 on success
 */
static int
store_shmem_table_del_entry (
		STORE_SHMEM_T * shmemp
		)
{
	STORE_SHMEM_TABLE_T * tp = storeShmemTable, * tp1 = NULL;
	int res = -1;

	while ((res = pthread_mutex_lock (&store_shmem_table_mutex)) == -1) {
		if (errno == EINTR)
			continue;
		else {
			return (-1);
		}
	}

	while (tp != NULL) {
		if (strcmp (tp->shmemEntryP->name, shmemp->name) != 0) {
			tp1 = tp;
			tp = tp->next;
			continue;
		}
		if (tp->shmemEntryP->size != shmemp->size) {
			tp1 = tp;
			tp = tp->next;
			continue;
		} else {
			if (tp1 == NULL) { /* element is at head of list */
				storeShmemTable = tp->next;
			} else { /* tp1 points to the previous element */
				tp1->next = tp->next;
			}
			free (tp);
			break;
		}
		tp1 = tp;
		tp = tp->next;
	}

	while ((res = pthread_mutex_unlock (&store_shmem_table_mutex)) == -1) {
		if (errno == EINTR)
			continue;
		else {
			return (-1);
		}
	}

	return (res);
}

/***************************************************************************
 * store_create_shmem - create a shared memory block
 */
STORE_SHMEM_T *
store_create_shmem (
		char * name,
		int size
		)
{
	STORE_SHMEM_T * shmemp;
	//printf("NPTL Shmem : creating %s\n",name);

	if ((shmemp = (STORE_SHMEM_T *) calloc (1, sizeof (STORE_SHMEM_T)))
			== NULL) {
		return (NULL);
	}

	if ((shmemp->id = shm_open (name, O_RDWR | O_CREAT | O_EXCL, 0664)) 
			== -1) {
		free (shmemp);
		return (NULL);
	}

	if (ftruncate (shmemp->id, size) == -1) {
		close (shmemp->id);
		free (shmemp);
		return (NULL);
	}

	if ((shmemp->d = (unsigned char *) mmap (NULL, size, PROT_READ |
					PROT_WRITE, MAP_SHARED, shmemp->id, 0)) == MAP_FAILED) {
		close (shmemp->id);
		free (shmemp);
		return (NULL);
	}
	if ((shmemp->name = strdup (name)) == NULL) {
		close (shmemp->id);
		munmap (shmemp->d, size);
		free (shmemp);
		return (NULL);
	}
	shmemp->size = size;
	shmemp->pid = getpid ();
	memset (shmemp->d, 0, shmemp->size);
	if (store_shmem_table_add_entry (shmemp, 1) == -1) {
		close (shmemp->id);
		munmap (shmemp->d, size);
		free (shmemp->name);
		free (shmemp);
		return (NULL);
	}

	return (shmemp);
}

/***************************************************************************
 * store_destroy_shmem - destroy a shared memory block
 */
int
store_destroy_shmem (
		STORE_SHMEM_T * shmemp
		)
{
	//printf("NPTL Shmem : destroying %s\n",shmemp->name);
	if (close (shmemp->id) == -1) {
		return (-1);
	}

	if (munmap (shmemp->d, shmemp->size)) {
		return (-1);
	}

	if (shm_unlink (shmemp->name) == -1) {
		return (-1);
	}

	store_shmem_table_del_entry (shmemp);

	free (shmemp->name);
	free (shmemp);
	return (0);
}

/***************************************************************************
 * store_link_shmem - link to an existing shared memory block
 */
STORE_SHMEM_T *
store_link_shmem (
		char * name,
		int id,
		int size
		)
{
	STORE_SHMEM_T * shmemp;
	//printf("NPTL Shmem : linking to %s\n",name);

	/* If the shmem block has already been mapped, use it */
	if ((shmemp = store_shmem_table_get_entry (name, size))
			!= NULL)
		return (shmemp);

	/* Shmem block has not been mapped, so create an entry, map it in etc */
	if ((shmemp = (STORE_SHMEM_T *) calloc (1, sizeof (STORE_SHMEM_T)))
			== NULL) {
		return (NULL);
	}

	if ((shmemp->id = shm_open (name, O_RDWR, 0664)) 
			== -1) {
		free (shmemp);
		return (NULL);
	}

	if ((shmemp->d = (unsigned char *) mmap (NULL, size, PROT_READ |
					PROT_WRITE, MAP_SHARED, shmemp->id, 0)) == MAP_FAILED) {
		close (shmemp->id);
		free (shmemp);
		return (NULL);
	}
	if ((shmemp->name = strdup (name)) == NULL) {
		close (shmemp->id);
		munmap (shmemp->d, size);
		free (shmemp);
		return (NULL);
	}
	shmemp->size = size;
	shmemp->pid = getpid ();
	if (store_shmem_table_add_entry (shmemp, 0) == -1) {
		close (shmemp->id);
		munmap (shmemp->d, size);
		free (shmemp->name);
		free (shmemp);
		return (NULL);
	}
	return (shmemp);
}

/***************************************************************************
 * store_unlink_shmem - unlink an existing shared memory block
 */
int
store_unlink_shmem (
		STORE_SHMEM_T * shmemp
		)
{
	int errs = 0;

	if (close (shmemp->id) == -1) {
		rtx_error_errno ("store_unlink_shmem: close(%s) failed",
				shmemp->name);
		errs++;
	}

	if (munmap (shmemp->d, shmemp->size)) {
		rtx_error_errno ("store_unlink_shmem: munmap(%s) failed",
				shmemp->name);
		errs++;
	}

	store_shmem_table_del_entry (shmemp);

	free (shmemp->name);
	free (shmemp);
	if (errs)
		return (-1);
	return (0);
}

/***************************************************************************
 * Semaphore stuff
 ***************************************************************************
 */

/***************************************************************************
 * global variables
 */

static STORE_SHMEM_T * semPoolShmemPtr;
static unsigned char store_sem_pool[STORE_SEM_POOL_SIZE];
static int store_sem_pool_taken = 0;
static int store_sem_pool_next = 0;
pthread_mutex_t store_sem_pool_mutex;

/***************************************************************************
 * store_init_sem_pool - initialize the global semaphore pool
 */
	int 
store_init_sem_pool (void)
{
	char name[64];
	pthread_mutexattr_t attr;

#if defined (i486_lynxos) && defined (__OSREV_3_0_1)
	if (pthread_mutexattr_create (&attr) == -1) {
		return (-1);
	}
	if (pthread_mutex_init (&store_sem_pool_mutex, attr) == -1) {
		pthread_mutexattr_delete (&attr);
		return (-1);
	}
	if (pthread_mutexattr_delete (&attr) == -1) {
		pthread_mutex_destroy (&store_sem_pool_mutex);
	}
#else
	if (pthread_mutexattr_init (&attr) == -1) {
		return (-1);
	}
	if (pthread_mutex_init (&store_sem_pool_mutex, &attr) == -1) {
		pthread_mutexattr_destroy (&attr);
		return (-1);
	}
	if (pthread_mutexattr_destroy (&attr) == -1) {
		pthread_mutex_destroy (&store_sem_pool_mutex);
	}
#endif
	sprintf (name, "/%s-%d", STORE_SEM_POOL_NAME, (int) getpid ());
	if ((semPoolShmemPtr = store_create_shmem (name, STORE_SEM_POOL_SIZE *
					sizeof (sem_t))) == NULL) {
		pthread_mutex_destroy (&store_sem_pool_mutex);
		return (-1);
	}

	return (0);
}

/***************************************************************************
 * store_destroy_sem_pool - destroy the global semaphore pool
 */
	int 
store_destroy_sem_pool (void)
{
	if (pthread_mutex_destroy (&store_sem_pool_mutex) == -1) {
	}
	if (store_destroy_shmem (semPoolShmemPtr) == -1) {
		return (-1);
	}
	return (0);
}

/***************************************************************************
 * store_get_sem - allocate a semaphore from the global pool
 */
	static int 
store_get_sem (void)
{
	int res, ctr = 0, retVal;

	while ((res = pthread_mutex_lock (&store_sem_pool_mutex)) == -1) {
		if (errno == EINTR)
			continue;
		else {
			perror("store_get_sem: lock");
			return (-1);
		}
	}
	if (store_sem_pool_taken >= STORE_SEM_POOL_SIZE) {
		printf("store_get_sem: store_sem_pool_taken >= STORE_SEM_POOL_SIZE\n");
		return (-1);
	}

	store_sem_pool_taken++;
	if (rtx_getopt_get_verbose(0)) {
		printf("%d over %d semaphores in use\n",
				store_sem_pool_taken, STORE_SEM_POOL_SIZE);
	}
	retVal = store_sem_pool_next++;
	store_sem_pool[retVal] = 1;
	if (store_sem_pool_taken < STORE_SEM_POOL_SIZE) {
		while (ctr < STORE_SEM_POOL_SIZE) {
			if (store_sem_pool_next >= STORE_SEM_POOL_SIZE)
				store_sem_pool_next = 0;
			if (store_sem_pool[store_sem_pool_next]) {
				store_sem_pool_next++;
			} else {
				break;
			}
			ctr++;
		}
		if (ctr >= STORE_SEM_POOL_SIZE) {
			printf("store_get_sem: ctr >= STORE_SEM_POOL_SIZE\n");
			return (-1);
		}
	}
	while ((res = pthread_mutex_unlock (&store_sem_pool_mutex)) == -1) {
		if (errno == EINTR)
			continue;
		else {
			perror("store_get_sem: lock");
			return (-1);
		}
	}
	return (retVal);
}

/***************************************************************************
 * store_free_sem - free a semaphore in the global pool
 */
static int 
store_free_sem (
		int semid
		)
{
	int res = -1;

	if ((semid < 0) || (semid >= STORE_SEM_POOL_SIZE)) {
		return (-1);
	}
	while ((res = pthread_mutex_lock (&store_sem_pool_mutex)) == -1) {
		if (errno == EINTR)
			continue;
		else {
			return (-1);
		}
	}
	if (store_sem_pool_taken >= STORE_SEM_POOL_SIZE) {
		store_sem_pool_next = semid;
	}
	store_sem_pool_taken--;
	if (rtx_getopt_get_verbose(0)) {
		printf("%d over %d semaphores in use\n",
				store_sem_pool_taken, STORE_SEM_POOL_SIZE);
	}
	store_sem_pool[semid] = 0;
	while ((res = pthread_mutex_unlock (&store_sem_pool_mutex)) == -1) {
		if (errno == EINTR)
			continue;
		else {
			return (-1);
		}
	}
	return (0);

}

/***************************************************************************
 * store_create_sem - create a semaphore in the global pool
 */
	STORE_SEM_T * 
store_create_sem (void)
{
	STORE_SEM_T * semp;
	sem_t * semptr;

	if ((semp = (STORE_SEM_T *) calloc (1, sizeof (STORE_SEM_T))) == NULL) {
		perror("store_create_sem: malloc");
		return NULL;
	}

	if ((semp->id = store_get_sem ()) == -1) {
		perror("store_create_sem: get_sem");
		free (semp);
		return NULL;
	}
	semptr = (sem_t *) semPoolShmemPtr->d;
	semp->semp = (void *) &(semptr[semp->id]);
	if (sem_init ((sem_t *) semp->semp, 1, 0) == -1) {
		perror("store_create_sem: init_sem");
		free (semp);
		return NULL;
	}
	semp->pid = getpid ();
	semp->conn = NULL;
	return (semp);
}

/***************************************************************************
 * store_destroy_sem - destroy a semaphore in the global pool
 */
int 
store_destroy_sem (
		STORE_SEM_T * semp
		)
{
	if (sem_destroy (semp->semp) == -1) {
	}
	if (store_free_sem (semp->id) == -1) {
		return (-1);
	}

	free (semp);
	return (0);
}

/***************************************************************************
 * store_link_sem - link to an existing semaphore
 */
int 
store_link_sem (
		STORE_SEM_T * semp
		)
{
	STORE_SHMEM_T * shmemp;
	sem_t * semptr;
	char name[32];

	sprintf (name, "/%s-%d", STORE_SEM_POOL_NAME, (int) semp->pid);
	if ((shmemp = store_link_shmem (name, 0, STORE_SEM_POOL_SIZE *
					sizeof (sem_t))) == NULL) {
		return (-1);
	}

	semptr = (sem_t *) shmemp->d;
	semp->semp = (void *) &(semptr[semp->id]);
	return (0);
}

/***************************************************************************
 * store_unlink_sem - unlink from an existing semaphore
 */
int 
store_unlink_sem (
		STORE_SEM_T * semp
		)
{
	return (0);
}

/***************************************************************************
 * store_post_sem - post a semaphore
 */
int 
store_post_binary_sem (
		STORE_SEM_T * semp
		)
{
	int val;

	if (sem_getvalue (semp->semp, &val)) {
		rtx_message_warning ("store_post_sem: sem_getvalue: %s", strerror (errno));
		return (sem_post (semp->semp));
	} else {
		if (val > 0)
			return (0);
		return (sem_post (semp->semp));
	}
	return (0);
}

/***************************************************************************
 * store_post_sem - post a semaphore
 */
int 
store_post_sem (
		STORE_SEM_T * semp
		)
{
	return (sem_post (semp->semp));
}

/***************************************************************************
 * store_wait_sem - wait on a semaphore
 */
int 
store_wait_sem (
		STORE_SEM_T * semp
		)
{
	int result;

	while ((result = sem_wait (semp->semp)) == -1) {
		if (errno == EINTR)
			continue;
		else
			break;
	}
	return (result);
}

/***************************************************************************
 * store_wait_sem - wait on a semaphore
 */
typedef struct {
	STORE_SEM_T *sem;
	pthread_mutex_t mutex;
	pthread_cond_t cond;
	pthread_t thread;
} TimedWaitStruct;


void waiter_cleanup(void * arg) {
	TimedWaitStruct * tws = (TimedWaitStruct*)arg;
	/* this cleanup can only be called in three situations:
	 * - the condition has been signaled 
	 * - the condition has timed out
	 * in both case waiter thread should be terminated
	 * - a cancellation request has been received
	 * */
	pthread_cancel(tws->thread);
	pthread_join(tws->thread,NULL);
	/* the mutex is garanteed to be lock for the cleanup,
	 * according to doc */
	pthread_mutex_unlock(&tws->mutex);
	pthread_mutex_destroy(&tws->mutex);
	pthread_cond_destroy(&tws->cond);
	free(tws);
}


void * waiter_thread(void * arg) {
	int ret = 0;
	TimedWaitStruct * tws = (TimedWaitStruct*)arg;
	/* we want to be sure that a cancellation request 
	 * can be received, but only at the right place */
	pthread_setcancelstate(PTHREAD_CANCEL_ENABLE,NULL);
	pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED,NULL);

	pthread_mutex_lock(&tws->mutex); 
	pthread_mutex_unlock(&tws->mutex);
	/* cancellation point */
	ret = store_wait_sem(tws->sem);
	if (ret==0) pthread_cond_signal(&tws->cond);
	/* I used pthread_exit here to be sure that no cancellation can be received 
	 * in the middle of exiting. Paranoid? I know... */
	pthread_exit(NULL);

	/* not reachable */
	return NULL;
}




int 
store_timed_wait_sem (
		STORE_SEM_T * semp,
		RtxTime * timeout
		)
{
	int oldtype = 0, oldstate = 0;
	int ret = 0;
	RtxTime rts;
	struct timespec ts;
	TimedWaitStruct *tws;

	/* compute the maximum absolute time for waiting the condition */
	rtx_time_get(&rts);
	rtx_time_add(&rts,&rts,timeout);
	ts.tv_sec = rts.seconds;
	ts.tv_nsec = rts.nanoSeconds;

	/* we want to be sure that a cancellation request 
	 * can be received, but only at the right place */
	pthread_setcancelstate(PTHREAD_CANCEL_ENABLE,&oldstate);
	pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED,&oldtype);
	
	tws = (TimedWaitStruct*)malloc(sizeof(TimedWaitStruct));

	tws->sem = semp;
	pthread_mutex_init(&tws->mutex,NULL);
	pthread_cond_init(&tws->cond,NULL);

	/* Lock the mutex and create the waiting thread. The thread 
	 * will be waiting on the mutex to be unlocked */
	pthread_mutex_lock(&tws->mutex);
	ret = pthread_create(&tws->thread,NULL,waiter_thread,tws);
	if (ret != 0) return rtx_error_errno("store_timed_wait_sem: failed to create waiter thread");
	

	/* Prepare a cleanup function in case we get a cancellation
	 * request while waiting */
	pthread_cleanup_push(waiter_cleanup,tws);
	ret = pthread_cond_timedwait(&tws->cond,&tws->mutex,&ts);
	/* Remove the cleanup function and execute it. If we reach this point, then
	 * it cannot have been executed yet */
	pthread_cleanup_pop(1);

	/* Pu the thread back in its old state */
	pthread_setcancelstate(oldstate,NULL);
	pthread_setcanceltype(oldtype,NULL);

	switch (ret) {
		case 0: 
			return 0;
		case ETIMEDOUT : 
			return +1;
		default: 
			break;
	}

	return rtx_error_errno("store_timed_wait_sem: pthread_cond_timedwait failed");
}

/**************************************************
 * store_cleanup_ipc - cleanup required ipc data
 * */
int 
store_cleanup_ipc(void) {

	return store_destroy_shmem_table();
}



#else // USE_SYSV_API

#define _SVID_SOURCE

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <string.h> /* required by Linux */
#include <errno.h>
#include <pthread.h>
#include <fcntl.h>


#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/shm.h>
#include "store-ipc.h"

#include "rtx/error.h"

/***************************************************************************
 * Shared memory stuff
 ***************************************************************************
 */

/***************************************************************************
 * Typedefs
 */


/* Entry structure for the global shared memory table */

typedef struct _store_shmem_table {
	struct _store_shmem_table        * next;
	STORE_SHMEM_T                    * shmemEntryP;
	int                              created;
} STORE_SHMEM_TABLE_T;

/***************************************************************************
 * global variables
 */

/* Global shared memory table - entries are dynamically created */

static STORE_SHMEM_TABLE_T * storeShmemTable = NULL;
pthread_mutex_t store_shmem_table_mutex;


/***************************************************************************
 * functions
 */

/***************************************************************************
 * store_init_shmem_table - initilaize the global shared memory table
 */
	int
store_init_shmem_table (void)
{
	if (pthread_mutex_init (&store_shmem_table_mutex, NULL) == -1) {
		return (-1);
	}
	return (0);
}

/***************************************************************************
 * store_destroy_shmem_table - destroy the global shared memory table
 */
	int 
store_destroy_shmem_table (void)
{
	if (pthread_mutex_destroy (&store_shmem_table_mutex) == -1) {
	}
	return (0);
}

/***************************************************************************
 * store_shmem_table_add_entry - adds an entry to the global shared memory
 *                               table
 */
static int
store_shmem_table_add_entry (
		STORE_SHMEM_T * shmemp,
		int created
		)
{
	STORE_SHMEM_TABLE_T * shmemTabP;
	int res, errs = 0;

	while ((res = pthread_mutex_lock (&store_shmem_table_mutex)) == -1) {
		if (errno == EINTR)
			continue;
		else {
			return (-1);
		}
	}

	if ((shmemTabP = (STORE_SHMEM_TABLE_T *) calloc (1,
					sizeof (STORE_SHMEM_TABLE_T))) == NULL) {
		errs++;
	} else {
		shmemTabP->shmemEntryP = shmemp;
		shmemTabP->created = created;
		shmemTabP->next = storeShmemTable;
		storeShmemTable = shmemTabP;
	}

	while ((res = pthread_mutex_unlock (&store_shmem_table_mutex)) == -1) {
		if (errno == EINTR)
			continue;
		else {
			rtx_error_errno_null ("store_shmem_table_add_entry store__sem: unable to unlock mutex\n");
			return -1;
		}
	}

	if (errs)
		return (-1);
	return (0);
}

/***************************************************************************
 * store_shmem_table_get_entry - finds an entry in the global shared memory
 *                               table; return -1 if not found, 0 on success
 */
static STORE_SHMEM_T *
store_shmem_table_get_entry (
		char * name,
		int size
		)
{
	STORE_SHMEM_TABLE_T * tp = storeShmemTable;
	STORE_SHMEM_T * shmemp = NULL;
	int res;

	while ((res = pthread_mutex_lock (&store_shmem_table_mutex)) == -1) {
		if (errno == EINTR)
			continue;
		else {
			return (NULL);
		}
	}

	while (tp != NULL) {
		if (strcmp (tp->shmemEntryP->name, name) != 0) {
			tp = tp->next;
			continue;
		}
		if (tp->shmemEntryP->size != size) {
			tp = tp->next;
			continue;
		} else {
			shmemp = tp->shmemEntryP;
		}
		tp = tp->next;
	}

	while ((res = pthread_mutex_unlock (&store_shmem_table_mutex)) == -1) {
		if (errno == EINTR)
			continue;
		else {
			return rtx_error_errno_null("store_shmem_table_get_entry: store__sem: unable to unlock mutex\n");
		}
	}

	return (shmemp);
}

/***************************************************************************
 * store_shmem_table_del_entry - delete an entry in the global shared memory
 *                               table; return -1 if not found, 0 on success
 */
static int
store_shmem_table_del_entry (
		STORE_SHMEM_T * shmemp
		)
{
	STORE_SHMEM_TABLE_T * tp = storeShmemTable, * tp1 = NULL;
	int res = -1;

	while ((res = pthread_mutex_lock (&store_shmem_table_mutex)) == -1) {
		if (errno == EINTR)
			continue;
		else {
			return (-1);
		}
	}

	while (tp != NULL) {
		if (strcmp (tp->shmemEntryP->name, shmemp->name) != 0) {
			tp1 = tp;
			tp = tp->next;
			continue;
		}
		if (tp->shmemEntryP->size != shmemp->size) {
			tp1 = tp;
			tp = tp->next;
			continue;
		} else {
			if (tp1 == NULL) { /* element is at head of list */
				storeShmemTable = tp->next;
			} else { /* tp1 points to the previous element */
				tp1->next = tp->next;
			}
			free (tp);
			break;
		}
		tp1 = tp;
		tp = tp->next;
	}

	while ((res = pthread_mutex_unlock (&store_shmem_table_mutex)) == -1) {
		if (errno == EINTR)
			continue;
		else {
			fprintf (stderr, "store__sem: unable to unlock mutex\n");
			return (-1);
		}
	}

	return (res);
}

/***************************************************************************
 * store_create_shmem - create a shared memory block
 */
STORE_SHMEM_T *
store_create_shmem (
		char * name,
		int size
		)
{
	STORE_SHMEM_T * shmemp;
	key_t key;
	unsigned int len = strlen(name);
	//printf("SYSV Shmem : creating %s\n",name);
	{
		int fp,pid;
		char tmp[len+10];
		sscanf(name,"/%[^-]-%d",tmp,&pid);
		sprintf(tmp,"/tmp/%s",name);
		fp = creat(tmp,0644);
		if (fp < 0) {
			return rtx_error_errno_null("store_create_shmem: creat");
		}
		close(fp);
		key = ftok(tmp,pid);
		//printf("Key : %d\n",(int)key);
	}

	if ((shmemp = (STORE_SHMEM_T *) malloc (sizeof (STORE_SHMEM_T))) == NULL) {
		return rtx_error_errno_null ("store_create_shmem: malloc");
	}

	if ((shmemp->id = shmget (key, size, IPC_CREAT | IPC_EXCL | 0666)) == -1) {
		free (shmemp);
		return rtx_error_errno_null("store_create_shmem: shmget");
	}

	if ((int) (shmemp->d = (unsigned char *) shmat (shmemp->id, NULL, 0)) == -1) {
		free (shmemp);
		return rtx_error_errno_null("store_create_shmem: shmat");
	}

	shmemp->size = size;
	shmemp->name = strdup (name);
	shmemp->pid = getpid ();
	store_shmem_table_add_entry (shmemp, 1);

	return (shmemp);
}

/***************************************************************************
 * store_destroy_shmem - destroy a shared memory block
 */
int
store_destroy_shmem (
		STORE_SHMEM_T * shmemp
		)
{
	unsigned int len = strlen(shmemp->name);
	struct shmid_ds shmds;
	//printf("SYSV Shmem : destroying %s\n",shmemp->name);
	{
		char tmp[len+10];
		sprintf(tmp,"/tmp/%s",shmemp->name);
		unlink(tmp);
	}

	if (shmdt (shmemp->d) == -1) {
		rtx_error_errno_null("store_destroy_shmem: shmdt:");
		return -1;
	}

	if (shmctl (shmemp->id, IPC_RMID, &shmds) == -1) {
		rtx_error_errno_null("store_destroy_shmem: shmctl:");
		return -1;
	}

	store_shmem_table_del_entry (shmemp);

	free (shmemp->name);
	free (shmemp);

	
	return (0);
}

/***************************************************************************
 * store_link_shmem - link to an existing shared memory block
 */
STORE_SHMEM_T *
store_link_shmem (
		char * name,
		int id,
		int size
		)
{
	STORE_SHMEM_T * shmemp;
	struct shmid_ds shmds;
	key_t key;
	unsigned int len = strlen(name);
	//printf("SYSV Shmem : linking to %s\n",name);
	{
		int pid;
		char tmp[len+10];
		sscanf(name,"/%[^-]-%d",tmp,&pid);
		sprintf(tmp,"/tmp/%s",name);
		key = ftok(tmp,pid);
	}

	/* If the shmem block has already been mapped, use it */
	if ((shmemp = store_shmem_table_get_entry (name, size))
			!= NULL)
		return (shmemp);

	/* Shmem block has not been mapped, so create an entry, map it in etc */
	if ((shmemp = (STORE_SHMEM_T *) malloc (sizeof (STORE_SHMEM_T))) == NULL) {
		return rtx_error_null("store_link_shm: shmemp malloc:");
		return (NULL);
	}

	if ((shmemp->id = shmget (key, 0, 0)) == -1) {
		perror ("store_link_shmem: shmget");
		free (shmemp);
		return (NULL);
	}

	if (shmctl (shmemp->id, IPC_STAT, &shmds) == -1) {
		perror("shmctl STAT");
		free (shmemp);
		return (NULL);
	}

	if ((int) (shmemp->d = (unsigned char *) shmat (shmemp->id, 0, 0)) == -1) {
		perror("shmat");
		free (shmemp);
		return (NULL);
	}

	shmemp->size = shmds.shm_segsz;
	shmemp->name = strdup (name);
	shmemp->pid = getpid ();
	store_shmem_table_add_entry (shmemp, 0);
	return (shmemp);
}

/***************************************************************************
 * store_unlink_shmem - unlink an existing shared memory block
 */
int
store_unlink_shmem (
		STORE_SHMEM_T * shmemp
		)
{
	if (shmdt (shmemp->d) == -1) {
		return (-1);
	}

	store_shmem_table_del_entry (shmemp);

	free (shmemp->name);
	free (shmemp);
	return (0);
}

/***************************************************************************
 * Semaphore stuff
 ***************************************************************************
 */

/***************************************************************************
 * Typedefs
 */

#if (defined(__GNU_LIBRARY__) && !defined(_SEM_SEMUN_UNDEFINED)) || defined ppc_darwin
/* l'union semun est dfinie par l'inclusion de <sys/sem.h> */
#else
/* d'aprs X/OPEN il faut la dfinir nous-mmes */
union semun {
	int val;                  /* valeur pour SETVAL */
	struct semid_ds *buf;     /* buffer pour IPC_STAT, IPC_SET */
	unsigned short *array;    /* table  pour GETALL, SETALL */
	/* Spcificit Linux : */
	struct seminfo *__buf;    /* buffer pour IPC_INFO */
};
#endif


/***************************************************************************
 * global variables
 */
static int semPoolsemid;
static unsigned char store_sem_pool[STORE_SEM_POOL_SIZE];
static int store_sem_pool_taken = 0;
static int store_sem_pool_next = 0;
pthread_mutex_t store_sem_pool_mutex;


/* sem pool would not be required for SysV IPC since all objects are system-wide.
 * Nevertheless, it is required in order to make a clean exit. We cannot rely
 * on clients to properly clean there semaphores on exit.
 */

/***************************************************************************
 * store_init_sem_pool - initialize the global semaphore pool
 */
	int 
store_init_sem_pool (void)
{
	pthread_mutexattr_t attr;

#if defined (i486_lynxos) && defined (__OSREV_3_0_1)
	if (pthread_mutexattr_create (&attr) == -1) {
		return (-1);
	}
	if (pthread_mutex_init (&store_sem_pool_mutex, attr) == -1) {
		pthread_mutexattr_delete (&attr);
		return (-1);
	}
	if (pthread_mutexattr_delete (&attr) == -1) {
		pthread_mutex_destroy (&store_sem_pool_mutex);
	}
#else
	if (pthread_mutexattr_init (&attr) == -1) {
		return (-1);
	}
	if (pthread_mutex_init (&store_sem_pool_mutex, &attr) == -1) {
		pthread_mutexattr_destroy (&attr);
		return (-1);
	}
	if (pthread_mutexattr_destroy (&attr) == -1) {
		pthread_mutex_destroy (&store_sem_pool_mutex);
	}
#endif
	if ((semPoolsemid = semget (IPC_PRIVATE, STORE_SEM_POOL_SIZE, 0666)) == -1) {
		rtx_error_errno_null("store_init_sem_pool: semget");
		return -1;
	}
	//printf("Sem Pool Created %d\n",semPoolsemid);
	
	return (0);
}

/***************************************************************************
 * store_destroy_sem_pool - destroy the global semaphore pool
 */
	int 
store_destroy_sem_pool (void)
{
	/* Required for a compatible API */
	union semun semst;
	if (pthread_mutex_destroy (&store_sem_pool_mutex) == -1) {
		rtx_error_errno_null("store_destroy_sem_pool: mutex_destroy:");
	}
	if (semctl (semPoolsemid, 0, IPC_RMID, semst) == -1) {
		rtx_error_errno_null("store_destroy_sem_pool: semctl:");
		return -1;
	}
	//printf("Sem Pool Destroyed %d\n",semPoolsemid);
	return (0);
}

/***************************************************************************
 * store_get_sem - allocate a semaphore from the global pool
 */
	static int 
store_get_sem (void)
{
	int res, ctr = 0, retVal;

	while ((res = pthread_mutex_lock (&store_sem_pool_mutex)) == -1) {
		if (errno == EINTR)
			continue;
		else {
			return (-1);
		}
	}
	if (store_sem_pool_taken >= STORE_SEM_POOL_SIZE)
		return (-1);

	store_sem_pool_taken++;
	retVal = store_sem_pool_next++;
	store_sem_pool[retVal] = 1;
	if (store_sem_pool_taken < STORE_SEM_POOL_SIZE) {
		while (ctr < STORE_SEM_POOL_SIZE) {
			if (store_sem_pool_next >= STORE_SEM_POOL_SIZE)
				store_sem_pool_next = 0;
			if (store_sem_pool[store_sem_pool_next]) {
				store_sem_pool_next++;
			} else {
				break;
			}
			ctr++;
		}
		if (ctr >= STORE_SEM_POOL_SIZE)
			return (-1);
	}
	while ((res = pthread_mutex_unlock (&store_sem_pool_mutex)) == -1) {
		if (errno == EINTR)
			continue;
		else {
			return (-1);
		}
	}
	return (retVal);
}

/***************************************************************************
 * store_free_sem - free a semaphore in the global pool
 */
static int 
store_free_sem (
		int semid
		)
{
	int res = -1;

	if ((semid < 0) || (semid >= STORE_SEM_POOL_SIZE)) {
		return (-1);
	}
	while ((res = pthread_mutex_lock (&store_sem_pool_mutex)) == -1) {
		if (errno == EINTR)
			continue;
		else {
			return (-1);
		}
	}
	if (store_sem_pool_taken >= STORE_SEM_POOL_SIZE) {
		store_sem_pool_next = semid;
	}
	store_sem_pool_taken--;
	store_sem_pool[semid] = 0;
	while ((res = pthread_mutex_unlock (&store_sem_pool_mutex)) == -1) {
		if (errno == EINTR)
			continue;
		else {
			return (-1);
		}
	}
	return (0);

}

/***************************************************************************
 * store_create_sem - create a semaphore in the global pool
 */
	STORE_SEM_T * 
store_create_sem (void)
{
	union semun semst;
	STORE_SEM_T * semp;

	if ((semp = (STORE_SEM_T *) malloc (sizeof (STORE_SEM_T))) == NULL) {
		return rtx_error_errno_null("store_create_sem: malloc");
	}
	semp->semid = semPoolsemid;
	if ((semp->id = store_get_sem ()) == -1) {
		free (semp);
		return NULL;
	}
	//printf("Store Created %d-%d\n",semp->semid,semp->id);

	semst.val = 0;
	if (semctl (semp->semid, semp->id, SETVAL, semst) == -1) {
		free (semp);
		return rtx_error_errno_null("store_create_sem: semctl");
	}

	semp->pid = getpid ();
	semp->sync_pid = 0;
	return (semp);
}

/***************************************************************************
 * store_destroy_sem - destroy a semaphore in the global pool
 */
int 
store_destroy_sem (
		STORE_SEM_T * semp
		)
{
	if (store_free_sem (semp->id) == -1) {
		return (-1);
	}
	//printf("Store Destroyed %d-%d\n",semp->semid,semp->id);

	free (semp);
	return (0);
}

/***************************************************************************
 * store_link_sem - link to an existing semaphore
 */
int 
store_link_sem (
		STORE_SEM_T * semp
		)
{
	/* Required for compatibility */
	return (0);
}

/***************************************************************************
 * store_unlink_sem - unlink from an existing semaphore
 */
int 
store_unlink_sem (
		STORE_SEM_T * semp
		)
{
	/* Required for compatibility */
	return (0);
}

/***************************************************************************
 * store_post_sem - post a semaphore
 */
int 
store_post_sem (
		STORE_SEM_T * semp
		)
{
	struct sembuf op[1];
	int result = -1;

	op[0].sem_num = semp->id;
	op[0].sem_op = 1;
	op[0].sem_flg = 0;

	while ((result = semop (semp->semid, op, 1)) == -1) {
		if (result == -1) {
			if (errno == EINTR)
				continue;
			else
				break;
		}
	}
	return (result);
}

int store_post_binary_sem(STORE_SEM_T * semp)
{
	union semun su;
	struct sembuf op[1];
	int result = -1;

	if ((result = semctl (semp->semid, semp->id, GETVAL, su)) == -1) {
		perror ("store_post_sem: semctl");
	} else {
		if (result > 0)
			return (0);
	}

	op[0].sem_num = semp->id;
	op[0].sem_op = 1;
	op[0].sem_flg = 0;

	while ((result = semop (semp->semid, op, 1)) == -1) {
		if (result == -1) {
			if (errno == EINTR)
				continue;
			else
				break;
		}
	}
	return (result);
}

/***************************************************************************
 * store_wait_sem - wait on a semaphore
 */
int 
store_wait_sem (
		STORE_SEM_T * semp
		)
{
	struct sembuf op[1];
	int result;

	op[0].sem_num = semp->id;
	op[0].sem_op = -1;
	op[0].sem_flg = 0;

	//printf("Waiting %d-%d\n",semp->semid,semp->id);
	while ((result = semop (semp->semid, op, 1)) == -1) {
		if (errno == EINTR) {
			continue;
		} else {
			break;
		}
	}
	if (result != 0) {
		rtx_error_errno("sysv_timed_wait :");
	}
	return (result);
}


/***************************************************************************
 * store_wait_sem - wait on a semaphore
 */
int 
store_timed_wait_sem (
		STORE_SEM_T * semp, RtxTime * timeout
		)
{
	struct sembuf op[1];
	int result;

	op[0].sem_num = semp->id;
	op[0].sem_op = -1;
	op[0].sem_flg = 0;

	//printf("Waiting %d-%d\n",semp->semid,semp->id);
	while ((result = semop (semp->semid, op, 1)) == -1) {
		if (errno == EINTR) {
			continue;
		} else {
			break;
		}
	}
	if (result != 0) {
		rtx_error_errno("sysv_timed_wait :");
	}
	return (result);
}

/**************************************************
 * store_cleanup_ipc - cleanup required ipc data
 * */
int 
store_cleanup_ipc(void) {
	return store_destroy_shmem_table();
}



#endif
