
/***********************************************************************
 * store-handlers.c - ddx store message handler functions
 *
 * CSIRO MANUFACTURING SCIENCE & TECHNOLOGY
 * HIGH-SPEED SERIAL INTERFACE PROJECT
 * 
 * $Id: store-handlers.c 2302 2008-01-07 06:02:30Z roy029 $
 * 
 * Copyright (c) CSIRO Manufacturing Science & Technology.
 *
 ***********************************************************************
 */

#ifdef __GNUC__
#define DDX_STATIC_ATTRIBUTE __attribute__((unused))
#else
#define DDX_STATIC_ATTRIBUTE
#endif
static char *rcsid DDX_STATIC_ATTRIBUTE = "$Id: store-handlers.c 2302 2008-01-07 06:02:30Z roy029 $";

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <pthread.h>

#include <rtx/message.h>
#include <rtx/time.h>
#include <rtx/timer.h>
#include <rtx/sem.h>
#include <rtx/thread.h>
#include <rtx/error.h>
#include <rtx/parse.h>
#include <rtx/hash.h>
#include <rtx/inet.h>

#include "store-ipc.h"
#include "store-msg.h"
#include "store-messages.h"
#include "catalog-messages.h"
#include "store-mem.h"
#include "store-main.h"
#include "store-handlers.h"
#include "store-common.h"

	MSG *
	store_get_catalog_response (
			char * recvBuf,
			char * sendBuf,
			int timeout,
			STORE * store
			)
{
	int numChars;
	MSG * mp;
	RtxTime t0;

	if (store->verbose)
		rtx_message_routine ("store_get_catalog_response (timeout %d) [%s]", 
				timeout, sendBuf);
	if ((numChars = rtx_inet_write (store->catalog.server->sock, sendBuf, 
					strlen (sendBuf), NULL)) == -1) {
		return (rtx_error_errno_null ("store_get_catalog_response: "
					"rtx_inet_write: %s", sendBuf));
	}

	rtx_time_get (&t0);
	while ((store->catalogMsgQ->n == 0) && 
			(rtx_time_get_delta (&t0) < timeout))
		usleep (100000);
	if (store->catalogMsgQ->n == 0)
		return (rtx_error_null ("store_get_catalog_response: timedout"));
	if ((mp = (MSG *) rtx_list_dequeue (store->catalogMsgQ)) == NULL)
		return (rtx_error_null ("store_get_catalog_response: "
					"rtx_list_dequeue"));
	if (store->verbose)
		rtx_message_routine ("store_get_catalog_response: "
				"msg id [%d]", mp->id);
	return (mp);
}

void store_clean_sync_semaphores(STORE * store, RtxInetConn * inetConn)
{
	if (store_semQ_add (store->semQ, STORE_SEM_Q_ACTION_CLEAN,
				0, inetConn) == -1) {
		rtx_error_flush ("store_clean_sync_semaphores): "
				"store_semQ_add() failed");
	}
}

STORE_SEM_T *
store_add_sync_sem (
		STORE * store,
		int varId,
		RtxInetConn * conn
		)
{
	char * varName = NULL;
	STORE_SEM_T * semp;
	VAR_INFO_ITEM * vInfo;

	vInfo = store->vars[varId];
	varName = vInfo->vEntry->v->name;
	if ((semp = store_create_sem ()) == NULL) {
		return (rtx_error_null ("store_add_sync_sem (%s): store_create_sem()"
					" failed", varName));
	}
	semp->conn = conn;
	if (store_semQ_add (store->semQ, STORE_SEM_Q_ACTION_ADD,
				varId, (void *) semp) == -1) {
		rtx_error_flush ("store_add_sync_sem (%s): "
				"store_semQ_add() failed", varName);
		return (NULL);
	}
	return (semp);
}

int
store_del_sync_sem (
		STORE * store,
		int varId,
		int syncSemId
		)
{
	char * varName = NULL;
	VAR_INFO_ITEM * vInfo;

	vInfo = store->vars[varId];
	varName = vInfo->vEntry->v->name;
	if (store_semQ_add (store->semQ, STORE_SEM_Q_ACTION_DEL,
				varId, (void *) syncSemId) == -1) {
		rtx_error_flush ("store_del_sync_sem (%s): "
				"store_semQ_add() failed", varName);
		return (-1);
	}
	return (0);
}

int
store_get_new_id (
		STORE * store
		)
{
	VAR_INFO_ITEM ** v;
	int ret = -1;

	if (store->curVarId >= store->sizeVarList) {
		if ((v = (VAR_INFO_ITEM **) calloc (store->sizeVarList + 
						SIZE_VAR_INFO_LIST, sizeof (VAR_INFO_ITEM *))) == NULL) {
			return (rtx_error_errno ("store_get_new_id: calloc() failed"));
		}
		memcpy ((char *) v, (char *) store->vars, 
				sizeof (VAR_INFO_ITEM *) * store->sizeVarList);
		free (store->vars);
		store->vars = v;
		store->sizeVarList += SIZE_VAR_INFO_LIST;
	}
	ret = store->curVarId++;
	return (ret);
}

int
store_add_var_info_item (
		RtxParseVarTableEntry * v,
		int varId,
		int offset,
		int headerOffset,
		char * multiAddr,
		int multiPort,
		STORE_SEM_T * lockSemP,
		STORE * store
		)
{
	VAR_INFO_ITEM * vInfo;

	if ((vInfo = (VAR_INFO_ITEM *) calloc (1, sizeof (VAR_INFO_ITEM))) == NULL)
		return (rtx_error ("store_add_var_info_item (%s): calloc() "
					"failed", v->name));
	if ((vInfo->syncSemList = rtx_list_init ()) == NULL)
		return (rtx_error ("store_add_var_info_item (%s): "
					"rtx_list_init() failed", v->name));
	vInfo->vEntry = v;
	vInfo->varId = varId;
	vInfo->offset = offset;
	vInfo->headerOffset = headerOffset;
	vInfo->lockSemP = lockSemP;
	vInfo->multiPort = multiPort;
	vInfo->multiAddr = multiAddr;
	vInfo->store = store;
	store->vars[varId] = vInfo;
	if (store->verbose)
		rtx_list_print (store->varList);
	if (rtx_list_add (store->varList, v->name, vInfo) == -1)
		return (rtx_error ("store_add_var_info_item: rtx_list_add()"
					" failed"));
	return (0);
}

int
store_del_var_info_item (
		int varId,
		STORE * store
		)
{
	VAR_INFO_ITEM * vInfo;

	vInfo = store->vars[varId];
	store->vars[varId] = NULL;
	if (rtx_list_del (store->varList, vInfo->vEntry->name, 1) == NULL)
		return (rtx_error ("store_del_var_info_item (%s): not found",
					vInfo->vEntry->name));
	return (0);
}

int
store_send_error_response (
		char * msg,
		int keyId,
		char * txbuf,
		RtxInetConn * fromAddr,
		STORE * store
		)
{
	int numChars;

	sprintf (txbuf, "%s %s = %s %s = $$storeToken_%s\n",
			keyTable[MSG_TOK_STORE_ERROR_REPLY].name,
			keyTable[MSG_TOK_RESPONSE].name, msg,
			keyTable[MSG_TOK_MESSAGE_ID].name, keyTable[keyId].name);
	if ((numChars = rtx_inet_write (fromAddr, txbuf, 
					strlen (txbuf), NULL)) == -1) {
		return (rtx_error ("store_send_error_response: "
					"rtx_inet_write() failed"));
	}
	return (0);
}

int
store_send_ok_response (
		int keyId,
		char * txbuf,
		RtxInetConn * fromAddr,
		STORE * store
		)
{
	int numChars;

	sprintf (txbuf, "%s %s = %s\n",
			keyTable[keyId].name,
			keyTable[MSG_TOK_RESPONSE].name, "OK");
	if ((numChars = rtx_inet_write (fromAddr, txbuf, 
					strlen (txbuf), NULL)) == -1) {
		return (rtx_error ("store_send_ok_response: "
					"rtx_inet_write() failed"));
	}
	return (0);
}

int
store_handle_open_request (
		MSG * mp, char * rxBuf, char * txBuf,
		RtxInetConn * fromAddr,
		STORE * store
		)
{
	int numChars;

	sprintf (txBuf, "%s %s = %d %s = %s %s = %d %s = %s %s = %d\n",
			keyTable[MSG_TOK_STORE_OPEN_REPLY].name,
			keyTable[MSG_TOK_STORE_PID].name, store->pid,
			keyTable[MSG_TOK_SHMEM_NAME].name, STORE_SHMEM_NAME,
			keyTable[MSG_TOK_SHMEM_SIZE].name, store->shmemSize,
			keyTable[MSG_TOK_SEM_POOL_NAME].name, STORE_SEM_POOL_NAME,
			keyTable[MSG_TOK_SEM_POOL_SIZE].name, STORE_SEM_POOL_SIZE);
	if ((numChars = rtx_inet_write (fromAddr, txBuf, 
					strlen (txBuf), NULL)) == -1) {
		return (rtx_error ("store_handle_open_request: "
					"rtx_inet_write() failed"));
	}
	return (0);
}

int
store_handle_close_request (
		MSG * mp, char * rxBuf, char * txBuf,
		RtxInetConn * fromAddr,
		STORE * store
		)
{
	int numChars;

	sprintf (txBuf, "%s %s = %s\n",
			keyTable[MSG_TOK_STORE_CLOSE_REPLY].name,
			keyTable[MSG_TOK_RESPONSE].name, "OK");
	if ((numChars = rtx_inet_write (fromAddr, txBuf, 
					strlen (txBuf), NULL)) == -1) {
		return (rtx_error ("store_handle_close_request: "
					"rtx_inet_write() failed"));
	}
	return (0);
}

int
store_handle_register_const_request (
		MSG * mp, char * rxBuf, char * txBuf,
		RtxInetConn * fromAddr,
		STORE * store
		)
{
	int constVal, numChars;
	char * constValP, * constNameP;
	RtxParseConstTableEntry * stEntry;
	int i, errs = 0;

	if ((constValP = get_message_part (keyTable, mp, MSG_TOK_VALUE)) == NULL) {
		store_send_error_response ("unable to extract value",
				MSG_TOK_STORE_REGISTER_CONST_REPLY,
				txBuf, fromAddr,store);
		return (rtx_error ("store_handle_register_const_request: unable to get "
					"const value"));
	}
	if ((constNameP = get_message_part (keyTable, mp, MSG_TOK_NAME)) == NULL) {
		store_send_error_response ("unable to extract name",
				MSG_TOK_STORE_REGISTER_CONST_REPLY,
				txBuf, fromAddr,store);
		return (rtx_error ("store_handle_register_const_request: unable to get "
					"const name"));
	}
	constVal = atoi (constValP);
	if (rtx_mutex_lock (&store->mutex) == -1)
		rtx_error_flush ("store_handle_register_const_request (%s): "
				"rtx_mutex_lock() failed", constNameP);
	if ((stEntry = (RtxParseConstTableEntry *) rtx_hash_find
	     (store->mySymTabs->constTable, constNameP, (rtx_error_t)NULL)) != NULL) {
		if (stEntry->value != constVal) {
			if (rtx_mutex_unlock (&store->mutex) == -1)
				rtx_error_flush ("store_handle_register_const_request (%s): "
						"rtx_mutex_unlock() failed", constNameP);
			store_send_error_response ("name already exists",
					MSG_TOK_STORE_REGISTER_CONST_REPLY,
					txBuf, fromAddr,store);
			return (rtx_error ("store_handle_register_const_request: const"
						" exists, desired val = %d, existing val = %d", 
						constVal, stEntry->value));
		} else {
			if (rtx_mutex_unlock (&store->mutex) == -1)
				rtx_error_flush ("store_handle_register_const_request (%s): "
						"rtx_mutex_unlock() failed", constNameP);
			sprintf (txBuf, "%s %s = %s\n",
					keyTable[MSG_TOK_STORE_REGISTER_CONST_REPLY].name,
					keyTable[MSG_TOK_RESPONSE].name, "OK");
			if ((numChars = rtx_inet_write (fromAddr, txBuf, 
							strlen (txBuf), NULL)) == -1) {
				return (rtx_error ("store_handle_register_const_request: "
							"rtx_inet_write() failed"));
			}
			return (0);
		}
	}
	for (i=0; i<STORE_ARCH_MAX_NUM; i++) {
		if (rtx_parse_add_const_symbol (constNameP, constVal, store->symTabs[i]->constTable)
				== -1) {
			store_send_error_response ("unable to add to symbol table",
					MSG_TOK_STORE_REGISTER_CONST_REPLY,
					txBuf, fromAddr,store);
			rtx_error ("store_handle_register_const_request: "
					"add_const_symbol failed");
			errs++;
			break;
		}
	}
	if (rtx_mutex_unlock (&store->mutex) == -1)
		rtx_error_flush ("store_handle_register_const_request (%s): "
				"rtx_mutex_unlock() failed", constNameP);
	if (errs)
		return (-1);
	sprintf (txBuf, "%s %s = %s\n",
			keyTable[MSG_TOK_STORE_REGISTER_CONST_REPLY].name,
			keyTable[MSG_TOK_RESPONSE].name, "OK");
	if ((numChars = rtx_inet_write (fromAddr, txBuf, 
					strlen (txBuf), NULL)) == -1) {
		return (rtx_error ("store_handle_register_const_request: "
					"rtx_inet_write() failed"));
	}
	if (store->verbose > 1)
		rtx_parse_print_const_table (store->mySymTabs->constTable);
	return (0);
}

int
store_handle_register_type_request (
		MSG * mp, char * rxBuf, char * txBuf,
		RtxInetConn * fromAddr,
		STORE * store
		)
{
	char * typeDecl;
	RtxParseToken curToken;
	int typeStrIndex = 0, numChars;
	RtxParseVar * v;
	RtxParseTypeTableEntry * stEntry;
	int i, errs = 0;

	if ((v = calloc (1, sizeof (RtxParseVar))) == NULL) {
		store_send_error_response ("unable to allocate memory",
				MSG_TOK_STORE_REGISTER_TYPE_REPLY,
				txBuf, fromAddr,store);
		return (rtx_error_errno ("store_handle_register_type_request: "
					"calloc() failed"));
	}
	if ((typeDecl = get_message_part (keyTable, mp, MSG_TOK_TYPE_DEFINITION))
			== NULL) {
		store_send_error_response ("unable to extract type definition",
				MSG_TOK_STORE_REGISTER_TYPE_REPLY,
				txBuf, fromAddr,store);
		return (rtx_error ("store_handle_register_type_request: unable to get "
					"type definition"));
	}
	if (store->verbose > 1)
		rtx_parse_print_type_table (store->mySymTabs->typeTable);
	if (rtx_parse_decl (typeDecl, &typeStrIndex, &curToken, v, store->mySymTabs) <= 0) {
		store_send_error_response ("unable to parse type definition",
				MSG_TOK_STORE_REGISTER_TYPE_REPLY,
				txBuf, fromAddr,store);
		return (rtx_error ("store_handle_register_type_request: unable to "
					"parse type definition"));
	}
	if (store->verbose > 1)
		rtx_parse_print_decl (v, 0);
	if (rtx_mutex_lock (&store->mutex) == -1)
		rtx_error_flush ("store_handle_register_type_request (%s): "
				"rtx_mutex_lock() failed", typeDecl);
	if ((stEntry = (RtxParseTypeTableEntry *) rtx_hash_find 
	     (store->mySymTabs->typeTable, v->name, (rtx_error_t)NULL)) != NULL) {
		if (rtx_parse_cmp_types (v, stEntry->t)) {
			if (rtx_mutex_unlock (&store->mutex) == -1)
				rtx_error_flush ("store_handle_register_type_request (%s): "
						"rtx_mutex_unlock() failed", typeDecl);
			store_send_error_response ("name already exists",
					MSG_TOK_STORE_REGISTER_TYPE_REPLY,
					txBuf, fromAddr,store);
			rtx_parse_free_var (v);
			return (rtx_error ("store_handle_register_type_request: type "
						"exists, requested type definition differs "
						"from that in the store"));
		} else {
			if (rtx_mutex_unlock (&store->mutex) == -1)
				rtx_error_flush ("store_handle_register_type_request (%s): "
						"rtx_mutex_unlock() failed", typeDecl);
			if (store->verbose > 1)
				rtx_parse_print_type_table (store->mySymTabs->typeTable);
			store_send_ok_response (MSG_TOK_STORE_REGISTER_TYPE_REPLY,
					txBuf, fromAddr,store);
			rtx_parse_free_var (v);
			return (0);
		}
	}
	rtx_parse_free_var (v);
	for (i=0; i<STORE_ARCH_MAX_NUM; i++) {
		if ((v = calloc (1, sizeof (RtxParseVar))) == NULL) {
			store_send_error_response ("unable to allocate memory",
					MSG_TOK_STORE_REGISTER_TYPE_REPLY,
					txBuf, fromAddr,store);
			rtx_error_errno ("store_handle_register_type_request: "
					"calloc() failed");
			errs++;
			break;
		}
		typeStrIndex = 0;
		memset (&curToken, 0, sizeof (curToken));
		if (rtx_parse_decl (typeDecl, &typeStrIndex, &curToken, v,
					store->symTabs[i]) <= 0) {
			store_send_error_response ("unable to parse type definition",
					MSG_TOK_STORE_REGISTER_TYPE_REPLY,
					txBuf, fromAddr,store);
			rtx_error ("store_handle_register_type_request: unable to "
					"parse type definition");
			errs++;
			break;
		}
		if (rtx_parse_add_type_symbol (v, 0, store->symTabs[i]->typeTable, 
					&store->symTabs[i]->typeNumber) == -1) {
			store_send_error_response ("unable to add type",
					MSG_TOK_STORE_REGISTER_TYPE_REPLY,
					txBuf, fromAddr,store);
			rtx_error ("store_handle_register_type_request: add_type_symbol"
					"failed");
			errs++;
			break;
		}
	}
	if (rtx_mutex_unlock (&store->mutex) == -1)
		rtx_error_flush ("store_handle_register_type_request (%s): "
				"rtx_mutex_unlock() failed", typeDecl);
	if (errs)
		return (-1);
	if (store->verbose > 1)
		rtx_parse_print_type_table (store->mySymTabs->typeTable);
	sprintf (txBuf, "%s %s = %s\n",
			keyTable[MSG_TOK_STORE_REGISTER_TYPE_REPLY].name,
			keyTable[MSG_TOK_RESPONSE].name, "OK");
	if ((numChars = rtx_inet_write (fromAddr, txBuf, 
					strlen (txBuf), NULL)) == -1) {
		return (rtx_error ("store_handle_register_type_request: "
					"rtx_inet_write() failed"));
	}
	return (0);
}

/***********************************************************************/

int
store_lookup_var_in_catalog (
		RtxParseVar * v,
		char ** varDecl,
		char ** multiAddr,
		int * multiPortNum,
		int varDeclProvided,
		char * rxBuf,
		char * txBuf,
		STORE * store
		)
{
	MSG * mp = NULL;
	char * msgParm;
	char * decl;
	char * addr;
	int multiPort;
	int multicastReqd = 0;
	int errs = 0;

	store->buf[0] = '\0';
	if (varDeclProvided) {
		if (rtx_parse_generate_var_decl (v, store->buf) == -1) {
			rtx_parse_free_var (v);
			rtx_error_flush ("store_lookup_var_in_catalog (%s): "
					"generate_var_decl() failed", v->name);
			errs++;
		}
		if (! errs)
			sprintf (txBuf, "%s %s = %s %s = %s %s = %s\n",
					catKeyTable[MSG_TOK_CATALOG_LOOKUP_VAR_REQUEST].name,
					catKeyTable[MSG_TOK_CATALOG_VAR_NAME].name, v->name,
					catKeyTable[MSG_TOK_CATALOG_VAR_DECL].name, store->buf,
					catKeyTable[MSG_TOK_CATALOG_HOST_NAME].name, store->myHostName);
	} else {
		sprintf (txBuf, "%s %s = %s %s = %s\n",
				catKeyTable[MSG_TOK_CATALOG_LOOKUP_VAR_REQUEST].name,
				catKeyTable[MSG_TOK_CATALOG_VAR_NAME].name, v->name,
				catKeyTable[MSG_TOK_CATALOG_HOST_NAME].name, store->myHostName);
	}
	if (! errs)
		if ((mp = store_get_catalog_response (rxBuf, txBuf, 5, 
						store)) == NULL) {
			rtx_error_flush ("store_lookup_var_in_catalog (%s): no response "
					"(5s) from catalog", v->name);
			errs++;
		}
	if (errs)
		return (-1);
	if (mp->id != MSG_CATALOG_LOOKUP_VAR_REPLY) {
		free_message (catKeyTable, mp);
		return (rtx_error ("store_lookup_var_in_catalog (%s): invalid/error response "
					"from catalog", v->name));
	}
	if ((msgParm = get_message_part 
				(catKeyTable, mp, MSG_TOK_CATALOG_MULTICAST_PORT)) == NULL) {
		free_message (catKeyTable, mp);
		return (rtx_error ("store_lookup_var_in_catalog (%s): unable to get multicast "
					"port from catalog", v->name));
	}
	multiPort = atoi (msgParm);
	store_debug (("store_lookup_var_in_catalog (%s): multicast port = %d",
				v->name, multiPort));
	if (multiPort < 0) {
		free_message (catKeyTable, mp);
		return (rtx_error ("store_lookup_var_in_catalog (%s): invalid multicast "
					"port (%d) from catalog", v->name, multiPort));
	}
	if ((addr = get_message_part 
				(catKeyTable, mp, MSG_TOK_CATALOG_MULTICAST_ADDR)) == NULL) {
		free_message (catKeyTable, mp);
		return (rtx_error ("store_lookup_var_in_catalog (%s): unable to get multicast "
					"address from catalog", v->name));
	}
	if (multiAddr != NULL)
		if ((* multiAddr = strdup (addr)) == NULL) {
			free_message (catKeyTable, mp);
			return (rtx_error ("store_lookup_var_in_catalog (%s): strdup() failed",
						v->name));
		}
	if ((decl = get_message_part 
				(catKeyTable, mp, MSG_TOK_CATALOG_VAR_DECL)) == NULL) {
		free_message (catKeyTable, mp);
		return (rtx_error ("store_lookup_var_in_catalog (%s): unable to get var "
					"decl from catalog", v->name));
	}
	if (varDecl != NULL)
		if ((* varDecl = strdup (decl)) == NULL) {
			free_message (catKeyTable, mp);
			return (rtx_error ("store_lookup_var_in_catalog (%s): strdup() failed",
						v->name));
		}
	if (multiPortNum != NULL)
		* multiPortNum = multiPort;
	if ((msgParm = get_message_part (catKeyTable, mp, MSG_TOK_CATALOG_MULTICAST_ON))
			!= NULL) {
		multicastReqd = 1;
	}
	free_message (catKeyTable, mp);
	return (multicastReqd);
}

RtxParseVarTableEntry *
store_create_var (
		RtxParseVar * v,
		char * typeDefn,
		char * rxBuf,
		char * txBuf,
		STORE * store
		)
{
	RtxParseVarTableEntry * stEntry;
	int offset;
	int headerOffset;
	int varId;
	int multiPort;
	char * multiAddr;
	STORE_SEM_T * semp;
	STORE_VAR_HEADER * svhp;
	VAR_INFO_ITEM * vInfo;
	char * varDecl;
	int multicastReqd = 0;
	RtxParseToken curToken;
	int varStrIndex = 0;
	RtxParseVar * v1;
	int i, errs = 0;

	store_debug (("store_create_var (%s): checking with catalog\nType is '%s'",
				v->name,typeDefn));
	if (typeDefn != NULL) {
		if ((multicastReqd = store_lookup_var_in_catalog (v, NULL, &multiAddr, &multiPort, 1, rxBuf, txBuf, store)) == -1) {
			return (rtx_error_null ("store_create_var (%s): "
						"store_lookup_var_in_catalog() failed", v->name));
		}
		store_debug(("store_create_var (%s): created decl '%s' -> '%s'\n",v->name,typeDefn,store->buf));
	} else {
		if ((multicastReqd = store_lookup_var_in_catalog (v, &varDecl, &multiAddr, &multiPort, 0, rxBuf, txBuf, store)) == -1) {
			return (rtx_error_null ("store_create_var (%s): "
						"store_lookup_var_in_catalog() failed", v->name));
		}
		memset (v, 0, sizeof (RtxParseVar));
		memset (&curToken, 0, sizeof (RtxParseToken));
		if (rtx_parse_decl (varDecl, &varStrIndex, &curToken, v, store->mySymTabs) <= 0) {
			return (rtx_error_null ("store_create_var (%s): parse "
						"failed", varDecl));
		}
		if (rtx_parse_expand_var_definition (v, store->mySymTabs->typeTable) == -1) {
			rtx_parse_free_var (v);
			return (rtx_error_null ("store_create_var (%s): expand_var "
						"failed", v->name));
		}
		if (rtx_parse_generate_var_decl (v, store->buf) == -1) {
			rtx_parse_free_var (v);
			rtx_error_null ("store_create_var (%s): "
					"generate_var_decl() failed", v->name);
			errs++;
		}
		store_debug(("store_create_var (%s): created decl '%s'\n",v->name,store->buf));
	}
	store_debug (("store_create_var (%s): multicast address = %s:%d",
				v->name, multiAddr, multiPort));
	/* have all the info needed from the catalog */
	if ((headerOffset = store_alloc_mem (store->myHeaderVarP->size)) == -1) {
		return (rtx_error_null ("store_create_var (%s): store_alloc_mem (header, %d) "
					"failed", v->name, store->myHeaderVarP->size));
	}
	if ((offset = store_alloc_mem (v->size)) == -1) {
		return (rtx_error_null ("store_create_var (%s): store_alloc_mem (var, %d) "
					"failed", v->name, v->size));
	}
	if ((varId = store_get_new_id (store)) == -1) {
		return (rtx_error_null ("store_create_var (%s): store_get_new_id() "
					"failed", v->name));
	}
	if ((semp = store_create_sem ()) == NULL) {
		return (rtx_error_null ("store_create_var (%s): store_create_sem() "
					"failed", v->name));
	}
	if ((store_post_sem (semp)) == -1) {
		store_destroy_sem(semp);
		return (rtx_error_null ("store_create_var (%s): store_post_sem() "
					"failed", v->name));
	}
	store_debug (("store_create_var (%s): var creation mostly done", v->name));
	memset (&(store->shmemP->d[offset]), 0, v->size);
	memset (&(store->shmemP->d[headerOffset]), 0, store->myHeaderVarP->size);
	svhp = (STORE_VAR_HEADER *) &(store->shmemP->d[headerOffset]);
	svhp->lockSemId = semp->id;
	svhp->lockSemSysVId = semp->semid;
	svhp->multicastRequired = 0;
    svhp->queueLength = 0;
	svhp->shmOffset = offset; /* store the offset so that client can receive updates */
	if (multicastReqd)
		svhp->multicastRequired = 1;
	for (i=0; i<STORE_ARCH_MAX_NUM; i++) {
		if ((v1 = calloc (1, sizeof (RtxParseVar))) == NULL)
			return (rtx_error_null ("store_create_var: calloc() failed"));
		memset (&curToken, 0, sizeof (RtxParseToken));
		varStrIndex = 0;
		if (rtx_parse_decl (store->buf, &varStrIndex, &curToken, v1,
					store->symTabs[i]) <= 0) {
			return (rtx_error_null ("store_create_var (%s): parse "
						"failed", store->buf));
		}
		if (rtx_parse_expand_var_definition (v1, store->symTabs[i]->typeTable) == -1) {
			rtx_parse_free_var (v1);
			return (rtx_error_null ("store_create_var (%s): expand_var "
						"failed", v->name));
		}
		if (rtx_parse_add_var_symbol (v1, varId, offset, headerOffset, store->buf,
					store->symTabs[i]->varTable) == -1) {
			return (rtx_error_null ("store_create_var (%s): add_var_symbol() "
						"failed", v->name));
		}
	}
	store_debug (("store_create_var (%s): var added to symbol table", v->name));
	if (store->verbose > 1)
		rtx_parse_print_var_table (store->mySymTabs->varTable);
	if ((stEntry = (RtxParseVarTableEntry *) rtx_hash_find
	     (store->mySymTabs->varTable, v->name, (rtx_error_t)NULL)) == NULL) {
		return (rtx_error_null ("store_create_var (%s): lookup_symbol() failed - "
					"serious problem, since the add just succeeded",
					v->name));
	}
	if (store_add_var_info_item (stEntry, varId, offset, headerOffset, 
				multiAddr, multiPort, semp, store) == -1) {
		return (rtx_error_null ("store_create_var (%s): store_add_var_info_item() "
					"failed", v->name));
	}
	store_debug (("store_create_var (%s): var added to store var table", 
				stEntry->v->name));
	vInfo = store->vars[stEntry->varId];
	/* create local producer thread */
	vInfo->multiAddr = multiAddr;
	vInfo->multiPort = multiPort;
	/*
	   if (rtx_thread_create_legacy (&vInfo->localProducerThreadId, 21, 16384,
	   (void * (*)(void *)) store_local_producer_thread, 
	   (void *) vInfo) == -1) {
	   return (rtx_error_null ("store_create_var (%s): "
	   "rtx_thread_create_legacy() failed", v->name));
	   }
	   */
	if ((vInfo->localProducerThreadId = rtx_thread_create 
				(v->name, store->verbose, RTX_THREAD_SCHED_OTHER, RTX_THREAD_PRIO_MIN, 16384,
				 RTX_THREAD_CANCEL_DEFERRED,
				 (void * (*)(void *)) store_local_producer_thread,
				 (void *) vInfo, NULL, NULL)) == NULL) {
		return (rtx_error_null ("store_create_var (%s): "
					"rtx_thread_create_legacy() failed", v->name));
	}

	return (stEntry);
}

int
store_delete_var (
		STORE * store,
		int varId
		)
{
	RtxParseVarTableEntry * stEntry;
	char * varName;
	int offset;
	int headerOffset;
	STORE_VAR_HEADER * svhp;

	headerOffset = store->vars[varId]->headerOffset;
	offset = store->vars[varId]->offset;
	store_debug (("store_deregister_var: varId = %d", varId));
	svhp = (STORE_VAR_HEADER *) 
		&(store->shmemP->d[store->vars[varId]->headerOffset]);
	stEntry = store->vars[varId]->vEntry;
	varName = stEntry->v->name;
	store_debug (("store_deregister_var (%s):", varName));
	if (svhp->queueLength>0) {
	  store_free_mem (offset, (stEntry->v->size + sizeof(svhp->ts))*svhp->queueLength);
	} else {
	  store_free_mem (offset, stEntry->v->size);
	}
	store_free_mem (headerOffset, store->myHeaderVarP->size);
	rtx_parse_del_var_symbol (stEntry->v->name, store->mySymTabs->varTable);
	store_del_var_info_item (varId, store);
	return (0);
}

void *
store_local_producer_thread (
		VAR_INFO_ITEM * v
		)
{
	RtxInet * multicastConn;
	STORE * store;
	RtxInetEndpoint fromAddr;
	RtxInetEndpoint myAddr;
	STORE_MULTICAST_HEADER * mhp;
	unsigned char * dataBuf;
	STORE_MULTICAST_HEADER * rxMhp;
	unsigned char * rxDataBuf;
	STORE_MULTICAST_HEADER * localMhp;
	unsigned char * localDataBuf;
	int dataLen, n;
	int myStatus = 0;
	STORE_ARCH_TYPE remoteArchType;
	STORE_VAR_HEADER * svhp;
	unsigned char * dp;
	int done = 0, i;
	char * varName;
	char * myAddrStr, * fromAddrStr;
	int varId, errs;
	char errorInitStr[BUFSIZ];
	RtxParseVar * myVarP;
	RtxParseVar * varP[STORE_ARCH_MAX_NUM];
	RtxParseVar * remoteVarP;
	RtxParseVarTableEntry * stp;
	int offset;
	int headerOffset;

	myVarP = v->vEntry->v;
	varName = v->vEntry->v->name;
	varId = v->vEntry->varId;
	store = v->store;

	sprintf (errorInitStr, "store[local_producer_thread(%s)]", varName);
	rtx_error_init (errorInitStr, RTX_ERROR_STDERR | RTX_ERROR_MESSAGE, NULL);

	for (i=0; i<STORE_ARCH_MAX_NUM; i++) {
		if ((stp = (RtxParseVarTableEntry *) rtx_hash_find 
		     (store->symTabs[i]->varTable, varName, (rtx_error_t)NULL)) == NULL) {
			rtx_error_flush ("lookup_symbol() failed");
			pthread_exit ((void *) myStatus);
		}
		varP[i] = stp->v;
	}
	rxMhp = (STORE_MULTICAST_HEADER *) &(v->buf[8]);
	if ((localMhp = (STORE_MULTICAST_HEADER *) calloc (1, 
					sizeof (STORE_MULTICAST_HEADER))) == NULL) {
		rtx_error_flush ("calloc() failed");
		pthread_exit ((void *) myStatus);
	}
	if ((localDataBuf = calloc (1, v->vEntry->v->size)) == NULL) {
		rtx_error_flush ("calloc() failed");
		pthread_exit ((void *) myStatus);
	}

	myStatus = 1;
	store_debug (("store_local_producer_thread (%s): starting, varId = %d",
				varName, varId));
	if ((multicastConn = rtx_inet_init (RTX_INET_UDP_MULTICAST,
					NULL, 0, v->multiAddr,
					v->multiPort, 
					NULL, NULL, NULL)) == NULL) {
		rtx_error_flush ("rtx_inet_init(multicast - %s:%d) failed for [%s]",
				v->multiAddr, v->multiPort, varName);
		pthread_exit ((void *) myStatus);
	}
	if (rtx_inet_set_multicast_ttl (multicastConn, 
				store->multicastTtl) == -1) {
		rtx_error_flush ("rtx_inet_set_multicast_ttl(multicast - %s:%d)"
				" failed for [%s]", v->multiAddr, v->multiPort,
				varName);
	}
	myStatus++;
	dataLen = v->vEntry->v->size;
	rxDataBuf = (unsigned char *) &(v->buf[sizeof (STORE_MULTICAST_HEADER) + 8]);
	svhp = (STORE_VAR_HEADER *) 
		&(store->shmemP->d[v->vEntry->headerOffset]);
	if (rtx_inet_get_net_addr (store->myHostName, 0, &myAddr) == -1) {
		rtx_error_errno_flush ("rtx_inet_get_net_addr() failed");
	}
	myAddrStr = strdup (rtx_inet_print_net_addr (&myAddr));
	store_debug (("store_local_producer_thread (%s): listening at %s:%d "
				"on %s (%s)", varName, v->multiAddr, v->multiPort, 
				myAddr.ifname, myAddrStr));
	while (! done) {
		errs = 0;
		if ((n = rtx_inet_read (multicastConn->sock, v->buf, STORE_BUF_SIZE, 
						&fromAddr)) == -1) {
			rtx_error_flush ("rtx_inet_read() failed");
		}
		if (n == 0) {
			rtx_message_warning ("store_local_producer_thread (%s): "
					"empty packet", varName);
		}
		if (n <= 0) {
			rtx_timer_sleep (0.01);
			continue;
		}
		remoteArchType = v->buf[0];
		fromAddrStr = rtx_inet_print_net_addr (&fromAddr);
		store_debug (("store_local_producer_thread (%s): read %d bytes, "
					" from %s (local is %s), remote arch %d", varName, n,
					fromAddrStr, myAddrStr, remoteArchType));
		if (strcmp (myAddrStr, fromAddrStr) == 0) {
			store_debug (("store_local_producer_thread (%s): local data", 
						varName));
			continue;
		}
		if (remoteArchType == store->myArchType) {
			mhp = rxMhp;
			dataBuf = rxDataBuf;
			remoteVarP = myVarP;
		} else {
			/* data translation required */
			remoteVarP = varP[remoteArchType];

			/** This looks fishy, myHeaderVarP is a STORE_VAR_HEADER, but rxMhp
			 * is a STORE_MULTICAST_HEADER **/
			rtx_parse_translate_data (localMhp, store->myHeaderVarP, rxMhp, 
					store->headerVarP[remoteArchType], 1);
			rtx_parse_translate_data (localDataBuf, myVarP, rxDataBuf, 
					remoteVarP, 1);
			mhp = localMhp;
			dataBuf = localDataBuf;
		}
		if (mhp->size != remoteVarP->size) {
		  store_debug (("store_local_producer_thread (%s): remote size (%d) != "
				"local size (%d)", varName, mhp->size, remoteVarP->size));
		  continue;
		}
		if ((sizeof (STORE_MULTICAST_HEADER) + remoteVarP->size + 8) != n) {
		  store_debug (("store_local_producer_thread (%s): rx pkt size = %d, "
				"expected pkt size = %d", varName, n, 
				myVarP->size + sizeof (STORE_MULTICAST_HEADER) + 8));
		  continue;
		}
		store_debug (("store_local_producer_thread (%s): valid data :)", varName));
		if (store_wait_sem (v->lockSemP) == -1) {
			rtx_error ("store_wait_sem() failed");
			errs++;
		}

        /* grab offsets after sem locked */
        headerOffset = store->vars[varId]->headerOffset;
        offset = store->vars[varId]->offset;

        /* todo: v->vEntry->headerOffset and v->vEntry->offset are not valid after queue changes */
        svhp = (STORE_VAR_HEADER *) 
          &(store->shmemP->d[headerOffset]);
		dp = &(store->shmemP->d[offset]); 
        /* rtx_message("base %p, offset %d %d",store->shmemP->d,offset, dp-store->shmemP->d); */

		memcpy (&(svhp->ts), &(mhp->ts), sizeof (svhp->ts));
		store_queue_write_entry(svhp, dp, dataBuf, dataLen);
		svhp->count++;

		if (store_post_sem (v->lockSemP) == -1) {
			rtx_error ("store_post_sem() failed");
			errs++;
		}
		/* Now let everyone know that there is data */
		if (store_semQ_add (store->semQ, STORE_SEM_Q_ACTION_POST,
					varId, NULL) == -1) {
			errs++;
		}
		if (errs)
			rtx_error_flush ("errors in this cycle");
	}
	store_debug (("store_local_producer_thread (%s): done", varName));
	return ((void *) myStatus);
}

int
store_handle_lookup_var_request (
		MSG * mp, char * rxBuf, char * txBuf,
		RtxInetConn * fromAddr,
		STORE * store
		)
{
	char * varDefn;
	char * typeDefn;
	int localLookup = 0;
	RtxParseToken curToken;
	int varStrIndex = 0, numChars;
	RtxParseVar * v, * vParse;
	RtxParseVarTableEntry * stEntry = NULL;
	STORE_SEM_T * semp;
	VAR_INFO_ITEM * vInfo = NULL;
	STORE_VAR_HEADER * svhp;
	int multicastReqd = 0;
	int varInLocalStore = 0;
	int errs = 0;

	/* Determine message type */
	if ((varDefn = get_message_part (keyTable, mp, MSG_TOK_VAR_DEFINITION))
			== NULL) {
		store_send_error_response ("unable to extract var definition",
				MSG_TOK_STORE_LOOKUP_VAR_REPLY,
				txBuf, fromAddr,store);
		return (rtx_error ("store_handle_lookup_var_request: unable to extract "
					"var definition"));
	}
	typeDefn = get_message_part (keyTable, mp, MSG_TOK_TYPE_DEFINITION);
	if (get_message_part (keyTable, mp, MSG_TOK_LOCAL_LOOKUP) != NULL) {
		localLookup = 1;
	}
	if (typeDefn == NULL) {
		store_debug (("store_handle_lookup_var_request (%s): local = %d", 
					varDefn, localLookup));
	} else {
		store_debug (("store_handle_lookup_var_request (%s): type = %s, "
					"local = %d", varDefn, typeDefn, localLookup));
	}
	if ((v = calloc (1, sizeof (RtxParseVar))) == NULL) {
		store_send_error_response ("unable to allocate memory",
				MSG_TOK_STORE_LOOKUP_VAR_REPLY,
				txBuf, fromAddr,store);
		return (rtx_error_errno ("store_handle_lookup_var_request (%s): "
					"calloc() failed", varDefn));
	}
	memset (v->name, 0, MAX_TOK_LEN);
	/* Check if lookup is local */
	if (localLookup) {
		/* just need to check if the variable is in the store */
		if ((vParse = rtx_parse_name_list (varDefn, &varStrIndex, &curToken, v, store->mySymTabs))
				== NULL) {
			store_send_error_response ("name not found",
					MSG_TOK_STORE_LOOKUP_VAR_REPLY,
					txBuf, fromAddr,store);
			rtx_parse_free_var (v);
			return (rtx_error ("store_handle_lookup_var_request (%s): "
						"name not found", varDefn));
		}
		rtx_parse_free_var (vParse);
		if ((stEntry = (RtxParseVarTableEntry *) rtx_hash_find 
		     (store->mySymTabs->varTable, v->name, (rtx_error_t)NULL)) == NULL) {
			store_send_error_response ("name not found",
					MSG_TOK_STORE_LOOKUP_VAR_REPLY,
					txBuf, fromAddr,store);
			rtx_parse_free_var (v);
			return (rtx_error ("store_handle_lookup_var_request (%s): serious"
						" error - parse_name_list succeeded but "
						"lookup_symbol failed", varDefn));
		}
		vInfo = store->vars[stEntry->varId];
		sprintf (txBuf, "%s %s = %s %s = %d %s = %d %s = %ld %s = %d"
				" %s = %d %s = %d %s = %d %s = %s %s = %s %s = %s "
				"%s = %d\n",
				keyTable[MSG_TOK_STORE_LOOKUP_VAR_REPLY].name,
				keyTable[MSG_TOK_SHMEM_NAME].name, store->shmemP->name,
				keyTable[MSG_TOK_SHMEM_ID].name, store->shmemP->id,
				keyTable[MSG_TOK_SHMEM_SIZE].name, store->shmemP->size,
				keyTable[MSG_TOK_SHMEM_PID].name, (unsigned long) store->shmemP->pid,
				keyTable[MSG_TOK_SHMEM_VAR_OFFSET].name, 
                 vInfo->offset, /* stEntry->offset, */
				keyTable[MSG_TOK_SHMEM_VAR_HEADER_OFFSET].name,
                 vInfo->headerOffset, /* stEntry->headerOffset, */
				keyTable[MSG_TOK_SHMEM_VAR_ID].name, 
				stEntry->varId,
				keyTable[MSG_TOK_SHMEM_VAR_SIZE].name, 
				stEntry->v->size,
				keyTable[MSG_TOK_VAR_DEFINITION].name, 
				stEntry->decl,
				keyTable[MSG_TOK_NAME].name, v->name,
				keyTable[MSG_TOK_VAR_MULTICAST_ADDRESS].name,
				vInfo->multiAddr,
				keyTable[MSG_TOK_VAR_MULTICAST_PORT].name,
				vInfo->multiPort);
		if ((numChars = rtx_inet_write (fromAddr, txBuf, 
						strlen (txBuf), NULL)) == -1) {
			rtx_parse_free_var (v);
			return (rtx_error ("store_handle_lookup_var_request (%s): "
						"rtx_inet_write failed", varDefn));
		}
		rtx_parse_free_var (v);
		return (0);
	}

	/* Lookup is not local. There are several possibilities:
	 * - var is already in store
	 * - var is not in store but in catalog
	 * - var does not exist at all
	 * 
	 * Depending on the situation, the var might need to be
	 * created in the store and possibly in the catalog.
	 * Therefore, only one thread should be allowed to proceed
	 * beyond this point.
	 */

	/* lock the store mutex to prevent any other threads from
	 * proceeding
	 */
	if (rtx_mutex_lock (&store->mutex) == -1)
		rtx_error_flush ("store_handle_lookup_var_request (%s): rtx_mutex_lock() failed", varDefn);

	/* Check if var is in local store */
	if (typeDefn == NULL) {
		/* just need to check if the variable is in the store */
		if ((vParse = rtx_parse_name_list (varDefn, &varStrIndex, &curToken, v, store->mySymTabs))
				!= NULL) {
			if ((stEntry = (RtxParseVarTableEntry *) rtx_hash_find 
			     (store->mySymTabs->varTable, v->name, (rtx_error_t)NULL)) != NULL) {
				varInLocalStore = 1;
			}
			rtx_parse_free_var (vParse);
		}
	} else {
		store_debug (("store_handle_lookup_var_request (%s): parsing", varDefn));
		/* typeDefn provided, construct var definition */
		sprintf (store->buf, "%s %s ;", typeDefn, varDefn);
		if (rtx_parse_decl (store->buf, &varStrIndex, &curToken, v, store->mySymTabs) <= 0) {
			rtx_error ("store_handle_lookup_var_request (%s): parse "
					"failed", varDefn);
			errs++;
		}
		if (!errs) {
			if (v->name != NULL)
				store_debug (("store_handle_lookup_var_request (%s): v->name = %s", varDefn, v->name));
			if (rtx_parse_expand_var_definition (v, store->mySymTabs->typeTable)  == -1) {
				rtx_error ("store_handle_lookup_var_request (%s): "
						"parse failed", varDefn);
				errs++;
			} else {
				store_debug (("store_handle_lookup_var_request (%s): checking locally", varDefn));
				if (v->name != NULL)
					store_debug (("store_handle_lookup_var_request (%s): v->name = %s", varDefn, v->name));
				if ((stEntry = (RtxParseVarTableEntry *) rtx_hash_find
				     (store->mySymTabs->varTable, v->name, (rtx_error_t)NULL)) != NULL) {
					varInLocalStore = 1;
				}
			}
		}
	}
	if (errs) {
		if (rtx_mutex_unlock (&store->mutex) == -1)
			rtx_error ("store_handle_lookup_var_request (%s): rtx_mutex_unlock() failed", varDefn);
		rtx_parse_free_var (v);
		return (-1);
	}
	/* we have now successfully queried the local store without
	 * errors
	 */
	if (varInLocalStore) {
		/* var exists in store */
		vInfo = store->vars[stEntry->varId];
		if (typeDefn != NULL)
			if (rtx_parse_cmp_types (v, stEntry->v) != 0) {
				rtx_parse_free_var (v);
				rtx_error ("store_handle_lookup_var_request (%s): var "
						"exists", varDefn);
				if (rtx_mutex_unlock (&store->mutex) == -1)
					rtx_error ("store_handle_lookup_var_request (%s): rtx_mutex_unlock() failed", varDefn);
				return (-1);
			}
		/* add sync sem */
		if ((semp = store_add_sync_sem (store, 
						stEntry->varId,fromAddr)) == NULL) {
			store_send_error_response ("unable to add semaphore",
					MSG_TOK_STORE_LOOKUP_VAR_REPLY,
					txBuf, fromAddr,store);

			if (rtx_mutex_unlock (&store->mutex) == -1)
				rtx_error ("store_handle_lookup_var_request (%s): rtx_mutex_unlock() failed", varDefn);
			rtx_parse_free_var (v);
			return (rtx_error ("store_handle_lookup_var_request (%s): "
						"store_add_sync_sem() failed", varDefn));
		}
		svhp = (STORE_VAR_HEADER *) 
			&(store->shmemP->d[vInfo->headerOffset]);
		/* inform the catalog that a client is interested only if the current
		 * number of clients is zero */
		if (vInfo->numClients == 0) {
			if ((multicastReqd = store_lookup_var_in_catalog (v, NULL, NULL, NULL, 1, rxBuf, txBuf, store)) == -1) {
				store_send_error_response ("catalog lookup failed",
						MSG_TOK_STORE_LOOKUP_VAR_REPLY,
						txBuf, fromAddr,store);
				if (rtx_mutex_unlock (&store->mutex) == -1)
					rtx_error ("store_handle_lookup_var_request (%s): rtx_mutex_unlock() failed", varDefn);
				rtx_parse_free_var (v);
				return (rtx_error ("store_handle_lookup_var_request (%s): "
							"store_lookup_var_in_catalog() failed", varDefn));
			}
			svhp->multicastRequired = multicastReqd;
		}
		vInfo->numClients++;
		sprintf (txBuf, "%s %s = %s %s = %d %s = %d %s = %ld %s = %d"
				" %s = %d %s = %d %s = %d %s = %d %s = %d %s = %s %s = %s "
				"%s = %s %s = %d\n",
				keyTable[MSG_TOK_STORE_LOOKUP_VAR_REPLY].name,
				keyTable[MSG_TOK_SHMEM_NAME].name, store->shmemP->name,
				keyTable[MSG_TOK_SHMEM_ID].name, store->shmemP->id,
				keyTable[MSG_TOK_SHMEM_SIZE].name, store->shmemP->size,
				keyTable[MSG_TOK_SHMEM_PID].name, (unsigned long) store->shmemP->pid,
				keyTable[MSG_TOK_SHMEM_VAR_OFFSET].name, 
				vInfo /* stEntry */ ->offset,
				keyTable[MSG_TOK_SHMEM_VAR_HEADER_OFFSET].name,
				vInfo /* stEntry */ ->headerOffset,
				keyTable[MSG_TOK_SHMEM_VAR_ID].name,
				stEntry->varId,
				keyTable[MSG_TOK_SHMEM_VAR_SIZE].name, 
				stEntry->v->size,
				keyTable[MSG_TOK_SYNC_SEM_ID].name, semp->id,
				keyTable[MSG_TOK_SYNC_SYSVSEM_ID].name, semp->semid,
				keyTable[MSG_TOK_VAR_DEFINITION].name, 
				stEntry->decl,
				keyTable[MSG_TOK_NAME].name, v->name,
				keyTable[MSG_TOK_VAR_MULTICAST_ADDRESS].name,
				vInfo->multiAddr,
				keyTable[MSG_TOK_VAR_MULTICAST_PORT].name,
				vInfo->multiPort);
		if ((numChars = rtx_inet_write (fromAddr, txBuf, 
						strlen (txBuf), NULL)) == -1) {
			rtx_parse_free_var (v);
			if (rtx_mutex_unlock (&store->mutex) == -1)
				rtx_error ("store_handle_lookup_var_request (%s): rtx_mutex_unlock() failed", varDefn);
			return (rtx_error ("store_handle_lookup_var_request (%s): "
						"rtx_inet_write failed", varDefn));
		}
		rtx_parse_free_var (v);
		if (rtx_mutex_unlock (&store->mutex) == -1)
			rtx_error_flush ("store_handle_lookup_var_request (%s): rtx_mutex_unlock() failed", varDefn);
		return (0);
	}
	/* var is not in local store. Send request to catalog and
	 * then create in local store.
	 */
	store_debug (("store_handle_lookup_var_request (%s): creating", varDefn));
	if ((stEntry = store_create_var (v, typeDefn, rxBuf, txBuf, store)) == NULL) {
		rtx_parse_free_var (v);
		if (rtx_mutex_unlock (&store->mutex) == -1)
			rtx_error ("store_handle_lookup_var_request (%s): rtx_mutex_unlock() failed", varDefn);
		return (rtx_error ("store_handle_lookup_var_request (%s): "
					"store_create_var() failed", varDefn));
	}
	vInfo = store->vars[stEntry->varId];
	vInfo->numClients = 1;
	store_debug (("store_handle_lookup_var_request (%s): created", varDefn));
	/* the var is now in the store locally. Send response back to client */
	/* add sync sem */
	if ((semp = store_add_sync_sem (store, 
					stEntry->varId, fromAddr)) == NULL) {
		store_send_error_response ("unable to add semaphore",
				MSG_TOK_STORE_LOOKUP_VAR_REPLY,
				txBuf, fromAddr,store);
		rtx_parse_free_var (v);
		if (rtx_mutex_unlock (&store->mutex) == -1)
			rtx_error ("store_handle_lookup_var_request (%s): rtx_mutex_unlock() failed", varDefn);
		return (rtx_error ("store_handle_lookup_var_request (%s): "
					"store_add_sync_sem() failed", varDefn));
	}
	store_debug (("store_handle_lookup_var_request (%s): sending response", 
				varDefn));
	sprintf (txBuf, "%s %s = %s %s = %d %s = %d %s = %ld %s = %d"
			" %s = %d %s = %d %s = %d %s = %d %s = %d %s = %s %s = %s "
			"%s = %s %s = %d\n",
			keyTable[MSG_TOK_STORE_LOOKUP_VAR_REPLY].name,
			keyTable[MSG_TOK_SHMEM_NAME].name, store->shmemP->name,
			keyTable[MSG_TOK_SHMEM_ID].name, store->shmemP->id,
			keyTable[MSG_TOK_SHMEM_SIZE].name, store->shmemP->size,
			keyTable[MSG_TOK_SHMEM_PID].name, (unsigned long) store->shmemP->pid,
			keyTable[MSG_TOK_SHMEM_VAR_OFFSET].name, 
			vInfo /* stEntry */ ->offset,
			keyTable[MSG_TOK_SHMEM_VAR_HEADER_OFFSET].name,
			vInfo /* stEntry */ ->headerOffset,
			keyTable[MSG_TOK_SHMEM_VAR_ID].name,
			stEntry->varId,
			keyTable[MSG_TOK_SHMEM_VAR_SIZE].name, 
			stEntry->v->size,
			keyTable[MSG_TOK_SYNC_SEM_ID].name, semp->id,
			keyTable[MSG_TOK_SYNC_SYSVSEM_ID].name, semp->semid,
			keyTable[MSG_TOK_VAR_DEFINITION].name, 
			stEntry->decl,
			keyTable[MSG_TOK_NAME].name, v->name,
			keyTable[MSG_TOK_VAR_MULTICAST_ADDRESS].name,
			vInfo->multiAddr,
			keyTable[MSG_TOK_VAR_MULTICAST_PORT].name,
			vInfo->multiPort);
	if ((numChars = rtx_inet_write (fromAddr, txBuf, 
					strlen (txBuf), NULL)) == -1) {
		rtx_parse_free_var (v);
		if (rtx_mutex_unlock (&store->mutex) == -1)
			rtx_error ("store_handle_lookup_var_request (%s): rtx_mutex_unlock() failed", varDefn);
		return (rtx_error ("store_handle_lookup_var_request (%s): "
					"rtx_inet_write failed", varDefn));
	}
	rtx_parse_free_var (v);
	if (rtx_mutex_unlock (&store->mutex) == -1)
		rtx_error_flush ("store_handle_lookup_var_request (%s): rtx_mutex_unlock() failed", varDefn);
	return (0);
}

int
store_handle_done_var_request (
		MSG * mp, char * rxBuf, char * txBuf,
		RtxInetConn * fromAddr,
		STORE * store
		)
{
	RtxParseVar * v;
	char * msgParm, * varName;
	int varId, syncSemId = -1;
	int numChars;
	RtxParseVarTableEntry * stEntry;
	STORE_VAR_HEADER * svhp;
	VAR_INFO_ITEM * vInfo;
	MSG * catMsg;

	store_debug (("store_handle_done_var_request:"));
	if ((varName = get_message_part (keyTable, mp, MSG_TOK_NAME))
			== NULL) {
		store_send_error_response ("unable to extract var name",
				MSG_TOK_STORE_DONE_VAR_REPLY,
				txBuf, fromAddr,store);
		return (rtx_error ("store_handle_done_var_request: unable to extract "
					"var name"));
	}
	store_debug (("store_handle_done_var_request (%s): looking", varName));
	if ((stEntry = (RtxParseVarTableEntry *) rtx_hash_find
	     (store->mySymTabs->varTable, varName, (rtx_error_t)NULL)) == NULL) {
		store_send_error_response ("name not found",
				MSG_TOK_STORE_DONE_VAR_REPLY,
				txBuf, fromAddr,store);
		return (rtx_error ("store_handle_done_var_request (%s): lookup_symbol "
					"failed", varName));
	}
	varId = stEntry->varId;
	v = stEntry->v;
	vInfo = store->vars[varId];
	svhp = (STORE_VAR_HEADER *) 
		&(store->shmemP->d[vInfo /* stEntry */ ->headerOffset]);
	store_debug (("store_handle_done_var_request (%s): found", v->name));
	if ((msgParm = get_message_part (keyTable, mp, MSG_TOK_SYNC_SEM_ID))
			== NULL) {
		return (rtx_error ("store_handle_done_var_request (%s): unable to get "
					"sync sem id", varName));
	}
	if ((syncSemId = atoi (msgParm)) < 0) {
		store_send_error_response ("invalid_sync_sem_id",
				MSG_TOK_STORE_DONE_VAR_REPLY,
				txBuf, fromAddr,store);
		return (rtx_error ("store_handle_done_var_request (%s): invalid sync sem "
					"id (%d)", varName, syncSemId));
	}
	/* delete sem */
	store_debug (("store_handle_done_var_request (%s): "
				"deleting syncSem", varName));
	if (store_del_sync_sem (store, varId, syncSemId)) {
		store_send_error_response ("unable to del semaphore",
				MSG_TOK_STORE_DONE_VAR_REPLY,
				txBuf, fromAddr,store);
		return (rtx_error ("store_handle_done_var_request (%s): store_del_sync_sem ()"
					" failed", varName));
	}
	if (rtx_mutex_lock (&store->mutex) == -1)
		rtx_error_flush ("store_handle_done_var_request (%s): rtx_mutex_lock() failed", varName);
	if (vInfo->numClients > 0)
		vInfo->numClients--;
	if (vInfo->numClients == 0) {
		sprintf (txBuf, "%s %s = %s %s = %s\n",
				catKeyTable[MSG_TOK_CATALOG_DONE_VAR_REQUEST].name,
				catKeyTable[MSG_TOK_CATALOG_VAR_NAME].name, v->name,
				catKeyTable[MSG_TOK_CATALOG_HOST_NAME].name, store->myHostName);
		if ((catMsg = store_get_catalog_response (rxBuf, txBuf, 5, 
						store)) == NULL) {
			if (rtx_mutex_unlock (&store->mutex) == -1)
				rtx_error ("store_handle_done_var_request (%s): rtx_mutex_unlock() failed", varName);
			return (rtx_error ("store_handle_done_var_request (%s): no response (5s) "
						"from catalog", v->name));
		} else if (catMsg->id != MSG_CATALOG_DONE_VAR_REPLY) {
			free_message (catKeyTable, catMsg);
			if (rtx_mutex_unlock (&store->mutex) == -1)
				rtx_error ("store_handle_done_var_request (%s): rtx_mutex_unlock() failed", varName);
			return (rtx_error ("store_handle_done_var_request (%s): invalid/error "
						"response from catalog", v->name));
		}
		if ((msgParm = get_message_part (catKeyTable, catMsg,
						MSG_TOK_CATALOG_MULTICAST_OFF)) != NULL) {
			svhp->multicastRequired = 0;
		}
		free_message (catKeyTable, catMsg);
	}
	/* all done */
	if (rtx_mutex_unlock (&store->mutex) == -1)
		rtx_error_flush ("store_handle_done_var_request (%s): rtx_mutex_unlock() failed", varName);
	store_debug (("store_handle_done_var_request (%s): "
				"all done", varName));
	sprintf (txBuf, "%s %s = %s\n",
			keyTable[MSG_TOK_STORE_DONE_VAR_REPLY].name,
			keyTable[MSG_TOK_RESPONSE].name, "OK");
	if ((numChars = rtx_inet_write (fromAddr, txBuf, 
					strlen (txBuf), NULL)) == -1) {
		return (rtx_error ("store_handle_done_var_request (%s): rtx_inet_write() "
					"failed", varName));
	}
	return (0);
}

/*******************************************************************/

int
store_handle_get_catalog_request (
		MSG * mp, char * rxBuf, char * txBuf,
		RtxInetConn * fromAddr,
		STORE * store
		)
{
	int numChars;

	sprintf (txBuf, "%s %s = %s %d\n",
			keyTable[MSG_TOK_STORE_GET_CATALOG_REPLY].name,
			keyTable[MSG_TOK_NAME].name, 
			store->catalog.server->sock->remote.ifname,
			store->catalog.server->sock->remote.portNum);
	if ((numChars = rtx_inet_write (fromAddr, txBuf, 
					strlen (txBuf), NULL)) == -1) {
		return (rtx_error ("store_handle_get_catalog_request: "
					"rtx_inet_write() failed"));
	}
	return (0);
}

static void
store_handle_table_dump (void *key,
			 void *data,
			 int i, int j,
			 int *stop_iter,
			 void *context) {
	strcat (context, key);
	strcat (context, " ");
}

int
store_handle_get_var_list_request (
		MSG * mp, char * rxBuf, char * txBuf,
		RtxInetConn * fromAddr,
		STORE * store
		)
{
	int numChars;
	char numbuf[64];
	VAR_INFO_ITEM * vp;
	char * msgParm = NULL;
	int initStrLen = 0;

	store_debug (("store_handle_get_var_list_request: handling"));

	if ((msgParm = get_message_part (keyTable, mp, 
					MSG_TOK_ITEM_TYPE_VAR)) != NULL) {
		store_debug (("store_handle_get_var_list_request: var"));
		sprintf (txBuf, "%s %s = ",
				keyTable[MSG_TOK_STORE_GET_VAR_LIST_REPLY].name,
				keyTable[MSG_TOK_LIST].name);
		initStrLen = strlen (txBuf);
		while ((vp = (VAR_INFO_ITEM *) rtx_list_iterate
					(store->varList)) != NULL) {
			if ((vp->flags & STORE_VAR_ENTRY_FIELD) == 0) {
				strcat (txBuf, vp->vEntry->name);
				strcat (txBuf, " ");
				sprintf (numbuf, "%d", vp->varId);
				strcat (txBuf, numbuf);
				strcat (txBuf, " ");
			}
		}
		if (strlen(txBuf) == initStrLen) {
			sprintf (txBuf, "%s",
					keyTable[MSG_TOK_STORE_GET_VAR_LIST_REPLY].name);
		}
		strcat (txBuf, "\n");
	} else if ((msgParm = get_message_part 
				(keyTable, mp, MSG_TOK_ITEM_TYPE_TYPE)) != NULL) {
		store_debug (("store_handle_get_var_list_request: type"));
		sprintf (txBuf, "%s %s = ",
				keyTable[MSG_TOK_STORE_GET_VAR_LIST_REPLY].name,
				keyTable[MSG_TOK_LIST].name);
		initStrLen = strlen (txBuf);
		rtx_hash_iter(store->mySymTabs->typeTable, 
			      store_handle_table_dump, txBuf);
		if (strlen (txBuf) == initStrLen) 
			sprintf (txBuf, "%s ",
					keyTable[MSG_TOK_STORE_GET_VAR_LIST_REPLY].name);
		strcat (txBuf, "\n");
	} else if ((msgParm = get_message_part 
				(keyTable, mp, MSG_TOK_ITEM_TYPE_CONST)) != NULL) {
		store_debug (("store_handle_get_var_list_request: const"));
		sprintf (txBuf, "%s %s = ",
				keyTable[MSG_TOK_STORE_GET_VAR_LIST_REPLY].name,
				keyTable[MSG_TOK_LIST].name);
		initStrLen = strlen (txBuf);
		rtx_hash_iter(store->mySymTabs->constTable, 
			      store_handle_table_dump, txBuf);
		store_debug (("store_handle_get_var_list_request: len = "
					"(%d, %d)", initStrLen, strlen (txBuf)));
		if (strlen (txBuf) == initStrLen) {
			store_debug (("store_handle_get_var_list_request: "
						"empty list"));
			sprintf (txBuf, "%s ",
					keyTable[MSG_TOK_STORE_GET_VAR_LIST_REPLY].name);
		}
		strcat (txBuf, "\n");
	} else {
		store_debug (("store_handle_get_var_list_request: var (def)"));
		sprintf (txBuf, "%s %s = ",
				keyTable[MSG_TOK_STORE_GET_VAR_LIST_REPLY].name,
				keyTable[MSG_TOK_LIST].name);
		initStrLen = strlen (txBuf);
		while ((vp = (VAR_INFO_ITEM *) rtx_list_iterate
					(store->varList)) != NULL) {
			if ((vp->flags & STORE_VAR_ENTRY_FIELD) == 0) {
				strcat (txBuf, vp->vEntry->name);
				strcat (txBuf, " ");
				sprintf (numbuf, "%d", vp->varId);
				strcat (txBuf, numbuf);
				strcat (txBuf, " ");
			}
		}
		if (strlen(txBuf) == initStrLen) {
			sprintf (txBuf, "%s",
					keyTable[MSG_TOK_STORE_GET_VAR_LIST_REPLY].name);
		}
		strcat (txBuf, "\n");
	}
	if ((numChars = rtx_inet_write (fromAddr, txBuf, 
					strlen (txBuf), NULL)) == -1) {
		return (rtx_error ("store_handle_get_var_list_request: rtx_inet_write() failed"));
	}
	return (0);
}

int
store_handle_get_var_status_request (
		MSG * mp, char * rxBuf, char * txBuf,
		RtxInetConn * fromAddr,
		STORE * store
		)
{
	int numChars;
	char * msgParm = NULL;
	char * varName = NULL;
	int initStrLen = 0;
	RtxParseVarTableEntry * vEntry = NULL;
	RtxParseConstTableEntry * cEntry = NULL;
	RtxParseTypeTableEntry * tEntry = NULL;
	STORE_VAR_HEADER * svhp;

	store_debug (("store_handle_get_var_status_request: handling"));

	if ((varName = get_message_part (keyTable, mp, MSG_TOK_NAME))
			== NULL) {
		store_send_error_response ("unable to extract var definition",
				MSG_TOK_STORE_GET_VAR_STATUS_REPLY,
				txBuf, fromAddr,store);
		return (rtx_error ("store_handle_get_var_status_request: "
					"unable to get var name"));
	}
	if ((msgParm = get_message_part (keyTable, mp, 
					MSG_TOK_ITEM_TYPE_VAR)) != NULL) {
		if ((vEntry = (RtxParseVarTableEntry *) rtx_hash_find
		     (store->mySymTabs->varTable, varName, (rtx_error_t)NULL)) == NULL) {
			store_send_error_response ("name not found",
					MSG_TOK_STORE_GET_VAR_STATUS_REPLY,
					txBuf, fromAddr,store);
			return (rtx_error ("store_handle_get_var_status_request (%s): "
						"lookup_symbol() failed", varName));
		}
		svhp = (STORE_VAR_HEADER *) 
			&(store->shmemP->d[store->vars[vEntry->varId]->headerOffset]);
		sprintf (txBuf, "%s %s = %d %s = %d %s = ",
				keyTable[MSG_TOK_STORE_GET_VAR_STATUS_REPLY].name,
				keyTable[MSG_TOK_VALUE].name, svhp->count,
				keyTable[MSG_TOK_SHMEM_VAR_SIZE].name, vEntry->v->size,
				keyTable[MSG_TOK_VAR_DEFINITION].name);
		initStrLen = strlen (txBuf);
		rtx_parse_generate_var_decl (vEntry->v, &(txBuf[initStrLen]));
		strcat (txBuf, "\n");
	} else if ((msgParm = get_message_part 
				(keyTable, mp, MSG_TOK_ITEM_TYPE_TYPE)) != NULL) {
		if ((tEntry = (RtxParseTypeTableEntry *) rtx_hash_find
		     (store->mySymTabs->typeTable, varName, (rtx_error_t)NULL)) == NULL) {
			store_send_error_response ("name not found",
					MSG_TOK_STORE_GET_VAR_STATUS_REPLY,
					txBuf, fromAddr,store);
			return (rtx_error ("store_handle_get_var_status_request (%s): "
						"lookup_symbol() failed", varName));
		}
		sprintf (txBuf, "%s %s = %d %s = %d %s = ",
				keyTable[MSG_TOK_STORE_GET_VAR_STATUS_REPLY].name,
				keyTable[MSG_TOK_VALUE].name, 0,
				keyTable[MSG_TOK_SHMEM_VAR_SIZE].name, tEntry->t->size,
				keyTable[MSG_TOK_VAR_DEFINITION].name);
		initStrLen = strlen (txBuf);
		rtx_parse_expand_var_definition(tEntry->t, store->mySymTabs->typeTable);
		rtx_parse_generate_var_decl (tEntry->t, &(txBuf[initStrLen]));
		strcat (txBuf, "\n");
	} else if ((msgParm = get_message_part 
				(keyTable, mp, MSG_TOK_ITEM_TYPE_CONST)) != NULL) {
		if ((cEntry = (RtxParseConstTableEntry *) rtx_hash_find
		     (store->mySymTabs->constTable, varName, (rtx_error_t)NULL)) == NULL) {
			store_send_error_response ("name not found",
					MSG_TOK_STORE_GET_VAR_STATUS_REPLY,
					txBuf, fromAddr,store);
			return (rtx_error ("store_handle_get_var_status_request (%s): "
						"lookup_symbol() failed", varName));
		}
		sprintf (txBuf, "%s %s = %d %s = %d %s = %s",
				keyTable[MSG_TOK_STORE_GET_VAR_STATUS_REPLY].name,
				keyTable[MSG_TOK_VALUE].name, cEntry->value,
				keyTable[MSG_TOK_SHMEM_VAR_SIZE].name, 0,
				keyTable[MSG_TOK_VAR_DEFINITION].name, varName);
		strcat (txBuf, "\n");
	} else {
		if ((vEntry = (RtxParseVarTableEntry *) rtx_hash_find
		     (store->mySymTabs->varTable, varName, (rtx_error_t)NULL)) == NULL) {
			store_send_error_response ("name not found",
					MSG_TOK_STORE_GET_VAR_STATUS_REPLY,
					txBuf, fromAddr,store);
			return (rtx_error ("store_handle_get_var_status_request (%s): "
						"lookup_symbol() failed", varName));
		}
		svhp = (STORE_VAR_HEADER *) 
			&(store->shmemP->d[store->vars[vEntry->varId]->headerOffset]);
		sprintf (txBuf, "%s %s = %d %s = %d %s = ",
				keyTable[MSG_TOK_STORE_GET_VAR_STATUS_REPLY].name,
				keyTable[MSG_TOK_VALUE].name, svhp->count,
				keyTable[MSG_TOK_SHMEM_VAR_SIZE].name, vEntry->v->size,
				keyTable[MSG_TOK_VAR_DEFINITION].name);
		initStrLen = strlen (txBuf);
		rtx_parse_generate_var_decl (vEntry->v, &(txBuf[initStrLen]));
		strcat (txBuf, "\n");
	}
	if ((numChars = rtx_inet_write (fromAddr, txBuf, 
					strlen (txBuf), NULL)) == -1) {
		return (rtx_error ("store_handle_get_var_list_request: rtx_inet_write() failed"));
	}
	return (0);
}

int
store_handle_get_var_value_request (
		MSG * mp, char * rxBuf, char * txBuf,
		RtxInetConn * fromAddr,
		STORE * store
		)
{
	char * msgParm;
	int numChars;
	int varId = -1;
	char * varName;
	VAR_INFO_ITEM * vInfo;
	RtxParseVarTableEntry * stEntry = NULL;
	STORE_VAR_HEADER * svhp;
	unsigned char * d;
	int vCount;
	STORE_TIMESPEC ts;

	store_debug (("store_handle_get_var_value_request:"));
	if ((varName = get_message_part (keyTable, mp, MSG_TOK_NAME))
			== NULL) {
		store_send_error_response ("unable to extract var definition",
				MSG_TOK_STORE_GET_VAR_VALUE_REPLY,
				txBuf, fromAddr,store);
		return (rtx_error ("store_handle_get_var_value_request: unable to get var "
					"name"));
	}
	store_debug (("store_handle_get_var_value_request: name = %s", varName));
	if ((msgParm = get_message_part (keyTable, mp, MSG_TOK_ID)) != NULL) {
		varId = atoi (msgParm);
		store_debug (("store_handle_get_var_value_request: id = %s", msgParm));
	}
	if (varId < 0) { /* need to lookup varId */
		if ((stEntry = (RtxParseVarTableEntry *) rtx_hash_find
		     (store->mySymTabs->varTable, varName, (rtx_error_t)NULL)) == NULL) {
			store_send_error_response ("name not found",
					MSG_TOK_STORE_GET_VAR_VALUE_REPLY,
					txBuf, fromAddr,store);
			return (rtx_error ("store_handle_get_var_value_request (%s): "
						"lookup_symbol() failed", varName));
		}
		varId = stEntry->varId;
	} else
		stEntry = store->vars[varId]->vEntry;
	store_debug (("store_handle_get_var_value_request: varId = %d", varId));
	vInfo = store->vars[varId];
	svhp = (STORE_VAR_HEADER *) 
		&(store->shmemP->d[vInfo->headerOffset]);
	if (store_wait_sem (vInfo->lockSemP) == -1) {
		return (rtx_error ("store_handle_get_var_value_request (%s): "
					"store_wait_sem() failed", varName));
	}
	d = &(store->shmemP->d[vInfo->offset]);	/* get offset after sem locked */
	vCount = svhp->count;
	ts = svhp->ts;
	if (stEntry->v->size <= STORE_BUF_SIZE) {
	  store_queue_read_current(svhp, d, vInfo->buf, stEntry->v->size);
	} else {
		return rtx_error("store_handle_get_var_value_request (%s):"
				"variable too big, can't print values", varName);
	}
	if (store_post_sem (vInfo->lockSemP) == -1) {
		return (rtx_error ("store_handle_get_var_value_request (%s):"
					"store_post_sem() failed", varName));
	}
	sprintf (txBuf, "%s %s = %d %lu %ld ",
			keyTable[MSG_TOK_STORE_GET_VAR_VALUE_REPLY].name,
			keyTable[MSG_TOK_VALUE].name,
			vCount, ts.tv_sec, ts.tv_nsec);
	if (rtx_parse_print_var (stEntry->v, vInfo->buf, 
				&(txBuf[strlen (txBuf)])) == -1) {
		store_send_error_response ("unable to print var",
				MSG_TOK_STORE_GET_VAR_VALUE_REPLY,
				txBuf, fromAddr,store);
		return (rtx_error ("store_handle_get_var_value_request (%s): rtx_parse_print_var()"
					" failed", varName));
	}
	strcat (txBuf, "\n");
	if ((numChars = rtx_inet_write (fromAddr, txBuf, 
					strlen (txBuf), NULL)) == -1) {
		return (rtx_error ("store_handler_get_var_value_request (%s): rtx_inet_write() "
					"failed", varName));
	}
	return (0);
}

int
store_handle_set_var_value_request (
		MSG * mp, char * rxBuf, char * txBuf,
		RtxInetConn * fromAddr,
		STORE * store
		)
{
	char * msgParm;
	int numChars;
	int varId = -1;
	char * varName;
	VAR_INFO_ITEM * vInfo = NULL;
	RtxParseVarTableEntry * stEntry = NULL;
	STORE_VAR_HEADER * svhp;
	unsigned char * d;
	STORE_TIMESPEC ts;
	char * lasts;

	clock_gettime (CLOCK_REALTIME, (struct timespec *) &ts);
	store_debug (("store_handle_set_var_value_request:"));
	if ((varName = get_message_part (keyTable, mp, MSG_TOK_NAME))
			== NULL) {
		store_send_error_response ("unable to extract var definition",
				MSG_TOK_STORE_SET_VAR_VALUE_REPLY,
				txBuf, fromAddr,store);
		return (rtx_error ("store_handle_set_var_value_request: unable to get "
					"var name"));
	}
	store_debug (("store_handle_set_var_value_request: name = %s", varName));
	if ((msgParm = get_message_part (keyTable, mp, MSG_TOK_ID)) != NULL) {
		varId = atoi (msgParm);
	}
	store_debug (("store_handle_set_var_value_request: id = %s", msgParm));
	if (varId < 0) { /* need to lookup varId */
		if ((stEntry = (RtxParseVarTableEntry *) rtx_hash_find
		     (store->mySymTabs->varTable, varName, (rtx_error_t)NULL)) == NULL) {
			store_send_error_response ("name not found",
					MSG_TOK_STORE_SET_VAR_VALUE_REPLY,
					txBuf, fromAddr,store);
			return (rtx_error ("store_handle_set_var_value_request (%s): "
						"lookup_symbol() failed", varName));
		}
		varId = stEntry->varId;
	} else
		stEntry = store->vars[varId]->vEntry;
	store_debug (("store_handle_set_var_value_request: varId = %d", varId));
	if ((msgParm = get_message_part (keyTable, mp, MSG_TOK_VALUE)) == NULL) {
		store_send_error_response ("value not found",
				MSG_TOK_STORE_SET_VAR_VALUE_REPLY,
				txBuf, fromAddr,store);
		return (rtx_error ("store_handle_set_var_value_request (%s): unable to get "
					"var value", varName));
	}
	store_debug (("store_handle_set_var_value_request: varValue = %s", 
				msgParm));
	vInfo = store->vars[varId];
	svhp = (STORE_VAR_HEADER *) 
		&(store->shmemP->d[vInfo->headerOffset]);
	rtx_parse_scan_var (stEntry->v, vInfo->buf, msgParm, &lasts, 1);
	store_debug (("store_handle_set_var_value_request: varValue scanned"));
	if (store_wait_sem (vInfo->lockSemP) == -1) {
		return (rtx_error ("store_handle_set_var_value_request (%s): "
					"store_wait_sem() failed", varName));
	}
	d = &(store->shmemP->d[vInfo->offset]);	/* get offset after sem locked */
	svhp->ts = ts;
	store_queue_write_entry(svhp, d, vInfo->buf, stEntry->v->size);
	svhp->count++;

	if (store_post_sem (vInfo->lockSemP) == -1) {
		return (rtx_error ("store_handle_set_var_value_request (%s): "
					"store_post_sem() failed", varName));
	}

    /* Now let everyone know that there is data */
    if (store_semQ_add (store->semQ, STORE_SEM_Q_ACTION_POST,
                        varId, NULL) == -1) {
      return (rtx_error ("store_handle_set_var_value_request (%s): "
                         "store_semQ_add() failed", varName));
    }



	sprintf (txBuf, "%s %s = %s\n",
			keyTable[MSG_TOK_STORE_SET_VAR_VALUE_REPLY].name,
			keyTable[MSG_TOK_RESPONSE].name, "OK");
	if ((numChars = rtx_inet_write (fromAddr, txBuf, 
					strlen (txBuf), NULL)) == -1) {
		return (rtx_error ("store_handler_set_var_value_request (%s): "
					"rtx_inet_write() failed", varName));
	}
	return (0);
}

int 
store_handle_lookup_field_request (
		MSG * mp,

		char * rxBuf, 
		char * txBuf,
		RtxInetConn * fromAddr,
		STORE * store
		)
{
	int numChars;
	int varId = -1;
	char * varName;
	RtxParseVarTableEntry * stEntry = NULL, * myStEntry = NULL;
	RtxParseVar * v = NULL, * v1 = NULL;
	RtxParseToken curToken;
	int varStrIndex = 0;
	VAR_INFO_ITEM * vInfo;

	store_debug (("store_handle_lookup_field_request:"));
	if ((varName = get_message_part (keyTable, mp, MSG_TOK_NAME))
			== NULL) {
		store_send_error_response ("unable to extract var definition",
				MSG_TOK_STORE_SET_VAR_VALUE_REPLY,
				txBuf, fromAddr,store);
		return (rtx_error ("store_handle_lookup_field_request: unable to get "
					"var name"));
	}
	store_debug (("store_handle_lookup_field_request: name = %s", varName));
	if ((stEntry = (RtxParseVarTableEntry *) rtx_hash_find
	     (store->mySymTabs->varTable, varName, (rtx_error_t)NULL)) == NULL) {
		/* Not in table. First, parse the name */
		if ((v = calloc (1, sizeof (RtxParseVar))) == NULL) {
			store_send_error_response ("unable to allocate memory",
					MSG_TOK_STORE_LOOKUP_VAR_REPLY,
					txBuf, fromAddr,store);
			return (rtx_error_errno ("store_handle_lookup_field_request (%s): "
						"calloc() failed", varName));
		}
		if ((v1 = rtx_parse_name_list (varName, &varStrIndex, &curToken, v, store->mySymTabs))
				== NULL) {
			store_send_error_response ("name parse failed",
					MSG_TOK_STORE_LOOKUP_VAR_REPLY,
					txBuf, fromAddr,store);
			rtx_parse_free_var (v);
			return (rtx_error ("store_handle_lookup_field_request (%s): "
						"name not found", varName));
		}
		strcpy (v1->name, varName);
		if ((stEntry = (RtxParseVarTableEntry *) rtx_hash_find
		     (store->mySymTabs->varTable, v->name, (rtx_error_t)NULL)) == NULL) {
			store_send_error_response ("unable to lookup top-level var",
					MSG_TOK_STORE_LOOKUP_VAR_REPLY,
					txBuf, fromAddr,store);
			return (rtx_error_errno ("store_handle_lookup_field_request (%s): "
						"unable to lookup top-level var", v->name));
		}
		/* Have a valid parsed name. Now, add it to the tables. */
		if (rtx_mutex_lock (&store->mutex) == -1)
			rtx_error_flush ("store_handle_lookup_field_request (%s): "
					"rtx_mutex_lock() failed", varName);
		if ((varId = store_get_new_id (store)) == -1) {
			if (rtx_mutex_unlock (&store->mutex) == -1)
				rtx_error_flush ("store_handle_lookup_field_request (%s): "
						"rtx_mutex_unlock() failed", varName);
			return (rtx_error ("store_handle_lookup_field_request (%s): store_get_new_id() "
						"failed", v->name));
		}
		if (rtx_parse_add_var_symbol (v1, varId, stEntry->offset + v1->offset,
					stEntry->headerOffset, store->buf,
					store->mySymTabs->varTable) == -1) {
			if (rtx_mutex_unlock (&store->mutex) == -1)
				rtx_error_flush ("store_handle_lookup_field_request (%s): "
						"add_var_symbol() failed", varName);
			return (rtx_error ("store_handle_lookup_field_request (%s): add_var_symbol() "
						"failed", v->name));
		}
		if ((myStEntry = (RtxParseVarTableEntry *) rtx_hash_find
		     (store->mySymTabs->varTable, v1->name, (rtx_error_t)NULL)) == NULL) {
			if (rtx_mutex_unlock (&store->mutex) == -1)
				rtx_error_flush ("store_handle_lookup_field_request (%s): "
						"lookup_symbol() failed - serious problem, "
						"since the add just succeeded", varName);
			return (rtx_error ("store_handle_lookup_field_request (%s): "
						"lookup_symbol() failed - "
						"serious problem, since the add just succeeded",
						v->name));
		}
		vInfo = store->vars[stEntry->varId];
		if (store_add_var_info_item (myStEntry, varId, stEntry->offset + v1->offset,
					stEntry->headerOffset,
					NULL, 0, vInfo->lockSemP, store) == -1) {
			return (rtx_error ("store_create_var (%s): store_add_var_info_item() "
						"failed", v->name));
		}
		store_debug (("store_handle_lookup_field_request (%s): var added to store var table", 
					stEntry->v->name));
		if (rtx_mutex_unlock (&store->mutex) == -1)
			rtx_error_flush ("store_handle_lookup_field_request (%s): "
					"rtx_mutex_unlock() failed", varName);
		store->vars[varId]->flags |= STORE_VAR_ENTRY_FIELD;
		v1->offset = 0;
	} else {
		varId = stEntry->varId;
	}
	store_debug (("store_handle_set_var_value_request: varId = %d", varId));
	sprintf (txBuf, "%s %s = %d\n",
			keyTable[MSG_TOK_STORE_LOOKUP_FIELD_REPLY].name,
			keyTable[MSG_TOK_ID].name, varId);
	if ((numChars = rtx_inet_write (fromAddr, txBuf, 
					strlen (txBuf), NULL)) == -1) {
		return (rtx_error ("store_handler_lookup_field_request (%s): "
					"rtx_inet_write() failed", varName));
	}
	return (0);
}

int
store_handle_start_multicast_request (
		MSG * mp, char * rxBuf, char * txBuf,
		RtxInetConn * fromAddr,
		STORE * store
		)
{
	char * varName;
	RtxParseVarTableEntry * stEntry;
	int varId;
	VAR_INFO_ITEM * vInfo;
	STORE_VAR_HEADER * svhp;

	if ((varName = get_message_part (catKeyTable, mp, MSG_TOK_CATALOG_VAR_NAME))
			== NULL) {
		return (rtx_error ("store_handle_start_multicast_request: unable to extract "
					"var name"));
	}
	if ((stEntry = (RtxParseVarTableEntry *) rtx_hash_find
	     (store->mySymTabs->varTable, varName, (rtx_error_t)NULL)) == NULL) {
		return (rtx_error ("store_handle_start_multicast_request (%s): failed", 
					varName));
	}
	varId = stEntry->varId;
	vInfo = store->vars[varId];
	svhp = (STORE_VAR_HEADER *) 
		&(store->shmemP->d[vInfo->headerOffset]);
	svhp->multicastRequired = 1;
	return (0);
}

int
store_handle_stop_multicast_request (
		MSG * mp, char * rxBuf, char * txBuf,
		RtxInetConn * fromAddr,
		STORE * store
		)
{
	char * varName;
	RtxParseVarTableEntry * stEntry;
	int varId;
	VAR_INFO_ITEM * vInfo;
	STORE_VAR_HEADER * svhp;

	if ((varName = get_message_part (catKeyTable, mp, MSG_TOK_CATALOG_VAR_NAME))
			== NULL) {
		return (rtx_error ("store_handle_stop_multicast_request: unable to extract "
					"var name"));
	}
	if ((stEntry = (RtxParseVarTableEntry *) rtx_hash_find
	     (store->mySymTabs->varTable, varName, (rtx_error_t)NULL)) == NULL) {
		return (rtx_error ("store_handle_start_multicast_request (%s): failed",
					varName));
	}
	varId = stEntry->varId;
	vInfo = store->vars[varId];
	svhp = (STORE_VAR_HEADER *) 
		&(store->shmemP->d[vInfo->headerOffset]);
	svhp->multicastRequired = 0;
	return (0);
}

int
store_handle_set_var_queue_request (
		MSG * mp, char * rxBuf, char * txBuf,
		RtxInetConn * fromAddr,
		STORE * store
		)
{
	char * msgParm;
	int numChars;
	int varId = -1;
	char * varName;
	VAR_INFO_ITEM * vInfo = NULL;
	RtxParseVarTableEntry * stEntry = NULL;
	STORE_VAR_HEADER * svhp;
	STORE_TIMESPEC ts;
	int queue_length;
	int new_offset=-1;
	unsigned int queue_size;

	clock_gettime (CLOCK_REALTIME, (struct timespec *) &ts);
	store_debug (("store_handle_set_var_queue_request:"));
	if ((varName = get_message_part (keyTable, mp, MSG_TOK_NAME))
			== NULL) {
		store_send_error_response ("unable to extract var name",
				MSG_TOK_STORE_SET_VAR_QUEUE_REPLY,
				txBuf, fromAddr,store);
		return (rtx_error ("store_handle_set_var_queue_request: unable to get "
					"var name"));
	}
	store_debug (("store_handle_set_var_queue_request: name = %s", varName));
	/*
	if ((msgParm = get_message_part (keyTable, mp, MSG_TOK_ID)) != NULL) {
		varId = atoi (msgParm);
	}
	store_debug (("store_handle_set_var_queue_request: id = %s", msgParm));
	*/
	if (varId < 0) { /* need to lookup varId */
		if ((stEntry = (RtxParseVarTableEntry *) rtx_hash_find
		     (store->mySymTabs->varTable, varName, (rtx_error_t)NULL)) == NULL) {
			store_send_error_response ("name not found",
					MSG_TOK_STORE_SET_VAR_VALUE_REPLY,
					txBuf, fromAddr,store);
			return (rtx_error ("store_handle_set_var_queue_request (%s): "
						"lookup_symbol() failed", varName));
		}
		varId = stEntry->varId;
	} else
		stEntry = store->vars[varId]->vEntry;
	store_debug (("store_handle_set_var_queue_request: varId = %d", varId));
	if ((msgParm = get_message_part (keyTable, mp, MSG_TOK_VALUE)) == NULL) {
		store_send_error_response ("queue length not found",
				MSG_TOK_STORE_SET_VAR_VALUE_REPLY,
				txBuf, fromAddr,store);
		return (rtx_error ("store_handle_set_var_queue_request (%s): unable to get "
					"var value", varName));
	}
	queue_length = atoi (msgParm);
	store_debug (("store_handle_set_var_queue_request: queue length = %s", 
				msgParm));
	if (queue_length < 0) {
		store_send_error_response ("queue length invalid",
				MSG_TOK_STORE_SET_VAR_VALUE_REPLY,
				txBuf, fromAddr,store);
		return (rtx_error ("store_handle_set_var_queue_request (%s): queue length "
					"invalid", varName));
	}
	vInfo = store->vars[varId];
	svhp = (STORE_VAR_HEADER *) 
		&(store->shmemP->d[vInfo->headerOffset]);
	/* lock up the variable */
	if (store_wait_sem (vInfo->lockSemP) == -1) {
		return (rtx_error ("store_handle_set_var_queue_request (%s): "
					"store_wait_sem() failed", varName));
	}
	/* lock up the store */
	if (rtx_mutex_lock (&store->mutex) == -1)
	  rtx_error_flush ("store_handle_set_var_queue_request (%s): rtx_mutex_lock() failed", varName);

	/* only update if queue length has changed */
	if (svhp->queueLength != queue_length) {

	  /* free up the old data */
	  /* not done yet for simplicity */

	  /* allocate new data */
	  queue_size = queue_length * (sizeof(ts) + stEntry->v->size);
	  new_offset = store_alloc_mem (queue_size);
	  if (new_offset == -1) {
	    /* unlock the store */
	    if (rtx_mutex_unlock (&store->mutex) == -1)
	      rtx_error_flush ("store_handle_set_var_queue_request (%s): rtx_mutex_unlock() failed", varName);
	    /* unlock the variable */
	    if (store_post_sem (vInfo->lockSemP) == -1) {
	      return (rtx_error ("store_handle_set_var_queue_request (%s): "
				 "store_post_sem() failed", varName));
	    }
	    return (rtx_error ("store_handle_set_var_queue_request (%s): store_alloc_mem (var, %d) "
				    "failed", varName, queue_size));
	  }
	  store_debug (("store_handle_set_var_queue_request: new queue allocated. size = %d, offest=%d", 
			queue_size, new_offset));

	  /* reset counters */
	  svhp->count=0;
	  svhp->queueLength = queue_length;
	  
	  /* store the new offset */
	  svhp->shmOffset = new_offset;
	  vInfo->offset = new_offset;

      rtx_message("new shmOffset = %d",new_offset);
	  
	}
	/* unlock the store */
	if (rtx_mutex_unlock (&store->mutex) == -1)
	  rtx_error_flush ("store_handle_set_var_queue_request (%s): rtx_mutex_unlock() failed", varName);
	/* unlock the variable */
	if (store_post_sem (vInfo->lockSemP) == -1) {
		return (rtx_error ("store_handle_set_var_queue_request (%s): "
					"store_post_sem() failed", varName));
	}
	sprintf (txBuf, "%s %s = %s\n",
			keyTable[MSG_TOK_STORE_SET_VAR_QUEUE_REPLY].name,
			keyTable[MSG_TOK_RESPONSE].name, "OK");
	if ((numChars = rtx_inet_write (fromAddr, txBuf, 
					strlen (txBuf), NULL)) == -1) {
		return (rtx_error ("store_handler_set_var_queue_request (%s): "
					"rtx_inet_write() failed", varName));
	}
	return (0);
}
