
/***********************************************************************
 * catalog-common.c - ddx functions common to store and clients
 *
 * CSIRO MANUFACTURING SCIENCE & TECHNOLOGY
 * HIGH-SPEED SERIAL INTERFACE PROJECT
 * 
 * $Id: store-common.c 3068 2008-05-15 03:58:27Z roy029 $
 * 
 * Copyright (c) CSIRO Manufacturing Science & Technology.
 *
 ***********************************************************************
 */

#ifdef __GNUC__
#define DDX_STATIC_ATTRIBUTE __attribute__((unused))
#else
#define DDX_STATIC_ATTRIBUTE
#endif
static char *rcsid DDX_STATIC_ATTRIBUTE = "$Id: store-common.c 3068 2008-05-15 03:58:27Z roy029 $";

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <pthread.h>
#include <time.h>
#include <semaphore.h>
#ifndef i486_qnxrtp
#include <poll.h>
#else
#include <sys/time.h>
#endif

#include <rtx/message.h>
#include <rtx/error.h>
#include <rtx/time.h>

#include "store-main.h"
#include "store-msg.h"
#include "store-messages.h"
#include "store-ipc.h"
#include "store-client.h"
#include "store-common.h"

#include "ddx.h"

/**************************************************************************
 * send_buffer_tcp - send a buffer to the identified host through given TCP/IP s
ocket
 */
static int
send_buffer_tcp (
                 int sock,
                 char * buf,
                 int len
                 )
{
    int n = len, m;
    char * p = buf;

    while (n) {
        if ((m = write (sock, p, n)) <= 0) {
            return (-1);
        }
        n -= m;
        p += m;
    }
    return (len);
}

MSG *
store_get_response (
		    int sockfd,
		    char * recvBuf,
		    char * sendBuf,
		    int timeout,
		    DDX_STORE_ID * storeId,
		    KEYWORD_TABLE_ENTRY * keyTable,
		    MSG_TABLE_ENTRY * msgTable,
		    const char * srcStr,
		    int verbose
		    )
{
    int numChars;
    MSG * mp = NULL;
    int i, n;
    char ch;
#ifndef i486_qnxrtp
    struct pollfd fd[1] = {{0, 0, 0}};
#else
    struct timeval tv = {0, 0};
    fd_set myReadFdSet;
#endif
    int ret = -1;

    if (verbose)
        rtx_message_routine ("%s: store_get_response (timeout %d) [%s]", 
			     srcStr, timeout, sendBuf);
    if ((numChars = send_buffer_tcp (sockfd, sendBuf, 
				     strlen (sendBuf))) == -1) {
        return (rtx_error_errno_null ("%s: store_get_response: "
				      "send_buffer: %s", srcStr, sendBuf));
    }

#ifndef i486_qnxrtp
    fd[0].fd = sockfd;
    fd[0].events = POLLIN;
    if ((ret = poll (fd, 1, timeout * 1000)) == -1)
        return (rtx_error_errno_null ("%s: store_get_response: "
				      "poll() failed", srcStr));
    if (ret == 0)
        return (rtx_error_null ("%s: store_get_response: "
				"poll() timedout", srcStr));
#else
    FD_ZERO (&myReadFdSet);
    FD_SET (sockfd, &myReadFdSet);
    tv.tv_sec = timeout;
    if ((ret = select (sockfd+1, &myReadFdSet, NULL, NULL, &tv)) == -1)
        return (rtx_error_errno_null ("%s: store_get_response: "
				      "select() failed", srcStr));
    if (ret == 0)
        return (rtx_error_null ("%s: store_get_response: "
				"select() timedout", srcStr));
#endif
    /* there is a response from the other end, lets read it */

    i = 0; n = 1;
    while ((n > 0) && (i < (STORE_BUF_SIZE-1))) {
        if ((n = read (sockfd, &ch, 1)) < 1) {
		if (n < 0)
		    rtx_error_errno ("read failed");
		continue;
	}
	if (ch == '\n') {
	    recvBuf[i++] = '\0';
	    break;
	}
	recvBuf[i++] = ch;
    }
    if (n == -1)
        return (rtx_error_errno_null ("%s: store_get_response: "
				      "read() failed", srcStr));
    if (n == 0)
        return (rtx_error_errno_null ("%s: store_get_response: "
				      "client closed conn", srcStr));

    /* Have a complete request for processing */
    if (verbose)
        rtx_message ("%s: store_get_response: received %d chars, "
		     "reply = %s\n", srcStr, strlen (recvBuf),
		     recvBuf);

    if ((mp = parse_message (keyTable, recvBuf)) == NULL) {
        free_message (keyTable, mp);
        return (rtx_error_null ("%s: store_get_response: message "
				"parse failed [%s]", 
				srcStr, recvBuf));
    }
    if (identify_message (msgTable, mp) == -1) {
        free_message (keyTable, mp);
        return (rtx_error_null ("%s: store_get_response: invalid "
				"message [%s]", 
				srcStr, recvBuf));
    }
    return (mp);
}

int 
store_semQ_add (
		STORE_SEM_Q * semQ, 
		int action, 
		int varId,
		void * arg
		)
{
    int errs = 0;
    RtxTime ts;

    if (rtx_sem_wait_ignore_eintr (&(semQ->lockSem)) == -1) {
	return (rtx_error ("store_semQ_add: "
			   "rtx_sem_wait_ignore_eintr() failed"));
    }
    semQ->Q[semQ->nextEmptySlotInQ].varId = varId;
    semQ->Q[semQ->nextEmptySlotInQ].action = action;
    semQ->Q[semQ->nextEmptySlotInQ].arg = arg;
    rtx_time_get (&ts);
    semQ->Q[semQ->nextEmptySlotInQ].ts.tv_sec = ts.seconds;
    semQ->Q[semQ->nextEmptySlotInQ].ts.tv_nsec = ts.nanoSeconds;
    semQ->nextEmptySlotInQ++;
    if (semQ->nextEmptySlotInQ >= semQ->QSize)
	semQ->nextEmptySlotInQ = 0;
    if (semQ->nextEmptySlotInQ == semQ->nextVarInQ) {
        rtx_error ("store_semQ_add: sem Q full");
	errs++;
    }
    if (rtx_sem_post (&(semQ->lockSem)) == -1) {
        rtx_error ("store_semQ_add: rtx_sem_post(lockSem) failed");
	errs++;
    }
    if (rtx_sem_post (&(semQ->dataSem)) == -1) {
        rtx_error ("store_semQ_add: rtx_sem_post(dataSem) failed");
	errs++;
    }
    if (errs)
        return (-1);
    return (0);
}


/* writes data to the head of a queue, uses the timestamp from the header */
int store_queue_write_entry(STORE_VAR_HEADER *svhp, 
			     void *queue_data, void *entry_data, unsigned int entry_size)
{
  unsigned int queueLen =  svhp->queueLength;
  if (queueLen > 0) {
    unsigned int queueHead = svhp->count % queueLen;
    unsigned int ts_size = sizeof (svhp->ts);
    unsigned int size = entry_size + ts_size;
    /* copy queue element data */
    memcpy (((char*)queue_data) + queueHead*size, entry_data, entry_size);
    /* also copy the timestamp into the queue */
    memcpy (((char*)queue_data) + queueHead*size + entry_size, &(svhp->ts), ts_size);
  } else {
    memcpy (queue_data, entry_data, entry_size);
  }

  return queueLen;
}

/* reads data from the tail of a queue (does not check for overflow),
   also updates timestamp in header from queue entry */
int store_queue_read_entry(STORE_VAR_HEADER *svhp, 
			    void *queue_data, void *entry_data, unsigned int entry_size, 
			    int tailCount)
{
  unsigned int queueLen =  svhp->queueLength;
  if (queueLen > 0) {
    unsigned int queueInd = tailCount % queueLen;
    unsigned int ts_size = sizeof (svhp->ts);
    unsigned int size = entry_size + ts_size;
    /* copy queue element data */
    memcpy (entry_data, ((char*)queue_data) + queueInd*size, entry_size);
    /* also copy the timestamp into the queue */
    memcpy (&(svhp->ts), ((char*)queue_data) + queueInd*size + entry_size, ts_size);
    /* need to handle wraparound of count */
    if (svhp->count >= DDX_STORE_QUEUE_MAX_COUNT) {
      svhp->count -= (DDX_STORE_QUEUE_MAX_COUNT/queueLen)*queueLen;
    }
  } else {
    memcpy (entry_data, queue_data, entry_size);
  }

  return queueLen;
}

/* reads the lastest entry inserted into the queue */
int store_queue_read_current(STORE_VAR_HEADER *svhp, 
			    void *queue_data, void *entry_data, unsigned int entry_size)
{
  return store_queue_read_entry(svhp, queue_data, entry_data, entry_size, svhp->count-1);
}
