/*********************************************************************
 *
 * CSIRO Automation
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * www.cat.csiro.au/cmst
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 *********************************************************************/


static char *rcsid = "$Id: store-client.c 2256 2007-12-21 04:56:33Z rue006 $";

/**
 ********************************************************************
 *
 * \file store-client.c
 * \brief DDX client API
 * \author Pavan Sikka
 *
 ********************************************************************
 */                                                                                                                                  
/**
 * \page libddx libddx 
 *
 * The DDX library can be used by clients to communicate with the
 * store. \link store-client.c API details \endlink
 *
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <pthread.h>
#include <time.h>
#include <math.h>
#include <semaphore.h>

#include <rtx/message.h>
#include <rtx/sem.h>
#include <rtx/time.h>
#include <rtx/error.h>
#include <rtx/parse.h>
#include <rtx/mutex.h>
#include <rtx/thread.h>
#include <rtx/inet.h>

#include "store-msg.h"
#include "store-messages.h"
#include "store-ipc.h"
#include "store-client.h"
#include "store-common.h"
#include "store-arch.h"
#include "ddx.h"

const char *store_version = "$Revision: 2256 $ compiled " __DATE__ " " __TIME__;


#define ddx_client_debug(x) \
    if (storeClientVerbose) \
        rtx_message_routine x

static int storeClientVerbose = 0;
static pthread_once_t storeClientInitOnce = PTHREAD_ONCE_INIT;
static pthread_once_t storeClientExitOnce = PTHREAD_ONCE_INIT;
static int storeClientInited = 0;
static RtxParseVar * headerVarArray[STORE_ARCH_MAX_NUM];

static void
store_client_init_once_function (void)
{
	/* stupid thing to avoid a stupid unused warning */
	char * s; s = rcsid;
    if (store_init_shmem_table () == -1)
        return;
    storeClientInited = 1;
}

static void
store_client_exit_once_function (void)
{
    if (store_cleanup_ipc () == -1)
        return;
    storeClientInited = 0;
}

/**
 * Initialize shared memory data structures.
 * 
 * @return 0 on success, -1 on failure.
 *
 * This function should be called once by the client before any other calls
 * are made.
 */
int
ddx_client_init (
		 int verbose       /**< verbose level */
		 )
{
    pthread_once (&storeClientInitOnce, store_client_init_once_function);
    if (storeClientInited == 0)
        return (rtx_error ("ddx_client_init: store_init_shmem_table() failed"));
    storeClientVerbose = verbose;
    return (0);
}

/**
 * Destroy shared memory data structures.
 * 
 * @return 0 on success, -1 on failure.
 *
 * This function should be called once by the client after the connection to
 * the store has been closed the program is ready to terminate.
 */
int
ddx_client_done (void)
{
    pthread_once (&storeClientExitOnce, store_client_exit_once_function);
    if (storeClientInited != 0)
        return (rtx_error ("ddx_client_done: store_destroy_shmem_table() failed"));
    return (0);
}

static int ddx_store_init_header_array()
{
	int arch;
	for (arch=0;arch < STORE_ARCH_MAX_NUM; arch++) {
		RtxParseSymTabs *symTabs;
		RtxParseVar *v;
		int parseIndex;
		RtxParseToken t;
		
		symTabs = rtx_parse_init_remote(storeArchs[arch].dataAlign, storeArchs[arch].dataSize);

		parseIndex = 0;
		memset (&t, 0, sizeof (t));
		if ((v = calloc (1, sizeof (RtxParseVar))) == NULL)
			return (rtx_error_errno ("store_init_header: calloc() failed"));
		if (rtx_parse_decl (STORE_TIMESPEC_STR, &parseIndex, &t, v, symTabs) <= 0) {
			rtx_parse_free_var (v);
			return (rtx_error ("store_init_header: parse_decl "
						"(STORE_TIMESPEC_STR) failed"));
		}
		if (rtx_parse_add_type_symbol (v, 0, symTabs->typeTable,
					&(symTabs->typeNumber)) == -1) {
			return (rtx_error ("store_init_header: add_type (%s) failed",
						v->name));
		}

		parseIndex = 0;
		memset (&t, 0, sizeof (t));
		if ((v = calloc (1, sizeof (RtxParseVar))) == NULL)
			return (rtx_error_errno ("ddx_store_init_header_array: calloc() failed"));
		if (rtx_parse_decl (STORE_MULTICAST_HEADER_STR, &parseIndex, &t, v, symTabs) <= 0) {
			rtx_parse_free_var (v);
			return (rtx_error ("ddx_store_init_header_array: parse_decl "
						"(STORE_MULTICAST_HEADER_STR) failed"));
		}
		if (rtx_parse_add_type_symbol (v, 0, symTabs->typeTable,
					&(symTabs->typeNumber)) == -1) {
			return (rtx_error ("ddx_store_init_header_array: add_type (%s) failed",
						v->name));
		}
		parseIndex = 0;
		memset (&t, 0, sizeof (t));
		if ((v = calloc (1, sizeof (RtxParseVar))) == NULL)
			return (rtx_error_errno ("store_init_header: calloc() failed"));
		if (rtx_parse_decl (STORE_MULTICAST_HEADER_VAR, &parseIndex, &t, v, symTabs) <= 0) {
			rtx_parse_free_var (v);
			return (rtx_error ("store_init_header: parse_decl "
						"(STORE_MULTICAST_HEADER_VAR) failed"));
		}
		if (rtx_parse_expand_var_definition (v, symTabs->typeTable) == -1) {
			rtx_parse_free_var (v);
			return (rtx_error ("store_init_header: expand_var_definition "
						"(%s) failed", v->name));
		}
		headerVarArray[arch] = v;
		rtx_parse_finish(symTabs);
	}
	return 0;
}

static int ddx_store_terminate_header_array()
{
	int arch;
	for (arch=0;arch < STORE_ARCH_MAX_NUM; arch++) {
		rtx_parse_free_var(headerVarArray[arch]);
	}
	return 0;
}

/**
 * Open a connection the the store running on the local machine.
 *
 * @return A store handle on success, NULL on failure.
 */
DDX_STORE_ID * 
ddx_store_open (
		const char * storeName,        /**< ignored for now */
		int storePortNumber,     /**< store port number */
		int tmout                /**< timeout for the open request */
		)
{
    char * msgval;
    char * shmemName;
    int shmemSize;
    char * semPoolName;
    int semPoolSize;
    int storePid;
    char name[64];
    char locStoreName[RTX_INET_MAX_HOST_NAME_LEN];
    MSG * mp;
    DDX_STORE_ID * storeId;
    unsigned int mySwapWord = 0x01020304;

    fprintf(stderr,"ddx_store_open(): %s\n", store_version);

    if ((storeId = (DDX_STORE_ID *) calloc (1, sizeof (DDX_STORE_ID))) 
	    == NULL)
        return (rtx_error_errno_null ("ddx_store_open: calloc() failed"));

    if ((storeId->mutex = rtx_mutex_init (NULL, 0, 0)) == NULL)
        return (rtx_error_null ("ddx_store_open: rtx_mutex_init() failed"));
      
    memcpy (&(storeId->myArch.swapword), (unsigned char *) &mySwapWord, 
	    sizeof (mySwapWord));
    rtx_parse_get_compiler_parms (storeId->myArch.dataSize, storeId->myArch.dataAlign);
    if ((storeId->myArchType = store_get_arch (&storeId->myArch)) == -1)
        return (rtx_error_errno_null ("ddx_store_open: store_get_arch() failed"));

    storeId->verbose = storeClientVerbose;
    if (gethostname (locStoreName, RTX_INET_MAX_HOST_NAME_LEN) == -1)
        return (rtx_error_errno_null ("ddx_store_open: gethostname() failed"));
    if (storePortNumber == 0)
        storePortNumber = STORE_PORT_NUM;
    if ((storeId->server = rtx_inet_init (RTX_INET_TCP_CLIENT, NULL, 0, 
					    locStoreName, storePortNumber,
					    NULL, NULL, NULL)) == NULL) {
	free (storeId);
        return (rtx_error_null ("ddx_store_open: rtx_inet_init() failed"));
    }
    sprintf (storeId->buf, "%s\n", keyTable[MSG_TOK_STORE_OPEN_REQUEST].name);
    if ((mp = store_get_response (storeId->server->sock->sockfd, 
				  storeId->buf1, storeId->buf, 5, storeId,
				  keyTable, msgTable, "ddx_store_open",
				  storeId->verbose)) == NULL) {
   		free (storeId);
        return (rtx_error_null ("ddx_store_open: store timed out (5s)"));
    }
    if (mp->id != MSG_STORE_OPEN_REPLY) {
		free (storeId); free_message(keyTable,mp);
        return (rtx_error_null ("ddx_store_open: invalid/error response"
				" from store"));
    }
    if ((shmemName = get_message_part (keyTable, mp, MSG_TOK_SHMEM_NAME))
	    == NULL) {
		free (storeId); free_message(keyTable,mp);
        return (rtx_error_null ("ddx_store_open: unable to get shmem name"
				" from store"));
    }
    if ((msgval = get_message_part (keyTable, mp, MSG_TOK_SHMEM_SIZE))
	    == NULL) {
		free (storeId); free_message(keyTable,mp);
        return (rtx_error_null ("ddx_store_open: unable to get shmem size"
				" from store"));
    }
    shmemSize = atoi (msgval);
    if (shmemSize <= 0) {
		free (storeId); free_message(keyTable,mp);
        return (rtx_error_null ("ddx_store_open: invalid shmem size (%d)"
				" from store", shmemSize));
    }
    if ((semPoolName = get_message_part (keyTable, mp, MSG_TOK_SEM_POOL_NAME))
	    == NULL) {
		free (storeId); free_message(keyTable,mp);
        return (rtx_error_null ("ddx_store_open: unable to get sem pool name"
				" from store"));
    }
    if ((msgval = get_message_part (keyTable, mp, MSG_TOK_SEM_POOL_SIZE))
	    == NULL) {
		free (storeId); free_message(keyTable,mp);
        return (rtx_error_null ("ddx_store_open: unable to get sem pool size"
				" from store"));
    }
    semPoolSize = atoi (msgval);
    if (shmemSize <= 0) {
		free (storeId); free_message(keyTable,mp);
        return (rtx_error_null ("ddx_store_open: invalid sem pool size (%d)"
				" from store", semPoolSize));
    }
    if ((msgval = get_message_part (keyTable, mp, MSG_TOK_STORE_PID))
	    == NULL) {
		free (storeId); free_message(keyTable,mp);
        return (rtx_error_null ("ddx_store_open: unable to get process id"
				" from store"));
    }
    storePid = atoi (msgval);
    if (storePid <= 0) {
		free (storeId); free_message(keyTable,mp);
        return (rtx_error_null ("ddx_store_open: invalid process id (%d)"
				" from store", storePid));
    }
    storeId->pid = storePid;
    ddx_client_debug (("ddx_store_open: store pid = %d", storeId->pid));
    sprintf (name, "/%s-%d", shmemName, storePid);
    if ((storeId->shmemP = store_link_shmem (name, 0, shmemSize))
	    == NULL) {
		free (storeId); free_message(keyTable,mp);
        return (rtx_error_null ("ddx_store_open: store_link_shmem (%s, %d)"
				"failed", name, shmemSize));
    }
    sprintf (name, "/%s-%d", STORE_SEM_Q_NAME, storePid);
    if ((storeId->semQShmemP = store_link_shmem (name, 0, 
						 sizeof (STORE_SEM_Q)))
	    == NULL) {
		free (storeId); free_message(keyTable,mp);
        return (rtx_error_null ("ddx_store_open: store_link_shmem (%s, %d)"
				"failed", name, sizeof (STORE_SEM_Q)));
    }
    storeId->semQ = (STORE_SEM_Q *) storeId->semQShmemP->d;
    storeId->magicNum = DDX_MAGIC_NUM;
	free_message(keyTable,mp);

	ddx_store_init_header_array();

    return (storeId);
}

/**
 * Close the connection to the store.
 *
 * @return 0 on success, -1 on failure.
 */
int
ddx_store_close (
		 DDX_STORE_ID * storeId     /**< store handle */
		 )
{
    int errs = 0;

	ddx_store_terminate_header_array();

    if (storeId == NULL)
        return (rtx_error ("ddx_store_close: NULL handle"));
    if (storeId->magicNum != DDX_MAGIC_NUM)
        return (rtx_error ("ddx_store_close: invalid handle"));

    if (rtx_inet_done (storeId->server) == -1) {
        errs++;
        rtx_error ("ddx_store_close: rtx_inet_done() failed");
    }
    /*
    if (store_unlink_shmem (storeId->shmemP) == -1) {
        errs++;
        rtx_error ("ddx_store_close: store_unlink_shmem() failed");
    }
    */
    if (rtx_mutex_destroy (storeId->mutex) == -1) {
        errs++;
        rtx_error ("ddx_store_close: rtx_mutex_destroy() failed");
    }
    if (errs)
        return (-1);
    return (0);
}

/* General */

/**
 * Register a constant with the store.
 *
 * @return 0 on success, -1 on failure.
 */
int 
ddx_store_register_const (
			  DDX_STORE_ID * storeId,   /**< store handle */
			  const char * name,              /**< constant name */
			  int val                   /**< constant value */
			  )
{
    MSG * mp;
	int mpid;

    if (storeId == NULL)
        return (rtx_error ("ddx_store_register_const: NULL handle"));
    if (storeId->magicNum != DDX_MAGIC_NUM)
        return (rtx_error ("ddx_store_register_const: invalid handle"));
    if (rtx_mutex_lock (storeId->mutex) == -1)
        return (rtx_error ("ddx_store_register_const: rtx_mutex_lock() failed"));
    sprintf (storeId->buf, "%s %s = %s %s = %d\n", 
	     keyTable[MSG_TOK_STORE_REGISTER_CONST_REQUEST].name,
	     keyTable[MSG_TOK_NAME].name, name,
	     keyTable[MSG_TOK_VALUE].name, val);
    if ((mp = store_get_response (storeId->server->sock->sockfd, storeId->buf1, storeId->buf, 5, storeId, keyTable, msgTable,
				  "ddx_store_register_const", storeId->verbose))
	    == NULL)
        return (rtx_error ("ddx_store_register_const: store timed out (5s)"
			   " for %s (value = %d)", name, val));
	mpid = mp->id;
	free_message(keyTable,mp);
    if (rtx_mutex_unlock (storeId->mutex) == -1)
        return (rtx_error ("ddx_store_register_const: rtx_mutex_unlock() failed"));
    if (mpid != MSG_STORE_REGISTER_CONST_REPLY)
        return (rtx_error ("ddx_store_register_const: invalid/error response"
			   " from store for %s (value = %d)", name, val));
    return (0);
}

/**
 * Register a type with the store.
 *
 * @return 0 on success, -1 on failure.
 */
int 
ddx_store_register_type (
			 DDX_STORE_ID * storeId,    /**< store handle */
			 const char * type                /**< type definition */
			 )
{
    MSG * mp;
	int mpid;

    if (storeId == NULL)
        return (rtx_error ("ddx_store_register_type: NULL handle"));
    if (storeId->magicNum != DDX_MAGIC_NUM)
        return (rtx_error ("ddx_store_register_type: invalid handle"));
    if (rtx_mutex_lock (storeId->mutex) == -1)
        return (rtx_error ("ddx_store_register_type: rtx_mutex_lock() failed"));
    sprintf (storeId->buf, "%s %s = %s\n", 
	     keyTable[MSG_TOK_STORE_REGISTER_TYPE_REQUEST].name,
	     keyTable[MSG_TOK_TYPE_DEFINITION].name, type);
    if ((mp = store_get_response (storeId->server->sock->sockfd, storeId->buf1, storeId->buf, 5, storeId, keyTable, msgTable,
				  "ddx_store_register_type", storeId->verbose))
	    == NULL)
        return (rtx_error ("ddx_store_register_type: store timed out (5s)"
			   " for type [%s]", type));
	mpid = mp->id;
	free_message(keyTable,mp);
    if (rtx_mutex_unlock (storeId->mutex) == -1)
        return (rtx_error ("ddx_store_register_type: rtx_mutex_unlock() failed"));
    if (mpid != MSG_STORE_REGISTER_TYPE_REPLY)
        return (rtx_error ("ddx_store_register_type: invalid/error response"
			   " from store for type [%s]", type));
    return (0);
}

/**
 * Lookup an item in the store.
 *
 * @return Item handle on success, NULL on failure.
 *
 * This function is used by clients to lookup data items in the store. This
 * function creates the item in the store if the item doesnt already exist.
 *
 * The parameters varType and varSize are optional and can be specified as
 * NULL and 0, respectively. The varSize parameter is used simply as an
 * additional check on the validity of the item being looked up. The varType
 * parameter is also used as a check when it is specified. If the item does
 * not exist in the store then the varType item must be specified, otherwise
 * the function returns with an error.
 *
 * The varDecl parameter has different semantics based on the varType
 * parameter. If varType is set to NULL, then varDecl is used as the name
 * of the item to be looked up in the store. The syntax for this supports
 * sub-structures and array subscripts, thus allowing the user to lookup
 * parts of an item.
 *
 * If varType is not NULL then varType and varDecl are combined to provide
 * a 'C'-style declaration of the item being looked up.
 *
 */
DDX_STORE_ITEM * 
ddx_store_lookup_item (
		       DDX_STORE_ID * storeId,     /**< store handle */
		       const char * varDecl,             /**< name of the item */
		       const char * varType,             /**< type of the item */
		       int varSize                 /**< size of the item */
		       )
{
	MSG * mp;
	DDX_STORE_ITEM * itemP;
	int offset, hdrOffset, varId, storeVarSize, syncSemId, multiPort;
	char * msgval, * multiAddr;

	if (storeId == NULL)
		return (rtx_error_null ("ddx_store_lookup_item: NULL handle"));
	if (storeId->magicNum != DDX_MAGIC_NUM)
		return (rtx_error_null ("ddx_store_lookup_item: invalid handle"));
	if (rtx_mutex_lock (storeId->mutex) == -1)
		return (rtx_error_null ("ddx_store_lookup_item: rtx_mutex_lock() failed"));
	if (varType == NULL)
		sprintf (storeId->buf, "%s %s = %s\n", 
				keyTable[MSG_TOK_STORE_LOOKUP_VAR_REQUEST].name,
				keyTable[MSG_TOK_VAR_DEFINITION].name, varDecl);
	else
		sprintf (storeId->buf, "%s %s = %s %s = %s\n", 
				keyTable[MSG_TOK_STORE_LOOKUP_VAR_REQUEST].name,
				keyTable[MSG_TOK_VAR_DEFINITION].name, varDecl,
				keyTable[MSG_TOK_TYPE_DEFINITION].name, varType);
	if ((mp = store_get_response (storeId->server->sock->sockfd, storeId->buf1, storeId->buf, 5, storeId, keyTable, msgTable,
					"ddx_store_lookup_item", storeId->verbose))
			== NULL) {
		if (rtx_mutex_unlock (storeId->mutex) == -1) 
			rtx_error_flush ("ddx_store_lookup_item: rtx_mutex_unlock() failed");

		return (rtx_error_null ("ddx_store_lookup_item (%s): no response "
					"from store (5s)", varDecl));
	}
	if (rtx_mutex_unlock (storeId->mutex) == -1) {
		free_message(keyTable,mp);
		return (rtx_error_null ("ddx_store_lookup_item: rtx_mutex_unlock() failed"));
	}
	if (mp->id != MSG_STORE_LOOKUP_VAR_REPLY) {
		free_message(keyTable,mp);
		return (rtx_error_null ("ddx_store_lookup_item (%s): error/invalid"
					" response from store", varDecl));
	}

	if ((itemP = calloc (1, sizeof (DDX_STORE_ITEM))) == NULL) {
		free_message(keyTable,mp);
		return (rtx_error_errno_null ("ddx_store_lookup_item (%s): "
					"calloc(DDX_STORE_ITEM[%d]) failed", varDecl,
					sizeof (DDX_STORE_ITEM)));
	}

	if ((itemP->mutex = rtx_mutex_init (NULL, 0, 0)) == NULL) {
		free (itemP);
		free_message(keyTable,mp);
		return (rtx_error_null ("ddx_store_lookup_item (%s): rtx_mutex_init() failed"));
	}

	if ((msgval = get_message_part (keyTable, mp, MSG_TOK_SHMEM_VAR_OFFSET))
			== NULL) {
		free (itemP);
		free_message(keyTable,mp);
		return (rtx_error_null ("ddx_store_lookup_item (%s): unable to get "
					"shmem var offset from store", varDecl));
	}
	if ((offset = atoi (msgval)) < 0) {
		free (itemP);
		free_message(keyTable,mp);
		return (rtx_error_null ("ddx_store_lookup_item (%s): invalid shmem "
					"var offset (%d) from store", varDecl, offset));
	}
	if ((msgval = get_message_part (keyTable, mp, 
					MSG_TOK_SHMEM_VAR_HEADER_OFFSET)) == NULL) {
		free (itemP);
		free_message(keyTable,mp);
		return (rtx_error_null ("ddx_store_lookup_item (%s): unable to get "
					"shmem var header offset from store", varDecl));
	}
	if ((hdrOffset = atoi (msgval)) < 0) {
		free (itemP);
		free_message(keyTable,mp);
		return (rtx_error_null ("ddx_store_lookup_item (%s): invalid shmem "
					"var header offset (%d) from store", varDecl,
					hdrOffset));
	}
	if ((msgval = get_message_part (keyTable, mp, 
					MSG_TOK_SHMEM_VAR_ID)) == NULL) {
		free (itemP);
		free_message(keyTable,mp);
		return (rtx_error_null ("ddx_store_lookup_item (%s): unable to get "
					"shmem var id from store", varDecl));
	}
	if ((varId = atoi (msgval)) < 0) {
		free (itemP);
		free_message(keyTable,mp);
		return (rtx_error_null ("ddx_store_lookup_item (%s): invalid shmem "
					"var id (%d) from store", varDecl, varId));
	}
	if ((msgval = get_message_part (keyTable, mp, 
					MSG_TOK_SHMEM_VAR_SIZE)) == NULL) {
		free (itemP);
		free_message(keyTable,mp);
		return (rtx_error_null ("ddx_store_lookup_item (%s): unable to get "
					"shmem var size from store", varDecl));
	}
	if ((storeVarSize = atoi (msgval)) < 1) {
		free (itemP);
		free_message(keyTable,mp);
		return (rtx_error_null ("ddx_store_lookup_item (%s): invalid shmem "
					"var size (%d) from store", varDecl, 
					storeVarSize));
		return (rtx_error_null ("ddx_store_lookup_item (%s): ", varDecl));
	}
	if (varSize)
		if (varSize != storeVarSize) {
			free (itemP);
			free_message(keyTable,mp);
			return (rtx_error_null ("ddx_store_lookup_item (%s): var size from "
						"store (%d) disagrees with var size "
						"provided (%d)", varDecl, storeVarSize,
						varSize));
		}
	if ((msgval = get_message_part (keyTable, mp, 
					MSG_TOK_VAR_DEFINITION)) == NULL) {
		free (itemP);
		free_message(keyTable,mp);
		return (rtx_error_null ("ddx_store_lookup_item (%s): unable to get "
					"shmem var definition from store", varDecl));
	}
	if ((itemP->varDefinition = strdup (msgval)) == NULL) {
		free (itemP);
		free_message(keyTable,mp);
		return (rtx_error_null ("ddx_store_lookup_item (%s): strdup(varDefn) "
					"failed", varDecl));
	}
	if ((msgval = get_message_part (keyTable, mp, 
					MSG_TOK_VAR_MULTICAST_PORT)) == NULL) {
		free (itemP);
		free_message(keyTable,mp);
		return (rtx_error_null ("ddx_store_lookup_item (%s): unable to get "
					"shmem var multicast port from store", varDecl));
	}
	if ((multiPort = atoi (msgval)) < 1) {
		free (itemP);
		free_message(keyTable,mp);
		return (rtx_error_null ("ddx_store_lookup_item (%s): invalid multicast "
					"port (%d) from store", varDecl, 
					multiPort));
	}
	if ((msgval = get_message_part (keyTable, mp, 
					MSG_TOK_VAR_MULTICAST_ADDRESS)) == NULL) {
		free (itemP);
		free_message(keyTable,mp);
		return (rtx_error_null ("ddx_store_lookup_item (%s): unable to get "
					"shmem var muticast address from store", varDecl));
	}
	multiAddr = msgval;
	if ((msgval = get_message_part (keyTable, mp, 
					MSG_TOK_NAME)) == NULL) {
		free (itemP);
		free_message(keyTable,mp);
		return (rtx_error_null ("ddx_store_lookup_item (%s): unable to get "
					"shmem var offset from store", varDecl));
	}
	if ((itemP->multicastPort = rtx_inet_init (RTX_INET_UDP_MULTICAST,
					NULL, 0, multiAddr,
					multiPort, NULL, NULL,
					NULL)) == NULL) {
		free (itemP);
		free_message(keyTable,mp);
		return (rtx_error_null ("ddx_store_lookup_item (%s): rtx_inet_init() "
					"failed", varDecl));
	}
	if (rtx_inet_set_multicast_ttl (itemP->multicastPort, 10) == -1)
		rtx_error_flush ("ddx_store_lookup_item: "
				"rtx_inet_set_multicast_ttl"
				" failed (%s)", varDecl);
	if ((itemP->varName = strdup (msgval)) == NULL) {
		free (itemP);
		free_message(keyTable,mp);
		return (rtx_error_null ("ddx_store_lookup_item (%s): strdup(varName) "
					"failed", varDecl));
	}
	itemP->store = storeId;
	itemP->headerP = (STORE_VAR_HEADER *) &(storeId->shmemP->d[hdrOffset]);
	itemP->varId = varId;
	itemP->varSize = storeVarSize;
	itemP->lockSem.id = itemP->headerP->lockSemId;
	itemP->lockSem.semid = itemP->headerP->lockSemSysVId;
	itemP->lockSem.pid = storeId->pid;
	if ((itemP->d = ddx_store_var_pointer(itemP)) == NULL) {
		return rtx_error_null("ddx_store_lookup_item (%s): failed to get item pointer",itemP->varName);
	}
	if (store_link_sem (&(itemP->lockSem)) == -1) {
		free (itemP);
		free_message(keyTable,mp);
		return (rtx_error_null ("ddx_store_lookup_item (%s): store_link_sem"
					"(lockSem) failed", varDecl));
	}
	if ((msgval = get_message_part (keyTable, mp, 
					MSG_TOK_SYNC_SEM_ID)) == NULL) {
		free (itemP);
		free_message(keyTable,mp);
		return (rtx_error_null ("ddx_store_lookup_item (%s): unable to get "
					"var sync sem id from store", varDecl));
	}
	if ((syncSemId = atoi (msgval)) < 0) {
		free (itemP);
		free_message(keyTable,mp);
		return (rtx_error_null ("ddx_store_lookup_item (%s): invalid var sync "
					"sem id (%d) from store", varDecl, syncSemId));
	}
	itemP->syncSem.id = syncSemId;

	/** Request the sysv part, if needed **/
	if ((msgval = get_message_part (keyTable, mp, 
					MSG_TOK_SYNC_SYSVSEM_ID)) == NULL) {
		free (itemP);
		free_message(keyTable,mp);
		return (rtx_error_null ("ddx_store_lookup_item (%s): unable to get "
					"var sync sysv sem id from store", varDecl));
	}
	if ((syncSemId = atoi (msgval)) < 0) {
		free (itemP);
		free_message(keyTable,mp);
		return (rtx_error_null ("ddx_store_lookup_item (%s): invalid var sync "
					"sysv sem id (%d) from store", varDecl, syncSemId));
		return (rtx_error_null ("ddx_store_lookup_item (%s): ", varDecl));
	}
	itemP->syncSem.semid = syncSemId;

	itemP->syncSem.pid = storeId->pid;
	if (store_link_sem (&(itemP->syncSem)) == -1) {
		free (itemP);
		free_message(keyTable,mp);
		return (rtx_error_null ("ddx_store_lookup_item (%s): store_link_sem"
					"(syncSem) failed", varDecl));
	}
	itemP->magicNum = DDX_MAGIC_NUM;
	free_message(keyTable,mp);
	return (itemP);
}

/**
 * Indicate to the store that the item is no longer of interest.
 *
 * @return 0 on success, -1 on failure.
 *
 * This function is used by clients to indicate to the store that the item
 * is no longer of interest. This allows the store/catalog to free up resources
 * allocated for the corresponding lookup() request. This function also cleans
 * up at the client end.
 *
 */
int 
ddx_store_done_item (
		     DDX_STORE_ITEM * itemP        /**< item handle */
		     )
{
    MSG * mp = NULL;
	int mpid;
    DDX_STORE_ID * storeId;
    int errs = 0;

    if (itemP == NULL)
        return (rtx_error ("ddx_store_done_item: NULL handle"));
    if (itemP->magicNum != DDX_MAGIC_NUM)
        return (rtx_error ("ddx_store_done_item: invalid handle"));

    rtx_inet_done(itemP->multicastPort);
    
    storeId = itemP->store;
    if (rtx_mutex_lock (storeId->mutex) == -1)
        return (rtx_error ("ddx_store_done_item: rtx_mutex_lock() failed"));
    ddx_client_debug (("ddx_store_done_item (%s): sending request to store %s",
		       itemP->varName, rtx_inet_print_net_addr (&(storeId->server->sock->remote))));
    sprintf (storeId->buf, "%s %s = %s %s = %d\n", 
	     keyTable[MSG_TOK_STORE_DONE_VAR_REQUEST].name,
	     keyTable[MSG_TOK_NAME].name, itemP->varName,
	     keyTable[MSG_TOK_SYNC_SEM_ID].name, itemP->syncSem.id);
    ddx_client_debug (("ddx_store_done_item: request = %s", storeId->buf));
    if ((mp = store_get_response (storeId->server->sock->sockfd, storeId->buf1, storeId->buf, 5, storeId, keyTable, msgTable,
				  "ddx_store_done_item", storeId->verbose)) 
	    == NULL)
        return (rtx_error ("ddx_store_done_item (%s): no response (5s) from "
			   "store", itemP->varName));
	mpid = mp->id;
	free_message(keyTable,mp);
    if (rtx_mutex_unlock (storeId->mutex) == -1)
        return (rtx_error ("ddx_store_done_item: rtx_mutex_unlock() failed"));
    if (mpid != MSG_STORE_DONE_VAR_REPLY)
        return (rtx_error ("ddx_store_done_item (%s): error/incvalid response"
			   " from store", itemP->varName));
    if (rtx_mutex_destroy (itemP->mutex) == -1) {
        errs++;
	rtx_error ("ddx_store_done_item (%s): rtx_mutex_destroy() failed");
    }
    if (errs)
        return (-1);
    return (0);
}

/**
 * Indicate to the store that the item should be queued.
 *
 * @return 0 on success, -1 on failure.
 *
 * This function is used by clients to indicate to the store that the
 * item should be queued.  The store will reallocate the shared memory
 * to accomodate the queue size.
 *
 */
int 
ddx_store_item_set_queue (
		     DDX_STORE_ITEM * itemP,        /**< item handle */
		     int queueLength /**< the number of desired entries in the queue  */
		     )
{
    MSG * mp = NULL;
	int mpid;
    DDX_STORE_ID * storeId;
    int errs = 0;

    if (itemP == NULL)
        return (rtx_error ("ddx_store_item_set_queue: NULL handle"));
    if (itemP->magicNum != DDX_MAGIC_NUM)
        return (rtx_error ("ddx_store_item_set_queue: invalid handle"));

    storeId = itemP->store;
    if (rtx_mutex_lock (storeId->mutex) == -1)
        return (rtx_error ("ddx_store_item_set_queue: rtx_mutex_lock() failed"));
    ddx_client_debug (("ddx_store_item_set_queue (%s): sending request to store %s",
		       itemP->varName, rtx_inet_print_net_addr (&(storeId->server->sock->remote))));
    sprintf (storeId->buf, "%s %s = %s %s = %d\n", 
	     keyTable[MSG_TOK_STORE_SET_VAR_QUEUE_REQUEST].name,
	     keyTable[MSG_TOK_NAME].name, itemP->varName,
	     keyTable[MSG_TOK_VALUE].name, queueLength);
    ddx_client_debug (("ddx_store_item_set_queue: request = %s", storeId->buf));
    if ((mp = store_get_response (storeId->server->sock->sockfd, storeId->buf1, storeId->buf, 5, storeId, keyTable, msgTable,
				  "ddx_store_item_set_queue", storeId->verbose)) 
	    == NULL)
        return (rtx_error ("ddx_store_item_set_queue (%s): no response (5s) from "
			   "store", itemP->varName));
	mpid = mp->id;
	free_message(keyTable,mp);
    if (rtx_mutex_unlock (storeId->mutex) == -1)
        return (rtx_error ("ddx_store_item_set_queue: rtx_mutex_unlock() failed"));
    if (mpid != MSG_STORE_SET_VAR_QUEUE_REPLY)
        return (rtx_error ("ddx_store_item_set_queue (%s): error/incvalid response"
			   " from store", itemP->varName));
    if (errs)
        return (-1);
    return (0);
}

/**
 * Return the pointer to the share memory for direct access
 *
 * @return a valid pointer on success, NULL on failure.
 */
void* ddx_store_var_pointer(DDX_STORE_ITEM *storeItem)
{
	if (!storeItem) {
		return rtx_error_null("ddx_store_var_pointer: uninitialised item");
	}
	if (!(storeItem->store)) {
		return rtx_error_null("ddx_store_var_pointer: item->store should never be NULL");
	}
	if (storeItem->headerP->shmOffset + storeItem->varSize*storeItem->headerP->queueLength >= 
			storeItem->store->shmemP->size) {
		return rtx_error_null("ddx_store_var_pointer: item '%s' shared memory pointer have been corrupted",
				storeItem->varName);
	}
	return (void *) &(storeItem->store->shmemP->d[storeItem->headerP->shmOffset]);
}

/**
 * Read the specified item from the store.
 *
 * @return 0 on success, -1 on failure.
 *
 * This function is used by clients to read an item from the store. The
 * parameter skipCount moderates the behavior of the function as follows:
 *
 * (1) skipCount == 0
 *     The function works is an asynchronous manner. It gets a timestamp
 *     and then reads the data from the store. If the timestamp in the 
 *     store is older than the current timestamp by an amount greater
 *     than the parameter timeArg (staleness), it returns 1, otherwise
 *     it returns 0. If the parameter d is NULL, no data is copied. If
 *     the parameter ts is NULL, the timestamp from the store is not
 *     copied.
 * (2) skipCount > 0
 *     The function works is a synchronous manner. It waits for the semaphore
 *     skipCount times before proceeding to read the data from the store.
 *     If the parameter d is NULL, no data is copied. If
 *     the parameter ts is NULL, the timestamp from the store is not
 *     copied. If timeArg > 0, it is used as a timeout. The function returns
 *     1, if no new value is available withing timeout seconds. 
 *     If timeArg <= 0, an infinite blocking read is performed.
 *
 */
int 
ddx_store_read (
		DDX_STORE_ITEM * itemP,    /**< item handle */
		void * d,                  /**< data buffer */
		RtxTime * ts,              /**< timestamp buffer */
		double timeArg,            /**< time argument, staleness if
					        skipCount == 0, else timeout
						(currently not used) */
		int skipCount              /**< number of data cycles to skip */
		)
{
	RtxTime tsnow, timeout;
	double tdiff;
	int i = 0, oldCancel = -1;
	rtx_time_from_double(&timeout,timeArg);

	if (itemP == NULL)
		return (rtx_error ("ddx_store_read: NULL handle"));
	if (itemP->magicNum != DDX_MAGIC_NUM)
		return (rtx_error ("ddx_store_read: invalid handle"));
	rtx_time_get (&tsnow);
	if (skipCount < 0)
		return (rtx_error ("ddx_store_read (%s): invalid skip count (%d)",
					itemP->varName, skipCount));
	while (i++ < skipCount) {
		int ret = 0;
		if (timeArg>0) 
			ret = store_timed_wait_sem (&(itemP->syncSem),&timeout);
		else
			ret = store_wait_sem(&(itemP->syncSem));
		switch (ret) {
			case -1:
				return (rtx_error ("ddx_store_read (%s): store_wait_sem(syncSem)"
							" failed", itemP->varName));
			case +1:
				return (1);
			default :
				break;
		}
	}
	oldCancel = rtx_thread_setcancel (RTX_THREAD_CANCEL_DISABLE);
	if (store_wait_sem (&(itemP->lockSem)) == -1) {
		rtx_thread_setcancel (oldCancel);
		return (rtx_error ("ddx_store_read (%s): store_wait_sem(lockSem) failed", itemP->varName));
	}
	/* make sure that memory pointer is uptodate */
	if ((itemP->d = ddx_store_var_pointer(itemP)) == NULL) {
		return rtx_error("ddx_store_read (%s): failed to get item pointer");
	}

	if (d != NULL)
		store_queue_read_current(itemP->headerP, itemP->d, d, itemP->varSize);
	if (ts != NULL)
		memcpy (ts, &(itemP->headerP->ts), sizeof (* ts));
	if (store_post_sem (&(itemP->lockSem)) == -1) {
		rtx_thread_setcancel (oldCancel);
		return (rtx_error ("ddx_store_read (%s): store_post_sem(lockSem) failed", itemP->varName));
	}
	rtx_thread_setcancel (oldCancel);
	if (skipCount == 0) {
		tdiff = (tsnow.seconds - itemP->headerP->ts.tv_sec) +
			1.0e-9 * (tsnow.nanoSeconds - itemP->headerP->ts.tv_nsec);
		if (tdiff >= timeArg)
			return (1);
		return (0);
	}
	return (0);
}

/**
 * Write the specified item to the store.
 *
 * @return 0 on success, -1 on failure.
 *
 * This function is used by clients to write an item to the store. If d is
 * NULL, no data is copied. If tsp is NULL, the timestamp is provided by the
 * store.
 */
int 
ddx_store_write (
		 DDX_STORE_ITEM * itemP,      /**< item handle */
		 void * d,                    /**< data buffer */
		 RtxTime * tsp                /**< timestamp buffer */
		 )
{
    int numChars;
    RtxTime tsnow, * ts;
    DDX_STORE_ID * store;
    int varCount = 0;
    int errs = 0, oldCancel;

    ts = tsp;
    if (tsp == NULL) {
        rtx_time_get (&tsnow);
	ts = &tsnow;
    }

    if (itemP == NULL)
        return (rtx_error ("ddx_store_write: NULL handle"));
    if (itemP->magicNum != DDX_MAGIC_NUM)
        return (rtx_error ("ddx_store_write: invalid handle"));
    ddx_client_debug (("ddx_store_write (%s): writing", itemP->varName));
    oldCancel = rtx_thread_setcancel (RTX_THREAD_CANCEL_DISABLE);
    if (store_wait_sem (&(itemP->lockSem)) == -1) {
        rtx_thread_setcancel (oldCancel);
        return (rtx_error ("ddx_store_write (%s): store_wait_sem(lockSem) "
			   "failed", itemP->varName));
    }
    memcpy (&(itemP->headerP->ts), ts, sizeof (* ts));
    if (d != NULL) {
      /* make sure that memory pointer is uptodate */
	  if ((itemP->d = ddx_store_var_pointer(itemP)) == NULL) {
		  return rtx_error("ddx_store_write (%s): failed to get item pointer");
	  }
      /* rtx_message("c base %p, offset %d %d",itemP->store->shmemP->d, itemP->headerP->shmOffset, (unsigned char *)itemP->d - itemP->store->shmemP->d); */
      store_queue_write_entry(itemP->headerP, itemP->d, d, itemP->varSize);
    }
    varCount = ++(itemP->headerP->count);
    if (store_post_sem (&(itemP->lockSem)) == -1) {
        rtx_thread_setcancel (oldCancel);
        return (rtx_error ("ddx_store_write (%s): store_post_sem(lockSem) "
			   "failed", itemP->varName));
    }
    store = itemP->store;
    if (store_semQ_add (store->semQ, STORE_SEM_Q_ACTION_POST,
			itemP->varId, NULL) == -1) {
        rtx_thread_setcancel (oldCancel);
        return (rtx_error ("ddx_store_write (%s): store_semQ_add failed", itemP->varName));
    }
    /* multicast the data */
    if (itemP->headerP->multicastRequired) {
        if (rtx_mutex_lock (itemP->mutex) == -1) {
	    errs++;
	    rtx_error ("ddx_store_write (%s): rtx_mutex_lock() failed");
	}
        ddx_client_debug (("ddx_store_write (%s): multicasting at %s:%d", 
			   itemP->varName, 
			   itemP->multicastPort->sock->remote.ifname, 
			   itemP->multicastPort->sock->remote.portNum));
        itemP->multiHeader.count = varCount;
	itemP->multiHeader.size = itemP->varSize;
	memcpy (&(itemP->multiHeader.ts), ts, sizeof (* ts));
	itemP->buf[0] = store->myArchType;
	memcpy (&(itemP->buf[8]), &(itemP->multiHeader),
		sizeof (itemP->multiHeader));
	memcpy (&(itemP->buf[sizeof (itemP->multiHeader)+8]), d,
		itemP->varSize);
	if ((numChars = rtx_inet_write (itemP->multicastPort->sock,
					itemP->buf, 
					sizeof (itemP->multiHeader) + 
					itemP->varSize + 8,
					&(itemP->multicastPort->sock->remote)
					)) == -1) {
	    rtx_error ("ddx_store_write (%s): send_buffer() failed",
			       itemP->varName);
	    errs++;
	}
        if (rtx_mutex_unlock (itemP->mutex) == -1) {
	    errs++;
	    rtx_error ("ddx_store_write (%s): rtx_mutex_unlock() failed");
	}
	if (errs)
	    rtx_error_traceback ();
    }
    rtx_thread_setcancel (oldCancel);
    ddx_client_debug (("ddx_store_write (%s): written", itemP->varName));
    return (0);
}

/**
 * Read the specified item from a queue in the store.
 *
 * @returns the number of items left in the queue on success, -1 on
 * failure.
 *
 * This function is used by clients to read a queue item from the
 * store.  The caller must pass in a pointer to the tail count for the
 * queue, which will be updated.  If the tail is too old, the next
 * valid entry will be used, if the tail is -1, then the latest entry
 * will be returned (ie the head).  If the queue is empty, the
 * function will wait for at most timeOut before returning.  *
 */
int 
ddx_store_queue_read (
		DDX_STORE_ITEM * itemP,    /**< item handle */
		void * d,                  /**< data buffer */
		RtxTime * ts,              /**< timestamp buffer */
		double timeArg,            /**< timeout if queue is empty */
		int *tailP	           /**< pointer to tail count */
		)
{
  RtxTime tsnow, timeout;
  int oldCancel = -1;
  int headCount;
  int tailCount = *tailP;
  int queueLen, numElements;
  int overflow=0;
	
  rtx_time_from_double(&timeout,timeArg);

  if (itemP == NULL)
    return (rtx_error ("ddx_store_queue_read: NULL handle"));
  if (itemP->magicNum != DDX_MAGIC_NUM)
    return (rtx_error ("ddx_store_queue_read: invalid handle"));

  queueLen = itemP->headerP->queueLength;

  /* make sure that the variable is a queue */
  if (queueLen < 1) {
    return (rtx_error ("ddx_store_queue_read (%s): variable is not a queue", itemP->varName));
  }

  rtx_time_get (&tsnow);

  /* lock up the queue */
  oldCancel = rtx_thread_setcancel (RTX_THREAD_CANCEL_DISABLE);
  if (store_wait_sem (&(itemP->lockSem)) == -1) {
    rtx_thread_setcancel (oldCancel);
    return (rtx_error ("ddx_store_queue_read (%s): store_wait_sem(lockSem) failed", itemP->varName));
  }

  headCount = itemP->headerP->count;
  numElements = headCount-tailCount;
  if (numElements < -(DDX_STORE_QUEUE_MAX_COUNT/2) ) {
    /* head counter wrapped around */
    tailCount -= (DDX_STORE_QUEUE_MAX_COUNT/queueLen)*queueLen;
  } else if (tailCount == -DDX_STORE_QUEUE_MAX_COUNT ) {
    /* get the most recent entry */
    tailCount = headCount;
  } else if (numElements >= queueLen) {
    /* check for queue overflow */
    tailCount = headCount-queueLen+1;
    overflow = 1;
    ddx_client_debug(("ddx_store_queue_read: queue overflow, length %d >= %d",numElements,queueLen));
  }
	
  while (tailCount >= headCount) {
    int ret = 0;

    /* unlock the queue before waiting */
    if (store_post_sem (&(itemP->lockSem)) == -1) {
      rtx_thread_setcancel (oldCancel);
      return (rtx_error ("ddx_store_queue_read (%s): store_post_sem(lockSem) failed", itemP->varName));
    }

    if (timeArg>0) 
      ret = store_timed_wait_sem (&(itemP->syncSem),&timeout);
    else
      ret = store_wait_sem(&(itemP->syncSem));
    switch (ret) {
    case -1:
      rtx_thread_setcancel (oldCancel);
      return (rtx_error ("ddx_store_queue_read (%s): store_wait_sem(syncSem)"
			 " failed", itemP->varName));
    case +1:
      rtx_thread_setcancel (oldCancel);
      return (-2);
    default :
      /* lock it up again */
      if (store_wait_sem (&(itemP->lockSem)) == -1) {	
	rtx_thread_setcancel (oldCancel);
	return (rtx_error ("ddx_store_queue_read (%s): store_wait_sem(lockSem) failed", itemP->varName));
      }
      headCount = itemP->headerP->count;
      break;
    }
  }

  /* make sure that memory pointer is uptodate */
  if ((itemP->d = ddx_store_var_pointer(itemP)) == NULL) {
	  return rtx_error("ddx_store_queue_read (%s): failed to get item pointer");
  }

  if (d != NULL) 
    store_queue_read_entry(itemP->headerP, itemP->d, d, itemP->varSize, tailCount);

  if (ts != NULL)
    memcpy (ts, &(itemP->headerP->ts), sizeof (* ts));

  if (store_post_sem (&(itemP->lockSem)) == -1) {
    rtx_thread_setcancel (oldCancel);
    return (rtx_error ("ddx_store_queue_read (%s): store_post_sem(lockSem) failed", itemP->varName));
  }
  rtx_thread_setcancel (oldCancel);
  
  *tailP = ++tailCount;	/* increment the tail pointer */
  return headCount-tailCount;
}

/**
 * read a variable in the store
 * 
 * same function as ddx_store_read() except it does not copy any data
 *
 * @return 0 on success, -1 on failure.
 */
int 
ddx_store_read_direct (
		       DDX_STORE_ITEM * itemP,    /**< item handle */
		       RtxTime * ts,              /**< timestamp buffer */
		       double timeArg,            /**< time argument, 
						     staleness if
						     skipCount == 0, else
						     timeout
						     (currently not used) */
		       int skipCount              /**< number of data cycles 
						     to skip */
		       )
{
    RtxTime tsnow,timeout;
	rtx_time_from_double(&timeout,timeArg);
    double tdiff;
    int i = 0, oldCancel = -1;

    if (itemP == NULL)
        return (rtx_error ("ddx_store_read: NULL handle"));
    if (itemP->magicNum != DDX_MAGIC_NUM)
        return (rtx_error ("ddx_store_read: invalid handle"));
    rtx_time_get (&tsnow);
    if (skipCount < 0)
        return (rtx_error ("ddx_store_read (%s): invalid skip count (%d)",
			   itemP->varName, skipCount));
    while (i++ < skipCount) {
        int ret = 0;
		if (timeArg>0) 
			ret = store_timed_wait_sem (&(itemP->syncSem),&timeout);
		else
			ret = store_wait_sem(&(itemP->syncSem));
		switch (ret) {
			case -1:
				return (rtx_error ("ddx_store_read (%s): store_wait_sem(syncSem)"
							" failed", itemP->varName));
			case +1:
				return (1);
			default :
				break;
		}
	}
    oldCancel = rtx_thread_setcancel (RTX_THREAD_CANCEL_DISABLE);
    if (store_wait_sem (&(itemP->lockSem)) == -1) {
        rtx_thread_setcancel (oldCancel);
	return (rtx_error ("ddx_store_read (%s): store_wait_sem(lockSem) failed", itemP->varName));
    }
    if (ts != NULL)
        memcpy (ts, &(itemP->headerP->ts), sizeof (* ts));
    if (store_post_sem (&(itemP->lockSem)) == -1) {
        rtx_thread_setcancel (oldCancel);
	return (rtx_error ("ddx_store_read (%s): store_post_sem(lockSem) failed", itemP->varName));
    }
    rtx_thread_setcancel (oldCancel);
    if (skipCount == 0) {
        tdiff = (tsnow.seconds - itemP->headerP->ts.tv_sec) +
	    1.0e-9 * (tsnow.nanoSeconds - itemP->headerP->ts.tv_nsec);
	if (tdiff >= timeArg)
	    return (1);
	return (0);
    }
    return (0);
}

/**
 * Write the specified item to the store.
 *
 * same function as ddx_store_write() except it does not copy any data
 *
 * @return 0 on success, -1 on failure.
 *
 */
int 
ddx_store_write_direct (
			DDX_STORE_ITEM * itemP,      /**< item handle */
			RtxTime * tsp                /**< timestamp buffer */
			)
{
    RtxTime tsnow, * ts;
    DDX_STORE_ID * store;
    int varCount = 0;
    int oldCancel;

    ts = tsp;
    if (tsp == NULL) {
        rtx_time_get (&tsnow);
	ts = &tsnow;
    }

    if (itemP == NULL)
        return (rtx_error ("ddx_store_write: NULL handle"));
    if (itemP->magicNum != DDX_MAGIC_NUM)
        return (rtx_error ("ddx_store_write: invalid handle"));
    ddx_client_debug (("ddx_store_write (%s): writing", itemP->varName));
    oldCancel = rtx_thread_setcancel (RTX_THREAD_CANCEL_DISABLE);
    if (store_wait_sem (&(itemP->lockSem)) == -1) {
        rtx_thread_setcancel (oldCancel);
        return (rtx_error ("ddx_store_write (%s): store_wait_sem(lockSem) "
			   "failed", itemP->varName));
    }
    varCount = ++(itemP->headerP->count);
    memcpy (&(itemP->headerP->ts), ts, sizeof (* ts));
    if (store_post_sem (&(itemP->lockSem)) == -1) {
        rtx_thread_setcancel (oldCancel);
        return (rtx_error ("ddx_store_write (%s): store_post_sem(lockSem) "
			   "failed", itemP->varName));
    }
    store = itemP->store;
    if (store_semQ_add (store->semQ, STORE_SEM_Q_ACTION_POST,
			itemP->varId, NULL) == -1) {
        rtx_thread_setcancel (oldCancel);
        return (rtx_error ("ddx_store_write (%s): store_semQ_add failed", itemP->varName));
    }
    rtx_thread_setcancel (oldCancel);
    ddx_client_debug (("ddx_store_write (%s): written", itemP->varName));
    return (0);
}

static DDX_STRING_LIST* ddx_store_update_list(DDX_STORE_ID * id, 
		DDX_STRING_LIST* list, int varType) 
{
	MSG * mp;
	char * arg;
	char * tok;

	switch (varType) {
		case 1: /* type list */
			sprintf (id->buf1, "%s %s\n", 
					keyTable[MSG_TOK_STORE_GET_VAR_LIST_REQUEST].name,
					keyTable[MSG_TOK_ITEM_TYPE_TYPE].name);
			break;
		case 2: /* const list */
			sprintf (id->buf1, "%s %s\n", 
					keyTable[MSG_TOK_STORE_GET_VAR_LIST_REQUEST].name,
					keyTable[MSG_TOK_ITEM_TYPE_CONST].name);
			break;
		default: /* var list */
			sprintf (id->buf1, "%s\n", 
					keyTable[MSG_TOK_STORE_GET_VAR_LIST_REQUEST].name);
			break;
	}
	mp = store_get_response (id->server->sock->sockfd, 
			id->buf, id->buf1, 5, id, keyTable, 
			msgTable, "ddxInterface.ddxStore", id->verbose);
	if (mp == NULL) return list;
	if (mp->id != MSG_STORE_GET_VAR_LIST_REPLY) {
		free_message(keyTable,mp);
		return list;
	}
	arg = get_message_part (keyTable, mp, MSG_TOK_LIST);
	if (arg != NULL) {
		tok = strtok (arg, " \t");
		while (tok != NULL) {
			if ((strcmp (tok, "char") != 0) &&
					(strcmp (tok, "short") != 0) &&
					(strcmp (tok, "int") != 0) &&
					(strcmp (tok, "long") != 0) &&
					(strcmp (tok, "float") != 0) &&
					(strcmp (tok, "double") != 0) &&
					(strcmp (tok, "struct") != 0)) {
				// this hack to remove the infamous STORE_VAR_HEADER
				// from the type list
				if ((varType != 1)  || (strcmp(tok,"STORE_VAR_HEADER")!=0)) {
					list = ddx_stringlist_pushfront(list,tok);
					//printf("list %p name %p %s next %p\n",list,list->name,list->name,list->next);
				}
			}
			if (varType == 0) strtok (NULL, " \t");
			tok = strtok(NULL," \t");
		}
	}
	free_message(keyTable,mp);
	return list;
}

/**
 * Collect the list of constant names in the store
 *
 * @return a pointer to the list on success, NULL on failure.
 * The returned pointer must be freed by the user with ddx_stringlist_free
 *
 */
DDX_STRING_LIST * ddx_store_get_constant_list(DDX_STORE_ID * storeId)
{
	return ddx_store_update_list(storeId,NULL,2);
}

/**
 * Collect the list of type names in the store
 *
 * @return a pointer to the list on success, NULL on failure.
 * The returned pointer must be freed by the user with ddx_stringlist_free
 *
 */
DDX_STRING_LIST * ddx_store_get_type_list(DDX_STORE_ID * storeId)
{
	return ddx_store_update_list(storeId,NULL,1);
}

/**
 * Collect the list of variable names in the store
 *
 * @return a pointer to the list on success, NULL on failure.
 * The returned pointer must be freed by the user with ddx_stringlist_free
 *
 */
DDX_STRING_LIST * ddx_store_get_variable_list(DDX_STORE_ID * storeId)
{
	return ddx_store_update_list(storeId,NULL,0);
}

/**
 * Find the value of a constant
 *
 * @return 0 on success 
 * @param val is used to store the value
 *
 */
int ddx_store_lookup_const(DDX_STORE_ID * storeId, const char * name, int * val)
{
	MSG * mp;
	char * arg;

	sprintf (storeId->buf1, "%s %s %s = %s\n", 
			keyTable[MSG_TOK_STORE_GET_VAR_STATUS_REQUEST].name,
			keyTable[MSG_TOK_ITEM_TYPE_CONST].name,
			keyTable[MSG_TOK_NAME].name, name);
	mp = store_get_response (storeId->server->sock->sockfd, storeId->buf, 
			storeId->buf1, 5, storeId, keyTable,
			msgTable, "ddx_store_lookup_const",
			storeId->verbose);
	if (mp == NULL) {
		rtx_error("No response from store (5s)");
		goto error;
	}
	if (mp->id != MSG_STORE_GET_VAR_STATUS_REPLY) {
		free_message(keyTable,mp);
		rtx_error("Invalid response from store (5s)");
		goto error;
	}
	arg = get_message_part (keyTable, mp, MSG_TOK_VALUE);
	if (arg == NULL) {
		free_message(keyTable,mp);
		rtx_error("get_message_part() failed");
		goto error;
	}
	sscanf(arg," %d ",val);
	free_message(keyTable,mp);
	return 0;
error :
	return -1;
}

static DDX_STRING_LIST * buildDefinition(RtxParseVar * v,
		DDX_STRING_LIST * list, unsigned int indentLevel, int pretty,
		int typeonly)
{
	char tmp[1024];
	unsigned int i;

	if (v == NULL)
		return list;

	if (pretty) {
		for (i=0; i<indentLevel; i++) 
			tmp[i] = '\t';
		tmp[indentLevel] = 0;
		list = ddx_stringlist_attachafter(list,tmp);
	}
	if (v->type == rtx_struct_t) {
		RtxParseVar * vit = v->subVar;
		sprintf(tmp,"struct {%c",pretty?'\n':' ');
		list = ddx_stringlist_attachafter(list,tmp);
		while (vit != NULL) {
			list = buildDefinition (vit, list, indentLevel+1, pretty, 0);
			vit = vit->next;
		}
		if (pretty) {
			for (i=0; i<indentLevel; i++) 
				tmp[i] = '\t';
			strcpy(tmp+indentLevel,"} ");
			list = ddx_stringlist_attachafter(list,tmp);
		} else {
			list = ddx_stringlist_attachafter(list,"} ");
		}
	} else {
		switch (v->type) {
			case rtx_char_t : 
				list = ddx_stringlist_attachafter(list,"char ");
				break;
			case rtx_short_t : 
				list = ddx_stringlist_attachafter(list,"short ");
				break;
			case rtx_int_t : 
				list = ddx_stringlist_attachafter(list,"int ");
				break;
			case rtx_long_t : 
				list = ddx_stringlist_attachafter(list,"long ");
				break;
			case rtx_float_t : 
				list = ddx_stringlist_attachafter(list,"float ");
				break;
			case rtx_double_t : 
				list = ddx_stringlist_attachafter(list,"double ");
				break;
		}
	}
	if (typeonly) 
		return list;

	list = ddx_stringlist_attachafter(list,v->name);
	for (i=0; i<(unsigned int)(v->dim); i++) {
		sprintf (tmp,"[%d]", v->arrayDim[i]);
		list = ddx_stringlist_attachafter(list,tmp);
	}
	if (pretty)
		list = ddx_stringlist_attachafter(list,";\n");
	else
		list = ddx_stringlist_attachafter(list,"; ");
	return list;
}

/**
 * Build a textual representation of a type in the store
 *
 * @return the text on success. Returned pointer must be freed by user.
 *
 */
char* ddx_store_lookup_type (DDX_STORE_ID * storeId, const char * typeName)
{
	RtxParseSymTabs * symtabs = NULL;
	RtxParseVar * v;
	MSG * mp;
	char *arg, *result;
	DDX_STRING_LIST sentinel = {NULL,NULL};

	sprintf (storeId->buf1, "%s %s %s = %s\n", 
			keyTable[MSG_TOK_STORE_GET_VAR_STATUS_REQUEST].name,
			keyTable[MSG_TOK_ITEM_TYPE_TYPE].name,
			keyTable[MSG_TOK_NAME].name, typeName);
	mp = store_get_response (storeId->server->sock->sockfd, storeId->buf, 
			storeId->buf1, 5, storeId, keyTable,
			msgTable, "ddx_store_lookup_type",
			storeId->verbose);
	if (mp == NULL) {
		rtx_error("No response from store (5s)");
		goto error;
	}
	if (mp->id != MSG_STORE_GET_VAR_STATUS_REPLY) {
		free_message(keyTable,mp);
		rtx_error("Invalid response from store (5s)");
		goto error;
	}
	arg = get_message_part (keyTable, mp, MSG_TOK_VAR_DEFINITION);
	if (arg == NULL) {
		free_message(keyTable,mp);
		rtx_error("get_message_part() failed");
		goto error;
	}
	symtabs = rtx_parse_init();
	v = rtx_parse (arg, symtabs);
	rtx_parse_finish(symtabs);

	if (v == NULL) {
		free_message(keyTable,mp);
		rtx_error("rtx_parse failed");
		goto error;
	}
	buildDefinition(v,&sentinel,0,1,0);
	result = ddx_stringlist_concatenate(sentinel.next);
	ddx_stringlist_free(sentinel.next);
	//printf("Result type %s : %s\n",typeName,result);
	free_message(keyTable,mp);
	rtx_parse_free_var(v);
	return result;
error :
	return NULL;
}

/**
 * Returns a RtxParseVar corresponding to a given store Item.
 *
 * @return a pointer to a newly allocated var. Returned pointer must be freed by user with rtx_parse_free_var
 *
 */
RtxParseVar * ddx_store_item_get_parsed_var(DDX_STORE_ITEM * storeItem)
{
	RtxParseSymTabs * symtabs = NULL;
	RtxParseVar * var;
	if (storeItem == NULL) return NULL;
	symtabs = rtx_parse_init();
	var = rtx_parse(storeItem->varDefinition,symtabs);
	rtx_parse_finish(symtabs);
	if (var == NULL) {
		rtx_error("rtx_parse failed");
	}
	return var;
}

/**
 * Returns a RtxParseVar corresponding to a given store Item on a specified
 * architecture
 *
 * @return a pointer to a newly allocated var. Returned pointer must be freed by user with rtx_parse_free_var
 *
 */
RtxParseVar * ddx_store_item_get_remote_parsed_var(DDX_STORE_ITEM * storeItem,
		STORE_ARCH_TYPE arch)
{
	RtxParseSymTabs * symtabs = NULL;
	RtxParseVar * var;
	if (storeItem == NULL) return NULL;
	symtabs = rtx_parse_init_remote(storeArchs[arch].dataAlign, storeArchs[arch].dataSize);
	var = rtx_parse(storeItem->varDefinition,symtabs);
	rtx_parse_finish(symtabs);
	if (var == NULL) {
		rtx_error("rtx_parse failed");
	}
	return var;
}

/**
 * Build a textual definition of an item in the store
 *
 * @return the text on success. Returned pointer must be freed by user.
 * @param pretty activates pretty printing (indenting and linebreak).
 *
 */
char * ddx_store_item_get_definition(DDX_STORE_ITEM * storeItem, int pretty)
{
	DDX_STRING_LIST sentinel = {NULL,NULL};
	RtxParseSymTabs * symtabs = NULL;
	RtxParseVar * var;
	char * result;

	if (storeItem == NULL) return NULL;
	symtabs = rtx_parse_init();
	var = rtx_parse(storeItem->varDefinition,symtabs);
	rtx_parse_finish(symtabs);
	if (var == NULL) {
		rtx_error("rtx_parse failed");
		return NULL;
	}
	buildDefinition(var,&sentinel,0,pretty,0);
	result = ddx_stringlist_concatenate(sentinel.next);
	ddx_stringlist_free(sentinel.next);
	rtx_parse_free_var (var);
	return result;
	
	
}

/**
 * Build a textual definition of the type of an item in the store
 *
 * @return the text on success. Returned pointer must be freed by user.
 * @param pretty activates pretty printing (indenting and linebreak).
 *
 */
char * ddx_store_item_get_type_definition(DDX_STORE_ITEM * storeItem, int pretty)
{
	DDX_STRING_LIST sentinel = {NULL,NULL};
	RtxParseSymTabs * symtabs = NULL;
	RtxParseVar * var;
	char * result;

	if (storeItem == NULL) return NULL;
	symtabs = rtx_parse_init();
	var = rtx_parse(storeItem->varDefinition,symtabs);
	rtx_parse_finish(symtabs);
	if (var == NULL) {
		rtx_error("rtx_parse failed");
		return NULL;
	}
	buildDefinition(var,&sentinel,0,pretty,1);
	result = ddx_stringlist_concatenate(sentinel.next);
	ddx_stringlist_free(sentinel.next);
	rtx_parse_free_var (var);
	return result;
	
	
}



/* For item serialisation */

/**
 * Returns the required size of a buffer for storing itemP
 * Return -1 on failure.
 * */
int ddx_store_item_get_packsize(const DDX_STORE_ITEM * itemP)
{
	if (!itemP) return -1;
	return 8 + sizeof(STORE_MULTICAST_HEADER) + itemP->varSize;
}

/**
 * Serialize data contained in 'data' using item itemP into buffer.
 * itemP is needed here to provide a link on some meta-data
 * If not null, timestamp ts is used stored in buffer.
 */
int ddx_store_item_pack(const DDX_STORE_ITEM * itemP, 
		const void * data, const RtxTime * ts, 
		unsigned char * buffer)
{
	unsigned char * headp = buffer+8;
	unsigned char * datap = headp+sizeof(STORE_MULTICAST_HEADER);
	STORE_MULTICAST_HEADER hdr;
	if (!itemP) return -1;
    hdr.count = itemP->headerP->count;
	hdr.size = itemP->varSize;
	if (ts != NULL) {
		hdr.ts.tv_sec = ts->seconds;
		hdr.ts.tv_nsec = ts->nanoSeconds;
	} else {
		memcpy(&hdr.ts, &(itemP->headerP->ts), sizeof (STORE_TIMESPEC));
	}

	bzero(buffer,8 + sizeof(STORE_MULTICAST_HEADER) + itemP->varSize);
	buffer[0] = itemP->store->myArchType;
#if 0
	printf("Packing in arch %d\n",buffer[0]);
#endif
	memcpy (headp, &hdr, sizeof (STORE_MULTICAST_HEADER));
#if 0
	printf("hdr: %d bytes\n",sizeof(STORE_MULTICAST_HEADER));
	for (i=0;i<sizeof(STORE_MULTICAST_HEADER);i++) printf("%02X ",headp[i]);
	printf("\n");
#endif
	memcpy (datap, data, itemP->varSize);
#if 0
	printf("data: %d bytes \n",itemP->varSize);
	for (i=0;i<itemP->varSize;i++) printf("%02X ",((unsigned char*)data)[i]);
	printf("\n");
#endif
	return 0;
}


/**
 * De-serialize 'buffer' into 'data'. If not null, the timestamp stored in
 * buffer is recovered into ts. 
 * if there is a need for architecture conversion, and if they contains NULL, 
 * @param localVar and @param remoteVar will be used to store a structured
 * representation of the item on both architecture. If their content is not
 * NULL then it will be used for the translation.
 * User should free their content with rtx_parse_free_var.
 * */
int ddx_store_item_unpack(DDX_STORE_ITEM * itemP, 
		void * data, RtxTime * ts, 
		RtxParseVar ** localVar, RtxParseVar ** remoteVar,
		unsigned char * buffer)
{
	unsigned char * headp = buffer+8;
	unsigned char * datap = headp+sizeof(STORE_MULTICAST_HEADER);
	DDX_STORE_ID * store;
	STORE_MULTICAST_HEADER hdr;
	unsigned char remoteArchType = buffer[0];
	if (!itemP) return -1;
	store = itemP->store;

	if (remoteArchType == store->myArchType) {
		memcpy(&hdr,headp,sizeof(STORE_MULTICAST_HEADER));
		memcpy(data,datap,hdr.size);
	} else {
 	    /* data translation required */
		rtx_parse_translate_data (&hdr, headerVarArray[store->myArchType], headp,
				headerVarArray[remoteArchType], 1);

		if (*localVar == NULL) 
			*localVar = ddx_store_item_get_parsed_var(itemP);
		if (*remoteVar == NULL) 
			*remoteVar = ddx_store_item_get_remote_parsed_var(itemP, remoteArchType);

		rtx_parse_translate_data (data, *localVar, datap, *remoteVar, 1);
	}
	if (ts != NULL) {
		ts->seconds = hdr.ts.tv_sec;
		ts->nanoSeconds = hdr.ts.tv_nsec;
	}
	return 0;
}

