/** \page ddxweb ddxweb
 * Application that provides web access to store 
 * 
 * Connect to a computer running a store on port WEBSERVER_PORT (default 8000 - do not change it).
 * 1. Connection to localhost:8000 display the catalog page which show each connected store ;
 * 2. - You can see the beginning of each connected store information on which ddxweb is running ;
 *    - Connect to it and you'll see every variables (and their values), const and types registered !
 *
 * author: Sebastien VAUSSIER
 * mail: sebastien.vaussier@mail.rsise.anu.edu.au
 */

#define _TRY_
// Be careful ! If you change this value,
// change it on ALL ddxweb
#define WEBSERVER_PORT 8000

#define AUTOM_ETC "/data/aa/etc"
#define HOSTFILE AUTOM_ETC"/ddxweb_hosts"
#define PWDFILE AUTOM_ETC"/xmlrpcpasswd"

// Webpages is refreshed each update_webpage_timeout milliseconds
// Call to rand is used to prevent pages to updates at the very same
// time (it doesn't crash but the resulting pages doesn't look good)
// If you need to change this timeout, change only UPDATE_TIMEOUT (in second)
#define UPDATE_TIMEOUT 10

#include <assert.h>
#include <signal.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <string>
#include <ctype.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <strings.h>
#include <pthread.h>
#include <time.h>
#include <math.h>

#include <rtx/defines.h>
#include <rtx/message.h>
#include <rtx/time.h>
#include <rtx/error.h>
#include <rtx/command.h>
#include <rtx/parse.h>
#include <rtx/inet.h>
#include <rtx/main.h>
#include <rtx/httpd.h>
#include <ddx.h>

extern "C" {
#include "store-common.h"
#include "store-header.h"
#include "store-msg.h"
#include "store-messages.h"
#include "catalog-messages.h"
#include "catalog-main.h"
}

static char rcsid[] RTX_UNUSED = "$Id: ddxweb-main.cpp 3074 2008-05-15 04:58:18Z roy029 $";

#ifdef i486_lynxos310
extern char *optarg;
extern int opterr;
#endif

using namespace std;

int done = 0;
char storeName[128];
char catalogName[128];
DDX_STORE_ID * store;
DDX_STORE_ID catalog;
string update_info;
int update_webpage_timeout;

string buildDefinition(string & s, RtxParseVar * v,unsigned int dim,int * arrayDim,
		unsigned int indentLevel, int pretty, char * buffer);


static string intToString (int i) {
    char buf[20];
    sprintf(buf,"%d",i);
    return string(buf);
}

static string floatToString (float f) {
    char buf[20];
    sprintf(buf,"%.1f",f);
    return string(buf);
}


static int
ddxsh_store_get_catalog (void) {
    MSG * mp;
    char * catName;
    char * tok;

    sprintf (store->buf1, "%s\n",
             keyTable[MSG_TOK_STORE_GET_CATALOG_REQUEST].name);
    if ((mp = store_get_response (store->server->sock->sockfd,
                                  store->buf, store->buf1,
                                  5, store, keyTable,
                                  msgTable, "ddxsh",
                                  store->verbose)) == NULL)
        return (rtx_error ("ddxsh_store_get_catalog: "
                           "no response from store (5s)"));
    if (store->verbose)
        fprintf (stderr, "store-reply: %s\n", store->buf);
    if (mp->id != MSG_STORE_GET_CATALOG_REPLY)
        rtx_error ("ddxsh_store_get_catalog: "
                   "invalid/error reply from store");
    if ((catName = get_message_part (keyTable, mp, MSG_TOK_NAME))
            == NULL)
        rtx_error ("ddxsh_store_get_catalog: "
                   "get_message_part() failed");
    tok = strtok (catName, " ");
    strcpy (catalogName, tok);
    free_message (catKeyTable, mp);
    return (0);
}

string buildItemList (string what) {
	string response;
    DDX_STRING_LIST * list = NULL;
	//printf("buildItemList %s\n",what.c_str());
    if (what == "variable")
        list = ddx_store_get_variable_list(store);
    else if (what == "type")
        list = ddx_store_get_type_list(store);
    else if (what == "constant")
        list = ddx_store_get_constant_list(store);

    DDX_STRING_LIST * it = list;
	while (it != NULL) {
			//printf("it %p name %p %s next %p\n",it,it->name,it->name,it->next);
			assert(it->name != NULL);
			response +="<li><a target='_top' title='Click for more information about "+what;
			response += " \""+string(it->name)+"\"' href='"+what+"?name=";
			response += string(it->name)+"'>"+string(it->name)+"</a></li>\n";
			it = it->next;
	}
    ddx_stringlist_free(list);
	//printf("Returned : %s\n",response.c_str());
    return response;
}


string genValueString(RtxParseVar * V,char * buffer)
{
	string response;
	return buildDefinition(response,V,V->dim,V->arrayDim,0,1,buffer);
}

string genCatalogPage()
{
	string response;
	response +="<center><h1>Welcome to <span class='store'>"+string(catalogName);
	response +="</span>  catalog !</h1></center>\n"+update_info;

	MSG * mp;
	sprintf (catalog.buf1, "%s\n", catKeyTable[MSG_TOK_CATALOG_GET_STORE_LIST_REQUEST].name);

	if ((mp = store_get_response (catalog.server->sock->sockfd,
					catalog.buf, catalog.buf1,5,
					&catalog, catKeyTable, catMsgTable,
					"ddxweb: (show catalog)",catalog.verbose)) == NULL)
		response +="<b>Internal error:</b> no response from catalog (5s)";
	string storeList=string(catalog.buf);
	string stores;
	unsigned int aux;
	if ((aux=storeList.find("=")+2) == storeList.size()) // No store connected to the catalog
		response +="No store connected to the catalog.<br>\n";
	else {
		response +="The following store(s) are connected to the catalog:\n<ul>\n";
		// Catalog anszers a string like that:
		// $$blabla $blablabla = store_name1 store_name2
		stores=storeList.substr(aux,storeList.size()-aux-1);
		char sep[] =" ";
		char *ptr;
		char *storesNames=strdup(stores.c_str());
		string iFrames ="";
		ptr = strtok(storesNames, sep);
		while ( ptr != NULL ) {
			response +="<li>Go to <a target='_top' href='http://"+string(ptr)+":";
			response +=intToString(WEBSERVER_PORT)+"/store'>"+string(ptr)+" !</a></li>";

			iFrames += "<iframe style='background-color:#F0F0F0;border:2px solid black'";
			iFrames += " src='http://"+string(ptr)+":"+intToString(WEBSERVER_PORT);
			iFrames += "/store' width='95%' height='400'>You browser does not support ";
			iFrames += "<i>iframe</i> ! Not a problem: use the above";
			iFrames += " links to check connected store(s).</iframe>\n<br><br><br><br>\n";
			ptr = strtok(NULL, sep);
		}
		response +="</ul>\n<div style='border-top:3px double black;padding:20px'>&nbsp;</div>";
		response +="\n<center>"+iFrames+"</center>\n";
		free(storesNames);
	}
	free_message(catKeyTable,mp);

	return response;
}

string genStorePage()
{
	string response;
	response +="<center><h1>Welcome to <a target='_top' href='http://"+string(storeName)+":";
	response +=intToString(WEBSERVER_PORT)+"/store'>"+string(storeName)+"</a> store !</h1></center>\n";
	response +="<h5>[ <a target='_top' href='/' target='_top'>Back to catalog</a> ]</h5>\n";
	response +=update_info;
	response +="<table align='center' width='100%'>\n<tr>\n\t<td width='33%'>\n";
	response +="<h2>Var</h2>\n<ul>\n";
	// Get var
	response += buildItemList("variable");

	// Get struct
	response +="</ul>\n\t</td>\n\t<td width='33%'>\n<h2>Type</h2>\n<ul>";
	response += buildItemList("type");

	// Get const
	response +="</ul>\n\t</td>\n\t<td width='33%'>\n<h2>Const</h2>\n<ul>";
	response += buildItemList("constant");

	response +="\t</td>\n</tr>\n</table>\n<br>";
	return response;
}

string genVariablePage(string name)
{
	string response;
	//printf("Variable %s\n",name.c_str());
	DDX_STORE_ITEM * item = ddx_store_lookup_item(store,
			name.c_str(),NULL,0);
	if (item == NULL) {
		response += "Undefined variable";
		return response;
	}
	char * buffer = (char*)malloc(item->varSize);
	RtxParseVar * v = ddx_store_item_get_parsed_var(item);
	RtxTime ts,tsc,tsa;
	ddx_store_read(item,buffer,&ts,1,0);
	rtx_time_get(&tsc);
	rtx_time_subtract(&tsa,&tsc,&ts);

	char tmp[128];
	response += "<center>";
	sprintf(tmp,"<em>Current time:</em> <font color=blue>%ld.%ld</font>s<br>\n",tsc.seconds,tsc.nanoSeconds);
	response += tmp;
	response += "</center>";
	response += "<table align='center' style='width:90%;'>\n<tr>\t<td>\n";
#if 0
	char * type = ddx_store_item_get_definition(item,1);
	response += "<h2>Type definition</h2>\n";
	response += "<pre>\n";
	response += type;
	response += "</pre>\n";
	response+="\t</td><td>";
	free(type);
#endif
	
	response += "<h2>Value</h2>\n";

	response += "<ul>";
	sprintf(tmp,"<li><em>Time Stamp:</em> <font color=blue>%ld.%ld</font>s</li>\n",ts.seconds,ts.nanoSeconds);
	response += tmp;
	sprintf(tmp,"<li><em>Variable Age:</em> <font color=blue>%ld.%ld</font>s</li>\n",tsa.seconds,tsa.nanoSeconds);
	response += tmp;
	response += "</ul><pre>";
	response += genValueString(v,buffer);
	rtx_parse_free_var(v);
	free(buffer);
	ddx_store_done_item(item);
	
	response +="\n</pre>\n\n";
	response +="</td>\n";
	response +="</tr>\n</table>\n";
		
	return response;
}

string genTypePage(string name)
{
	string response;
	char * type = ddx_store_lookup_type(store,name.c_str());
	response += "<table align='center' style='width:50%;'>\n<tr>\t<td>\n";
	response += "<h2>Type definition</h2>\n";
	response += "<pre>\n";
	response += type;
	response += "</pre>\n";
	response +="</td>\n";
	response +="</tr>\n</table>\n";
		
	free(type);
	return response;
}

string genConstPage(string name)
{
	string response;
	int val = 0;
	ddx_store_lookup_const(store,name.c_str(),&val);
	response += "<table align='center' style='width:40%;'>\n<tr>\t<td>\n";
	response += "<h2>Constant value</h2>\n\n";
	response += "<center>\n";
	response += intToString(val);
	response += "</center>\n";
	response +="</td>\n";
	response +="</tr>\n</table>\n";
	
	return response;
}

//static int handler_call = 0;
int
ddxweb_handler(RtxHttpdReq * r) {
    const char *name_key = "name";
    char * name;

    bool index_requested,store_requested,var_requested,type_requested,const_requested;
	string response;

    if (!(r->method == RTX_HTTPD_M_GET))
    return 0;
    // printf("Handler %8d\n",handler_call++);
    var_requested=(strcmp(r->url,"/variable") == 0)?true:false;
    store_requested=(strcmp(r->url,"/store") ==0)?true:false;
    type_requested=(strcmp(r->url,"/type")==0)?true:false;
    const_requested=(strcmp(r->url,"/constant")==0)?true:false;
    // If a client asks for a unknown page, index.html page is returned    
    index_requested=!(var_requested || store_requested || type_requested || const_requested);
       
    string title; // webpage's title
    if (index_requested)
        title=string(catalogName);
    else
        title=string(storeName);
    // HTML headers
    response="<html>\n<head>\n<title>Welcome to "+title+"</title>\n</head>\n";
    // CSS stuff to make pages look beautiful
    response += "<style>\na {text-decoration:none;}\na:hover {background-color:#FFFF00;}\n";
    response += "a:visited {color:blue;}\na:link: {color:blue;}\n";
    response += "H2 {text-align:center;background-color:#AAAAFF;display:block;}\n";
    response += "TD {vertical-align:top;border:1px solid black;padding:5px;}\n";
    response += "H5 {text-align:center}\n\n";
    response += ".store {font-weight:bold;font-style:italic;color:darkred;}\n";
    response += ".about {font-style:italic;}\n</style>\n";
    // This javascript updates this webpages each update_webpage_timeout (10000ms default)
    response += "<body onLoad='window.setTimeout(\"location.reload()\",";
    //response += intToString(100)+");'>\n";
    response += intToString(update_webpage_timeout)+");'>\n";
    update_info = "<h5><center>This page is updated every ";
    update_info +=floatToString(update_webpage_timeout/1000.0)+" second(s)</center></h5>\n";
    // We want to see /var or /type or /const
    if (var_requested || type_requested || const_requested) {
      if (! (name = (char *)rtx_hash_find(r->args, (void *)name_key, (rtx_error_t)NULL))) {
            response += "<h1><center>About <span class='about'>"+string(name)+"</span></center></h1>\n";


            string tmp=response; // tmp will be rewritten at the end of the HTML file
            response ="<h5>";
            //ddxsh_store_get_var_list (1,cmd);
            response += " [ <a target='_top' href='/store'>Back to store</a> ]";
            response += " -=- [ <a target='_top' href='/'>Back to catalog</a> ]</h5>\n";
            swap (tmp,response);
            response +=tmp+update_info;

            if (var_requested) {
				response += genVariablePage(name);
            } else if (type_requested) {
				response += genTypePage(name);
            } else if (const_requested) {
				response += genConstPage(name);
            }
            response +=tmp+"<br>\n";
        } else {
            var_requested=false;
            index_requested=true; // no name given => let's go to the index page !

        }
    }
    if (index_requested) {
		response += genCatalogPage();
    }

    if (store_requested) {
		response += genStorePage();
    }

    if(rtx_httpd_reply_send_status(r,200,NULL)!=0) {
        return rtx_error("Could not set HTTP response status");
    }
    if(rtx_httpd_reply_send_header(r,"Content-Type","text/html")!=0) {
        return rtx_error("Could not set Content Type");
    }


    response+="</body>\n</html>\n";
    if(rtx_httpd_reply_send_body(r,(void *) response.c_str(),response.size())!=0)
        return rtx_error("Could not send response");
#ifdef DEBUG
    printf("Request %d for %s handled\n",r->method, r->url);
#endif

    return 0;
}

int main(int argc, char **argv) {

	srand48(time(NULL));
	update_webpage_timeout = (int)round((UPDATE_TIMEOUT * (0.9+drand48()*0.2)*1000));
    // store & catalog's initialisation

    rtx_error_init ("ddxweb", RTX_ERROR_STDERR, NULL);
	ddx_client_init(0);
    if (gethostname (storeName, RTX_INET_MAX_HOST_NAME_LEN) == -1) {
        rtx_error_flush ("gethostname() failed");
        exit (2);
    }


	store = ddx_store_open(NULL,STORE_PORT_NUM,5);
	if (store == NULL) {
        rtx_error_flush ("ddx_store_open() failed");
        exit (4);
	}
    if (ddxsh_store_get_catalog () == -1) {
        rtx_error_flush ("ddxsh_store_get_catalog() failed");
        exit (4);
    }
    if ((catalog.server = rtx_inet_init (RTX_INET_TCP_CLIENT,
                                         NULL, 0, catalogName,
                                         CATALOG_PORT_NUM,
                                         NULL, NULL, NULL)) == NULL) {
        rtx_error_flush ("rtx_inet_init (catalog) failed");
        exit (5);
    }

    // end initialisation


	RtxHttpd *server;

    fprintf(stderr, "Initialising simple HTTPD server on port %d\n",WEBSERVER_PORT);
	server = rtx_httpd_server_init(NULL,WEBSERVER_PORT,0,ddxweb_handler);
    if (!server) {
        fprintf(stderr, "httpd_server_init() failed.\n");
        exit(1);
    }

	rtx_main_wait_shutdown(0);

	rtx_httpd_server_destroy(server);

	ddx_store_close(store);
	ddx_client_done();
    exit(0);
}


#define MIN(A,B) ((unsigned int)(((signed)(A)<(signed)(B))?(A):(B)))
string buildDefinition(string & s, RtxParseVar * v,unsigned int dim,int * arrayDim,
		unsigned int indentLevel, int pretty, char * buffer)
{
	char tmp[1024];
	unsigned int i;

	if (v == NULL)
		return s;

	if (pretty) {
		for (i=0; i<indentLevel; i++) 
			tmp[i] = '\t';
		tmp[indentLevel] = 0;
		s += tmp;
	}
	if (v->type == rtx_struct_t) {
		RtxParseVar * vit = v->subVar;
		sprintf(tmp,"<em>struct</em> {%c",pretty?'\n':' ');
		s += tmp;
		while (vit != NULL) {
			s = buildDefinition (s, vit, vit->dim, vit->arrayDim, indentLevel+1, 
					pretty,(buffer==NULL)?NULL:(buffer + vit->offset));
			vit = vit->next;
		}
		if (pretty) {
			for (i=0; i<indentLevel; i++) 
				tmp[i] = '\t';
			strcpy(tmp+indentLevel,"} ");
			s += tmp;
		} else {
			s += "} ";
		}
	} else {
		switch (v->type) {
			case rtx_char_t : 
				s += "<em>char</em> ";
				break;
			case rtx_short_t : 
				s += "<em>short</em> ";
				break;
			case rtx_int_t : 
				s += "<em>int</em> ";
				break;
			case rtx_long_t : 
				s += "<em>long</em> ";
				break;
			case rtx_float_t : 
				s += "<em>float</em> ";
				break;
			case rtx_double_t : 
				s += "<em>double</em> ";
				break;
		}
	}
	s += "<b>";
	s += v->name;
	s += "</b><em>";
	for (i=0; i<(unsigned int)(dim); i++) {
		sprintf (tmp,"[%d]", arrayDim[i]);
		s += tmp;
	}
	s += "</em>";
	if ((v->type == rtx_struct_t) && (dim>0)) {
		s += " <u>displaying first object only</u>";
	}
	if (buffer != NULL) {
		switch (v->type) {
			case rtx_char_t : 
				{
					char *c = (char *)(buffer);
					switch (dim) {
						case 0: 
							if (isprint(*c)) {
								sprintf(tmp," = <font color=blue>%d 0x%02X '%c'</font>",*c,*c,*c); 
							} else {
								sprintf(tmp," = <font color=blue>%d 0x%02X </font>",*c,*c); 
							}
							s += tmp;
							break;
						case 1: 
							s += " = [<font color=blue> ";
							for (i=0;i<MIN(arrayDim[0],10);i++) {
								sprintf(tmp,"%02X ",c[i]); 
								s += tmp;
							}
							if (arrayDim[0] > 10) { 
								s += "</font>...] = '<font color=blue>";
							} else {
								s += "</font>] = '<font color=blue>";
							}
							for (i=0;i<MIN(arrayDim[0],10);i++) {
								if (isprint(c[i])) {
									s += c[i];
								} else {
									sprintf(tmp,"\\%03o",c[i]);
									s += tmp;
								}
							}
							if (arrayDim[0] > 10) { 
								s += "</font>...'";
							} else {
								s += "</font>'";
							}
							break;
						default : break;
					}
				}
				break;
			case rtx_short_t : 
				{
					short * c = (short *)(buffer);
					switch (dim) {
						case 0: 
							sprintf(tmp," = <font color=blue>%d 0x%04X</font>",*c,*c); 
							s += tmp;
							break;
						case 1: 
							s += " = [<font color=blue> ";
							for (i=0;i<MIN(arrayDim[0],10);i++) {
								sprintf(tmp,"%d ",c[i]); 
								s += tmp;
							}
							if (arrayDim[0] > 10) { 
								s += "</font>...]";
							} else {
								s += "</font>]";
							}
							break;
						default : break;
					}
				}
				break;
			case rtx_int_t : 
				{
					int * c = (int *)(buffer);
					switch (dim) {
						case 0: 
							sprintf(tmp," = <font color=blue>%d</font>",*c); 
							s += tmp;
							break;
						case 1: 
							s += " = [<font color=blue> ";
							for (i=0;i<MIN(arrayDim[0],10);i++) {
								sprintf(tmp,"%d ",c[i]); 
								s += tmp;
							}
							if (arrayDim[0] > 10) { 
								s += "</font>...]";
							} else {
								s += "</font>]";
							}
							break;
						default : break;
					}
				}
				break;
			case rtx_long_t : 
				{
					long * c = (long *)(buffer);
					switch (dim) {
						case 0: 
							sprintf(tmp," = <font color=blue>%ld 0x%08lX</font>",*c,*c); 
							s += tmp;
							break;
						case 1: 
							s += " = [<font color=blue> ";
							for (i=0;i<MIN(arrayDim[0],10);i++) {
								sprintf(tmp,"%ld ",c[i]); 
								s += tmp;
							}
							if (arrayDim[0] > 10) { 
								s += "</font>...]";
							} else {
								s += "</font>]";
							}
							break;
						default : break;
					}
				}
				break;
			case rtx_float_t :
				{
					float * c = (float *)(buffer);
					switch (dim) {
						case 0: 
							sprintf(tmp," = <font color=blue>%e</font>",*c); 
							s += tmp;
							break;
						case 1: 
							s += " = [<font color=blue> ";
							for (i=0;i<MIN(arrayDim[0],10);i++) {
								sprintf(tmp,"%e ",c[i]); 
								s += tmp;
							}
							if (arrayDim[0] > 10) { 
								s += "</font>...]";
							} else {
								s += "</font>]";
							}
							break;
						default : break;
					}
				}
				break;
			case rtx_double_t :
				{
					double * c = (double *)(buffer);
					switch (dim) {
						case 0: 
							sprintf(tmp," = <font color=blue>%e</font>",*c); 
							s += tmp;
							break;
						case 1: 
							s += " = [<font color=blue> ";
							for (i=0;i<MIN(arrayDim[0],10);i++) {
								sprintf(tmp,"%e ",c[i]); 
								s += tmp;
							}
							if (arrayDim[0] > 10) { 
								s += "</font>...]";
							} else {
								s += "</font>]";
							}
							break;
						default : break;
					}
				}
				break;
		}
	}
	if (pretty)
		s += ";\n";
	else
		s += "; ";
	return s;
}
