/*********************************************************************
 *
 * CSIRO Automation
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * www.cat.csiro.au/cmit
 *
 * Copyright (c) CSIRO Manufacturing & Infrastructure Technology
 *
 *********************************************************************/

static char *rcsid = "$Id: ddxsiggen-main.c 3865 2008-08-28 08:48:54Z roy029 $";

/**
 * \file ddxsiggen-main.c
 * \brief DDX Signal Generator
 * \author Charles Brooking
 */

/**
 * \page ddxsiggen ddxsiggen
 *
 * The DDX Signal Generator is a software version of a standard laboratory
 * function generator. It accepts the defintion of several functions, described
 * in a configuration file, and evaluates these at regular intervals for output
 * as variables in the DDX store. Functions are characterised by: their name,
 * which appears as the variable name in the DDX store; their type, such as
 * sine, pulse, triangle, etc; and several parameters, including period, and
 * amplitude. Each signal generated appears as a normal DDX variable, and hence
 * the target machine may receive it using the standard store intereface.

 \section Summary
The DDX Signal Generator is a software version of a standard laboratory function generator. It accepts the defintion of several functions, described in a configuration file, and evaluates these at regular intervals for output as variables in the DDX store. Functions are characterised by: their name, which appears as the variable name in the DDX store; their type, such as sine, pulse, triangle, etc; and several parameters, including period, and amplitude. Each signal generated appears as a normal DDX variable, and hence the target machine may receive it using the standard store intereface.
    </p>

\section Command Line
    <p>
      The command line is of the form
    </p>
    <blockquote>
      ddxsiggen -config <i>configFile</i>
    </blockquote>
    <p>
      where the following are valid examples:
    </p>
    <blockquote>
      <font face="courier" size="-1">ddxsiggen -config file.conf</font><br>
      <font face="courier" size="-1">ddxsiggen -config path/file.conf</font>
    </blockquote>

    <h2>Configuration File</h2>
    <p>
      The following options are specified through the configuration file, which is identified in the command line:
    </p>
    <ul>
      <li><b>Update interval</b> - Required interval for updating function values.</li>
      <li><b>Signal names</b> - Variable names that will appear in the DDX store for each function.</li>
      <li><b>Function types</b> - The general class of each function, and parameters determining the specific shape.</li>
    </ul>

    <p>
      The syntax for the configuration file is described by the following set of regular expressions.<br>
    </p>
    <table border="1" bgcolor="#EEEEEE" cellpadding="5" cellspacing="a">
    <tr>
    <td>
      <font face="times new roman">
        <i>Configuration</i> = <i>Interval</i> <i>Signal</i>*<br>
        <br>
        <i>Interval</i> = interval <i>RealNumber</i><br>
        <br>
        <i>Signal</i> = <i>SignalName</i> <i>SignalType</i> <i>SignalParams</i>?<br>
        <i>SignalName</i> = signal <i>Identifier</i><br>
        <i>SignalType</i> = type (sine | cosine | square | pulse | sawtooth | triangle | constant | gauss | uniform)<br>
        <i>SignalParams</i> = params (<i>Identifier</i> = <i>Value</i>)+<br>
        <br>
        <i>Comment</i> = % <i>Line</i><br>
        <br>
        (Assuming patterns <i>RealNumber</i>, <i>Identifier</i>, <i>Value</i> and <i>Line</i> are defined reasonably.)
      </font>
    </td>
    </tr>
    </table>
    <p>
      Layout is free-form, but with the strict condition that white space appears between all tokens. This means, for example, that "f = 5" is a valid parameter assignment but "f=5" is not. You should take great care to ensure the configuration file is valid, because there is only minimal support for handling errors.
    </p>
    <p>
      An example configuration file (with two signals specified) might be:
    </p>
    <table border="1" bgcolor="#EEEEEE" cellpadding="5" cellspacing="a">
    <tr>
    <td>
      <font face="courier" size="-1">
        % Comments may appear at the end of any line<br>
        <br>
        interval 0.01<br>
        <br>
        signal harry<br>
        type sine<br>
        params<br>
        &nbsp;&nbsp;A = 5<br>
        &nbsp;&nbsp;f = 1000<br>
        &nbsp;&nbsp;C = 0<br>
        <br>
        signal sarah<br>
        type triangle<br>
        params<br>
        &nbsp;&nbsp;A = -1&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;% Lower bound<br>
        &nbsp;&nbsp;B = 1&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;% Upper bound<br>
        &nbsp;&nbsp;T = 0.0001&nbsp;&nbsp;% Period 0.1 ms<br>
      </font>
    </td>
    </tr>
    </table>

    <h2>Types of Functions</h2>
    <p>
      The table below describes the types of functions that are supported.
    </p>
    <table border="1" cellpadding="2">
      <tr>
        <th bgcolor="#CCCCCC" align="left"><b>Function<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Description<b></th>
      </tr>
      <tr>
        <td>Sine</td>
        <td>As you would expect.</td>
      </tr>
      <tr>
        <td>Cosine</td>
        <td>Some people prefer cosine.</td>

      </tr>
      <tr>
        <td>Square</td>
        <td>Standard square wave.</td>
      </tr>
      <tr>
        <td>Pulse</td>
        <td>Similar to square wave, but with specified dute cycle.</td>
      </tr>
      <tr>
        <td>Sawtooth</td>
        <td>Standard sawtooth pattern.</td>
      </tr>
      <tr>
        <td>Triangle</td>
        <td>Similar to sawtooth, except also linearly decreasing.</td>
      </tr>
      <tr>
        <td>Constant</td>
        <td>Standard constant function.</td>
      </tr>
      <tr>
        <td>Gaussian Noise</td>
        <td>Random values from particular distribution.</td>
      </tr>
      <tr>
        <td>Uniform Noise</td>
        <td>Random values from uniform distribution.</td>
      </tr>
    </table>

    <h2>Function Parameters</h2>
    <p>
      The characteristics of each generated signal need to be supplied by the user in the configuration file. These depend on the function type chosen, as described for each below. All measurements are expressed using SI units.
    </p>
    <p>
      Specification of any parameter is optional, with the program assuming a particular default value if necessary. If fact, the entire <i>params</i> clause of a signal definition may be ommited from the configuration file if the user only requires standard behaviour. Default characteristics for each function type are described in the tables that follow.
    </p>
    <p>
      If the frequency is provided, then the period should not be (this point also applies conversely). If both are present, then the program will use the assignment that appears lowest in the configuration file.
    </p>

    <h3>Sine and Cosine</h3>
    <p>
      General mathematical form:
    </p>
    <blockquote>
      <i>sinfunc</i> (<i>t</i>) = <i>A</i> sin(<i>B</i> <i>t</i> + <i>C</i>) + <i>D</i><br>
      <i>cosfunc</i> (<i>t</i>) = <i>A</i> cos(<i>B</i> <i>t</i> + <i>C</i>) + <i>D</i>
    </blockquote>
    <p>
      Have the following parameters:
    </p>
    <table border="1" cellpadding="2" width="550">
      <tr>
        <th bgcolor="#CCCCCC" align="left"><b>Parameter<b></th>
        <th bgcolor="#CCCCCC" align="left" width="100%"><b>Description<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Units<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Default<b></th>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">A</font></td>
        <td>Amplitude.</td>
        <td>undefined</td>
        <td>1</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">B</font></td>
        <td>The angular frequency.</td>
        <td>rad&middot;s<sup>-1</sup></td>
        <td>1</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">C</font></td>
        <td>The phase angle.</td>
        <td>rad</td>
        <td>0</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">D</font></td>
        <td>Offset value.</td>
        <td>undefined</td>
        <td>0</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">f</font></td>
        <td>Frequency.</td>
        <td>Hz</td>
        <td>1 / 2&pi;</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">T</font></td>
        <td>Period.</td>
        <td>s</td>
        <td>2&pi;</td>
      </tr>
    </table>
    <p>
      Note that if <font face="courier" size="-1">B</font> is specified, then neither <font face="courier" size="-1">f</font> nor <font face="courier" size="-1">T</font> should be provided (and inversely).
    </p>

    <h3>Square</h3>
    <p>
      General mathematical form:
    </p>
    <blockquote>
      <i>square</i> (<i>t</i>) = <i>A</i> when (<i>t</i> mod <i>T</i>) is less than 0.5 <i>T</i>.<br>
      <i>square</i> (<i>t</i>) = <i>B</i> when (<i>t</i> mod <i>T</i>) is equal to or greater than 0.5 <i>T</i>.
    </blockquote>
    <p>
      Have the following parameters:
    </p>
    <table border="1" cellpadding="2" width="550">
      <tr>
        <th bgcolor="#CCCCCC" align="left"><b>Parameter<b></th>
        <th bgcolor="#CCCCCC" align="left" width="100%"><b>Description<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Units<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Default<b></th>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">A</font></td>
        <td>Low value.</td>
        <td>undefined</td>
        <td>0</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">B</font></td>
        <td>High value.</td>
        <td>undefined</td>
        <td>1</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">f</font></td>
        <td>Frequency.</td>
        <td>Hz</td>
        <td>1 / 2&pi;</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">T</font></td>
        <td>Period.</td>
        <td>s</td>
        <td>2&pi;</td>
      </tr>
    </table>
    <p>
      Note that <i>square</i> represents a particular case of <i>pulse</i> where the signal is high for exactly half of the period.
    </p>

    <h3>Pulse</h3>
    <p>
      General mathematical form:
    </p>
    <blockquote>
      <i>pulse</i> (<i>t</i>) = <i>A</i> when (<i>t</i> mod <i>T</i>) is less than (1 - <i>C</i>) <i>T</i>.<br>
      <i>pulse</i> (<i>t</i>) = <i>B</i> when (<i>t</i> mod <i>T</i>) is equal to or greater than (1 - <i>C</i>) <i>T</i>.
    </blockquote>
    <p>
      Have the following parameters:
    </p>
    <table border="1" cellpadding="2" width="550">
      <tr>
        <th bgcolor="#CCCCCC" align="left"><b>Parameter<b></th>
        <th bgcolor="#CCCCCC" align="left" width="100%"><b>Description<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Units<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Default<b></th>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">A</font></td>
        <td>Low value.</td>
        <td>undefined</td>
        <td>0</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">B</font></td>
        <td>High value.</td>
        <td>undefined</td>
        <td>1</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">C</font></td>
        <td>Fraction of period that signal is high.</td>
        <td>undefined</td>
        <td>0.5</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">f</font></td>
        <td>Frequency.</td>
        <td>Hz</td>
        <td>1 / 2&pi;</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">T</font></td>
        <td>Period.</td>
        <td>s</td>
        <td>2&pi;</td>
      </tr>
    </table>

    <h3>Sawtooth</h3>
    <p>
      The sawtooth function is defined to increase linearly from some lower bound to an upper bound (and subsequently drop straight down again) over a certain period.
    </p>
    <p>
      General mathematical form:
    </p>
    <blockquote>
      <i>sawtooth</i> (<i>t</i>) = <i>A</i> when <i>t</i> is a multiple of <i>T</i>.<br>
      <i>sawtooth'</i> (<i>t</i>) = (<i>B</i> - <i>A</i>) / <i>T</i> for all other values of <i>t</i>.
    </blockquote>
    <p>
      Have the following parameters:
    </p>
    <table border="1" cellpadding="2" width="550">
      <tr>
        <th bgcolor="#CCCCCC" align="left"><b>Parameter<b></th>
        <th bgcolor="#CCCCCC" align="left" width="100%"><b>Description<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Units<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Default<b></th>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">A</font></td>
        <td>Low value.</td>
        <td>undefined</td>
        <td>-1</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">B</font></td>
        <td>High value.</td>
        <td>undefined</td>
        <td>1</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">f</font></td>
        <td>Frequency.</td>
        <td>Hz</td>
        <td>1 / 2&pi;</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">T</font></td>
        <td>Period.</td>
        <td>s</td>
        <td>2&pi;</td>
      </tr>
    </table>
    <p>
      Note that <i>sawtooth</i> is a special instance of <i>triangle</i> which is increasing for the entire period.
    </p>

    <h3>Triangle</h3>
    <p>
      The triangle wave behaves similarly to the sawtooth function, except that it decreases linearly rather than dropping instantaneously. A parameter is supplied to the function, specifying the fraction of the period spent increasing (the remainder is spent decreasing).
    </p>
    <p>
      General mathematical form:
    </p>
    <blockquote>
      <i>triangle</i> (<i>t</i>) = <i>A</i> when <i>t</i> is a multiple of <i>T</i>.<br>
      <i>triangle'</i> (<i>t</i>) = (<i>B</i> - <i>A</i>) / (<i>C</i> &times; <i>T</i>) for all other values satisfying (<i>t</i> mod <i>T</i>) &lt; <i>C</i> &times; <i>T</i>.<br>
      <br>
      <i>triangle</i> (<i>t</i>) = <i>B</i> at the point where (<i>t</i> mod <i>T</i>) = <i>C</i> &times; <i>T</i>.<br>
      <i>triangle'</i> (<i>t</i>) = (<i>A</i> - <i>B</i>) / ((1 - <i>C</i>) &times; <i>T</i>) for all other values satisfying (<i>t</i> mod <i>T</i>) &gt; <i>C</i> &times; <i>T</i>.
    </blockquote>
    <p>
      Have the following parameters:
    </p>
    <table border="1" cellpadding="2" width="550">
      <tr>
        <th bgcolor="#CCCCCC" align="left"><b>Parameter<b></th>
        <th bgcolor="#CCCCCC" align="left" width="100%"><b>Description<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Units<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Default<b></th>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">A</font></td>
        <td>Low value.</td>
        <td>undefined</td>
        <td>-1</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">B</font></td>
        <td>High value.</td>
        <td>undefined</td>
        <td>1</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">C</font></td>
        <td>Fraction of period that function is increasing.</td>
        <td>undefined</td>
        <td>0.5</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">f</font></td>
        <td>Frequency.</td>
        <td>Hz</td>
        <td>1 / 2&pi;</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">T</font></td>
        <td>Period.</td>
        <td>s</td>
        <td>2&pi;</td>
      </tr>
    </table>

    <h3>Constant</h3>
    <p>
      General mathematical form:
    </p>
    <blockquote>
      <i>constant</i> (<i>t</i>) = <i>A</i>
    </blockquote>
    <p>
      Have the following parameters:
    </p>
    <table border="1" cellpadding="2" width="550">
      <tr>
        <th bgcolor="#CCCCCC" align="left"><b>Parameter<b></th>
        <th bgcolor="#CCCCCC" align="left" width="100%"><b>Description<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Units<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Default<b></th>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">A</font></td>
        <td>Value.</td>
        <td>undefined</td>
        <td>1</td>
      </tr>
    </table>

    <h3>Gaussian Noise</h3>
    <p>
      General mathematical form:
    </p>
    <blockquote>
      <i>gaussian</i> (<i>t</i>) = <i>X</i>, where X taken randomly from specified normal distribution.
    </blockquote>
    <p>
      Have the following parameters:
    </p>
    <table border="1" cellpadding="2" width="550">
      <tr>
        <th bgcolor="#CCCCCC" align="left"><b>Parameter<b></th>
        <th bgcolor="#CCCCCC" align="left" width="100%"><b>Description<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Units<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Default<b></th>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">u</font></td>
        <td>Mean.</td>
        <td>undefined</td>
        <td>0</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">s</font></td>
        <td>Standard deviation.</td>
        <td>undefined</td>
        <td>1</td>
      </tr>
    </table>

    <h3>Uniform Noise</h3>
    <p>
      General mathematical form:
    </p>
    <blockquote>
      <i>uniform</i> (<i>t</i>) = <i>X</i>, where X selected randomly from between lower and upper bound.
    </blockquote>
    <p>
      Have the following parameters:
    </p>
    <table border="1" cellpadding="2" width="550">
      <tr>
        <th bgcolor="#CCCCCC" align="left"><b>Parameter<b></th>
        <th bgcolor="#CCCCCC" align="left" width="100%"><b>Description<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Units<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Default<b></th>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">A</font></td>
        <td>Lower bound.</td>
        <td>undefined</td>
        <td>0</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">B</font></td>
        <td>Upper bound.</td>
        <td>undefined</td>
        <td>1</td>
      </tr>
    </table>

    <br>

    <hr>
    <p>
      <i>Copyright &copy; CSIRO Manufacturing and Infrastructure Technology<i>
    </p>

 *
    <h2>Command Line</h2>
    <p>
      The command line is of the form
    </p>
    <blockquote>
      ddxsiggen -config <i>configFile</i>
    </blockquote>
    <p>
      where the following are valid examples:
    </p>
    <blockquote>
      <font face="courier" size="-1">ddxsiggen -config file.conf</font><br>
      <font face="courier" size="-1">ddxsiggen -config path/file.conf</font>
    </blockquote>

    <h2>Configuration File</h2>
    <p>
      The following options are specified through the configuration file, which is identified in the command line:
    </p>
    <ul>
      <li><b>Update interval</b> - Required interval for updating function values.</li>
      <li><b>Signal names</b> - Variable names that will appear in the DDX store for each function.</li>
      <li><b>Function types</b> - The general class of each function, and parameters determining the specific shape.</li>
    </ul>

    <p>
      The syntax for the configuration file is described by the following set of regular expressions.<br>
    </p>
    <table border="1" bgcolor="#EEEEEE" cellpadding="5" cellspacing="a">
    <tr>
    <td>
      <font face="times new roman">
        <i>Configuration</i> = <i>Interval</i> <i>Signal</i>*<br>
        <br>
        <i>Interval</i> = interval <i>RealNumber</i><br>
        <br>
        <i>Signal</i> = <i>SignalName</i> <i>SignalType</i> <i>SignalParams</i>?<br>
        <i>SignalName</i> = signal <i>Identifier</i><br>
        <i>SignalType</i> = type (sine | cosine | square | pulse | sawtooth | triangle | constant | gauss | uniform)<br>
        <i>SignalParams</i> = params (<i>Identifier</i> = <i>Value</i>)+<br>
        <br>
        <i>Comment</i> = % <i>Line</i><br>
        <br>
        (Assuming patterns <i>RealNumber</i>, <i>Identifier</i>, <i>Value</i> and <i>Line</i> are defined reasonably.)
      </font>
    </td>
    </tr>
    </table>
    <p>
      Layout is free-form, but with the strict condition that white space appears between all tokens. This means, for example, that "f = 5" is a valid parameter assignment but "f=5" is not. You should take great care to ensure the configuration file is valid, because there is only minimal support for handling errors.
    </p>
    <p>
      An example configuration file (with two signals specified) might be:
    </p>
    <table border="1" bgcolor="#EEEEEE" cellpadding="5" cellspacing="a">
    <tr>
    <td>
      <font face="courier" size="-1">
        % Comments may appear at the end of any line<br>
        <br>
        interval 0.01<br>
        <br>
        signal harry<br>
        type sine<br>
        params<br>
        &nbsp;&nbsp;A = 5<br>
        &nbsp;&nbsp;f = 1000<br>
        &nbsp;&nbsp;C = 0<br>
        <br>
        signal sarah<br>
        type triangle<br>
        params<br>
        &nbsp;&nbsp;A = -1&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;% Lower bound<br>
        &nbsp;&nbsp;B = 1&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;% Upper bound<br>
        &nbsp;&nbsp;T = 0.0001&nbsp;&nbsp;% Period 0.1 ms<br>
      </font>
    </td>
    </tr>
    </table>

    <h2>Types of Functions</h2>
    <p>
      The table below describes the types of functions that are supported.
    </p>
    <table border="1" cellpadding="2">
      <tr>
        <th bgcolor="#CCCCCC" align="left"><b>Function<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Description<b></th>
      </tr>
      <tr>
        <td>Sine</td>
        <td>As you would expect.</td>
      </tr>
      <tr>
        <td>Cosine</td>
        <td>Some people prefer cosine.</td>

      </tr>
      <tr>
        <td>Square</td>
        <td>Standard square wave.</td>
      </tr>
      <tr>
        <td>Pulse</td>
        <td>Similar to square wave, but with specified dute cycle.</td>
      </tr>
      <tr>
        <td>Sawtooth</td>
        <td>Standard sawtooth pattern.</td>
      </tr>
      <tr>
        <td>Triangle</td>
        <td>Similar to sawtooth, except also linearly decreasing.</td>
      </tr>
      <tr>
        <td>Constant</td>
        <td>Standard constant function.</td>
      </tr>
      <tr>
        <td>Gaussian Noise</td>
        <td>Random values from particular distribution.</td>
      </tr>
      <tr>
        <td>Uniform Noise</td>
        <td>Random values from uniform distribution.</td>
      </tr>
    </table>

    <h2>Function Parameters</h2>
    <p>
      The characteristics of each generated signal need to be supplied by the user in the configuration file. These depend on the function type chosen, as described for each below. All measurements are expressed using SI units.
    </p>
    <p>
      Specification of any parameter is optional, with the program assuming a particular default value if necessary. If fact, the entire <i>params</i> clause of a signal definition may be ommited from the configuration file if the user only requires standard behaviour. Default characteristics for each function type are described in the tables that follow.
    </p>
    <p>
      If the frequency is provided, then the period should not be (this point also applies conversely). If both are present, then the program will use the assignment that appears lowest in the configuration file.
    </p>

    <h3>Sine and Cosine</h3>
    <p>
      General mathematical form:
    </p>
    <blockquote>
      <i>sinfunc</i> (<i>t</i>) = <i>A</i> sin(<i>B</i> <i>t</i> + <i>C</i>) + <i>D</i><br>
      <i>cosfunc</i> (<i>t</i>) = <i>A</i> cos(<i>B</i> <i>t</i> + <i>C</i>) + <i>D</i>
    </blockquote>
    <p>
      Have the following parameters:
    </p>
    <table border="1" cellpadding="2" width="550">
      <tr>
        <th bgcolor="#CCCCCC" align="left"><b>Parameter<b></th>
        <th bgcolor="#CCCCCC" align="left" width="100%"><b>Description<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Units<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Default<b></th>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">A</font></td>
        <td>Amplitude.</td>
        <td>undefined</td>
        <td>1</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">B</font></td>
        <td>The angular frequency.</td>
        <td>rad&middot;s<sup>-1</sup></td>
        <td>1</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">C</font></td>
        <td>The phase angle.</td>
        <td>rad</td>
        <td>0</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">D</font></td>
        <td>Offset value.</td>
        <td>undefined</td>
        <td>0</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">f</font></td>
        <td>Frequency.</td>
        <td>Hz</td>
        <td>1 / 2&pi;</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">T</font></td>
        <td>Period.</td>
        <td>s</td>
        <td>2&pi;</td>
      </tr>
    </table>
    <p>
      Note that if <font face="courier" size="-1">B</font> is specified, then neither <font face="courier" size="-1">f</font> nor <font face="courier" size="-1">T</font> should be provided (and inversely).
    </p>

    <h3>Square</h3>
    <p>
      General mathematical form:
    </p>
    <blockquote>
      <i>square</i> (<i>t</i>) = <i>A</i> when (<i>t</i> mod <i>T</i>) is less than 0.5 <i>T</i>.<br>
      <i>square</i> (<i>t</i>) = <i>B</i> when (<i>t</i> mod <i>T</i>) is equal to or greater than 0.5 <i>T</i>.
    </blockquote>
    <p>
      Have the following parameters:
    </p>
    <table border="1" cellpadding="2" width="550">
      <tr>
        <th bgcolor="#CCCCCC" align="left"><b>Parameter<b></th>
        <th bgcolor="#CCCCCC" align="left" width="100%"><b>Description<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Units<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Default<b></th>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">A</font></td>
        <td>Low value.</td>
        <td>undefined</td>
        <td>0</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">B</font></td>
        <td>High value.</td>
        <td>undefined</td>
        <td>1</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">f</font></td>
        <td>Frequency.</td>
        <td>Hz</td>
        <td>1 / 2&pi;</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">T</font></td>
        <td>Period.</td>
        <td>s</td>
        <td>2&pi;</td>
      </tr>
    </table>
    <p>
      Note that <i>square</i> represents a particular case of <i>pulse</i> where the signal is high for exactly half of the period.
    </p>

    <h3>Pulse</h3>
    <p>
      General mathematical form:
    </p>
    <blockquote>
      <i>pulse</i> (<i>t</i>) = <i>A</i> when (<i>t</i> mod <i>T</i>) is less than (1 - <i>C</i>) <i>T</i>.<br>
      <i>pulse</i> (<i>t</i>) = <i>B</i> when (<i>t</i> mod <i>T</i>) is equal to or greater than (1 - <i>C</i>) <i>T</i>.
    </blockquote>
    <p>
      Have the following parameters:
    </p>
    <table border="1" cellpadding="2" width="550">
      <tr>
        <th bgcolor="#CCCCCC" align="left"><b>Parameter<b></th>
        <th bgcolor="#CCCCCC" align="left" width="100%"><b>Description<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Units<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Default<b></th>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">A</font></td>
        <td>Low value.</td>
        <td>undefined</td>
        <td>0</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">B</font></td>
        <td>High value.</td>
        <td>undefined</td>
        <td>1</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">C</font></td>
        <td>Fraction of period that signal is high.</td>
        <td>undefined</td>
        <td>0.5</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">f</font></td>
        <td>Frequency.</td>
        <td>Hz</td>
        <td>1 / 2&pi;</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">T</font></td>
        <td>Period.</td>
        <td>s</td>
        <td>2&pi;</td>
      </tr>
    </table>

    <h3>Sawtooth</h3>
    <p>
      The sawtooth function is defined to increase linearly from some lower bound to an upper bound (and subsequently drop straight down again) over a certain period.
    </p>
    <p>
      General mathematical form:
    </p>
    <blockquote>
      <i>sawtooth</i> (<i>t</i>) = <i>A</i> when <i>t</i> is a multiple of <i>T</i>.<br>
      <i>sawtooth'</i> (<i>t</i>) = (<i>B</i> - <i>A</i>) / <i>T</i> for all other values of <i>t</i>.
    </blockquote>
    <p>
      Have the following parameters:
    </p>
    <table border="1" cellpadding="2" width="550">
      <tr>
        <th bgcolor="#CCCCCC" align="left"><b>Parameter<b></th>
        <th bgcolor="#CCCCCC" align="left" width="100%"><b>Description<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Units<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Default<b></th>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">A</font></td>
        <td>Low value.</td>
        <td>undefined</td>
        <td>-1</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">B</font></td>
        <td>High value.</td>
        <td>undefined</td>
        <td>1</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">f</font></td>
        <td>Frequency.</td>
        <td>Hz</td>
        <td>1 / 2&pi;</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">T</font></td>
        <td>Period.</td>
        <td>s</td>
        <td>2&pi;</td>
      </tr>
    </table>
    <p>
      Note that <i>sawtooth</i> is a special instance of <i>triangle</i> which is increasing for the entire period.
    </p>

    <h3>Triangle</h3>
    <p>
      The triangle wave behaves similarly to the sawtooth function, except that it decreases linearly rather than dropping instantaneously. A parameter is supplied to the function, specifying the fraction of the period spent increasing (the remainder is spent decreasing).
    </p>
    <p>
      General mathematical form:
    </p>
    <blockquote>
      <i>triangle</i> (<i>t</i>) = <i>A</i> when <i>t</i> is a multiple of <i>T</i>.<br>
      <i>triangle'</i> (<i>t</i>) = (<i>B</i> - <i>A</i>) / (<i>C</i> &times; <i>T</i>) for all other values satisfying (<i>t</i> mod <i>T</i>) &lt; <i>C</i> &times; <i>T</i>.<br>
      <br>
      <i>triangle</i> (<i>t</i>) = <i>B</i> at the point where (<i>t</i> mod <i>T</i>) = <i>C</i> &times; <i>T</i>.<br>
      <i>triangle'</i> (<i>t</i>) = (<i>A</i> - <i>B</i>) / ((1 - <i>C</i>) &times; <i>T</i>) for all other values satisfying (<i>t</i> mod <i>T</i>) &gt; <i>C</i> &times; <i>T</i>.
    </blockquote>
    <p>
      Have the following parameters:
    </p>
    <table border="1" cellpadding="2" width="550">
      <tr>
        <th bgcolor="#CCCCCC" align="left"><b>Parameter<b></th>
        <th bgcolor="#CCCCCC" align="left" width="100%"><b>Description<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Units<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Default<b></th>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">A</font></td>
        <td>Low value.</td>
        <td>undefined</td>
        <td>-1</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">B</font></td>
        <td>High value.</td>
        <td>undefined</td>
        <td>1</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">C</font></td>
        <td>Fraction of period that function is increasing.</td>
        <td>undefined</td>
        <td>0.5</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">f</font></td>
        <td>Frequency.</td>
        <td>Hz</td>
        <td>1 / 2&pi;</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">T</font></td>
        <td>Period.</td>
        <td>s</td>
        <td>2&pi;</td>
      </tr>
    </table>

    <h3>Constant</h3>
    <p>
      General mathematical form:
    </p>
    <blockquote>
      <i>constant</i> (<i>t</i>) = <i>A</i>
    </blockquote>
    <p>
      Have the following parameters:
    </p>
    <table border="1" cellpadding="2" width="550">
      <tr>
        <th bgcolor="#CCCCCC" align="left"><b>Parameter<b></th>
        <th bgcolor="#CCCCCC" align="left" width="100%"><b>Description<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Units<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Default<b></th>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">A</font></td>
        <td>Value.</td>
        <td>undefined</td>
        <td>1</td>
      </tr>
    </table>

    <h3>Gaussian Noise</h3>
    <p>
      General mathematical form:
    </p>
    <blockquote>
      <i>gaussian</i> (<i>t</i>) = <i>X</i>, where X taken randomly from specified normal distribution.
    </blockquote>
    <p>
      Have the following parameters:
    </p>
    <table border="1" cellpadding="2" width="550">
      <tr>
        <th bgcolor="#CCCCCC" align="left"><b>Parameter<b></th>
        <th bgcolor="#CCCCCC" align="left" width="100%"><b>Description<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Units<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Default<b></th>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">u</font></td>
        <td>Mean.</td>
        <td>undefined</td>
        <td>0</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">s</font></td>
        <td>Standard deviation.</td>
        <td>undefined</td>
        <td>1</td>
      </tr>
    </table>

    <h3>Uniform Noise</h3>
    <p>
      General mathematical form:
    </p>
    <blockquote>
      <i>uniform</i> (<i>t</i>) = <i>X</i>, where X selected randomly from between lower and upper bound.
    </blockquote>
    <p>
      Have the following parameters:
    </p>
    <table border="1" cellpadding="2" width="550">
      <tr>
        <th bgcolor="#CCCCCC" align="left"><b>Parameter<b></th>
        <th bgcolor="#CCCCCC" align="left" width="100%"><b>Description<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Units<b></th>
        <th bgcolor="#CCCCCC" align="left"><b>Default<b></th>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">A</font></td>
        <td>Lower bound.</td>
        <td>undefined</td>
        <td>0</td>
      </tr>
      <tr>
        <td align="center"><font face="courier" size="-1">B</font></td>
        <td>Upper bound.</td>
        <td>undefined</td>
        <td>1</td>
      </tr>
    </table>

    <br>

 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>
#include <math.h>
#include <sys/stat.h>
#ifdef __linux__
#include <sys/file.h>
#include <libgen.h>
#else
#include <sys/lock.h>
#endif
#include <errno.h>
#include <signal.h>
#ifdef __sparc__
#include <libgen.h>
#endif

#include <rtx/message.h>
#include <rtx/error.h>
#include <rtx/thread.h>
#include <rtx/main.h>
#include <rtx/param.h>
#include <rtx/mutex.h>
#include <rtx/log.h>
#include <rtx/parse.h>
#include <rtx/signal.h>
#include <rtx/time.h>
#include <rtx/timer.h>
#include <rtx/getopt.h>
#include <rtx/list.h>

#include <ddx.h>

/* Function types. */

typedef enum _function_type {
    FUNCTION_TYPE_SINE,     // Used to represent COSINE (phase shifted)
    FUNCTION_TYPE_PULSE,    // Used to represent SQUARE function (special case)
    FUNCTION_TYPE_TRIANGLE, // Used to represent SAWTOOTH aswell (special case)
    FUNCTION_TYPE_CONSTANT,
    FUNCTION_TYPE_GAUSS,
    FUNCTION_TYPE_UNIFORM
} FunctionType;

/* 
 * Set of structures for representing characteristics of various function
 * types. The struct "SignalProfile" represents an asbtract superclass used
 * as a parent data type.  The ability for subclass structures to be upcast
 * relies on memory alignment (all have "name" and "type" as first fields).
 *
 */

typedef struct _signal_profile {
    char* name;
    FunctionType type;
    DDX_STORE_ITEM* item;
} SignalProfile;

typedef struct _sine_profile {
    char* name;
    FunctionType type;
    DDX_STORE_ITEM* item;
    double A;
    double B;
    double C; // Set to M_PI_2 for cosine.
    double D;
} SineProfile;

typedef struct _pulse_profile {
    char* name;
    FunctionType type;
    DDX_STORE_ITEM* item;
    double A;
    double B;
    double C; // Set to 0.5 for square wave.
    double T;
} PulseProfile;

typedef struct _triangle_profile {
    char* name;
    FunctionType type;
    DDX_STORE_ITEM* item;
    double A;
    double B;
    double C; // Set to 1.0 for sawtooth.
    double T;
} TriangleProfile;

typedef struct _constant_profile {
    char* name;
    FunctionType type;
    DDX_STORE_ITEM* item;
    double A;
} ConstantProfile;

typedef struct _gauss_profile {
    char* name;
    FunctionType type;
    DDX_STORE_ITEM* item;
    double u;
    double s;
} GaussProfile;

typedef struct _uniform_profile {
    char* name;
    FunctionType type;
    DDX_STORE_ITEM* item;
    double A;
    double B;
} UniformProfile;

/* 
 *  Structures needed because only one argument can be supplied to a function
 *  passed to a POSIX thread or timer. A single structure effectively contains
 *  many arguments.
 *
 */

/* Used for function linked to update timer. */

typedef struct _update_timer_arg {
    double interval;
    RtxList* signals;
} UpdateTimerArg;

/* Forward definitions. */

int main (int argc, char* argv[]);
static void* updateTimerFunc(void* arg);
static double evaluateSignal(SignalProfile* signal, double time);
static double evaluateSine(SineProfile* signal, double time);
static double evaluatePulse(PulseProfile* signal, double time);
static double evaluateTriangle(TriangleProfile* signal, double time);
static double evaluateConstant(ConstantProfile* signal, double time);
static double evaluateGauss(GaussProfile* signal, double time);
static double evaluateUniform(UniformProfile* signal, double time);
static int processCommandLine(int argc, char* argv[], char** conf);
static int processConfigFile(char* conf, double* interval, RtxList** signals);
static int parseInterval(char*** tokens, double* interval);
static SignalProfile* parseSignal(char*** tokens);
static char* parseSignalName(char*** tokens);
static char* parseSignalType(char*** tokens);
static RtxList* parseSignalParams(char*** tokens);
static SignalProfile* constructSine(char* name, RtxList* params, int cosine);
static SignalProfile* constructPulse(char* name, RtxList* params, int square);
static SignalProfile*
    constructTriangle(char* name, RtxList* params, int sawtooth);
static SignalProfile* constructConstant(char* name, RtxList* params);
static SignalProfile* constructGauss(char* name, RtxList* params);
static SignalProfile* constructUniform(char* name, RtxList* params);
static int accept(char* expected, char* actual);
static int parseDouble(char* token, double* result);
static char* parseString(char* token);
static char** tokeniseFile(char* fileName);

/**
 * Main program.
 *
 */
int main (int argc, char* argv[])
{
    char* conf;                   // Path of configuration file.
    RtxList* signals;             // List of signal profiles.
    SignalProfile* signal;        // Used for iterating over list.
    double interval;              // Interval (in seconds) for updating.
    RtxTimerThread* timerThread;  // Thread to perform periodic updating.
    UpdateTimerArg timerArg;      // Argument structure for update function.
    DDX_STORE_ID* storeId;        // The DDX store.

    /* Seed random number generator to make sequence unique. */

    srand(time(0));

    /* Initialise error handling. */

    rtx_main_init("ddxsiggen", RTX_ERROR_STDERR | RTX_ERROR_MESSAGE);

    /* Obtain signal characteristics from configuration file. */

    if(processCommandLine(argc, argv, &conf) == -1)
    {
        return rtx_error("main: processCommandLine() failed");
    }

    if(processConfigFile(conf, &interval, &signals) == -1)
    {
        return rtx_error("main: processConfigFile() failed");
    }

    /* Initialise DDX variables and connect to store. */

    if(ddx_client_init(0) == -1)
    {
        return rtx_error("main: ddx_client_init() failed");
    }
    if((storeId = ddx_store_open(NULL, 0, 5)) == NULL)
    {
        return rtx_error("main: ddx_store_open() failed");
    }

    while((signal = rtx_list_iterate(signals)) != NULL)
    {
        signal->item =
            ddx_store_lookup_item(
                storeId, strdup(signal->name), "double", sizeof(double)
            );
        if(signal->item == NULL)
        {
            return rtx_error("main: ddx_store_lookup_item() failed");
        }
    }

    /* Create thread to handle value updating. */

    timerArg.interval = interval;
    timerArg.signals = signals;

    timerThread =
        rtx_timer_create_thread(
            interval, interval, 0, (void(*)(void*))updateTimerFunc,
            &timerArg, RTX_TIMER_CLOCK_REALTIME
        );
    if(timerThread == NULL)
    {
        return rtx_error("main: rtx_time_create() failed");
    }

    /* Clean up before terminating. */

    if(rtx_main_wait_shutdown(0) == -1)
    {
        rtx_error_flush("main: rtx_main_wait_shutdown() failed");
    }
    if(rtx_timer_destroy_thread(timerThread) == -1)
    {
        rtx_error_flush("main: rtx_timer_destroy_thread() failed");
    }

    while((signal = rtx_list_iterate(signals)) != NULL)
    {
        if(ddx_store_done_item(signal->item) == -1)
        {
            rtx_error_flush("main: ddx_store_done_item() failed");
        }
    }
    if(ddx_store_close(storeId) == -1)
    {
        rtx_error_flush("main: ddx_store_close() failed");
    }
    if(ddx_client_done() == -1)
    {
        rtx_error_flush("main: ddx_client_done() failed");
    }

    rtx_list_destroy(signals, 1);  // Will also "free" signal data

    return 0;
}

/*
 * Called on each instance that the timer fires. This is where the periodic
 * update of signal values occurs.
 *
 */
static void* updateTimerFunc(void* arg)
{
    UpdateTimerArg* timerArg;        // To cast supplied argument.
    SignalProfile* signal;           // Used for iterating over list.
    static double numIntervals = 0;  // Number of intervals passed.
    double time;                     // Time passed (in seconds).
    double value;                    // Store value of each function.

    timerArg = (UpdateTimerArg*)arg;

    /* Calculate current time value. */

    time = numIntervals * timerArg->interval;

    /* Iteratively update function value for each signal. */

    while((signal = rtx_list_iterate(timerArg->signals)) != NULL)
    {
        value = evaluateSignal(signal, time);
        if(ddx_store_write(signal->item, &value, NULL) == -1)
        {
            rtx_error_flush("updateTimerFunc: ddx_store_write() failed");
        }
    }

    /* Increment interval count. */

    numIntervals++;

    return NULL;
}

/*
 * Evaluates signal at specified time. This function is a generic interface to
 * the more specific ones below.
 *
 */
static double evaluateSignal(SignalProfile* signal, double time)
{
    switch (signal->type)
    {
        case FUNCTION_TYPE_SINE:
            return evaluateSine((SineProfile*)signal, time);
            break;
        case FUNCTION_TYPE_PULSE:
            return evaluatePulse((PulseProfile*)signal, time);
            break;
        case FUNCTION_TYPE_TRIANGLE:
            return evaluateTriangle((TriangleProfile*)signal, time);
            break;
        case FUNCTION_TYPE_CONSTANT:
            return evaluateConstant((ConstantProfile*)signal, time);
            break;
        case FUNCTION_TYPE_GAUSS:
            return evaluateGauss((GaussProfile*)signal, time);
            break;
        case FUNCTION_TYPE_UNIFORM:
            return evaluateUniform((UniformProfile*)signal, time);
            break;
        default:
            return 0;
            break;
    }
}

/*
 * Evaluates sine function at specified time.
 *
 */
static double evaluateSine(SineProfile* signal, double time)
{
    return (signal->A) * sin((signal->B) * time + (signal->C)) + (signal->D);
}

/*
 * Evaluates pulse function at specified time.
 *
 */
static double evaluatePulse(PulseProfile* signal, double time)
{
    if(fmod(time, signal->T) < (1 - signal->C) * signal->T)
    {
        return (signal->A);
    }
    else
    {
        return (signal->B);
    }
}

/*
 * Evaluates triangle function at specified time.
 *
 */
static double evaluateTriangle(TriangleProfile* signal, double time)
{
    double timeMod;

    timeMod = fmod(time, signal->T);

    if(timeMod < (signal->C * signal->T))
    {
        return
            (signal->A) +
            (signal->B - signal->A) / (signal->C * signal->T) * timeMod;
    }
    else
    {
        return
            (signal->B) +
            (signal->A - signal->B) / ((1 - signal->C) * signal->T) *
            (timeMod - (signal->C * signal->T));
    }
}

/*
 * Evaluates constant function at specified time.
 *
 */
static double evaluateConstant(ConstantProfile* signal, double time)
{
    return (signal->A);
}

/*
 * Applies the central limit theorem to produce normal distribution for sum of
 * N (assumed independent and uniformly distributed) random numbers. This is
 * then adjusted to be a unit normal random, which is subsequently modified to
 * have the desired mean and standard deviation.
 *
 */
static double evaluateGauss(GaussProfile* signal, double time)
{
    int N = 1000; // Number of independent random variables (need at least 20)
    double X;     // Use to accumulate sum.
    int i;        // Looping variable.

    /* Accumulate sum of N randoms. */

    X = 0;
    for(i = 0; i < N; i++)
    {
        X = X + (1.0 * rand() / RAND_MAX);
    }

    /* Modify distribution to have desired properties. */

    X = (X - 1.0*N/2) / sqrt(1.0*N/12); // Now mean = 0 and variance = 1.
    X = signal->u + signal->s * X;      // Now mean = u and variable = s^2.

    return X;
}

/*
 * Generates a random number between certain lower and upper bound.
 *
 */
static double evaluateUniform(UniformProfile* signal, double time)
{
    return signal->A + (1.0 * rand() / RAND_MAX) * (signal->B - signal->A);
}

/*
 * Processes command-line arguments represented by argc and argv, resulting in
 * conf pointing to a string containing the path of the configuration file.
 *
 */
static int processCommandLine(int argc, char* argv[], char** conf)
{

    RtxGetopt ddxSigGenOpts[] =
    {
        RTX_GETOPT_END
    };

    char* ddxSigGenHelpStr = "Read the user manual.";

    rtx_getopt_disable_config(); // Use own config file parser

    if(RTX_GETOPT_CMD(ddxSigGenOpts, argc, argv, rcsid, ddxSigGenHelpStr) == -1)
    {
        return rtx_error("processCommandLine: RTX_GETOPT_CMD() failed");
    }

    *conf = rtx_getopt_get_config("ddxsiggen.conf");

    return 0;
}

/*
 * Parses configuration file, resulting in "interval" pointing to desired period
 * for updating values and "signals" pointing to a list of structures containing
 * the details of desired functions.
 *
 * Assumes the following grammar (using EBNF notation):
 *
 *   (slightly less strict that format described in user manual)
 *   (allows for "params" keyword to appear with no actual data)
 * 
 *   Configuration -> Interval {Signal}
 * 
 *   Interval -> interval RealNumber
 * 
 *   Signal -> SignalName SignalType [SignalParams]
 *   SignalName -> signal Identifier
 *   SignalType -> type (sine | cosine | square | pulse | ...)
 *   SignalParams -> params {Identifier = Value}
 *
 * Uses one-token lookahead parsing.
 *
 */
static int processConfigFile(char* conf, double* interval, RtxList** signals)
{
    char** tokens;          // Strings extracted from configuration file.
    char** working;         // Pointer for iterating over "tokens" array.
    SignalProfile* signal;  // Temporarily used before signal added to list. 

    /* Tokenise configuration file. */

    if((tokens = tokeniseFile(conf)) == NULL)
    {
        return rtx_error("processConfigFile: tokeniseFile() failed");
    }

    /* Perform parsing. */

    working = tokens; // Create new pointer to modify

    if(parseInterval(&working, interval) == -1)
    {
        return rtx_error("processConfigFile: parseInterval() failed");
    }

    if((*signals = rtx_list_init()) == NULL)
    {
        return rtx_error("processConfigFile: rtx_list_init() failed");
    }
    while(*working != NULL)
    {
        if((signal = parseSignal(&working)) == NULL)
        {
            return rtx_error("processConfigFile: parseSignal() failed");
        }
        if(rtx_list_add(*signals, strdup(signal->name), signal) == -1)
        {
            return rtx_error("processConfigFile: rtx_list_add() failed");
        }
    }

    /* Free memory allocated for tokens. */

    working = tokens;
    while(*working != NULL)
    {
        free(*(working++));
    }
    free(tokens);

    return 0;
}

/*
 * Parse update interval.
 *
 */
static int parseInterval(char*** tokens, double* interval)
{
    if(accept("interval", *((*tokens)++)) == -1)
    {
        return rtx_error("parseInterval: accept() failed");
    }
    if(parseDouble(*((*tokens)++), interval) == -1)
    {
        return rtx_error("parseInterval: parseDouble() failed");
    }
    return 0;
}

/*
 * Parse a signal.
 *
 */
static SignalProfile* parseSignal(char*** tokens)
{
    char* name;             // Name of signal.
    char* typeStr;          // String representing type of signal.
    RtxList* params;        // List for name-value pairs for parameters.
    SignalProfile* signal;  // Pointer to returned signal.

    /* Parse name of signal */

    if((name = parseSignalName(tokens)) == NULL)
    {
        return rtx_error_null("parseSignal: parseSignalName() failed");
    }

    /* Parse signal type */

    if((typeStr = parseSignalType(tokens)) == NULL)
    {
        return rtx_error_null("parseSignal: parseSignalType() failed");
    }

    /* Parse signal parameters (if any) */

    if((params = parseSignalParams(tokens)) == NULL)
    {
        return rtx_error_null("parseSignal: parseSignalParams() failed");
    }

    /* Now perform type-specific function building. */

    if(strcmp(typeStr, "sine") == 0)
    {
        signal = constructSine(name, params, 0);
    }
    if(strcmp(typeStr, "cosine") == 0)
    {
        signal = constructSine(name, params, 1);
    }
    else if(strcmp(typeStr, "pulse") == 0)
    {
        signal = constructPulse(name, params, 0);
    }
    else if(strcmp(typeStr, "square") == 0)
    {
        signal = constructPulse(name, params, 1);
    }
    else if(strcmp(typeStr, "triangle") == 0)
    {
        signal = constructTriangle(name, params, 0);
    }
    else if(strcmp(typeStr, "sawtooth") == 0)
    {
        signal = constructTriangle(name, params, 1);
    }
    else if(strcmp(typeStr, "constant") == 0)
    {
        signal = constructConstant(name, params);
    }
    else if(strcmp(typeStr, "gauss") == 0)
    {
        signal = constructGauss(name, params);
    }
    else if(strcmp(typeStr, "uniform") == 0)
    {
        signal = constructUniform(name, params);
    }

    rtx_list_destroy(params, 1); // Clean up

    /* Check for error in type-specific functions */

    if(signal == NULL)
    {
        return rtx_error_null("parseSignal: parseXSpecific failed");
    }

    return signal;
}

/*
 * Parse a signal name.
 *
 */
static char* parseSignalName(char*** tokens)
{
    char* name;
    if(accept("signal", *((*tokens)++)) == -1)
    {
        return rtx_error_null("parseSignalName: accept() failed");
    }
    if((name = parseString(*((*tokens)++))) == NULL)
    {
        return rtx_error_null("parseSignalName: parseString() failed");
    }
    return name;
}

/*
 * Parse a signal type.
 *
 */
static char* parseSignalType(char*** tokens)
{
    char* typeStr;
    if(accept("type", *((*tokens)++)) == -1)
    {
        return rtx_error_null("parseSignalType: accept() failed");
    }
    if((typeStr = parseString(*((*tokens)++))) == NULL)
    {
        return rtx_error_null("parseSignalType: parseString() failed");
    }
    return typeStr;
}

/*
 * Parse signal parameters.
 *
 */
static RtxList* parseSignalParams(char*** tokens)
{
    RtxList* params;
    char* name;
    double* value;

    if((params = rtx_list_init()) == NULL)
    {
        return rtx_error_null("parseSignalParams: rtx_list_init() failed");
    }

    if(strcmp(**tokens, "params") == 0)
    {
        if(accept("params", *((*tokens)++)) == -1)
        {
            return rtx_error_null("parseSignalParams: accept() failed");
        }
        while(**tokens != NULL && strcmp(**tokens, "signal") != 0)
        {
            if((name = parseString(*((*tokens)++))) == NULL)
            {
                rtx_list_destroy(params, 1);
                return
                    rtx_error_null("parseSignalParams: parseString() failed");
            }
            if(accept("=", *((*tokens)++)) == -1)
            {
                rtx_list_destroy(params, 1);
                return rtx_error_null("parseSignalParams: accept() failed");
            }
            value = (double*)calloc(1, sizeof(double));
            if(parseDouble(*((*tokens)++), value) == -1)
            {
                rtx_list_destroy(params, 1);
                free(value);
                return
                    rtx_error_null("parseSignalParams: parseDouble() failed");
            }
            if(rtx_list_add(params, name, value) == -1)
            {
                rtx_list_destroy(params, 1);
                return
                    rtx_error_null("parseSignalParams: rtx_list_add() failed");
            }
        }
    }

    return params;
}

/*
 * Construct a sine signal.
 *
 */
static SignalProfile* constructSine(char* name, RtxList* params, int cosine)
{
    SineProfile* signal;
    double* A;
    double* B;
    double* C;
    double* D;
    double* f;
    double* T;

    signal = (SineProfile*)calloc(1, sizeof(SineProfile));
    signal->name = strdup(name);
    signal->type = FUNCTION_TYPE_SINE;

    /* Default values. */

    signal->A = 1;
    signal->B = 1;
    signal->C = 0;
    signal->D = 0;

    /* Specified values. */

    if((A = rtx_list_lookup(params, "A")) != NULL) 
    {
        signal->A = *A;
    }
    if((B = rtx_list_lookup(params, "B")) != NULL) 
    {
        signal->B = *B;
    }
    if((C = rtx_list_lookup(params, "C")) != NULL) 
    {
        signal->C = *C;
    }
    if((D = rtx_list_lookup(params, "D")) != NULL) 
    {
        signal->D = *D;
    }
    if((f = rtx_list_lookup(params, "f")) != NULL) 
    {
        signal->B = 2 * M_PI * (*f);
    }
    if((T = rtx_list_lookup(params, "T")) != NULL) 
    {
        signal->B = 2 * M_PI / (*T);
    }

    /* Handle cosine case. */

    if(cosine)
        signal->C = signal->C + M_PI_2;

    return (SignalProfile*)signal;
}

/*
 * Construct a pulse signal.
 *
 */
static SignalProfile* constructPulse(char* name, RtxList* params, int square)
{
    PulseProfile* signal;
    double* A;
    double* B;
    double* C;
    double* f;
    double* T;

    signal = (PulseProfile*)calloc(1, sizeof(PulseProfile));
    signal->name = strdup(name);
    signal->type = FUNCTION_TYPE_PULSE;

    /* Default values. */

    signal->A = 0;
    signal->B = 1;
    signal->C = 0.5;
    signal->T = 2 * M_PI;

    /* Specified values. */
 
    if((A = rtx_list_lookup(params, "A")) != NULL) 
    {
        signal->A = *A;
    }
    if((B = rtx_list_lookup(params, "B")) != NULL) 
    {
        signal->B = *B;
    }
    if((C = rtx_list_lookup(params, "C")) != NULL) 
    {
        signal->C = *C;
    }
    if((f = rtx_list_lookup(params, "f")) != NULL) 
    {
        signal->T = 1 / (*f);
    }
    if((T = rtx_list_lookup(params, "T")) != NULL) 
    {
        signal->T = *T;
    }

    /* Handle square case. */

    if(square)
        signal->C = 0.5;

    return (SignalProfile*)signal;
}

/*
 * Construct a triangle signal.
 *
 */
static SignalProfile*
    constructTriangle(char* name, RtxList* params, int sawtooth)
{
    TriangleProfile* signal;
    double* A;
    double* B;
    double* C;
    double* f;
    double* T;

    signal = (TriangleProfile*)calloc(1, sizeof(TriangleProfile));
    signal->name = strdup(name);
    signal->type = FUNCTION_TYPE_TRIANGLE;

    /* Default values. */

    signal->A = -1;
    signal->B = 1;
    signal->C = 0.5;
    signal->T = 2 * M_PI;

    /* Specified values. */

    if((A = rtx_list_lookup(params, "A")) != NULL) 
    {
        signal->A = *A;
    }
    if((B = rtx_list_lookup(params, "B")) != NULL) 
    {
        signal->B = *B;
    }
    if((C = rtx_list_lookup(params, "C")) != NULL) 
    {
        signal->C = *C;
    }
    if((f = rtx_list_lookup(params, "f")) != NULL) 
    {
        signal->T = 1 / (*f);
    }
    if((T = rtx_list_lookup(params, "T")) != NULL) 
    {
        signal->T = *T;
    }

    /* Handle sawtooth case. */

    if(sawtooth)
        signal->C = 1;

    return (SignalProfile*)signal;
}

/*
 * Construct a constant signal.
 *
 */
static SignalProfile* constructConstant(char* name, RtxList* params)
{
    ConstantProfile* signal;
    double* A;

    signal = (ConstantProfile*)calloc(1, sizeof(ConstantProfile));
    signal->name = strdup(name);
    signal->type = FUNCTION_TYPE_CONSTANT;

    /* Default values. */

    signal->A = 1;

    /* Specified values. */

    if((A = rtx_list_lookup(params, "A")) != NULL) 
    {
        signal->A = *A;
    }

    return (SignalProfile*)signal;
}

/*
 * Construct a gaussian noise signal.
 *
 */
static SignalProfile* constructGauss(char* name, RtxList* params)
{
    GaussProfile* signal;
    double* u;
    double* s;

    signal = (GaussProfile*)calloc(1, sizeof(GaussProfile));
    signal->name = strdup(name);
    signal->type = FUNCTION_TYPE_GAUSS;

    /* Default values. */

    signal->u = 0;
    signal->s = 1;

    /* Specified values. */

    if((u = rtx_list_lookup(params, "u")) != NULL) 
    {
        signal->u = *u;
    }
    if((s = rtx_list_lookup(params, "s")) != NULL) 
    {
        signal->s = *s;
    }

    return (SignalProfile*)signal;
}

/*
 * Construct a uniform noise signal.
 *
 */
static SignalProfile* constructUniform(char* name, RtxList* params)
{
    UniformProfile* signal;
    double* A;
    double* B;

    signal = (UniformProfile*)calloc(1, sizeof(UniformProfile));
    signal->name = strdup(name);
    signal->type = FUNCTION_TYPE_UNIFORM;

    /* Default values. */

    signal->A = 0;
    signal->B = 1;

    /* Specified values. */

    if((A = rtx_list_lookup(params, "A")) != NULL) 
    {
        signal->A = *A;
    }
    if((B = rtx_list_lookup(params, "B")) != NULL) 
    {
        signal->B = *B;
    }

    return (SignalProfile*)signal;
}

/*
 * Helper function used in parsing to accept an expected token (eg keyword).
 *
 */
static int accept(char* expected, char* actual)
{
    if(actual == NULL)
        return rtx_error("accept: unexpected end of file");

    if(strcmp(expected, actual) == 0)
    {
        return 0;
    }
    else
    {
        return
            rtx_error(
                "accept: expected token \"%s\" but got \"%s\"",
                expected, actual
            );
    }
}

/*
 * Helper function to parse a double.
 * 
 * ERROR CONDITIONS DO NOT CAUSE EXPECTED VALUES FOR errno (commented out).
 *
 */
static int parseDouble(char* token, double* result)
{
    if(token == NULL)
    {
        return rtx_error("parseDouble: unexpected end of file");
    }
    else
    {
        *result = strtod(token, NULL);
        return 0;
    }

    // if(*result == 0 && (errno == EINVAL || errno == ERANGE))
    // {
    //     return rtx_error_errno("parseDouble: strtod() failed");
    // }
    // else
    // {
    //     return 0;
    // }
}

/*
 * Helper function to parse a string, checking for EOF.
 *
 */
static char* parseString(char* token)
{
    if(token != NULL)
    {
        return strdup(token);
    }
    else
    {
        return rtx_error_null("parseString: unexpected end of file");
    }
}

/*
 * Convert text file into tokens delimeted by spaces, tabs, or newlines.
 *
 */
static char** tokeniseFile(char* fileName)
{
    FILE* file;
    int c;
    char* token;
    int i;
    char** tokens;
    int tokenCount;

    file = fopen(fileName, "r");
    if(file == NULL)
    {
        return rtx_error_errno_null("tokeniseFile: fopen() failed");
    }

    tokens = (char**)calloc(256, sizeof(char*));
    if(tokens == NULL)
    {
        return rtx_error_errno_null("tokeniseFile: calloc() failed");
    }

    tokenCount = 0;
    i = 0;
    token = (char*)calloc(32, sizeof(char));
    if(token == NULL)
    {
        free(tokens);
        return rtx_error_errno_null("tokeniseFile: calloc() failed");
    }

    c = fgetc(file);
    do
    {
	if(c == ' ' || c == '\t' || c == '\n' || c == EOF || c == '%')
	{
            if(i > 0)
            {
		token[i++] = '\0';
		tokens[tokenCount++] = token;
		i = 0;
		token = (char*)calloc(80, sizeof(char));
                if(token == NULL)
                {
                    for(i = 0; i < tokenCount; i++)
                    {
                        free(tokens[i]);
                    }
                    free(tokens);
                    return
                        rtx_error_errno_null("tokeniseFile: calloc() failed");
                }
            }
            if(c == '%')
            {
                i = 0;
                while(c != '\n' && c != EOF)
                {
                    c = fgetc(file);
                }
            }
            if(c == EOF)
            {
		free(token);
            }
            c = fgetc(file);
	}
	else {
            token[i++] = c;
            c = fgetc(file);
	}
    } while(c != EOF);

    tokens[tokenCount] = NULL;

    if(fclose(file) == EOF)
    {
        return rtx_error_errno_null("tokeniseFile: fclose() failed");
    }

    return tokens;
}
