/*********************************************************************
 *
 * CSIRO Automation
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * www.cat.csiro.au/cmst
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 *********************************************************************/


static char *rcsid = "$Id: ddxsh-main.c 2302 2008-01-07 06:02:30Z roy029 $";

/**
 * \file ddxsh-main.c
 * \brief command-line interface to the store
 * \author Pavan Sikka
 */

/**
 * \page ddxsh ddxsh
 *
 * DDX commmand line interface can be used to query the status of the store. 
 * The following
 * commands are supported:
 *
 * - vlist\n
 *   list the variables in the store
 * - clist
 *   list the constants in the store
 * - tlist
 *   list the type definitions in the store
 * - vstatus <name>
 *   print the type definition of the var
 * - tstatus <name>
 *   print the type definition of the type
 * - cstatus <name>
 *   print the value of the constant
 * - value <name>
 *   print the value of the var
 *
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <pthread.h>
#include <time.h>

#include <rtx/message.h>
#include <rtx/time.h>
#include <rtx/error.h>
#include <rtx/command.h>
#include <rtx/parse.h>
#include <rtx/inet.h>

#include "store-common.h"
#include "store-header.h"
#include "store-msg.h"
#include "store-messages.h"
#include "catalog-messages.h"

#include "ddx.h"

int done = 0;
char cmdPrompt[1024];
char storeName[128];
char catalogName[128];
DDX_STORE_ID store;
DDX_STORE_ID catalog;
RtxParseSymTabs * symTabs;
char * list[1024];
int varId = 0;

/* ddxsh commands */

static int ddxsh_store_get_var_list (int argc, char * argv[]);
static int ddxsh_store_get_var_status (int argc, char * argv[]);
static int ddxsh_store_get_var_value (int argc, char * argv[]);
static int ddxsh_store_set_var_value (int argc, char * argv[]);
static int ddxsh_type (int argc, char * argv[]);
static int ddxsh_var (int argc, char * argv[]);
static int ddxsh_const (int argc, char * argv[]);

RtxCommandMenu ddxshCmds[] = {
  {"type", ddxsh_type, "list the typedefs in the store"},
  {"var", ddxsh_var, "list the variables in the store"},
  {"ls", ddxsh_var, "list the variables in the store"},
  {"const", ddxsh_const, "list the constants in the store"},
  {"vlist", ddxsh_store_get_var_list, "list the variables in the store"},
  {"clist", ddxsh_store_get_var_list, "list the constants in the store"},
  {"tlist", ddxsh_store_get_var_list, "list the type definitions in the store"},
  {"vstatus", ddxsh_store_get_var_status, "print the type definition of the var"},
  {"tstatus", ddxsh_store_get_var_status, "print the type definition of the type"},
  {"cstatus", ddxsh_store_get_var_status, "print the value of the constant"},
  {"value", ddxsh_store_get_var_value, "print the value of the var"},
  {"print", ddxsh_store_get_var_value, "print the value of the var"},
  {"get", ddxsh_store_get_var_value, "print the value of the var"},
  {"set", ddxsh_store_set_var_value, "set the value of the var"},
  {NULL, NULL, NULL}
};

static int
ddxsh_store_get_catalog (void)
{
    MSG * mp;
    char * catName;
    char * tok;

    sprintf (store.buf1, "%s\n", 
	     keyTable[MSG_TOK_STORE_GET_CATALOG_REQUEST].name);
    if ((mp = store_get_response (store.server->sock->sockfd, 
				  store.buf, store.buf1,
				  5, &store, keyTable,
				  msgTable, "ddxsh",
				  store.verbose)) == NULL)
	return (rtx_error ("ddxsh_store_get_catalog: "
			   "no response from store (5s)"));
    if (store.verbose)
        fprintf (stderr, "store-reply: %s\n", store.buf);
    if (mp->id != MSG_STORE_GET_CATALOG_REPLY)
        rtx_error ("ddxsh_store_get_catalog: "
		   "invalid/error reply from store");
    if ((catName = get_message_part (keyTable, mp, MSG_TOK_NAME))
	    == NULL)
        rtx_error ("ddxsh_store_get_catalog: "
		   "get_message_part() failed");
    tok = strtok (catName, " ");
    strcpy (catalogName, tok);
    free_message (catKeyTable, mp);
    return (0);
}

static int
sortfunc(void **ap, void **bp)
{
        char           *a = (char *) *ap;
        char           *b = (char *) *bp;
        return strcmp (a, b);
}

static int
ddxsh_var (int argc, char * argv[])
{
    char * cmd[4];
    MSG * mp;
    char * arg;
    char * tok;
    int i = 0, j;

    if (argc == 1) {
        cmd[0] = "vlist";
	ddxsh_store_get_var_list (1, cmd);
	return (0);
    }
    if ((argc == 2) && (strcmp (argv[1], "-l") == 0)) {
        /* get the list and then iterate */
        sprintf (store.buf1, "%s\n", 
		 keyTable[MSG_TOK_STORE_GET_VAR_LIST_REQUEST].name);
	if ((mp = store_get_response (store.server->sock->sockfd, store.buf, store.buf1,
				      5, &store, keyTable,
				      msgTable, "ddxsh",
				      store.verbose)) == NULL)
	    return (rtx_error ("ddxsh_store_get_var_list: "
			       "no response from store (5s)"));
	if (mp->id != MSG_STORE_GET_VAR_LIST_REPLY)
	    return (rtx_error ("ddxsh_store_get_var_list: "
			       "invalid/error reply from store"));
	if ((arg = get_message_part (keyTable, mp, MSG_TOK_LIST))
	    == NULL) {
	    printf ("\n");
	    return (0);
	}
	tok = strtok (arg, " \t");
	while (tok != NULL) {
	    if ((strcmp (tok, "char") != 0) &&
	            (strcmp (tok, "short") != 0) &&
	            (strcmp (tok, "int") != 0) &&
	            (strcmp (tok, "long") != 0) &&
	            (strcmp (tok, "float") != 0) &&
	            (strcmp (tok, "double") != 0) &&
	            (strcmp (tok, "struct") != 0)) {
	        if (i < 1024) {
		    list[i++] = tok;
		}
	    }
	    strtok (NULL, " \t");
	    tok = strtok (NULL, " \t");
	}
	qsort (list, i, sizeof (char *), 
	       (int (*)(const void *, const void *)) sortfunc);
	for (j=0; j<i; j++) {
	    cmd[0] = "vstatus";
	    cmd[1] = "-l";
	    cmd[2] = list[j];
	    ddxsh_store_get_var_status (3, cmd);
	}
	free_message (catKeyTable, mp);
	return (0);
    }
    if ((argc == 2) && (strcmp (argv[1], "-l") != 0)) {
        cmd[0] = "vstatus";
	cmd[1] = argv[1];
	ddxsh_store_get_var_status (2, cmd);
	return (0);
    }
    if (argc > 2) {
        if (strcmp (argv[1], "-l") == 0) {
	    for (i=2; i<argc; i++) {
	        cmd[0] = "vstatus";
		cmd[1] = argv[1];
		cmd[2] = argv[i];
		ddxsh_store_get_var_status (3, cmd);
	    }
	} else {
	    for (i=1; i<argc; i++) {
	        cmd[0] = "vstatus";
		cmd[1] = argv[i];
		ddxsh_store_get_var_status (2, cmd);
	    }
	}
	return (0);
    }
    printf ("invalid command\n");
    return (0);
}

static int
ddxsh_type (int argc, char * argv[])
{
    char * cmd[3];
    int i;

    if (argc == 1) {
        cmd[0] = "tlist";
	ddxsh_store_get_var_list (1, cmd);
	return (0);
    }

    if (argc > 1) {
        for (i=1; i<argc; i++) {
	    cmd[0] = "tstatus";
	    cmd[1] = argv[i];
	    ddxsh_store_get_var_status (2, cmd);
	}
	return (0);
    }
    printf ("invalid command\n");
    return (0);
}

static int
ddxsh_const (int argc, char * argv[])
{
    char * cmd[3];
    int i;

    if (argc == 1) {
        cmd[0] = "clist";
	ddxsh_store_get_var_list (1, cmd);
	return (0);
    }

    if (argc > 1) {
        for (i=1; i<argc; i++) {
	    cmd[0] = "cstatus";
	    cmd[1] = argv[i];
	    ddxsh_store_get_var_status (2, cmd);
	}
	return (0);
    }
    printf ("invalid command\n");
    return (0);
}

static int
ddxsh_store_get_var_list (int argc, char * argv[])
{
	MSG * mp;
	char * arg;
	char * tok;
	int varType, i = 0, j;

	varType = 0;
	if (strcmp (argv[0], "vlist") == 0)
		varType = 0;
	else if (strcmp (argv[0], "tlist") == 0)
		varType = 1;
	else if (strcmp (argv[0], "clist") == 0)
		varType = 2;
	switch (varType) {
		case 0: /* var list */
			sprintf (store.buf1, "%s\n", 
					keyTable[MSG_TOK_STORE_GET_VAR_LIST_REQUEST].name);
			break;
		case 1: /* type list */
			sprintf (store.buf1, "%s %s\n", 
					keyTable[MSG_TOK_STORE_GET_VAR_LIST_REQUEST].name,
					keyTable[MSG_TOK_ITEM_TYPE_TYPE].name);
			break;
		case 2: /* const list */
			sprintf (store.buf1, "%s %s\n", 
					keyTable[MSG_TOK_STORE_GET_VAR_LIST_REQUEST].name,
					keyTable[MSG_TOK_ITEM_TYPE_CONST].name);
			break;
		default: /* var list */
			sprintf (store.buf1, "%s\n", 
					keyTable[MSG_TOK_STORE_GET_VAR_LIST_REQUEST].name);
			break;
	}
	if ((mp = store_get_response (store.server->sock->sockfd, store.buf, store.buf1,
					5, &store, keyTable,
					msgTable, "ddxsh",
					store.verbose)) == NULL)
		return (rtx_error ("ddxsh_store_get_var_list: "
					"no response from store (5s)"));
	if (store.verbose)
		fprintf (stderr, "store-reply: %s\n", store.buf);
	if (mp->id != MSG_STORE_GET_VAR_LIST_REPLY) {
		free_message (catKeyTable, mp);
		return (rtx_error ("ddxsh_store_get_var_list: "
					"invalid/error reply from store"));
	}
	if ((arg = get_message_part (keyTable, mp, MSG_TOK_LIST))
			== NULL) {
		printf ("\n");
		free_message (catKeyTable, mp);
		return (0);
	}
	tok = strtok (arg, " \t");
	while (tok != NULL) {
		if ((strcmp (tok, "char") != 0) &&
				(strcmp (tok, "short") != 0) &&
				(strcmp (tok, "int") != 0) &&
				(strcmp (tok, "long") != 0) &&
				(strcmp (tok, "float") != 0) &&
				(strcmp (tok, "double") != 0) &&
				(strcmp (tok, "struct") != 0)) {
			if (i < 1024) {
				list[i++] = tok;
			}
		}
		if (varType == 0)
			strtok (NULL, " \t");
		tok = strtok (NULL, " \t");
	}
	qsort (list, i, sizeof (char *), 
			(int (*)(const void *, const void *)) sortfunc);
	for (j=0; j<i; j++)
		printf ("%s\n", list[j]);
	free_message (catKeyTable, mp);
	return (0);
}

int
ddxsh_get_arraysize (
		     RtxParseVar * v
		     )
{
    int i, n = 1;

    if (v->dim > 0) {
        for (i=0; i<v->dim; i++) {
  	    n *= v->arrayDim[i];
	}
    }
    return (n);
}

void
ddxsh_print_decl (
		  RtxParseVar * v,
		  int indentLevel
		  )
{
    int i;

    if (v == NULL)
        return;

    if (v->type == rtx_struct_t) {
        for (i=0; i<indentLevel; i++)
	    printf ("\t");
	printf ("struct {\n");
	ddxsh_print_decl (v->subVar, indentLevel+1);
        for (i=0; i<indentLevel; i++)
	    printf ("\t");
	printf ("} %s", v->name);
    } else {
        for (i=0; i<indentLevel; i++)
            printf ("\t");
        switch (v->type) {
            case rtx_char_t :
    	    printf ("char\t%s", v->name);
    	    break;
            case rtx_short_t :
    	    printf ("short\t%s", v->name);
    	    break;
            case rtx_int_t :
    	    printf ("int\t%s", v->name);
    	    break;
            case rtx_long_t :
    	    printf ("long\t%s", v->name);
    	    break;
            case rtx_float_t :
    	    printf ("float\t%s", v->name);
    	    break;
            case rtx_double_t :
    	    printf ("double\t%s", v->name);
    	    break;
        }
    }
    if (v->dim > 0)
        for (i=0; i<v->dim; i++)
	    printf ("[%d]", v->arrayDim[i]);
    printf (";\n");
    ddxsh_print_decl (v->next, indentLevel);
}

static int
ddxsh_store_get_var_status (int argc, char * argv[])
{
    MSG * mp;
    char * arg;
    int varSize;
    int varCount;
    int printSummary = 0;
    char * name;
    int varType;
    RtxParseVar * v = NULL;
    RtxParseVarTableEntry * vEntry = NULL;

    varType = 0;
    if (strcmp (argv[0], "vstatus") == 0) {
        if (argc == 2)
	    name = argv[1];      
	else if (argc == 3) {
	    name = argv[2];      
	    printSummary = 1;
	} else {
	    printf ("invalid command\n");
	    return (0);
	}
        varType = 0;
    } else if (strcmp (argv[0], "tstatus") == 0) {
	if (argc < 2) {
	    printf ("invalid command\n");
	    return (0);
	}
        varType = 1;
	name = argv[1];
    } else if (strcmp (argv[0], "cstatus") == 0) {
	if (argc < 2) {
	    printf ("invalid command\n");
	    return (0);
	}
	name = argv[1];
        varType = 2;
    } else {
        printf ("invalid command\n");
	return (0);
    }
    switch (varType) {
    case 0: /* var */
        if (printSummary == 0) {
	    if ((vEntry = (RtxParseVarTableEntry *) rtx_hash_find
		 (symTabs->varTable, name, (rtx_error_t)NULL)) != NULL) {
	        ddxsh_print_decl (vEntry->v, 0);
		return (0);
	    }
	}
        sprintf (store.buf1, "%s %s %s = %s\n", 
		 keyTable[MSG_TOK_STORE_GET_VAR_STATUS_REQUEST].name,
		 keyTable[MSG_TOK_ITEM_TYPE_VAR].name,
		 keyTable[MSG_TOK_NAME].name, name);
	break;
    case 1: /* type */
        sprintf (store.buf1, "%s %s %s = %s\n", 
		 keyTable[MSG_TOK_STORE_GET_VAR_STATUS_REQUEST].name,
		 keyTable[MSG_TOK_ITEM_TYPE_TYPE].name,
		 keyTable[MSG_TOK_NAME].name, name);
	break;
    case 2: /* const */
        sprintf (store.buf1, "%s %s %s = %s\n", 
		 keyTable[MSG_TOK_STORE_GET_VAR_STATUS_REQUEST].name,
		 keyTable[MSG_TOK_ITEM_TYPE_CONST].name,
		 keyTable[MSG_TOK_NAME].name, name);
	break;
    default: /* var */
        sprintf (store.buf1, "%s %s %s = %s\n", 
		 keyTable[MSG_TOK_STORE_GET_VAR_STATUS_REQUEST].name,
		 keyTable[MSG_TOK_ITEM_TYPE_VAR].name,
		 keyTable[MSG_TOK_NAME].name, name);
	break;
    }
    if ((mp = store_get_response (store.server->sock->sockfd, store.buf, store.buf1,
				  5, &store, keyTable,
				  msgTable, "ddxsh",
				  store.verbose)) == NULL)
	return (rtx_error ("ddxsh_store_get_var_status: "
			   "no response from store (5s)"));
    if (store.verbose)
        fprintf (stderr, "store-reply: %s\n", store.buf);
    if (mp->id != MSG_STORE_GET_VAR_STATUS_REPLY)
        return (rtx_error ("ddxsh_store_get_var_status: "
			   "invalid/error reply from store"));
    if ((arg = get_message_part (keyTable, mp, MSG_TOK_VALUE))
	    == NULL)
        return (rtx_error ("ddxsh_store_get_var_status: "
		   "get_message_part() failed"));
    varCount = atoi (arg);
    if ((arg = get_message_part (keyTable, mp, MSG_TOK_SHMEM_VAR_SIZE))
	    == NULL)
        return (rtx_error ("ddxsh_store_get_var_status: "
		   "get_message_part() failed"));
    varSize = atoi (arg);
    if ((arg = get_message_part (keyTable, mp, MSG_TOK_VAR_DEFINITION))
	    == NULL)
        return (rtx_error ("ddxsh_store_get_var_status: "
		   "get_message_part() failed"));
    if (varType != 2) {
        if ((v = rtx_parse (arg, symTabs)) == NULL)
	    printf ("rtx_parse: failed on [%s]\n", arg);
	if (varType == 0)
	    if (rtx_parse_add_var_symbol (v, varId++, 0, 0, arg, 
					  symTabs->varTable) == -1)
	        printf ("rtx_parse_add_var_symbol");
	if (printSummary == 0) {
	    ddxsh_print_decl (v, 0);
	} else {
	    printf ("%s\t[size = %d]\t[count = %d]\n",
		    name, varSize, varCount);
	}
    } else {
        printf ("%s\n", arg);
    }
    free_message (catKeyTable, mp);
    return (0);
}

char *
ddxsh_print_val (
		 RtxParseVar * v,
		 char * valStr,
		 int indentLevel
		 )
{
    int i, j, k, n;
    char * tok;

    if (v == NULL)
        return (NULL);

    n = ddxsh_get_arraysize (v);

    if (v->type == rtx_struct_t) {
        if (v->dim > 0) {
            for (j=0; j<n; j++) {
                for (i=0; i<indentLevel; i++)
		    printf ("\t");
		printf ("struct {\n");
		valStr = ddxsh_print_val (v->subVar, valStr, indentLevel+1);
                for (i=0; i<indentLevel; i++)
		    printf ("\t");
		printf ("} %s", v->name);
		for (k=0; k < v->dim; k++)
		    printf ("[%d]", v->arrayIndex[k]);
		printf (";\n");
		v->arrayIndex[v->dim-1]++;
		for (k=v->dim-1; k>=0; k--) {
		    if (v->arrayIndex[k] >= v->arrayDim[k]) {
			    v->arrayIndex[k] = 0;
			    if (k-1 >= 0)
			        v->arrayIndex[k-1]++;
		    }
		}
	    }
	} else {
	    for (i=0; i<indentLevel; i++)
	        printf ("\t");
	    printf ("struct {\n");
	    valStr = ddxsh_print_val (v->subVar, valStr, indentLevel+1);
	    for (i=0; i<indentLevel; i++)
	        printf ("\t");
	    printf ("} %s", v->name);
	    printf (";\n");
	}
    } else {
        if (v->dim > 0) {
	    for (j=0; j<n; j++) {
	    	for (i=0; i<indentLevel; i++)
	    	    printf ("\t");
	    	if ((tok = strtok (valStr, " \t")) == NULL)
	    	    return (NULL);
		valStr = NULL;
	    	switch (v->type) {
	    	    case rtx_char_t :
	 		printf ("char\t%s", v->name);
	 		break;
	    	    case rtx_short_t :
	 		printf ("short\t%s", v->name);
	 		break;
	    	    case rtx_int_t :
	 		printf ("int\t%s", v->name);
	 		break;
	    	    case rtx_long_t :
	 		printf ("long\t%s", v->name);
	 		break;
	    	    case rtx_float_t :
	 		printf ("float\t%s", v->name);
	 		break;
	    	    case rtx_double_t :
	 		printf ("double\t%s", v->name);
	 		break;
	    	}   
		for (k=0; k < v->dim; k++)
		    printf ("[%d]", v->arrayIndex[k]);
		printf ("\t= %s;\n", tok);
		v->arrayIndex[v->dim-1]++;
		for (k=v->dim-1; k>=0; k--) {
		    if (v->arrayIndex[k] >= v->arrayDim[k]) {
			    v->arrayIndex[k] = 0;
			    if (k-1 >= 0)
			        v->arrayIndex[k-1]++;
		    }
		}
	    }
	} else {
	    for (i=0; i<indentLevel; i++)
	        printf ("\t");
	    if ((tok = strtok (valStr, " \t")) == NULL)
	        return (NULL);
	    valStr = NULL;
	    switch (v->type) {
	        case rtx_char_t :
		    printf ("char\t%s", v->name);
		    break;
	        case rtx_short_t :
		    printf ("short\t%s", v->name);
		    break;
	        case rtx_int_t :
		    printf ("int\t%s", v->name);
		    break;
	        case rtx_long_t :
		    printf ("long\t%s", v->name);
		    break;
	    	case rtx_float_t :
		    printf ("float\t%s", v->name);
		    break;
	        case rtx_double_t :
		    printf ("double\t%s", v->name);
		    break;
	    }   
	    printf ("\t= %s;\n", tok);
	}
    }
    valStr = ddxsh_print_val (v->next, valStr, indentLevel);
    return (valStr);
}

static int
ddxsh_store_get_var_value (int argc, char * argv[])
{
	MSG * mp;
	char * arg;
	char * t1 = NULL, * t2 = NULL, * t3 = NULL;
	char * lasts = NULL;
	char * name;
	RtxParseVarTableEntry * vEntry = NULL;

	if (argc < 2)
		return (0);
	name = argv[1];
	if ((vEntry = (RtxParseVarTableEntry *) rtx_hash_find
	     (symTabs->varTable, name, (rtx_error_t)NULL)) == NULL) {
		char * back = argv[0];
		argv[0] = "vstatus";
		ddxsh_store_get_var_status (argc, argv);
		argv[0] = back;
		if ((vEntry = (RtxParseVarTableEntry *) rtx_hash_find
		     (symTabs->varTable, name, (rtx_error_t)NULL)) == NULL) {
			printf ("ddxsh_store_get_var_value: var still not in table\n");
			return (rtx_error ("ddxsh_store_get_var_value: serious error"));
		}
	}
	sprintf (store.buf1, "%s %s = %s\n", 
			keyTable[MSG_TOK_STORE_GET_VAR_VALUE_REQUEST].name,
			keyTable[MSG_TOK_NAME].name, name);
	if ((mp = store_get_response (store.server->sock->sockfd, store.buf, store.buf1,
					5, &store, keyTable,
					msgTable, "ddxsh",
					store.verbose)) == NULL)
		return (rtx_error ("ddxsh_store_get_var_value: "
					"no response from store (5s)"));
	if (store.verbose)
		fprintf (stderr, "store-reply: %s\n", store.buf);
	if (mp->id != MSG_STORE_GET_VAR_VALUE_REPLY)
		return (rtx_error ("ddxsh_store_get_var_value: "
					"invalid/error reply from store"));
	if ((arg = get_message_part (keyTable, mp, MSG_TOK_VALUE))
			== NULL)
		return (rtx_error ("ddxsh_store_get_var_value: "
					"get_message_part() failed"));
	t1 = strtok_r (arg, " \t", &lasts);
	t2 = strtok_r (NULL, " \t", &lasts);
	t3 = strtok_r (NULL, " \t", &lasts);
	printf ("%s.%s [%s]\n", t2, t3, t1);
	ddxsh_print_val (vEntry->v, lasts, 0);
	free_message (catKeyTable, mp);
	return (0);
}

static int
ddxsh_store_set_var_value (int argc, char * argv[])
{
    MSG * mp;
    char * arg;
    char * name;
    RtxParseVarTableEntry * vEntry = NULL;
    int i;

    if (argc < 3)
        return (0);
    name = argv[1];
    if ((vEntry = (RtxParseVarTableEntry *) rtx_hash_find
	 (symTabs->varTable, name, (rtx_error_t)NULL)) == NULL) {
		char * back = argv[0];
		argv[0] = "vstatus";
		ddxsh_store_get_var_status (argc, argv);
		argv[0] = back;
		if ((vEntry = (RtxParseVarTableEntry *) rtx_hash_find
		     (symTabs->varTable, name, (rtx_error_t)NULL)) == NULL) {
			printf ("ddxsh_store_set_var_value: var still not in table\n");
			return (rtx_error ("ddxsh_store_set_var_value: serious error"));
		}
	}
    sprintf (store.buf1, "%s %s = %s %s = %s", 
	     keyTable[MSG_TOK_STORE_SET_VAR_VALUE_REQUEST].name,
	     keyTable[MSG_TOK_NAME].name, name,
	     keyTable[MSG_TOK_VALUE].name, argv[2]);
    for (i=3; i<argc; i++) {
        strcat (store.buf1, " ");
	strcat (store.buf1, argv[i]);
    }
    strcat (store.buf1, "\n");
    if ((mp = store_get_response (store.server->sock->sockfd, store.buf, store.buf1,
				  5, &store, keyTable,
				  msgTable, "ddxsh",
				  store.verbose)) == NULL)
	return (rtx_error ("ddxsh_store_set_var_value: "
			   "no response from store (5s)"));
    if (store.verbose)
        fprintf (stderr, "store-reply: %s\n", store.buf);
    if (mp->id != MSG_STORE_SET_VAR_VALUE_REPLY)
        return (rtx_error ("ddxsh_store_set_var_value: "
			   "invalid/error reply from store"));
    if ((arg = get_message_part (keyTable, mp, MSG_TOK_RESPONSE))
	    == NULL)
        return (rtx_error ("ddxsh_store_set_var_value: "
			   "get_message_part() failed"));
    printf ("%s\n", arg);
    free_message (catKeyTable, mp);
    return (0);
}

static void
ddxsh_help (void)
{
    fprintf (stderr, "ddxsh [-s <hostname>] [-v] [-h]\n");
    fprintf (stderr, "\n");
    fprintf (stderr, "\t-s <hostname> host running the store [localhost]\n");
    fprintf (stderr, "\t-v            debug on [off]\n");
    fprintf (stderr, "\t-h            print help\n");
    fprintf (stderr, "\n");
    fprintf (stderr, "Commands:\n");
    fprintf (stderr, "\tvlist\t\tlist the variables in the store\n");
    fprintf (stderr, "\tclist\t\tlist the constants in the store\n");
    fprintf (stderr, "\ttlist\t\tlist the type definitions in the store\n");
    fprintf (stderr, "\tvstatus <name>\tprint the type definition of the var\n");
    fprintf (stderr, "\ttstatus <name>\tprint the type definition of the type\n");
    fprintf (stderr, "\tcstatus <name>\tprint the value of the constant\n");
    fprintf (stderr, "\tvalue <name>\tprint the value of the var\n");
    fprintf (stderr, "\thelp\t\tprint help\n");
    fprintf (stderr, "\tquit\n");
    fprintf (stderr, "\texit\n");
    fprintf (stderr, "\n");
    fprintf (stderr, "%s\n", rcsid);
    fprintf (stderr, "Build: " __DATE__ " " __TIME__ "\n");
    fprintf (stderr, "\n");
}

int
main (int argc, char * argv[])
{
	int errflg = 0, c;
	extern char * optarg;

	rtx_error_init ("ddxsh", RTX_ERROR_STDERR, NULL);
	if ((symTabs = rtx_parse_init ()) == NULL) {
		rtx_error ("rtx_parse_init");
		exit (1);
	}

	store.verbose = 0;
	if (gethostname (storeName, RTX_INET_MAX_HOST_NAME_LEN) == -1) {
		rtx_error_flush ("gethostname() failed");
		exit (2);
	}
	while ((c = getopt(argc, argv, "-s:vh")) != -1) {
		switch (c) {
			case 's':
				strcpy (storeName, optarg);
				break;
			case 'v':
				store.verbose = 1;
				break;
			case 'h':
			case '?':
				errflg++;
				break;
		}
	}
	if (errflg) {
		ddxsh_help ();
		exit (1);
	}

	if ((store.server = rtx_inet_init (RTX_INET_TCP_CLIENT,
					NULL, 0, storeName,
					STORE_PORT_NUM,
					NULL, NULL, NULL)) == NULL) {
		rtx_error_flush ("rtx_inet_init (store) failed");
		exit (3);
	}
	if (ddxsh_store_get_catalog () == -1) {
		rtx_error_flush ("ddxsh_store_get_catalog() failed");
		exit (4);
	}
#if 0
	/* nobody's using that for now, so don't annoy the catalog
	 * with a useless connection */
	if ((catalog.server = rtx_inet_init (RTX_INET_TCP_CLIENT,
					NULL, 0, catalogName,
					CATALOG_PORT_NUM,
					NULL, NULL, NULL)) == NULL) {
		rtx_error_flush ("rtx_inet_init (catalog) failed");
		exit (5);
	}
#else
	catalog.server = NULL;
#endif

	sprintf (cmdPrompt, "store@%s: ", storeName);
	rtx_command_handler (ddxshCmds, cmdPrompt);

	exit (0);
}
