/*********************************************************************
 *
 * CSIRO Automation
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * www.cat.csiro.au/cmst
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 *********************************************************************/

static char *rcsid = "$Id: ddxreplay-main.c 3134 2008-05-19 06:25:40Z roy029 $";
static char *rcsRevision = "$Revision: 3134 $";

/**
 *
 * \file ddxreplay-main.c
 * \brief DDX log file playback application
 * \author Pavan Sikka
 *
 */

/**
 * \page ddxreplay ddxreplay
 *
 * The DDX log file playback application (ddxreplay) can be used to read log files
 * generated by ddxlog and then replay them back to the store.
 *
 * The program can be used as follows:
 * - ddxreplay [options] <log-file> <var1> .... <varN>
 * - ddxreplay [options] [-1|-2|-3|-4|-5] <var1> ... <varN>
 *
 * ddxlogread supports the following options:
 * - -v\n
 *    turn debugging on
 * - -s\n
 *    step through file
 * - -q\n
 *    if a specified variable is not in the file, ignore the error and
 *    continue. The default behavior is to exit.
 * - -1\n
 * - -2\n
 * - -3\n
 * - -4\n
 * - -5\n
 *    sort all the log files in the directory by timestamp (in descending
 *    order) and then choose the one specified
 * - -h\n
 *    print a help message and exit
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <stdarg.h>
#include <strings.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <math.h>

#include <rtx/circbuf.h>
#include <rtx/logread.h>
#include <rtx/error.h>
#include <rtx/timer.h>
#include <rtx/time.h>
#include <rtx/main.h>
#include <ddx.h>

#define  dtime(ts)  ((ts).seconds + (ts).nanoSeconds*1e-9)

int             filetimestamp = 0;  // use timestamps from file
int				exacttimestamp = 1;
int				pauseSignal = 0;
int             endreplay = 0;
int				step = 0;	// Step through file
int             verbose;
int             quiet_mode;
char           *pname;
char           *handshake = NULL;
pid_t           pid;
RtxLogread     *readlog_log;
DDX_STORE_ID   *store;

char           *tabularfile;

unsigned int   numfiles = 0;
char           **logfilenames = NULL;
int             numTopVars;

DDX_STORE_TYPE(HAND_SHAKE, int);



void inthdl(int n)
{
	endreplay ++;
	if (endreplay >= 3) {
		kill(getpid(),SIGKILL);
	}
}

void pausehdl(int n)
{
	pauseSignal = 1;
}

void addFile(char * fname)
{
	numfiles += 1;
	logfilenames = (char**)realloc(logfilenames,numfiles*sizeof(char*));
	logfilenames[numfiles-1] = strdup(fname);
}

int isFile(char * fname)
{
	FILE * fp = fopen(fname,"r");
	if (fp == NULL) return 0;
	fclose(fp);
	return 1;
}

int isDirectory(char * fname)
{
	struct stat sstat;
	int r = stat(fname,&sstat);
	if (r != 0) return 0;
	if (S_ISDIR(sstat.st_mode)) return 1;
	return 0;
}

void freeFileArray(char ** array, int length)
{
	int i;
	for (i=0;i<length;i++)
		free(array[i]);
	free(array);
}

	int
main(int ac, char *av[])
{
	extern char    *optarg;
	extern int      optind;
	unsigned int	handshakeCount=0, localCount=0;
	int				optindex, filenum, i;
	int             c, count, errflag = 0;
	RtxLogreadVar   *rv;
	double         currenttime;
	double         timeFactor = 1;
	double         maxWait = 0;
	RtxTime		   offset = {0,0}, initialRealTime={0,0}, handshakeTime={0,0},
				   scaledInitialRealTime={0,0}, initialLogTime={0,0};
	DDX_STORE_ITEM *handshakeItem = NULL;

	pname = av[0];

	//fprintf (stderr, "ddxreplay: functionality not yet implemented\n");
	//exit (1);

	while ((c = getopt(ac, av, "m:y:w:sq12345vhxt")) != -1) {
		switch (c) {
			case 'y':
				handshake = strdup(optarg);
				break;
			case 's':
				step++;
				break;
			case 'x':
				exacttimestamp = 0;
				break;
			case 't':
				filetimestamp = 1;
				break;
			case 'm':
				sscanf(optarg," %le ",&timeFactor);
				if (timeFactor <= 0)
					timeFactor = 1;
				break;
			case 'w':
				sscanf(optarg," %le ",&maxWait);
				break;
			case 'v':
				verbose++;
				break;
			case 'q':
				quiet_mode++;
				break;
			case '1':
			case '2':
			case '3':
			case '4':
			case '5':
				{

					char logfilename[BUFSIZ];
					char          **files;
					int             nfiles;

					files = rtx_logread_get_files(".", &nfiles);
					if ((c - '1') >= nfiles) {
						fprintf(stderr, "only %d files to choose from\n",
								nfiles);
						exit(1);
					}
					sprintf(logfilename, "%s", files[c - '1']);
					addFile(logfilename);
					fprintf(stderr, "opening logfile %s\n", logfilename);
					freeFileArray(files,nfiles);
					break;
				}

			case 'h':
			case '?':
				errflag++;
				break;
		}
	}

	if (errflag || (ac == 1)) {
		fprintf(stderr, "Usage: %s [options] [filelist] [varlist]\n", pname);
		fprintf(stderr,
				"\t-q  \tquiet mode (ignore variables not in the file)\n"
				"\t-s  \tstep through file (default is to use timestamp)\n"
				"\t-x  \tdo not force timestamps\n"
				"\t-t  \tdo use timestamps from file\n"
				"\t-m {f} \tScale time by factor f (0.1 is ten times faster)\n"
				"\t-w {sec} \tWait for maximum of sec seconds between var readings\n"
				"\t-y {varname} \tAdd a synchronisation variable\n"
				"\t-{n}\tprocess the nth last log file in current directory, ie 1 to 5\n"
				"The log files are given explicitly as [filelist] or implicitly by the {n} option\n\n"
				"[filelist] can contains individual log files or directories. If a directory is\n"
				"specified, then all the sorted log files it contains are used.\n\n"
				"By default, all variables are replayed, however the variable list can be\n"
				"defined explicitly as [varlist] which contains the names of a top level vars.\n"
				"\nWarning:\n\tddxreplay will be confused by a first variable name\n"
				"\tequals to an existing file name.\n");
		fprintf(stderr, "\nVersion: %s\nRcsID: %s\n", rcsRevision,rcsid);
		exit(1);
	}

	optindex = optind;

	if (logfilenames == NULL) {

		/*
		 * no auto logfile name generated, take from command line
		 */
		while (optindex < ac) {
			if (isDirectory(av[optindex])) {
				char ** files;
				int nfiles;
				files = rtx_logread_get_files(av[optindex], &nfiles);
				for (i=0;i<nfiles;i++) {
					addFile(files[nfiles-i-1]);
				}
				freeFileArray(files,nfiles);
				//printf("Replaying all in '%s'\n",av[optindex]);
			} else if (isFile(av[optindex])) {
				addFile(av[optindex]);
			} else {
				break;
			}
			optindex += 1;
		}
	}
	if (numfiles == 0) {
		printf("No file to replay, aborting\n");
		exit(1);
	}
	printf("Log files to replay:\n");
	for (i=0;i<numfiles;i++) {
		printf("\t%s\n",logfilenames[i]);
	}
	printf("Time factor is %f. (Press Ctrl-Z to change)\n",timeFactor);

	signal(SIGINT,inthdl);
	signal(SIGTSTP,pausehdl);

	if (ddx_client_init(verbose) == -1)
		return (rtx_error("ddxreplay: ddx_client_init() failed"));

	if ((store = ddx_store_open (NULL, 0, 5)) == NULL)
		return (rtx_error("ddxreplay: ddx_store_open() failed"));

	if (handshake) {
		int initialIncr = 0;
		step = 0;
		DDX_STORE_REGISTER_TYPE(store, HAND_SHAKE);
		handshakeItem =
			ddx_store_lookup_item(store, handshake, "HAND_SHAKE", 0);
		if (handshakeItem == NULL) {
			return (rtx_error("ddxreplay: handshake variable creation failed"));
		}
		ddx_store_write(handshakeItem,&initialIncr,NULL);
	}


	for (filenum=0;filenum < numfiles;filenum++) {
		char * logfilename = logfilenames[filenum];
		int found_variable;
		if (endreplay) break;

		if ((readlog_log = rtx_logread_init(logfilename, verbose)) == NULL) {
			fprintf(stderr, "rtx_logread_init() failed for '%s'\n",logfilename);
			continue;
		}
		printf("Current file is '%s'\n",logfilename);
		if (verbose > 0)
			printf("%d top level variables in file\n", readlog_log->numVars);

		/*
		 * mark all variables that are required to be printed.
		 */
		errflag = 0;
		found_variable = 0;
		for (count = 0, i=optindex; i < ac; i++, count++) {
			if (rtx_logread_tag_var(readlog_log, av[i]) == NULL) {
				rtx_error_flush("variable <%s> not found", av[i]);
				if (quiet_mode)
					fprintf(stderr, "Continuing.\n");
				else {
					fprintf(stderr, "\n");
					errflag ++;
					break;
				}
			} else {
				found_variable += 1;
			}
		}
		if (errflag > 0) continue;

		/*
		 * if no variable specified on command line, tag all variables
		 */
		if (count == 0) {
			for (rv = readlog_log->vars; rv != NULL; rv = rv->next) {
				if (rtx_logread_tag_var(readlog_log, rv->v->name) == NULL) {
					rtx_error_flush("variable <%s> not found", rv->v->name);
					continue;
				} else {
					found_variable += 1;
				}
			}
		}
		if (found_variable == 0) {
			fprintf(stderr,"No usable variable\n");
			continue;
		}

		printf("Used variables: ");
		/* lookup all the tagged variables in the store */
		for (rv = readlog_log->vars; rv != NULL; rv = rv->next) {
			if (rv->v->attribs & VAR_ATTR_TAG) {
				char decl[4096]=""; /* Any struct should be smaller than that */
				rtx_parse_generate_var_decl(rv->v,decl);
				rv->userdata = ddx_store_lookup_item (store,rv->v->name,decl,0);
				if (verbose > 0)
					printf("Var %s: %s ->%p\n",rv->v->name,decl,rv->userdata);
				else
					printf("%s ",rv->v->name);
			}
		}
		printf("\n");

		/* Getting initial time */
		rv = rtx_logread_next_tagged(readlog_log);
		currenttime = dtime(rv->vh.ts);
		if (offset.seconds == 0) {
			/* first run, we conpute the offset between localtime and first time
			   stamp in the file
			   */
			rtx_time_get(&initialRealTime);
			scaledInitialRealTime = initialRealTime;
			initialLogTime = rv->vh.ts;
			rtx_time_subtract(&offset,&initialRealTime,&initialLogTime);
#if 0
			printf("Init %f Init Log %f Offset %f\n",
					rtx_time_to_double(&initialRealTime),
					rtx_time_to_double(&initialLogTime), rtx_time_to_double(&offset));
#endif
		}
		rtx_logread_rewind(readlog_log);

		if( step ) printf("Step Mode : Type CR to continue");

		while ((rv = rtx_logread_next_tagged(readlog_log)) != NULL) {
			RtxTime timestamp;

			int try_again;
			do {
				try_again = 0;
				if (pauseSignal) {
					RtxTime rt1,rt2;
					double t1,t2,irt,sirt;
					char line[1024];
					double tf = 1;
					rtx_time_get(&rt1);t1 = rtx_time_to_double(&rt1);
					sirt = rtx_time_to_double(&scaledInitialRealTime);
					irt = t1 - (t1 - sirt)/timeFactor;

					printf("Press <enter> to continue. Enter a new time factor if needed: ");
					fflush(stdout);
					fgets(line,1023,stdin);
					if ((sscanf(line," %le ", &tf) == 1) && (tf > 0)) {
						timeFactor = tf;
						printf("Time factor is %f\n",timeFactor);
					}
					rtx_time_get(&rt2); t2 = rtx_time_to_double(&rt2);
					irt += (t2-t1);
					rtx_time_from_double(&initialRealTime,irt);
					rtx_time_subtract(&offset,&initialRealTime,&initialLogTime);
					sirt = t2 - (t2 - irt)*timeFactor;
					rtx_time_from_double(&scaledInitialRealTime,sirt);
					//printf("Init %f/%f Init Log %f Offset %f\n",irt,sirt,
					//		rtx_time_to_double(&initialLogTime), rtx_time_to_double(&offset));
					pauseSignal = 0;
				}

				if (endreplay) break;

				if( step ) {
					getchar();
					if (verbose > 0)
						printf("%s: %f\n", rv->v->name, dtime(rv->vh.ts));
				} else {
					/* sleeping till next reading time */
					int res = 0xDEAD;
					RtxTime now,deltaReal,deltaLog,diff;
					rtx_time_get(&now);
					rtx_time_subtract(&deltaReal,&now,&scaledInitialRealTime);
					rtx_time_subtract(&deltaLog,&(rv->vh.ts),&initialLogTime);
					//printf("Dreal %f D log %f\n",rtx_time_to_double(&deltaReal),rtx_time_to_double(&deltaLog));
					double dl = rtx_time_to_double(&deltaLog) * timeFactor;
					rtx_time_from_double(&deltaLog,dl);
					rtx_time_subtract(&diff,&deltaLog,&deltaReal);
					double ddiff = rtx_time_to_double(&diff);
                    if (filetimestamp) {
                      // dont sleep for too long!
                      if (ddiff > 5.0) {
                        RtxTime five;
                        rtx_time_from_double(&five,5.0);
                        // reset tthe sirt
                        rtx_time_subtract(&diff, &now, &deltaLog);
                        rtx_time_subtract(&scaledInitialRealTime, &diff, &five);
                        ddiff = 5.0;
                      }
                    }
                    if ((maxWait > 0) && (ddiff > maxWait)) {
                    	//squeeze sleep time down to maxWait
                    	if (verbose > 1)
							printf("Squeezed sleep time from %f to %f\n", ddiff, maxWait);
						RtxTime mwt;
                        rtx_time_from_double(&mwt,maxWait);
                        // reset tthe sirt
                        rtx_time_subtract(&diff, &now, &deltaLog);
                        rtx_time_subtract(&scaledInitialRealTime, &diff, &mwt);
						ddiff = maxWait;
					}                    	
					if (verbose > 1)
						printf("%s: %f / %f sleeping %f\n",
								rv->v->name, dtime(timestamp), dtime(now), ddiff);
					if( ddiff > 0 ) res = rtx_timer_sleep(ddiff);
					if (verbose > 1) {
						rtx_time_get(&now);
						printf("Awaken at %f : %d\n",dtime(now),res);
					}
				}

				if (handshake) {
					int handshakeIncr;
					RtxTime hst;
					if (localCount >= handshakeCount)  {
						RtxTime rt1,rt2;
						double t1,t2,irt,sirt;
						rtx_time_get(&rt1);t1 = rtx_time_to_double(&rt1);
						sirt = rtx_time_to_double(&scaledInitialRealTime);
						irt = t1 - (t1 - sirt)/timeFactor;

						if (ddx_store_read(handshakeItem,
									&handshakeIncr, &hst, -1, 1) == 0) {
                            if (handshakeIncr < handshakeCount) localCount = 0; /* reset counter */
							handshakeCount = handshakeIncr;
							handshakeTime = hst;
						}
						rtx_time_get(&rt2); t2 = rtx_time_to_double(&rt2);
						irt += (t2-t1);
						rtx_time_from_double(&initialRealTime,irt);
						rtx_time_subtract(&offset,&initialRealTime,&initialLogTime);
						sirt = t2 - (t2 - irt)*timeFactor;
						rtx_time_from_double(&scaledInitialRealTime,sirt);
						if (localCount >= handshakeCount) {
							try_again = 1;
						}
					} else {
						ddx_store_read(handshakeItem,
								&handshakeIncr, &hst, -1, 0);
						if (rtx_time_cmp(&hst,&handshakeTime)>0) {
                            if (handshakeIncr < handshakeCount) localCount = 0; /* reset counter */
							handshakeCount = handshakeIncr;
							handshakeTime = hst;
						}
					}
				}
			} while (try_again);
			if (endreplay) break;

			/* writing data to store */
			if (filetimestamp)
				timestamp = rv->vh.ts;
			else
				rtx_time_add(&timestamp,&(rv->vh.ts),&offset);

			if (exacttimestamp)
				ddx_store_write((DDX_STORE_ITEM*)rv->userdata, rv->d, &timestamp);
			else
				ddx_store_write((DDX_STORE_ITEM*)rv->userdata, rv->d, NULL);
			localCount += 1;
			//printf("Written something, lc = %d hsc = %d\n",localCount,handshakeCount);
		}

		/* Clean up */
		for (rv = readlog_log->vars; rv != NULL; rv = rv->next) {
			if (rv->v->attribs & VAR_ATTR_TAG) {
				ddx_store_done_item((DDX_STORE_ITEM*)rv->userdata);
				rv->userdata = NULL;
			}
		}

		rtx_logread_close(readlog_log);
	}
	freeFileArray(logfilenames,numfiles);
	if (handshakeItem)
		ddx_store_done_item(handshakeItem);
	ddx_store_close(store);
	ddx_client_done();
	return (0);
}
