/*********************************************************************
 *
 * CSIRO Automation
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * www.cat.csiro.au/cmst
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 *********************************************************************/

static char *rcsid = "$Id: ddxlog-main.c 3642 2008-07-07 03:12:41Z zlo001 $";

/**
 * \file ddxlog-main.c
 * \brief DDX data logger
 * \author Pavan Sikka
 * \author Navid Nourani-Vatani 2007-02-28
 */

/**
 * \page ddxlog ddxlog
 *
 * The DDX data logger (ddxlog) can be used to log data being written to
 * the store. The data to be logged is specified in a configuration file.
 *
 * ddxlog supports the following options:
 * - -b\n
 *    run as a daemon
 * - -L\n
 *    inform launcher (historical, should not be used anymore)
 * - -v <level>\n
 *    debug information to be printed (default = 0)
 * - -c <filename>\n
 *    name of the configuration file (default = "./logger.conf")
 * - -d <log-dir>\n
 *    log file directory (default = "./")
 * - -n <log-file>\n
 *    log file name (default = ":.log")
 * - -f <filename>\n
 *    file to which the pid is written (default = "")
 * - -p <prio>\n
 *    priority of the logging threads
 * - -o\n
 *    start the logger with logging turned off
 * - -e\n
 *    print all debug messages to stderr
 * - -h\n
 *    print a help message and exit
 *
 * The option -n specifies the name of the log file. In this name, the
 * character ':' has a special meaning -- it is replaced by a timestamp
 * of the form "seconds.nano-seconds".
 *
 * The configuration file contains parameters for the logger in the form
 * of tag-value pairs separated by white space. Comments in the
 * configuration file are introduced by the '%' character.
 *
 * The following parameters are recognized:
 *
 * - logFileDir\n
 *    the log file directory
 * - logFileName\n
 *    the log file name
 * - fileExpireTime\n
 *    time (in seconds) after which the file is rolled over with suffix [sSmMhH]
 * - fileSizeLim\n
 *    maximum size of file before it is rolled over with suffix [kKmMgG]
 * - numBufs\n
 *    number of individual buffers in the data logger
 * - sizeBufs
 *    size of each individual data buffer with suffix [kKmMgG]
 * - queueLen
 *    for remote mode: number of variables in queue with suffix [kKmMgG]
 * - log\n
 *    name of the data item to be logged from the store. This
 *    parameter can appear as many times as needed.
 *    This parameter accepts the optional arguments:
 *      - prio=<prio>\n
 *          priority of the logging thread
 *      - skip=<count>\n
 *          one more than the number of records to skip before 
 *          logging to file.
 *
 * -- REMOTE LOGGING
 * New fetures are remote logging. This is achieved by starting ddxlog
 * in server mode "ddxlog --server" on one machine and starting another
 * ddxlog in remote mode "ddxlog --remote HOSTNAME" with the 
 * config file specifying the setup as described above. In this mode 
 * instead of saving the read data from store in a local file the data
 * is sent to the server on the remote machine. Using an internal queue
 * data are saved locally if the connection to the server is lost
 * temporary. The --port option can be used to specify a different
 * port for the server and remote.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/tcp.h>
#ifdef __linux__
#include <sys/file.h>
#include <libgen.h>
#else
#include <sys/lock.h>
#endif
#include <errno.h>
#include <signal.h>
#ifdef __sparc__
#include <libgen.h>
#endif

#include <rtx/circbuf.h>
#include <rtx/message.h>
#include <rtx/error.h>
#include <rtx/thread.h>
#include <rtx/main.h>
#include <rtx/param.h>
#include <rtx/mutex.h>
#include <rtx/log.h>
#include <rtx/logread.h>
#include <rtx/parse.h>
#include <rtx/signal.h>
#include <rtx/time.h>
#include <rtx/timer.h>
#include <rtx/getopt.h>
#include <rtx/list.h>
#include <rtx/inet.h>

#include <ddx.h>

#include "store-common.h"
#include "store-msg.h"
#include "store-messages.h"
#include "catalog-messages.h"

#define KILLER_PRIO      RTX_THREAD_PRIO_MIN
#define MAX_SHUTDOWN_WAIT_TIME    50

#define _TCP_SND_IDLE	1
#define _TCP_SND_INTVL	2
#define _TCP_SND_CNT	8

#define _TCP_RECV_IDLE	1
#define _TCP_RECV_INTVL	2
#define _TCP_RECV_CNT	9

/* sync_line = "%%.:|sync-line|:. 00001 $$";
 * the number is incremented at every sync
 */
#define SYNC_LEN 26

#define MAX_BUFF_LEN 8192 + sizeof(RtxLogItemHeader) + SYNC_LEN


/* Main logger data structure */

typedef struct _log_item_node {
  char                * name;
  RtxLogItemHeader     header;
  void                * d;
  DDX_STORE_ITEM      * itemP;
  RtxThread           * tid;
  RtxSem              * logSem;
  int                 prio;
  int                 skipCount;
  int                 num;
  int                 threadPaused;
  int                 threadReading;
  struct _logger      * logger;
} LOG_ITEM_NODE;

typedef struct _log_parms {
  int        basePrio;
  int        fileExpireTime;
  int        fileSizeLim;
  int        sizeBufs;
  int        gzip;
  char       * fileExpireTimeStr;
  char       * fileSizeLimStr;
  char       * sizeBufsStr;
  char       * queueLenStr;
  int        queueLen;
  int        numBufs;
  int        forkFlag;
  int        ignoreFlag;
  int        duration;
  int        verbose;
  char       ** defs;
  int        logOff;
  int        msgDestLocal;
  char       * path;
  char       * logFileDir;
  char       * logFileName;
  char       * configFileName;
  char       * indexFileName;
  char       * pidFileName;
  char       lockFile[BUFSIZ];
  unsigned char header[8192];
  int        headerLen;
  char       * hostName;
  int        port;
  int        server;
  int        remote;
  int        dots;
} LOG_PARMS;

typedef struct _logger {
  pid_t           pid;
  RtxList         * itemList;
  int             numThreads;
  LOG_PARMS       parms;
  DDX_STORE_ID    * store;
  RtxLog          * dataLogger;
  RtxLogStats     logStats;
  char            buf[BUFSIZ];
  RtxThread       * sigUsrHandler;
  RtxThread       * durationHandler;
  char            catalogName[128];
  /* Net vars */
  RtxInet         * tcpHandle;
  RtxCircbuf      * Q;
} LOGGER;

typedef struct _net_buffer_item {
  RtxLogItemHeader header;
  char * data;
  int data_size;
} NET_BUFFER_ITEM;


typedef struct _wlt_args {
  struct _logger 	* logger;
  RtxInetTcpClient 	* clnt;
  int				pfd;	/* pipe file descriptor */
} WLT_ARGS;

/* library functions/variables that need to be declared */

int plock();
double atof();

/* logger variables/parameters */

LOGGER logger;

static pthread_cond_t net_data_cond = PTHREAD_COND_INITIALIZER;
static pthread_mutex_t net_mutex = PTHREAD_MUTEX_INITIALIZER;

/* command-line options */

RtxGetopt ddxlogOpts[] = {
  {"-ignore", "ignore requests to log variables not in store", 
   {
     {RTX_GETOPT_SET, &logger.parms.ignoreFlag, ""},
     RTX_GETOPT_END_ARG
   }
  },
  {"-daemon", "fork and run as a daemon", 
   {
     {RTX_GETOPT_SET, &logger.parms.forkFlag, ""},
     RTX_GETOPT_END_ARG
   }
  },
  {"-nolog", "start up with logging turned off", 
   {
     {RTX_GETOPT_SET, &logger.parms.logOff, ""},
     RTX_GETOPT_END_ARG
   }
  },
  {"-stderr", "print messages to stderr", 
   {
     {RTX_GETOPT_SET, &logger.parms.msgDestLocal, ""},
     RTX_GETOPT_END_ARG
   }
  },
  {"-duration", "log for the specified duration", 
   {
     {RTX_GETOPT_INT, &logger.parms.duration, "duration (s)"},
     RTX_GETOPT_END_ARG
   }
  },
  {"-pidfile", "write pid in specified file", 
   {
     {RTX_GETOPT_STR, &logger.parms.pidFileName, "pid file name"},
     RTX_GETOPT_END_ARG
   }
  },
  {"-output", "create specified log file (a : is substituted by the timestamp)", 
   {
     {RTX_GETOPT_STR, &logger.parms.path, "path"},
     RTX_GETOPT_END_ARG
   }
  },
  {"-maxsize", "maximum file size before rollover as a number with suffix (kKmMgG)", 
   {
     {RTX_GETOPT_STR, &logger.parms.fileSizeLimStr, "max file size"},
     RTX_GETOPT_END_ARG
   }
  },
  {"-maxtime", "maximum time for file before rollover as a number with suffix (smh)", 
   {
     {RTX_GETOPT_STR, &logger.parms.fileExpireTimeStr, "max file age"},
     RTX_GETOPT_END_ARG
   }
  },
  {"-bufsize", "internal file buffer size as a number with suffix (kKmM)", 
   {
     {RTX_GETOPT_STR, &logger.parms.sizeBufsStr, "buffer size"},
     RTX_GETOPT_END_ARG
   }
  },
  {"-numbufs", "number of internal file buffers",
   {
     {RTX_GETOPT_INT, &logger.parms.numBufs, "numBufs"},
     RTX_GETOPT_END_ARG
   }
  },
  {"-queuelen", "for remote mode: number of variables in queue with suffix [kKmMgG]",
   {
     {RTX_GETOPT_STR, &logger.parms.queueLenStr, "queue length"},
     RTX_GETOPT_END_ARG
   }
  },
  {"-gzip", "gzip compression level", 
   {
     {RTX_GETOPT_INT, &logger.parms.gzip, "gzip"},
     RTX_GETOPT_END_ARG
   }
  },
  {"-index", "filename to add an index to", 
   {
     {RTX_GETOPT_STR, &logger.parms.indexFileName, "filename"},
     RTX_GETOPT_END_ARG
   }
  },
  {"-remote", "send data to remote ddxlog server", 
   {
     {RTX_GETOPT_SET, &logger.parms.remote, ""},
     {RTX_GETOPT_STR, &logger.parms.hostName, "host name"},
     RTX_GETOPT_END_ARG
   }
  },
  {"-server", "use this as a remote ddxlog server.", 
   {
     {RTX_GETOPT_SET, &logger.parms.server, ""},
     RTX_GETOPT_END_ARG
   }
  },
  {"-port", "use this to set a specific port number",
   {
     {RTX_GETOPT_INT, &logger.parms.port, "port"},
     RTX_GETOPT_END_ARG
   }
  },
  {"-dots", "use this for the process to print dots on the screen when active", 
   {
     {RTX_GETOPT_SET, &logger.parms.dots, ""},
     RTX_GETOPT_END_ARG
   }
  },
  RTX_GETOPT_END
};

char * ddxlogHelpStr = "Notes:\n"
"1. If log file name has a :, it is substituted by a timestamp\n";

/* forward defines */

static int log_init_store (LOGGER * logger);
static int log_create_log_threads (LOGGER * logger);
static int log_read_config_parms (LOGGER * logger);
static int log_build_init_item_list (LOGGER * logger);
static void log_print_parms (LOGGER * logger);
static int log_shutdown (LOGGER * logger);
static int log_start (LOGGER * logger);
static int log_stop (LOGGER * logger);
static void * log_handle_usr_sigs (void * arg);
static void * log_duration_handler (void * arg);
static int log_add_item (LOGGER * logger, char * name, int prio,
                         int skipCount);
static void * log_net_thread( LOGGER * logger );
static void * server_thread( void * arg1, void *arg2 );
static void * write_log_thread( WLT_ARGS * args );
int save_unsent_buffer( LOGGER * logger );
static int set_tcp_timeout(int sockfd, int idle, int intvl, int cnt, int verbose);

#ifdef __lynxos310
static char * dirname (char *path);
#endif

/** ********************************************************************************
 *  MAIN
 */
int
main (int argc, char *argv[])
{

  int        pid, i;
  int        skipCount, prio;
  char       * name;
  char       * path = NULL;
  char       b[3][64];
  char       ch;
  int        multFactor = 1;
#ifdef SHOULD_NOT_BE_REQUIRED
  int        fd;
  int        lockfd;
  int        oldmask;
#endif
  int        errs = 0;
  int        ret = -1;
  FILE       * fp;
  LOG_ITEM_NODE * p;

  RtxThread * tIdNet;

  rtx_main_init ("ddxlog", RTX_ERROR_STDERR | RTX_ERROR_MESSAGE);

  rtx_signal_block (SIGUSR1);
  rtx_signal_block (SIGUSR2);

  /* Initialize logger structure */
  logger.parms.indexFileName = NULL;
  logger.parms.fileExpireTimeStr = strdup ("1h");
  logger.parms.fileSizeLimStr = strdup ("1M");
  logger.parms.sizeBufsStr = strdup ("4k");
  logger.parms.queueLenStr = strdup ("12321");
  logger.parms.numBufs = 12;
  logger.parms.forkFlag = 0;
  logger.parms.duration = 0;
  logger.parms.path = strdup ("./:.log");
  logger.parms.pidFileName = strdup ("logger.pid");
  strcpy (logger.parms.lockFile, "");
  logger.pid = getpid ();
  if ((logger.itemList = rtx_list_init ()) == NULL) {
    rtx_error_flush ("rtx_list_init() failed");
    exit (1);
  }
  logger.parms.gzip = 0;
  logger.parms.remote = 0;
  logger.parms.server = 0;
  logger.parms.hostName = NULL;
  logger.parms.port = 15000;
  logger.parms.dots = 0;

  rtx_getopt_disable_config ();
  /* no command-line args, print help and exit */    
  if (argc == 1) {
    RTX_GETOPT_PRINT (ddxlogOpts, argv[0], rcsid, ddxlogHelpStr);
    exit (1);
  }

  if ((ret = RTX_GETOPT_CMD (ddxlogOpts, argc, argv, rcsid, ddxlogHelpStr)) == -1) {
    RTX_GETOPT_PRINT (ddxlogOpts, argv[0], rcsid, ddxlogHelpStr);
    exit (1);
  }
  if (ret == 0) {
    exit (1);
  }
  logger.parms.basePrio = rtx_getopt_get_priority (RTX_THREAD_PRIO_MIN);

  logger.parms.verbose = rtx_getopt_get_verbose (0);
  logger.parms.configFileName = rtx_getopt_get_str ("config");
  logger.parms.defs = rtx_getopt_get_strcat ("D");

  /* process config file if specified */
  if (logger.parms.configFileName != NULL) {
    /*
     * Read parameters from config file
     */
    if (logger.parms.verbose)
      rtx_message("main: reading params");
    if (log_read_config_parms (&logger) == -1) {
      rtx_error_flush ("main: log_read_config_parms() failed");
      exit(1);
    }
    /*
     * Read variable names from config file
     */
    if (logger.parms.verbose)
      rtx_message("main: reading variable names");
    if (log_build_init_item_list (&logger) == -1) {
      rtx_error_flush ("main: log_build_init_item_list() failed");
      exit(1);
    }
    /* process command-line options again to override config file params */
    if ((ret = RTX_GETOPT_CMD (ddxlogOpts, argc, argv, rcsid, ddxlogHelpStr)) == -1) {
      RTX_GETOPT_PRINT (ddxlogOpts, argv[0], rcsid, ddxlogHelpStr);
      exit (1);
    }
    if (ret == 0) {
      exit (1);
    }
  }
  /* add variables specified on the command-line */
  for (i=ret; i<argc; i++) {
    skipCount = 1;
    prio = logger.parms.basePrio;
    if (sscanf (argv[i], "%[^:]::%s", b[0], b[1]) == 2) {  
      /* x::3 */
      name = b[0];
      skipCount = atoi (b[1]);
      if (skipCount == 0)
        skipCount = 1;
    } else if (sscanf (argv[i], "%[^:]:%[^:]:%s", b[0], b[1], b[2]) == 3) {
      /* x:32:3 */
      name = b[0];
      prio = atoi (b[1]);
      skipCount = atoi (b[2]);
      if (skipCount == 0)
        skipCount = 1;
      if (prio <= 0)
        prio = logger.parms.basePrio;
    } else if (sscanf (argv[i], "%[^:]:%s", b[0], b[1]) == 2) {
      /* x:32 */
      name = b[0];
      prio = atoi (b[1]);
      if (prio <= 0)
        prio = logger.parms.basePrio;
    } else {
      /* x */
      name = argv[i];
    }
    if (log_add_item (&logger, name, prio, skipCount) == -1) {
      rtx_error_flush ("log_add_item() failed");
      exit(1);
    }
  }
  /* deal with other command-line options */
  /* process pathname for the log file */
  if ((path = strdup (logger.parms.path)) == NULL) {
    rtx_error_flush ("strdup() failed");
    exit(1);
  }
  logger.parms.logFileName = basename (path);
  if ((path = strdup (logger.parms.path)) == NULL) {
    rtx_error_flush ("strdup() failed");
    exit(1);
  }
  logger.parms.logFileDir = dirname (path);
  /* process file size limit for rollover */
  ch = logger.parms.fileSizeLimStr[strlen(logger.parms.fileSizeLimStr)-1];
  switch (ch) {
  case 'k':
  case 'K':
    multFactor = 1024;
    logger.parms.fileSizeLimStr[strlen(logger.parms.fileSizeLimStr)-1] = '\0';
    break;
  case 'm':
  case 'M':
    multFactor = 1024 * 1024;
    logger.parms.fileSizeLimStr[strlen(logger.parms.fileSizeLimStr)-1] = '\0';
    break;
  case 'g':
  case 'G':
    multFactor = 1024 * 1024 * 1024;
    logger.parms.fileSizeLimStr[strlen(logger.parms.fileSizeLimStr)-1] = '\0';
    break;
  default :
    multFactor = 1;
    break;
  }
  logger.parms.fileSizeLim = multFactor * atoi (logger.parms.fileSizeLimStr);
  if (logger.parms.fileSizeLim < 1024) {
    logger.parms.fileSizeLim = 1024;
  }
  /* process size of internal buffers */
  ch = logger.parms.sizeBufsStr[strlen(logger.parms.sizeBufsStr)-1];
  switch (ch) {
  case 'k':
  case 'K':
    multFactor = 1024;
    logger.parms.sizeBufsStr[strlen(logger.parms.sizeBufsStr)-1] = '\0';
    break;
  case 'm':
  case 'M':
    multFactor = 1024 * 1024;
    logger.parms.sizeBufsStr[strlen(logger.parms.sizeBufsStr)-1] = '\0';
    break;
  case 'g':
  case 'G':
    multFactor = 1024 * 1024 * 1024;
    logger.parms.sizeBufsStr[strlen(logger.parms.sizeBufsStr)-1] = '\0';
    break;
  default :
    multFactor = 1;
    break;
  }
  logger.parms.sizeBufs = multFactor * atoi (logger.parms.sizeBufsStr);
  if (logger.parms.sizeBufs < 1024) {
    logger.parms.sizeBufs = 1024;
  }
  /* process queue length */
  ch = logger.parms.queueLenStr[strlen(logger.parms.queueLenStr)-1];
  switch (ch) {
  case 'k':
  case 'K':
    multFactor = 1000;
    logger.parms.queueLenStr[strlen(logger.parms.queueLenStr)-1] = '\0';
    break;
  case 'm':
  case 'M':
    multFactor = 1000 * 1000;
    logger.parms.queueLenStr[strlen(logger.parms.queueLenStr)-1] = '\0';
    break;
  case 'g':
  case 'G':
    multFactor = 1000 * 1000 * 1000;
    logger.parms.queueLenStr[strlen(logger.parms.queueLenStr)-1] = '\0';
    break;
  default :
    multFactor = 1;
    break;
  }
  logger.parms.queueLen = multFactor * atoi (logger.parms.queueLenStr);
  if (logger.parms.queueLen < 1000) {
    logger.parms.queueLen = 1000;
  }
  /* process age of log file for rollover */
  ch = logger.parms.fileExpireTimeStr[strlen(logger.parms.fileExpireTimeStr)-1];
  switch (ch) {
  case 's':
  case 'S':
    multFactor = 1;
    logger.parms.fileExpireTimeStr[strlen(logger.parms.fileExpireTimeStr)-1] = '\0';
    break;
  case 'm':
  case 'M':
    multFactor = 60;
    logger.parms.fileExpireTimeStr[strlen(logger.parms.fileExpireTimeStr)-1] = '\0';
    break;
  case 'h':
  case 'H':
    multFactor = 60 * 60;
    logger.parms.fileExpireTimeStr[strlen(logger.parms.fileExpireTimeStr)-1] = '\0';
    break;
  default :
    multFactor = 1;
    break;
  }
  logger.parms.fileExpireTime = multFactor * atoi (logger.parms.fileExpireTimeStr);
  if (logger.parms.fileExpireTime < 60) {
    logger.parms.fileExpireTime = 60;
  }

  /* if rollovers are not required, set corresponding parms */
  if (strchr (logger.parms.logFileName, ':') == NULL) {
    logger.parms.fileExpireTime = 0;
    logger.parms.fileSizeLim = 0;
  }

#ifdef SHOULD_NOT_BE_REQUIRED
  /*
   * Check for other loggers already running using same config_file
   */
  oldmask = umask (0);
  sprintf (logger.parms.lockFile, "/tmp/locks/%s", 
           basename (logger.parms.configFileName));
  if ((lockfd = open (logger.parms.lockFile ,
                      O_CREAT | O_RDWR, 0666)) == -1) {
    rtx_error_errno_flush ("main: open(%s) failed", logger.parms.lockFile); 
    exit (1);
  }
  umask (oldmask);
  if (flock (lockfd, LOCK_EX | LOCK_NB)) {
    rtx_error_errno_flush ("main: flock(%s) failed", logger.parms.lockFile); 
    exit(1);
  }
#endif

  /*      
   * Fork the process
   */     
  if (logger.parms.forkFlag) {
    pid = fork();
    if (pid < 0) {
      rtx_error_errno_flush ("main: fork() failed"); 
      exit(1);
    } else if (pid > 0) {
      exit(0);
    }
  }

  /*
   * Lock the process in memory to prevent swapping
   */
#ifdef __Lynx__
  if (plock () != 0) {
    rtx_error_errno_flush ("main: unable to lock process");
  }
#endif


    if (logger.parms.verbose) {
      log_print_parms (&logger);
    }
    
  // ---------------------------------------------------------------------
  // If this is a server, start the server thread and skip everything else
  if( logger.parms.server ) {
    rtx_message( "Starting in server mode..." );
    logger.parms.headerLen = 0;
    logger.parms.hostName = NULL;
    logger.tcpHandle = 
      rtx_inet_init( RTX_INET_TCP_SERVER, 
                     logger.parms.hostName, logger.parms.port, 
                     NULL, 0, 
                     server_thread, NULL, &logger );
    if( logger.tcpHandle == NULL ) {
      rtx_error( "main: rtx_inet_init() failed." );
      return -1;
    }
    
    rtx_message( "Host name: %s, host port: %d", 
                 logger.tcpHandle->sock->hostname,
                 logger.tcpHandle->sock->local.portNum );
    rtx_message( "Waiting for remote connection..." );
  }
  /* *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** */
  else {
    /*
     * Lookup the variable names from the store
     */
    if (logger.parms.verbose)
      rtx_message("main: initializing connection to store and looking "
                  "up variable names");
    if (log_init_store (&logger) == -1) {
      rtx_error_flush ("main: log_init_store() failed");
      exit(1);
    }

    /* Deals with comment file */
    if (logger.parms.indexFileName) {
      FILE *comment = fopen(logger.parms.indexFileName,"a");
      if (comment == NULL) {
        rtx_message("main: appending to comment file failed");
      } else {
        char tbuffer[128];
        RtxTime ts;
        rtx_time_get(&ts);
        rtx_time_conv_date_time(tbuffer,127,"","-",": ",&ts);
        fprintf(comment,tbuffer);
        while ((p = (LOG_ITEM_NODE *) rtx_list_iterate (logger.itemList)) != NULL) {
          fprintf (comment, "%s(skip %d) ",
                   p->name, p->skipCount); 
        }
        fprintf(comment,"\n");
        fclose(comment);
        rtx_message("Appended comment to '%s'",logger.parms.indexFileName);
      }
    }

    // -------------------------------------------------------------
    // Setup remote network stuff
    if( logger.parms.remote ) {
      rtx_message( "Logging remotely to %s, Q length: %d", logger.parms.hostName, logger.parms.queueLen);
      if (logger.parms.queueLen == 12321) {
      	rtx_message( "WARNING! Q length not specifically set! Use option \"--queuelen <len[kKmMgG]>\" to set queue length");
      }

      int numQBufs = logger.parms.queueLen;
      logger.Q = rtx_circbuf_init( numQBufs, sizeof(NET_BUFFER_ITEM),
                                   NULL, NULL, 0, 0);
      if( (tIdNet = rtx_thread_create( "log_net_thread",
                                       logger.parms.verbose,
                                       RTX_THREAD_SCHED_OTHER,
                                       logger.parms.basePrio,
                                       16384,
                                       RTX_THREAD_CANCEL_DEFERRED,
                                       (void *)log_net_thread,
                                       &logger,
                                       NULL, NULL)) == NULL ) {
        return( rtx_error( "Unable to create log_net_thread" ) );
      }

      // Check for existing unsent buffers
      char ** existingFiles = NULL;
      int numFiles = 0;
      char unsentDir[256];
      sprintf( unsentDir, "%s/unsent/%s", logger.parms.logFileDir, logger.parms.hostName );
      existingFiles = rtx_logread_get_files( unsentDir, &numFiles );
      if( numFiles && existingFiles ) {
        rtx_message( "Num unsent log files is %d", numFiles );
        int i;
        // Read the unsent buffers and add their data
        // to the Q
        for( i = 0; i < numFiles; i++ ) {
          rtx_message( "Reading unsent file %d/%d: %s", i+1, numFiles, existingFiles[i] );
			    
          // Read unsent file and add to Q
          RtxLogread * unsentLog = rtx_logread_init( existingFiles[i], 0 );
          if( ! unsentLog ) {
            rtx_error( "Could not retrieve next log. Skipping." );
            continue;
          }
          NET_BUFFER_ITEM item;
          item.data = NULL;
          item.data_size = 0;

          RtxLogreadVar * readVar = NULL;
          while( (readVar = rtx_logread_next( unsentLog )) != NULL ) {
            item.data = (char *)malloc( sizeof(char) * readVar->v->size );

            // copy data from read var to item and
            // add to Q
            memcpy( &(item.header), &(readVar->vh), sizeof(RtxLogItemHeader) );
            memcpy( item.data, readVar->d, readVar->v->size );
            item.data_size = readVar->v->size;
            pthread_mutex_lock( &net_mutex );
            if( rtx_circbuf_put( logger.Q, &item, sizeof(NET_BUFFER_ITEM) ) != 0 ) {
              // if adding to buffer unsuccessfull, free the memory
              rtx_error( "Could not add variable to queue. Skipping." );
              free(item.data);
            }
            else {
            	pthread_cond_broadcast( &net_data_cond );
            }
            item.data = NULL;
            pthread_mutex_unlock( &net_mutex );
          }

          // remove unsent file from unsent folder
          char sysCmd[128];
          sprintf( sysCmd, "rm %s", existingFiles[i] );
          system( sysCmd );
        }

        rtx_message( "Finished loading data from unsent files\n" );
      }

      logger.dataLogger = NULL;
    }
    else {
      // Initialize the file writing module when not in
      //   remote mode.
      if (logger.parms.verbose) {
        rtx_message("main: initializing the file writing module");
      }
      if ((logger.dataLogger = rtx_log_open (
                                             logger.parms.logFileDir,
                                             logger.parms.logFileName,
                                             logger.parms.fileExpireTime / 60,
                                             logger.parms.fileSizeLim / 1024,
                                             logger.parms.numBufs,
                                             logger.parms.sizeBufs / 1024,
                                             logger.parms.basePrio, 
                                             logger.parms.gzip, 
                                             logger.parms.verbose, 
                                             logger.parms.header,
                                             logger.parms.headerLen,
                                             "ddxlog", 
                                             1, 
                                             logger.parms.indexFileName)) == NULL) {
        rtx_error_flush ("main: rtx_log_open() failed");
        exit(1);
      }
		    
    }
    /*
     * Create logger threads
     */
    if (logger.parms.verbose)
      rtx_message("main: createing logger threads");
    if (log_create_log_threads (&logger) == -1) {
      rtx_error_flush ("main: log_create_log_threads() failed");
      exit(1);
    }

    /* If logging is not turned off, signal each thread to start logging */
    if (logger.parms.logOff == 0) {
      if (log_start (&logger)) {
        rtx_error_flush ("main: log_create_log_threads() failed");
        exit(1);
      }   
    }

    if ((fp = fopen (logger.parms.pidFileName, "w")) != NULL) {
      fprintf (fp, "%d\n", (int) logger.pid);
      fclose (fp);
    }
    if( ! logger.parms.remote ) {
      rtx_message( "logging to file: %s", logger.dataLogger->pathname);
    }
    while ((p = (LOG_ITEM_NODE *) rtx_list_iterate (logger.itemList)) != NULL)
      rtx_message( "logging %s (prio = %d, skip = %d, tid = %lu)",
                   p->name, p->prio, p->skipCount, 
                   ((unsigned long int)(p->tid->id)));

    /* start the handler thread for SIGUSR1/2 signals */
    if ((logger.sigUsrHandler = rtx_thread_create ("log_handle_usr_sigs thread",
                                                   logger.parms.verbose,
                                                   RTX_THREAD_SCHED_OTHER,
                                                   logger.parms.basePrio,
                                                   16384,
                                                   RTX_THREAD_CANCEL_DEFERRED,
                                                   log_handle_usr_sigs,
                                                   &logger,
                                                   NULL,
                                                   NULL)) == NULL) {
      return (rtx_error ("unable to create log_handle_usr_sigs thread"));
    }

    if (logger.parms.duration > 0) {
      if ((logger.durationHandler = rtx_thread_create 
           ("duration thread",
            logger.parms.verbose,
            RTX_THREAD_SCHED_OTHER,
            logger.parms.basePrio,
            16384,
            RTX_THREAD_CANCEL_DEFERRED,
            log_duration_handler,
            &logger,
            NULL,
            NULL)) == NULL) {
        return (rtx_error ("unable to create duration handler thread"));
      }
    }
  } // end else (= if(!server))	

#ifdef __Lynx__
  if (rtx_main_wait_shutdown (KILLER_PRIO)) {
    rtx_error ("main: rtx_init_wait_shutdown() failed");
    errs++;
  }
#else
  if (rtx_main_wait_shutdown (0)) {
    rtx_error ("main: rtx_init_wait_shutdown() failed");
    errs++;
  }
#endif

  rtx_message ("shutting down...");
  // Only do this when not in remote mode
  if( ! logger.parms.server ) {
    if (log_shutdown (&logger)) {
      rtx_error ("main: logger_shutdown() failed");
      errs++;
    }
    rtx_message ("log_shutdown() done");
	  
	  
    if( logger.parms.remote ) {
      rtx_message( "Entering remote control area" );
      // At this point all threads except the net thread should
      //   be stopped. We stay in this mode until time is up or
      //   the net data queue is empty, if we have net
      //   connection.
      if( logger.Q->numInBuf != 0 ) {
        rtx_message( "Net data queue size is %d.", logger.Q->numInBuf );
	      
        RtxTime t0;
        RtxTime t_curr;
        rtx_time_get(&t0);
        while( rtx_time_get(&t_curr) ) {
          if( t_curr.seconds - t0.seconds > MAX_SHUTDOWN_WAIT_TIME ) {
            rtx_message( "Times up. No more time to transfer logged data." );
            break;
          }
		
          if( logger.Q->numInBuf == 0 ) {
            rtx_message( "Net data queue is empty." );
            break;
          }
		
          if( logger.tcpHandle == NULL ) {
            rtx_message( "No connection to server. Can't transfer buffered data." );
          }
		
          rtx_timer_sleep( 0.1 );
        }
	      
        // At this point the Net thread is also stopped. If any
        // data left in buffer this is written to file.
        save_unsent_buffer( &logger );
      }

      //TODO: free memory of circbuf Q here
      //add free() to rtx_circbuf_done 
      //rtx_circbuf_done(logger.Q);
        
      // Close connection to server if it exists
      if( logger.tcpHandle != NULL ) {
        rtx_inet_done( logger.tcpHandle );
      }
      // Distroy net thread
      rtx_thread_destroy_sync( tIdNet ); 
    }// if( remote )
	    
    unlink (logger.parms.pidFileName);
    if( ! logger.parms.remote ) { 
      while ((p = (LOG_ITEM_NODE *) rtx_list_iterate (logger.itemList)) != NULL)
        rtx_message( "logged %s [%d records]", p->name, p->num);
      rtx_message( "bytes written to file [%d]", logger.logStats.bytesToFile);
      rtx_message( "files written [%d]", logger.logStats.filesCreated);
      if (logger.logStats.writeErrors)
        rtx_message( "%d write errors", logger.logStats.writeErrors);
      if (logger.logStats.openErrors)
        rtx_message( "%d open errors", logger.logStats.openErrors);
      if (logger.logStats.closeErrors)
        rtx_message( "%d close errors", logger.logStats.closeErrors);
      if (logger.parms.verbose)
        rtx_message("main: done");
      if (errs) {
        rtx_error_flush ("main: errors during shutdown");
        exit (1);
      }
    }
  } // if( ! server )
  else {
    rtx_inet_done( logger.tcpHandle );
  }
  exit(0);
}

/** ********************************************************************************
 *
 */ 
static void
log_item_thread (
                 LOG_ITEM_NODE * p
                 )
{


  RtxTime ts = {0, 0};
  RtxLogWriteBufs datalogWriteBufs;
  int errs = 0;
  int tailCount=0;
  int queueLen=0;

  NET_BUFFER_ITEM item;

  rtx_signal_block (SIGUSR1);
  rtx_signal_block (SIGUSR2);

  datalogWriteBufs.buf[0] = (unsigned char *) &(p->header);
  datalogWriteBufs.bufLen[0] = sizeof (p->header);
  datalogWriteBufs.buf[1] = p->d;
  datalogWriteBufs.bufLen[1] = p->itemP->varSize;
  datalogWriteBufs.buf[2] = NULL;
  datalogWriteBufs.bufLen[2] = 0;

  p->logger->numThreads++;

  if (p->skipCount < 0) {
    /* setup queuelength */
    rtx_message ("setting %s queuelength = %d", p->name, -p->skipCount);

    ddx_store_item_set_queue(p->itemP, -p->skipCount);
  }

  while (1) {
    rtx_thread_testcancel ();
    errs = 0;
    if (rtx_sem_wait (p->logSem) == -1) {
      rtx_error_flush ("rtx_sem_wait() failed, proceeding with log");
      rtx_timer_sleep (0.01);
    }
    p->threadPaused = 0;
    while (p->logger->parms.logOff == 0) {
      rtx_thread_testcancel ();
      if (p->logger->parms.logOff) {
        continue;
      }
      p->threadReading = 1;
      queueLen = p->itemP->headerP->queueLength;
      //if ( p->skipCount>=0) {
      if( queueLen == 0 ){
      switch (ddx_store_read (p->itemP, p->d, &(p->header.ts), 1.0,
                                p->skipCount)) {
        case -1:
          rtx_error_flush ("ddx_store_sync_read() failed");
          continue;
        case +1:
          continue;
        default :
          break;
        }
      } else {
        /* variable is a queue */
        switch (ddx_store_queue_read (p->itemP, p->d, &(p->header.ts), 1.0,
                                      &tailCount)) {
        case -1:
          rtx_error_flush ("ddx_store_queue_read() failed");
          continue;
        case -2:
          continue;
        default :
          break;
        }
      }
      if (p->logger->parms.verbose > 1) {
        rtx_message ("read %s [%d]", p->name, p->num);
      }
      p->threadReading = 0;
      if (p->logger->parms.logOff)
        continue;
      if (memcmp (&ts, &(p->header.ts), sizeof (ts)) == 0) {
        if (p->logger->parms.verbose) {
          rtx_message_error ("duplicate timestamp");
        }
        errs++;
      } else if( ! p->logger->parms.remote ){
        rtx_thread_setcancel (RTX_THREAD_CANCEL_DISABLE);
        if (rtx_log_write_bufs (p->logger->dataLogger, &datalogWriteBufs)
            == -1) {
          rtx_error_flush ("datalog_write_bufs() failed");
          errs++;
        }
        p->num++;
        ts = p->header.ts;
        rtx_thread_setcancel (RTX_THREAD_CANCEL_DEFERRED);
      }

      /* Add to the net Q */
      if( p->logger->parms.remote ) {
        if (p->logger->parms.verbose > 1) {
          rtx_message( "got new data from ddx..." );
        }
		rtx_thread_setcancel (RTX_THREAD_CANCEL_DISABLE);
        item.data = (char *)malloc( sizeof(char) * p->itemP->varSize );
        memcpy( &(item.header), &(p->header), sizeof(RtxLogItemHeader) );
        memcpy( item.data, p->itemP->d, p->itemP->varSize );
        item.data_size = p->itemP->varSize;
        if( p->logger->parms.verbose ) {
          if( item.data == NULL ) {
            rtx_message( "Data is NULL!!" );
          }
          else if (p->logger->parms.verbose > 1) {
            rtx_message( "Read data..." );
          }
        }
        pthread_mutex_lock( &net_mutex );
				
        // Check if buffer is full. If write
        // add data to Q.
        if( p->logger->Q->numInBuf < (p->logger->Q->n) ) {
          if( rtx_circbuf_put( p->logger->Q, &item, sizeof(NET_BUFFER_ITEM) ) == 0 ) {
            item.data = NULL;
            p->num++;
            if (p->logger->parms.verbose > 1) {
              rtx_message( "Added %d bytes, Q size %d", item.data_size, p->logger->Q->numInBuf );
            }
          }
          else {
            rtx_error( "Could not add data to Q!" );
          }
        }
        if (item.data) {
          // free up unused data
          free(item.data);
          item.data = NULL;
        }
        pthread_cond_broadcast( &net_data_cond );
        pthread_mutex_unlock( &net_mutex );
        rtx_thread_setcancel (RTX_THREAD_CANCEL_DEFERRED);
      }

      /*
        Moved above. Navid 2007-02-21
        if (! errs) {
        p->num++; 
        ts = p->header.ts; 
        }
      */
    }
    p->threadPaused = 1;
  }
}

/** ********************************************************************************
 *
 */ 
static int
log_create_log_threads (
                        LOGGER * logger
                        )
{
  LOG_ITEM_NODE * p;
  char loggerThName[BUFSIZ];

  while ((p = (LOG_ITEM_NODE *) rtx_list_iterate (logger->itemList)) != NULL) {
    if (p->d != NULL) {
      if ((p->logSem = rtx_sem_init (NULL, 0, 0)) == NULL) {
        return (rtx_error ("unable to create logging semaphore"));
      }
      sprintf (loggerThName, "ddxlog[%s]", p->itemP->varName);
      if ((p->tid = rtx_thread_create (loggerThName,
                                       logger->parms.verbose,
                                       RTX_THREAD_SCHED_OTHER,
                                       p->prio,
                                       16384,
                                       RTX_THREAD_CANCEL_DEFERRED,
                                       (void * (*)(void *)) log_item_thread, 
                                       p,
                                       NULL,
                                       NULL)) == NULL) {
        return (rtx_error ("unable to create logging thread"));
      }
    }
  }
  return (0);
}
/** ********************************************************************************
 *
 */ 
static int
log_create_header (
                   LOGGER * logger
                   )
{
  LOG_ITEM_NODE * p;
  int n, endianInt = RTX_LOG_HEADER_MAGIC_NUMBER;
  int dataSize[8];
  int dataAlign[8];

  rtx_parse_get_compiler_parms (dataSize, dataAlign);
  sprintf ((char*) logger->parms.header,
           "%%%%primitiveDataSize: %d %d %d %d %d %d\n"
           "%%%%primitiveDataAlignment: %d %d %d %d %d %d\n",
           dataSize[2], dataSize[3], dataSize[4], dataSize[5], 
           dataSize[6], dataSize[7], dataAlign[2], dataAlign[3], 
           dataAlign[4], dataAlign[5], dataAlign[6], dataAlign[7]);

  strcat ((char*) logger->parms.header, "%%CatalogName: ");
  strcat ((char*) logger->parms.header, logger->catalogName);
  strcat ((char*) logger->parms.header, "\n");

  while ((p = (LOG_ITEM_NODE *) rtx_list_iterate (logger->itemList)) != NULL) {
    if (p->d != NULL) {
      strcat ((char*) logger->parms.header, p->itemP->varDefinition);
      strcat ((char*) logger->parms.header, "\n");
    }
  }
  strcat ((char*) logger->parms.header, RTX_LOG_HEADER_EOH);
  n = strlen ((char*) logger->parms.header);
  memcpy (&(logger->parms.header[n]), &endianInt, sizeof (int));
  logger->parms.headerLen = n + sizeof (int);
  return (0);
}
/** ********************************************************************************
 *
 */ 
static int
log_init_store (
                LOGGER * logger
                )
{
  LOG_ITEM_NODE * p;
  int itemNum = 0;

  if (ddx_client_init (logger->parms.verbose) == -1) {
    return (rtx_error ("log_init_store: ddx_client_init() failed"));
  }
  if ((logger->store = ddx_store_open (NULL, 0, 5)) == NULL) {
    return (rtx_error ("log_init_store: ddx_store_open() failed"));
  }

  while ((p = (LOG_ITEM_NODE *) rtx_list_iterate (logger->itemList)) != NULL) {
    if ((p->itemP = ddx_store_lookup_item (logger->store, p->name,
                                           NULL, 0)) == NULL) {
      if (logger->parms.ignoreFlag) {
        rtx_message ("ddxlog: %s not in store", p->name);
        continue;
      } else {
        return (rtx_error ("log_init_store: ddx_store_lookup_item(%s)"
                           " failed", p->name));
      }
    }
    if ((p->d = (void *) calloc (1, p->itemP->varSize)) == NULL) {
      return (rtx_error_errno ("log_init_store: calloc()"
                               " failed (%s)", p->name));
    }
    p->header.id = itemNum++;
  }
  log_create_header (logger);
  if (logger->parms.verbose) {
    rtx_message ("ddxlog: header length = %d", logger->parms.headerLen);
    rtx_message ("ddxlog: header = %s", logger->parms.header);
  }
  return (0);
}
/** ********************************************************************************
 * This function is called by killer_wait() when it receives the kill signal
 */
static int
log_shutdown (
              LOGGER * logger
              )
{
  LOG_ITEM_NODE * p;
  int    errors = 0;

  /* this is for debugging the shutdown problem */
  /*
    logger->parms.verbose = 1;
    logger->store->verbose = 1;
    logger->dataLogger->parms.verbose = 1;
  */

  if (logger->parms.verbose)
    rtx_message ("log_shutdown: shutting down logger");

  rtx_thread_destroy_sync (logger->sigUsrHandler);
  while ((p = (LOG_ITEM_NODE *) rtx_list_iterate (logger->itemList)) != NULL) {
    if (logger->parms.verbose)
      rtx_message ("log_shutdown: destroying thread %s", p->name);
    /* for debugging shutdown problem */
    /*
      p->tid->debug = 1;
    */
    rtx_thread_destroy_sync (p->tid);
    if (logger->parms.verbose)
      rtx_message ("log_shutdown: store done var %s", p->name);
    ddx_store_done_item (p->itemP);
  }

  if (logger->parms.verbose)
    rtx_message ("log_shutdown: closing connection to store");

  ddx_store_close (logger->store);
  ddx_client_done ();

  if( logger->dataLogger ) {
    rtx_log_close (logger->dataLogger, &logger->logStats);
  }
  if (logger->parms.verbose)
    rtx_message ("log_shutdown: LOGGER SHUT DOWN");

  if (errors)
    return (-1);
  else
    return (0);
}
/** ********************************************************************************
 *
 */ 
static int
log_start (
           LOGGER * logger
           )
{
  LOG_ITEM_NODE * p;
  int errs = 0;

  rtx_message( "log_start: starting logger");
  logger->parms.logOff = 0;
  while ((p = (LOG_ITEM_NODE *) rtx_list_iterate (logger->itemList)) != NULL) {
    if (p->d != NULL) {
      if (rtx_sem_post (p->logSem)) {
        errs++;
        rtx_error ("log_start: rtx_sem_post() failed");
      }
    }
  }
  if (errs)
    return (-1);
  return (0);
}
/** ********************************************************************************
 *
 */ 
static int
log_stop (
          LOGGER * logger
          )
{
  LOG_ITEM_NODE * p;
  int threadsPaused = 0;
  int threadsReading = 0;
  RtxTime ts;

  logger->parms.logOff = 1;
  rtx_message( "log_stop: pausing logger, %d threads", 
               logger->numThreads);
  /* Number of threads explicitly paused may be less than the
   * number of threads since some threads might be waiting in
   * the ddx_store_read() function.
   */
  rtx_time_get (&ts);
  rtx_timer_sleep (0.1);
  while (threadsPaused + threadsReading != logger->numThreads) {
    rtx_timer_sleep (0.1);
    while ((p = (LOG_ITEM_NODE *) rtx_list_iterate (logger->itemList)) != NULL) {
      threadsPaused += p->threadPaused;
      threadsReading += p->threadReading;
    }
    /*
      rtx_message( "log_stop: %d threads, %d reading, %d paused",
      logger->numThreads, threadsReading, threadsPaused);
    */
    if (threadsPaused + threadsReading == logger->numThreads) {
      rtx_message( "log_stop: logging paused");
      break;
    }
    if (rtx_time_get_delta (&ts) > 1.0) {
      rtx_error_flush ("log_stop: waiting too long (> 1s)");
      break;
    }
  }
  if( logger->dataLogger ) {
    rtx_log_rollover_log (logger->dataLogger, NULL, 0, 0);
  }
  return (0);
}
/** ********************************************************************************
 *
 */ 
static int
log_add_item (
              LOGGER * logger,
              char * name,
              int prio,
              int skipCount
              )
{
  LOG_ITEM_NODE * item = NULL;

  if ((item = (LOG_ITEM_NODE *) calloc (1, sizeof (LOG_ITEM_NODE))) 
      == NULL) {
    return (rtx_error_errno ("log_add_item: calloc()"
                             " failed (%s)", name));
  }
  if ((item->name = strdup (name)) == NULL) {
    return (rtx_error_errno ("log_add_item: strdup()"
                             " failed (%s)", name));
  }
  if (item->name[strlen (item->name) - 1] == '\n')
    item->name[strlen (item->name) - 1] = '\0';
  item->logger = logger;
  item->prio = prio;
  item->skipCount = skipCount;
  if (rtx_list_add (logger->itemList, item->name, item) == -1)
    return (rtx_error ("log_add_item: rtx_list_add() failed"));
  return (0);
}
/** ********************************************************************************
 *
 */ 
static int
log_build_init_item_list (
                          LOGGER * logger
                          )
{
  FILE * fp;
  char * tok;
  char * name;
  int prio;
  int skipCount;
  int numDefs = 0, i;
  char b[BUFSIZ];

  numDefs = rtx_getopt_get_num_strings (logger->parms.defs);
#ifdef sparc_solaris
  sprintf (b, "/usr/local/bin/cpp -P ");
#else
  sprintf (b, "/usr/bin/cpp -P ");
#endif
  for (i=0; i<numDefs; i++) {
    strcat (b, "-D");
    strcat (b, logger->parms.defs[i]);
    strcat (b, " ");
  }
  strcat (b, logger->parms.configFileName);
  if ((fp = popen (b, "r")) == NULL)
    return (rtx_error_errno ("log_build_init_item_list: popen(%s)"
                             " failed", b));
  if (logger->parms.verbose)
    rtx_message ("log_build_init_item_list: opened config file %s", b);
  while (fgets (logger->buf, BUFSIZ, fp) != NULL) {
    if (logger->parms.verbose > 1)
      rtx_message ("log_build_init_item_list: got line: %s", logger->buf);
    tok = strtok (logger->buf, " \t");
    if (strcmp (tok, "log"))
      continue;
    tok = strtok (NULL, " \t");
    if (tok == NULL)
      continue;
    name = tok;
    prio = logger->parms.basePrio;
    skipCount = 1;
    while (tok != NULL) {
      if ((tok = strtok (NULL, " \t=")) == NULL)
        continue;
      if (strcmp (tok, "prio") == 0) {
        if ((tok = strtok (NULL, " \t=")) == NULL)
          continue;
        prio = atoi (tok);
      }
      if (strcmp (tok, "skip") == 0) {
        if ((tok = strtok (NULL, " \t=")) == NULL)
          continue;
        skipCount = atoi (tok);
      }
      if (prio <= 0)
        prio = logger->parms.basePrio;
      if (skipCount == 0)
        skipCount = 1;
    }
    if (logger->parms.verbose)
      rtx_message ("log_build_init_item_list: logging: %s (%d, %d)",
                   name, prio, skipCount);
    if (log_add_item (logger, name, prio, skipCount) == -1)
      return (rtx_error ("log_build_init_item_list: log_add_item() "
                         "failed"));
  }
  pclose (fp);
  return (0);
}

/** ********************************************************************************
 * This kludge is needed to deal with the param module errors
 * when a parameter is not in the file.
 */
static int validParamError = 0;

static void
log_param_err_func (
                    const char * name,
                    const char * errStr
                    )
{
  if (strcmp (errStr, "PARAMETER NOT FOUND") != 0)
    validParamError++;
}
/** ********************************************************************************
 *
 */ 
static int
log_read_config_parms (
                       LOGGER * logger
                       )
{
  RtxParamStream * pf;
  int errs = 0, numDefs = 0;
  char * logFile = NULL;
  char * fileExpireTimeStr = NULL;
  char * fileSizeLimStr = NULL;
  char * sizeBufsStr = NULL;
  char * queueLenStr = NULL;

  numDefs = rtx_getopt_get_num_strings (logger->parms.defs);
  if ((pf = rtx_param_open_preprocessed (logger->parms.configFileName, 
                                         logger->parms.verbose, log_param_err_func,
                                         numDefs, logger->parms.defs)) == NULL) {
    return (rtx_error ("log_read_config_parms: rtx_param_open(%s)"
                       " failed", logger->parms.configFileName));
  }
  if ((logFile = rtx_param_get_string (pf, "logFile", NULL, 0))
      == NULL) {
    if (validParamError) {
      rtx_error ("log_read_config_parms: rtx_param_get_string(logFileName) failed");
      errs++;
    }
  } else
    logger->parms.path = logFile;
  if ((fileExpireTimeStr = rtx_param_get_string (pf, "fileExpireTime", NULL, 0)) == NULL) {
    if (validParamError) {
      rtx_error ("log_read_config_parms: rtx_param_get_string(fileExpireTime) failed");
      errs++;
    }
  } else
    logger->parms.fileExpireTimeStr = fileExpireTimeStr;
  if ((fileSizeLimStr = rtx_param_get_string (pf, "fileSizeLim", NULL, 0)) == NULL) {
    if (validParamError) {
      rtx_error ("log_read_config_parms: rtx_param_get_string(fileSizeLim) failed");
      errs++;
    }
  } else
    logger->parms.fileSizeLimStr = fileSizeLimStr;
  if ((sizeBufsStr = rtx_param_get_string (pf, "sizeBufs", NULL, 0)) == NULL) {
    if (validParamError) {
      rtx_error ("log_read_config_parms: rtx_param_get_string(sizeBufs) failed");
      errs++;
    }
  } else
    logger->parms.sizeBufsStr = sizeBufsStr;
  if ((queueLenStr = rtx_param_get_string (pf, "queueLen", NULL, 0)) == NULL) {
    if (validParamError) {
      rtx_error ("log_read_config_parms: rtx_param_get_string(queueLen) failed");
      errs++;
    }
  } else
    logger->parms.queueLenStr = queueLenStr;
  if (rtx_param_get_int (pf, "numBufs", &(logger->parms.numBufs), 1)) {
    if (validParamError) {
      rtx_error ("log_read_config_parms: rtx_param_get_int(numBufs) failed");
      errs++;
    }
  }
  rtx_param_close (pf);
  if (errs)
    return (-1);
  return (0);
}
/** ********************************************************************************
 *
 */ 
static void
log_print_parms (
                 LOGGER * logger
                 )
{
  if (logger->parms.remote) {
  	rtx_message( "queue length = %d items", logger->parms.queueLen);
  	rtx_message( "remote host = %s, port %d", logger->parms.hostName, logger->parms.port);
  } else {
  	rtx_message ("logFileDir = %s", logger->parms.logFileDir);
  	rtx_message( "internal buffer size (sizeBufs)= %d bytes", logger->parms.sizeBufs);
  	rtx_message( "number of internal buffers (numBufs) = %d bytes", logger->parms.sizeBufs);
  	if (logger->parms.fileExpireTime == 0) {
  		rtx_message( "log file rollover turned off");
  	} else {
  		rtx_message( "log file rollover after %d seconds", logger->parms.fileExpireTime);
  		rtx_message( "file size limit for rollover = %d bytes", logger->parms.fileSizeLim);
  	}
  }
}
/** ********************************************************************************
 *
 */ 
void *
log_handle_usr_sigs (
                     void * arg
                     )
{
  LOGGER * l = (LOGGER *) arg;
  int retVal, signo = -1;
  sigset_t signalSet;

  if(sigemptyset(&signalSet)) {
    return (rtx_error_errno_null ("log_handle_usr_sigs: "
                                  "sigemptyset () failed: "));
  }
  if(sigaddset (&signalSet, SIGUSR1)) {
    return (rtx_error_errno_null ("log_handle_usr_sigs: "
                                  "sigaddset () failed: "));
  }
  if(sigaddset (&signalSet, SIGUSR2)) {
    return (rtx_error_errno_null ("log_handle_usr_sigs: "
                                  "sigaddset () failed: "));
  }
  do {
    retVal = sigwait (&signalSet, &signo);
    if (retVal == -1) {
      return (rtx_error_errno_null ("log_handle_usr_sigs: "
                                    "sigwait () failed"));
    }
    if (signo == SIGUSR1)
      log_start (l);
    if (signo == SIGUSR2)
      log_stop (l);
  } while (1);

  return (NULL);
}
/** ********************************************************************************
 *
 */ 
static void * 
log_duration_handler (
                      void * arg
                      )
{
  LOGGER * l = (LOGGER *) arg;

  rtx_timer_sleep (l->parms.duration);
  rtx_main_signal_shutdown ();
  return (NULL);
}

#ifdef __lynxos310

/* LynxOS doesn't seem to have this function!  This is a copy from the source */
static char *
dirname (char *path)
{
  static const char dot[] = ".";
  char *last_slash;

  /* Find last '/'.  */
  last_slash = path != NULL ? strrchr (path, '/') : NULL;

  if (last_slash == path)
    /* The last slash is the first character in the string.  We have to
       return "/".  */
    ++last_slash;
  else if (last_slash != NULL && last_slash[1] == '\0')
    /* The '/' is the last character, we have to look further.  */
    last_slash = memchr (path, last_slash - path, '/');

  if (last_slash != NULL)
    /* Terminate the path.  */
    last_slash[0] = '\0';
  else
    /* This assignment is ill-designed but the XPG specs require to
       return a string containing "." in any case no directory part is
       found and so a static and constant string is required.  */
    path = (char *) dot;

  return path;
}
#endif

static int 
set_tcp_timeout(int sockfd, int idle, int intvl, int cnt, int verbose) 
{
	//set appropriate TCP timeout options
	int optval;
	socklen_t optlen = sizeof(optval);	
	/* overrides tcp_keepalive_probes (default: 9)*/
	/* the number of unacknowledged probes to send before considering the connection dead
	 * and notifying the application layer
	 */
	optval = cnt;
	if (setsockopt (sockfd, SOL_TCP, TCP_KEEPCNT, &optval, optlen) < 0) {
		close (sockfd);
		rtx_error("setsockopt(TCP_KEEPCNT)");
		return -1;
	}
	/* overrides tcp_keepalive_time (default: 7200 sec)*/
	/* the interval between the last data packet sent (simple ACKs are not considered data) 
	 * and the first keepalive probe; after the connection is marked to need keepalive, 
	 * this counter is not used any further 
	 */
	optval = idle;
	if (setsockopt (sockfd, SOL_TCP, TCP_KEEPIDLE, &optval, optlen) < 0) {
		close (sockfd);
		rtx_error("setsockopt(TCP_KEEPIDLE)");
		return -1;
	}
	/* overrides tcp_keepalive_intvl (default: 75 sec)*/
	/* the interval between subsequential keepalive probes, regardless of what the connection 
	 * has exchanged in the meantime
	 */
	optval = intvl;
	if (setsockopt (sockfd, SOL_TCP, TCP_KEEPINTVL, &optval, optlen) < 0) {
		close (sockfd);
		rtx_error("setsockopt(TCP_KEEPINTVL)");
		return -1;
	}
	/* Check the status for the keepalive option */
	if(getsockopt(sockfd, SOL_SOCKET, SO_KEEPALIVE, &optval, &optlen) < 0) {
		close(sockfd);
		rtx_error("getsockopt(TCP_KEEPALIVE)");
		return -1;
	}
	if( verbose ) {
		rtx_message("TCP KEEPALIVE is %s! %d sec idletime, %d sec interval, %d probes\n", (optval ? "ON" : "OFF"), idle, intvl, cnt);
	}
	
	return 0;
}

/** ********************************************************************************
 *
 */ 
static void *
log_net_thread( LOGGER * logger )
{
	
	int wait_time = 1;
	int n = 0;
	const int INET_MAX_WAIT_TIME = 10;
	int bufLevel = 0;
	char sync_line[32] = "%%.:|sync-line|:. 00001 $$";
	unsigned int sync_n = 1;
	int varSent = 0;
	
	int nbytes = 0;
	char snd_buff[MAX_BUFF_LEN];
	
	//if send times out after SOCK_SND_TIMEOUT sleep for SLEEP_TIME
	//try again MAX_RETRY times at the most
	int tryAgain = 0;
	#define MAX_RETRY 4
	#define SOCK_SND_TIMEOUT 2
	#define SLEEP_TIME 1.5
	
	while(1) {
		rtx_thread_testcancel ();
		logger->tcpHandle = NULL;
		sync_n = 1;
		
		// Try connecting to Base
		bufLevel = (logger->Q->numInBuf * 100.0) / (logger->Q->n);
		rtx_message( "Trying to connect to %s:%d. Buffer level %d%%", 
					logger->parms.hostName, logger->parms.port,
					bufLevel );
		logger->tcpHandle = rtx_inet_init( RTX_INET_TCP_CLIENT, NULL, 0, 
										logger->parms.hostName, 
										logger->parms.port, 
										NULL, NULL, NULL );
		// If failed sleep and try again. Increase wait_time until INET_MAX_WAIT_TIME
		if( logger->tcpHandle == NULL ) {
			//rtx_error( "rtx_inet_init() failed." );	  
			bufLevel = (logger->Q->numInBuf * 100.0) / (logger->Q->n);
			
			rtx_message( "Couldn't connect to %s:%d. Will try again in %d s. Buffer level %d%%", 
						logger->parms.hostName, logger->parms.port,
						wait_time,
						bufLevel );
		
			rtx_timer_sleep( wait_time );
			if( (wait_time *= 2) > INET_MAX_WAIT_TIME ) {
				wait_time = INET_MAX_WAIT_TIME;
			}
		
			continue;
		}
		//set appropriate TCP timeout options
		set_tcp_timeout(logger->tcpHandle->sock->sockfd, _TCP_SND_IDLE, _TCP_SND_INTVL, _TCP_SND_CNT, logger->parms.verbose);
		//set socket send timeout
		struct timeval optval = {SOCK_SND_TIMEOUT, 0};
		socklen_t optlen = sizeof(optval);
		if (setsockopt (logger->tcpHandle->sock->sockfd, SOL_SOCKET, SO_SNDTIMEO, &optval, optlen) < 0) {
			close (logger->tcpHandle->sock->sockfd);
			rtx_error("setsockopt(SO_SNDTIMEO)");
		}
		// When connected Reset waiting time
		wait_time = 1;
		rtx_message( "Connected to %s:%d", 
					logger->parms.hostName, logger->parms.port );
		//allow TCP keepalive to kick in
		rtx_timer_sleep(2);
	
		// Send header once when connected
		if( logger->parms.verbose ) {
		rtx_message( "Sending file header... " );
		}
		if( rtx_inet_write( logger->tcpHandle->sock, (char *)logger->parms.header , 
							logger->parms.headerLen, NULL ) == -1 ) {
			switch (errno) {
				case ECONNRESET:
					rtx_message( "log_net_thread: Connection reset by peer."  );
					break;
				case ETIMEDOUT:
					rtx_message( "log_net_thread: TCP connection timeout."  );
					break;
				case EAGAIN:
					rtx_message( "log_net_thread: Send timeout."  );
					break;
				case EPIPE:
					rtx_message( "log_net_thread: Connection closed unexpectedly."  );
					break;
				default:
					rtx_error( "log_net_thread: Unexpected error."  );
					break;
			}
			rtx_message( "rtx_inet_write( FILE HEADER ) failed." );
			rtx_inet_done( logger->tcpHandle );
			continue;
		}
		if( logger->parms.verbose ) {
			rtx_message( "Header sent. %d bytes.", logger->parms.headerLen );
		}
		// Write synchronization line
		snprintf(sync_line, sizeof(sync_line), "%%%%.:|sync-line|:. %05u $$", sync_n);
		if( rtx_inet_write( logger->tcpHandle->sock, sync_line , 
							strlen(sync_line), NULL ) == -1 ) {
			switch (errno) {
				case ECONNRESET:
					rtx_message( "log_net_thread: Connection reset by peer." );
					break;
				case ETIMEDOUT:
					rtx_message( "log_net_thread: TCP connection timeout." );
					break;
				case EAGAIN:
					rtx_message( "log_net_thread: Send sync line: timeout." );
					break;
				case EPIPE:
					rtx_message( "log_net_thread: Connection closed unexpectedly." );
					break;
				default:
					rtx_error( "log_net_thread: Unexpected error." );
					break;
			}
			rtx_message( "rtx_inet_write( sync line ) failed." );
			rtx_inet_done( logger->tcpHandle );
			continue;
		}
		if (sync_n < 65535 ) {
			sync_n++;
		} else {
			sync_n = 1;
		}
		
		rtx_timer_sleep( 0.01 );
		
		// Send data in data queue
		while(1) {
			rtx_thread_testcancel ();
			
			// if data queue is emtpy: 
			//   Wait for data (use pthread condition for safe locking
			//   Wait for mutex
			pthread_mutex_lock( &net_mutex );
			if( logger->Q->numInBuf == 0 ) {
				if( logger->parms.verbose > 1) {
					rtx_message( "No data in Q. Waiting..." );
				}
				pthread_cond_wait( &net_data_cond, &net_mutex );
			}
			// Get the location of the next item in buffer
			NET_BUFFER_ITEM * item = (NET_BUFFER_ITEM *)rtx_circbuf_peek( logger->Q );
			bufLevel = (logger->Q->numInBuf * 100.0) / (logger->Q->n);
			pthread_mutex_unlock( &net_mutex );
			
			//assemble complete variable (var header + var data + sync-line) in one memory block
			memcpy(snd_buff, &(item->header), sizeof(RtxLogItemHeader));
			nbytes = sizeof(RtxLogItemHeader);
			memcpy(&snd_buff[nbytes], item->data, item->data_size);
			nbytes += item->data_size;
			snprintf(sync_line, sizeof(sync_line), "%%%%.:|sync-line|:. %05u $$", sync_n);
			memcpy(&snd_buff[nbytes], sync_line, strlen(sync_line));
			nbytes += strlen(sync_line);
			
			if (logger->parms.verbose) {
				rtx_message( "Got from Q: bufLevel %d%%: id=%d, size=%d. Sending ... ", 
							bufLevel, item->header.id, item->data_size );
			}
			varSent = 0;
			tryAgain = 0;
			// Write variable
			while (!varSent) {
				if( (n = rtx_inet_write( logger->tcpHandle->sock, snd_buff, nbytes, NULL )) == -1 ) {
					switch (errno) {
						case ECONNRESET:
							rtx_message( "log_net_thread: Connection reset by peer." );
							break;
						case ETIMEDOUT:
							rtx_message( "log_net_thread: TCP connection timeout." );
							break;
						case EAGAIN:
							rtx_message( "log_net_thread: Send timeout." );
							tryAgain++;
							break;
						case EPIPE:
							rtx_message( "log_net_thread: Connection closed unexpectedly." );
							break;
						default:
							rtx_error( "log_net_thread: Unexpected error." );
							break;
					}
					if (tryAgain && (tryAgain <= MAX_RETRY)) {
						rtx_message( "log_net_thread: Trying again after %f sec. %d of %d", SLEEP_TIME, tryAgain, MAX_RETRY );
						rtx_timer_sleep(SLEEP_TIME);
						continue;
					}
					rtx_message( "log_net_thread: rtx_inet_write() failed."  );
					break;
				}
				varSent = 1;
				tryAgain = 0;
			}
			if (!varSent) {
				//close and try connecting to base again
				rtx_inet_done( logger->tcpHandle );
				break;
			}
			if (n != nbytes) {
				rtx_message("%d bytes sent instead of %d", n, nbytes);
				//increase sync count -> receiver will discard that data
				sync_n++;
			} else {
				if (logger->parms.verbose > 1) {
					rtx_message("Var sent. %d bytes", nbytes);
				}
			}
			if (sync_n < 65535 ) {
				sync_n++;
			} else {
				sync_n = 1;
			}
			
			if( logger->parms.dots ) {
				printf( "." );
				fflush(stdout);
			}
			
			// Free memory from data
			if (item->data) {
				free( item->data );
				item->data = NULL;
			}
			
			// take item out of Q
			pthread_mutex_lock( &net_mutex );
			rtx_circbuf_next( logger->Q );
			pthread_mutex_unlock( &net_mutex );
		}
	}
	
	return NULL;
}

/** ********************************************************************************
 *
 */
static void *
server_thread( void * arg1, void *arg2 ) {
	int pfd[2];
	int n = 0;
	char read_buff[MAX_BUFF_LEN];
	int pos = 0;
	int cleanup_and_exit = 0;
	int resync = 0;
	int err = 0;
	unsigned int sync_n = 0;
	unsigned int sync_n_in = 0;
	
	#define VAR_UNCOMPLETE	1
	
	//sync_line = "%%.:|sync-line|:. 00001 $$";
	//the number is incremented at every sync	
	char sync_buff[32] = "";
	
	RtxInetTcpClient * clnt = (RtxInetTcpClient *) arg1;
	LOGGER * logger = (LOGGER *) arg2;
	
	RtxThread * tIdWrite = NULL;
	WLT_ARGS wlt_args;
	
	//set appropriate TCP timeout options
	set_tcp_timeout(clnt->sock->sockfd, _TCP_RECV_IDLE, _TCP_RECV_INTVL, _TCP_RECV_CNT, logger->parms.verbose);
		
	rtx_message( "server_thread: Got connection from %s", clnt->sock->remote.ifname );
	
	while(1) {
		pos = 0;
		
		while(VAR_UNCOMPLETE) {
			n = 0;
			// If there is data read it
			if( rtx_inet_wait_data( clnt->sock, 0.001 ) == 0 ) {
				n = rtx_inet_read( clnt->sock, &read_buff[pos], 1, NULL );
				if( n < 0 ) {
					switch (errno) {
						case ECONNRESET:
							rtx_message("server_thread: Connection reset by peer.");
							cleanup_and_exit = 1;
							break;
						case ETIMEDOUT:
							rtx_message("server_thread: Connection timed out.");
							cleanup_and_exit = 1;
							break;
						case EINTR:
							if ( err <= 3 ) {
								rtx_message("server_thread: recv() was interrupted by a signal. Try again.");
								err++;
							} else {
								rtx_message("server_thread: recv() was interrupted by a signal. Close socket.");
								cleanup_and_exit = 1;
							}
							break;
						case EIO:
							if ( err <= 3 ) {
								rtx_message("server_thread: recv() I/O error occured. Try again.");
								err++;
							} else {
								rtx_message("server_thread: recv() I/O error occured. Close socket.");
								cleanup_and_exit = 1;
							}
							break;
						default:
							rtx_error_flush( "server_thread: rtx_inet_read failed" );
							cleanup_and_exit = 1;
					}
					if (cleanup_and_exit) {
						break;
					} else {
						rtx_error_flush( "server_thread: Should not happen. Reading again." );
						continue;
					}
				}
				// connection closed
				else if( n == 0 ) {
					rtx_message( "server_thread: Connection closed" );
					cleanup_and_exit = 1;
					break;
				}
				//byte received
				if ((read_buff[pos] == '$') && (pos >= SYNC_LEN ) && (read_buff[pos-1] == '$')) {
					//second $ in a row
					strncpy(sync_buff, &read_buff[pos - SYNC_LEN + 1], SYNC_LEN);
					if (sscanf(sync_buff, "%%%%.:|sync-line|:. %u $$", &sync_n_in) == 1) {
						//found synchronization line
						if ((sync_n == 0) && (sync_n_in != 1)) {
							//header not completely received -> close connection
							rtx_message("server_thread: Only partial log header received. Close connection.");
							cleanup_and_exit = 1;
							break;
						}
						//increase internal sync counter
						if (sync_n < 65535 ) {
							sync_n++;
						} else {
							sync_n = 1;
						}
						//check for correct sync numbers
						if (sync_n_in != sync_n) {
							//sync numbers don't match -> discard last data
							if (logger->parms.verbose) {
								rtx_message("sync failed, skipping message...");
							}
						} else if (resync) {
							//re-synchronized after an overlong packet
							if (logger->parms.verbose) {
								rtx_message("re-synchronized...");
							}
						} else {
							//synchronized and var ok
							break;
						}
						//synchronized again --> read next message
						sync_n = sync_n_in;
						pos = 0;
						resync = 0;
						continue;
					}
				}
				//end of message not reached, keep on reading till syncline is found
				pos++;
				if (pos > (MAX_BUFF_LEN - 1)) {
					rtx_message("message to long, discarted...");
					pos = 0;
					resync = 1;
				}
			}
		}// end while(VAR_UNCOMPLETE)
		
		if (cleanup_and_exit) {
			break;
		}
		
		//if first data start write_log_thread
		if (tIdWrite == NULL) {
			if (pipe(pfd) == -1) {
				rtx_message( "server_thread: Failed to create pipe!");
				return NULL;
			}
			
			wlt_args.clnt = clnt;
			wlt_args.logger = logger;
			wlt_args.pfd = pfd[0];
			
			//start write_log_thread
			if( (tIdWrite = rtx_thread_create( "write_log_thread",
						logger->parms.verbose,
						RTX_THREAD_SCHED_OTHER,
						logger->parms.basePrio,
						16384,
						RTX_THREAD_CANCEL_DEFERRED,
						(void *)write_log_thread,
						&wlt_args,
						NULL, NULL)) == NULL ) {
				rtx_error( "Unable to create write_log_thread" );
				break;
			}
		}
		
		//send data to pipe (without the sync line at the end)
		if ( write(pfd[1], read_buff, (pos + 1 - SYNC_LEN)) != (pos + 1 - SYNC_LEN) ) {
			rtx_message( "server_thread: Failed to write to pipe!");
			break;
		}
		
		if( logger->parms.dots ) {
			printf( "." );
			fflush(stdout);
		}
	}//end while(1)

	rtx_message( "server_thread: Lost connection to %s", clnt->sock->remote.ifname );
	
	//cleanup
	//close write end of pipe
	close(pfd[1]);
	if (tIdWrite) {
		rtx_thread_detach( tIdWrite );
	} else {
		//close read end of pipe
		close(pfd[0]);
	}
	
	return NULL;
}

/** ********************************************************************************
 *
 */
static void *
write_log_thread( WLT_ARGS * args ) {
	char appName[256]; 
	RtxLogread     *logread;
	RtxLogreadVar   *rv;
	
	RtxLog * dataLogger;
	RtxLogStats logStats;
	RtxLogWriteBufs datalogWriteBufs;
	
	datalogWriteBufs.buf[0] = NULL;
	datalogWriteBufs.bufLen[0] = 0;
	datalogWriteBufs.buf[1] = NULL;
	datalogWriteBufs.bufLen[1] = 0;
	datalogWriteBufs.buf[2] = NULL;
	datalogWriteBufs.bufLen[2] = 0;
	
	unsigned char header[8192];
	int        headerLen = 0;
	int n, endianInt = RTX_LOG_HEADER_MAGIC_NUMBER;
	
	//get args
	LOGGER * logger = (LOGGER *) args->logger;
	RtxInetTcpClient * clnt = (RtxInetTcpClient *) args->clnt;
	int pfd = args->pfd;

	//read the header and parse the var definitions
	if ((logread = rtx_logread_init_fd(pfd, 0)) == NULL) {
		rtx_error("rtx_logread_init_fd() failed for '%s'\n",clnt->sock->remote.ifname);
		return NULL;
	}
	
	//rtx_logread reads the header but discards the EOH and endianInt, put it back
	memcpy (header, logread->header, logread->headerLen);
	strcat ((char*) header, RTX_LOG_HEADER_EOH);
	n = strlen ((char*) header);
	memcpy (&header[n], &endianInt, sizeof (int));
	headerLen = n + sizeof (int);
	
	char decl[4096]=""; /* Any struct should be smaller than that */
	for (rv = logread->vars; rv != NULL; rv = rv->next) {
		rtx_parse_generate_var_decl(rv->v,decl);
		if (logger->parms.verbose)
			printf("Var %s ->%p\n", rv->v->name, decl);
		else
			printf("%s ",rv->v->name);
	}
	printf("\n");
	snprintf( appName, sizeof(appName), "ddxlog: data from %s", clnt->sock->remote.ifname );
	if (logger->parms.verbose) {
		rtx_message("server_thread: initializing the file writing module");
	}
	if ((dataLogger = rtx_log_open (
							logger->parms.logFileDir,
							logger->parms.logFileName,
							logger->parms.fileExpireTime / 60,
							logger->parms.fileSizeLim / 1024,
							logger->parms.numBufs,
							logger->parms.sizeBufs / 1024,
							logger->parms.basePrio, 
							logger->parms.gzip, 
							0, 
							header,
							headerLen,
							appName, 
							1, 
							logger->parms.indexFileName)) == NULL) {
		rtx_error_flush ("main: rtx_log_open() failed");
		rtx_logread_close(logread);
		return NULL;
	}

	
	//read the variable data 
	while ((rv = rtx_logread_next(logread)) != NULL) {
		datalogWriteBufs.buf[0] = (unsigned char *) &(rv->vh);
		datalogWriteBufs.bufLen[0] = sizeof(rv->vh);
		datalogWriteBufs.buf[1] = rv->d;
		datalogWriteBufs.bufLen[1] = rv->v->size;
		datalogWriteBufs.buf[2] = NULL;
		datalogWriteBufs.bufLen[2] = 0;
		if (rtx_log_write_bufs (dataLogger, &datalogWriteBufs) == -1) {
				rtx_error_flush ("datalog_write_bufs() failed");
				break;
		}
	}
  
	//also closes read end of pipe:
	rtx_logread_close(logread);

	rtx_log_close(dataLogger, &logStats);
	
	if( logger->parms.verbose ) { 
		rtx_message( "bytes written to file [%d]", logStats.bytesToFile);
		rtx_message( "files written [%d]", logStats.filesCreated);
		if (logStats.writeErrors)
			rtx_message( "%d write errors", logStats.writeErrors);
		if (logStats.openErrors)
			rtx_message( "%d open errors", logStats.openErrors);
		if (logStats.closeErrors)
			rtx_message( "%d close errors", logStats.closeErrors);
	}
	
	return NULL;
}

int save_unsent_buffer( LOGGER * logger ) 
{

  if( logger->Q->numInBuf <= 0 ) {
	return -1;
  }
    
  /**
   * At this point the Net thread is also stopped. If any
   *   data left in buffer this is written to file.
   */
  char sysCmd[128] = "";
  char net_log_file_name[128] = "";
  char time_buff[128] = "";
  RtxTime ts;
  // Make the relevant dirs
  snprintf( sysCmd, sizeof(sysCmd), "mkdir %s/unsent", logger->parms.logFileDir );	      
  printf( "%s\n", sysCmd );
  system( sysCmd );
  snprintf( sysCmd, sizeof(sysCmd), "mkdir %s/unsent/%s", logger->parms.logFileDir, logger->parms.hostName );
  printf( "%s\n", sysCmd );
  system( sysCmd );
  // create log file name from curr time
  rtx_time_get(&ts);
  rtx_time_conv_date_time( time_buff, 127, "", "", "",&ts);
  snprintf( net_log_file_name, sizeof(net_log_file_name), "%s/unsent/%s/%s.log", logger->parms.logFileDir, logger->parms.hostName, time_buff );
  rtx_message( "file name: \"%s\"", net_log_file_name );
  FILE * net_log_fp = fopen( net_log_file_name, "w" );
  if( net_log_fp == NULL) {
	rtx_error( "main: opening %s file failed", net_log_file_name );
  } 
  else {
	// Write header
	fwrite( logger->parms.header, sizeof(char), logger->parms.headerLen, net_log_fp );
	// White data in buffer
	while( logger->Q->numInBuf > 0 ) {

      // Get a copy of the next item in buffer
      pthread_mutex_lock( &net_mutex );
      NET_BUFFER_ITEM * item = (NET_BUFFER_ITEM *)rtx_circbuf_peek( logger->Q );	      
      rtx_circbuf_next( logger->Q );
      pthread_cond_broadcast( &net_data_cond );
      pthread_mutex_unlock( &net_mutex );

      // Write header
      fwrite( (char *)&item->header, sizeof(char), sizeof(RtxLogItemHeader), net_log_fp );
      // Write data
      fwrite( (char *)item->data, sizeof(char), item->data_size, net_log_fp );
      // Free memory from data
      free( item->data );
      item->data = NULL;
	}

	fprintf( net_log_fp,"\n");
	fclose( net_log_fp );
  }

  return 0;
}
