/*********************************************************************
 *
 * CSIRO Automation
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * www.cat.csiro.au/cmst
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 *********************************************************************/

static char *rcsid = "$Id: ddxiio-main.c 1370 2007-02-05 00:18:17Z pra077 $";

/**
 * \file ddxiio-main.c
 * \brief DDX IIO/store client
 * \author Pavan Sikka
 */

/**
 * \page ddxiio ddxiio 
 *
 * The DDX IIO/store client can be used to interface between IIO channels and
 * the store. It can read an IIO channel and write it to the store at the specified
 * rate. It can also read data from the store and write it to an IIO channel
 * at the specified rate. The behavior is controlled by command-line arguments
 * and/or a configuration file.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>
#include <math.h>
#include <sys/stat.h>
#ifdef __linux__
#include <sys/file.h>
#include <libgen.h>
#else
#include <sys/lock.h>
#endif
#include <errno.h>
#include <signal.h>
#ifdef __sparc__
#include <libgen.h>
#endif

#include <rtx/message.h>
#include <rtx/error.h>
#include <rtx/thread.h>
#include <rtx/main.h>
#include <rtx/param.h>
#include <rtx/mutex.h>
#include <rtx/log.h>
#include <rtx/parse.h>
#include <rtx/signal.h>
#include <rtx/time.h>
#include <rtx/timer.h>
#include <rtx/getopt.h>
#include <rtx/list.h>

#include <iio.h>

#include <ddx.h>

#define KILLER_PRIO      90

#define DDXIIO_ITEMTYPE_INPUT     1
#define DDXIIO_ITEMTYPE_OUTPUT    2
#define DDXIIO_ITEMTYPE_DELAY     3

typedef struct _ddxiio_item {
    char * iioName;
    char * storeName;
    char * storeType;
    int type;
    double delay;
    DDX_STORE_ITEM * storeItem;
    IIO iioChannel;
    void * buf;
} DDXIIO_ITEM;

typedef struct _ddxiio_group_item {
    double delay;
    RtxList * varList;
    RtxThread * th;
    RtxTimer * tmr;
    RtxTimerThread * timerTh;
    struct _ddxiio * p;
} DDXIIO_GROUP_ITEM;

typedef struct _ddxiio {
    RtxList * groupList;
    char * configFile;
    double delay;
    int verbose;
    DDX_STORE_ID * store;
} DDXIIO;

DDXIIO ddxIIO;

/* forward definitions */

int ddxiio_init (DDXIIO * p);
int ddxiio_done (DDXIIO * p);
int ddxiio_read_config_file (DDXIIO * p);
void ddxiio_print_config (DDXIIO * p);
void * ddxiio_var_thread (void * arg);
void ddxiio_var_thread_cleanup (void * arg);
void * ddxiio_var_thread_async (void * arg);

/**
 * main program
 */
int
main (int argc, char * argv[])
{
    int errs = 0, ret = -1;

    rtx_getopt_disable_config ();
    /* no command-line args, print help and exit */    
    if (argc == 1) {
        RTX_GETOPT_PRINT (NULL, argv[0], rcsid, NULL);
	exit (1);
    }

    if ((ret = RTX_GETOPT_CMD (NULL, argc, argv, rcsid, NULL)) == -1) {
        RTX_GETOPT_PRINT (NULL, argv[0], rcsid, NULL);
	exit (1);
    }
    if (ret == 0)
        exit (1);
    ddxIIO.delay = rtx_getopt_get_sample (0.1);
    ddxIIO.verbose = rtx_getopt_get_verbose (0);
    ddxIIO.configFile = rtx_getopt_get_str ("config");
    if (ddxIIO.configFile == NULL) {
        fprintf (stderr, "ddxiio: must specify configuration file "
		 "with -config option\n");
	exit (1);
    }

    rtx_main_init ("ddxiio", RTX_ERROR_STDERR | RTX_ERROR_MESSAGE);

    if (ddxiio_init (&ddxIIO) == -1) {
        rtx_error_flush ("main: ddxiio_init(%s) failed", ddxIIO.configFile);
	errs++;
    }

    if (errs == 0)
        ddxiio_print_config (&ddxIIO);

    if (errs == 0) {
#ifdef __Lynx__
	if (rtx_main_wait_shutdown (KILLER_PRIO)) {
	    rtx_error ("main: rtx_init_wait_shutdown() failed");
	    errs++;
	}
#else
	if (rtx_main_wait_shutdown (0)) {
	    rtx_error ("main: rtx_init_wait_shutdown() failed");
	    errs++;
	}
#endif
    }

    rtx_message ("shutting down...");

    if (ddxiio_done (&ddxIIO) == -1) {
        rtx_error_flush ("main: ddxiio_done() failed");
	errs++;
    }

    return (0);
}

int
ddxiio_init (
	     DDXIIO * p
	     )
{
    DDXIIO_GROUP_ITEM * groupItemP = NULL;
    DDXIIO_ITEM * itemP = NULL;
    char ddxiioThName[BUFSIZ];
    int groupNum = 0;

    if ((p->groupList = rtx_list_init ()) == NULL)
        return (rtx_error ("ddxiio_init: rtx_list_init() failed"));
    if (ddxiio_read_config_file (p) == -1)
        return (rtx_error ("ddxiio_init: ddxiio_read_config_file() failed"));
    /* Now open all the IIO channels */
    if (iio_init (iio_standard, iio_iflag_none))
        return (rtx_error ("ddxiio_init: iio_init() failed: %s",
			   iio_emessage_get ()));
    while ((groupItemP = rtx_list_iterate (p->groupList)) != NULL)
	while ((itemP = rtx_list_iterate (groupItemP->varList)) != NULL)
	    if (iio_open (itemP->iioName, iio_oflag_none, &(itemP->iioChannel)))
	        return (rtx_error ("ddxiio_init: iio_open(%s) failed: %s",
				   itemP->iioName, iio_emessage_get ()));
    if (ddx_client_init ((p->verbose>0)?(p->verbose - 1):0) == -1)
        return (rtx_error ("ddxiio_init: ddx_client_init() failed"));
    if ((p->store = ddx_store_open (NULL, 0, 5)) == NULL)
        return (rtx_error ("ddxiio_init: ddx_store_open() failed"));
    while ((groupItemP = rtx_list_iterate (p->groupList)) != NULL) {
        while ((itemP = rtx_list_iterate (groupItemP->varList)) != NULL) {
	    if ((itemP->storeItem = ddx_store_lookup_item (p->store, itemP->storeName,
							   itemP->storeType, 0)) == NULL)
	        return (rtx_error ("ddxiio_init: ddx_store_lookup_item(%s) failed",
				   itemP->storeName));
	    if ((itemP->buf = (void *) calloc (1, itemP->storeItem->varSize)) == NULL)
	        return (rtx_error ("ddxiio_init: calloc(%d) failed",
				   itemP->storeItem->varSize));
	}
    }
    /* now create all the threads */
    while ((groupItemP = rtx_list_iterate (p->groupList)) != NULL) {
        sprintf (ddxiioThName, "ddxiio[%d]", ++groupNum);
        if (fabs (groupItemP->delay) < 0.0001) {
	    if ((groupItemP->th = rtx_thread_create (ddxiioThName,
					     groupItemP->p->verbose,
					     RTX_THREAD_SCHED_OTHER,
					     RTX_THREAD_PRIO_MIN,
					     16384,
					     RTX_THREAD_CANCEL_DEFERRED,
					     ddxiio_var_thread_async, 
					     groupItemP,
					     NULL,
					     NULL)) == NULL)
	        return (rtx_error ("ddxiio_init: rtx_thread_create() failed"));
	} else {
	    if (p->verbose)
	        rtx_message ("ddxiio_init: creating timer thread");
	    if ((groupItemP->th = rtx_thread_create (ddxiioThName,
					     groupItemP->p->verbose,
					     RTX_THREAD_SCHED_OTHER,
					     RTX_THREAD_PRIO_MIN,
					     16384,
					     RTX_THREAD_CANCEL_DEFERRED,
					     ddxiio_var_thread, 
					     groupItemP,
					     ddxiio_var_thread_cleanup,
					     groupItemP)) == NULL)
	        return (rtx_error ("ddxiio_init: rtx_thread_create() failed"));
	}
    }
    return (0);
}

void
ddxiio_var_thread_cleanup (
			   void * arg
			   )
{
    DDXIIO_GROUP_ITEM * groupItemP = (DDXIIO_GROUP_ITEM *) arg;

    rtx_timer_stop (groupItemP->tmr);
    rtx_timer_destroy (groupItemP->tmr);
}

/**
 * do the IIO/store thing
 */
void *
ddxiio_var_thread (
		   void * arg
		   )
{
    DDXIIO_GROUP_ITEM * groupItemP = (DDXIIO_GROUP_ITEM *) arg;
    DDXIIO_ITEM * itemP = NULL;
    RtxTime ts;
    int errs = 0;
    unsigned int loopCount = 0;

    if ((groupItemP->tmr = rtx_timer_create (groupItemP->delay, groupItemP->delay,
					     NULL, NULL, 
					     RTX_TIMER_CLOCK_REALTIME)) == NULL) {
        rtx_error_flush ("rtx_timer_create failed");
	return (NULL);
    }
    if (rtx_timer_start (groupItemP->tmr) == -1) {
	rtx_error_flush ("rtx_timer_start failed");
	return (NULL);
    }
    if (groupItemP->p->verbose)
        rtx_message ("ddxiio_var_thread: starting");
    while (1) {
        rtx_thread_testcancel ();
        if (rtx_timer_wait (groupItemP->tmr) == -1) {
	    rtx_error_flush ("rtx_timer_wait failed");
	    rtx_timer_sleep (0.1);
	}
        rtx_thread_testcancel ();
	if (groupItemP->p->verbose > 1)
	    rtx_message ("ddxiio_var_thread: starting loop");
	rtx_thread_setcancel (RTX_THREAD_CANCEL_DISABLE);
	loopCount++;
    	while ((itemP = (DDXIIO_ITEM *) rtx_list_iterate (groupItemP->varList)) != NULL) {
    	    if (groupItemP->p->verbose > 1)
 		rtx_message ("ddxiio_var_thread: doing %s", itemP->storeName);
    	    if (itemP->type == DDXIIO_ITEMTYPE_INPUT) {
 		/* read from IIO, write to store */	  
	        if (groupItemP->p->verbose > 1)
		    rtx_message ("ddxiio_var_thread: reading %s from IIO", itemP->storeName);
 		if (strcmp (itemP->storeType, "int") == 0) {
 		    if (iio_operate (itemP->iioChannel, iio_op_read, itemP->buf)) {
 			rtx_error ("ddxiio_var_thread: iio_operate(%s) failed", itemP->storeName);
 			errs++;
 		    }
 		} else if (strcmp (itemP->storeType, "double") == 0) {
 		    if (iio_operate_real (itemP->iioChannel, iio_op_read, (double *) itemP->buf)) {
 			rtx_error ("ddxiio_var_thread: iio_operate(%s) failed", itemP->storeName);
 			errs++;
 		    }
 		}
		if (groupItemP->p->verbose > 1)
		    rtx_message ("ddxiio_var_thread: %s read from IIO", itemP->storeName);
 		if (ddx_store_write (itemP->storeItem, itemP->buf, NULL) == -1) {
 		    rtx_error ("ddxiio_var_thread: ddx_store_write (%s) failed",
 			       itemP->storeName);
 		    errs++;
 		}
		if (groupItemP->p->verbose > 1)
		    rtx_message ("ddxiio_var_thread: %s written to store", itemP->storeName);
 	    } else if (itemP->type == DDXIIO_ITEMTYPE_OUTPUT) {
 		/* read from store, write to IIO */
 		if (ddx_store_async_read (itemP->storeItem, itemP->buf, &ts, 1.0) == -1) {
 		    rtx_error ("ddxiio_var_thread: ddx_store_read (%s) failed",
 			       itemP->storeName);
 		    errs++;
 		}
 		if (strcmp (itemP->storeType, "int") == 0) {
 		    if (iio_operate (itemP->iioChannel, iio_op_write, itemP->buf)) {
 			rtx_error ("ddxiio_var_thread: iio_operate(%s) failed", itemP->storeName);
 			errs++;
 		    }
 		} else if (strcmp (itemP->storeType, "double") == 0) {
 		    if (iio_operate_real (itemP->iioChannel, iio_op_write, (double *) itemP->buf)) {
 			rtx_error ("ddxiio_var_thread: iio_operate(%s) failed", itemP->storeName);
 			errs++;
 		    }
 		}
 	    }
    	}
	rtx_thread_setcancel (RTX_THREAD_CANCEL_DEFERRED);
	if (errs) {
	    rtx_error_flush ("ddxiio_var_thread: errors, loop count = %d", 
			     loopCount);
	    errs = 0;
	}
    }
}

void * 
ddxiio_var_thread_async (
		   void * arg
		   )
{
    DDXIIO_GROUP_ITEM * groupItemP = (DDXIIO_GROUP_ITEM *) arg;
    DDXIIO_ITEM * itemP = NULL;
    RtxTime ts;
    int errs = 0;

    if (groupItemP->p->verbose)
        rtx_message ("ddxiio_var_thread_async: starting");

    while (1) {
	if (groupItemP->p->verbose > 1)
	    rtx_message ("ddxiio_var_thread_async: starting loop");
    	while ((itemP = (DDXIIO_ITEM *) rtx_list_iterate (groupItemP->varList)) != NULL) {
    	    if (groupItemP->p->verbose > 1)
 		rtx_message ("ddxiio_var_thread_async: doing %s", itemP->storeName);
 	    if (itemP->type == DDXIIO_ITEMTYPE_OUTPUT) {
 		/* read from store, write to IIO */
 		if (ddx_store_sync_read (itemP->storeItem, itemP->buf, &ts, -1.0) == -1) {
 		    rtx_error ("ddxiio_var_thread_async: ddx_store_read (%s) failed",
 			       itemP->storeName);
 		    errs++;
 		}
 		if (strcmp (itemP->storeType, "int") == 0) {
 		    if (iio_operate (itemP->iioChannel, iio_op_write, itemP->buf)) {
 			rtx_error ("ddxiio_var_thread_async: iio_operate(%s) failed", itemP->storeName);
 			errs++;
 		    }
 		} else if (strcmp (itemP->storeType, "double") == 0) {
 		    if (iio_operate_real (itemP->iioChannel, iio_op_write, (double *) itemP->buf)) {
 			rtx_error ("ddxiio_var_thread_async: iio_operate(%s) failed", itemP->storeName);
 			errs++;
 		    }
 		}
 	    }
    	}   
    }

    return (NULL);
}


#define LSTATE_START      1
#define LSTATE_INOUT      2
#define LSTATE_DELAY      3
#define LSTATE_DELAY_N    4
#define LSTATE_TYPE       5
#define LSTATE_NAME       6
#define LSTATE_DONE       7

DDXIIO_ITEM *
ddxiio_process_line (
		     DDXIIO * p,
		     char * line
		     )
{
    char * tok = NULL;
    int lstate = LSTATE_START, done = 0;
    DDXIIO_ITEM item, * itemP = NULL;

    memset (&item, 0, sizeof (item));
    item.delay = p->delay;
    if (p->verbose)
        rtx_message ("processing line: %s", line);
    tok = strtok (line, " \r\n\t=;");
    while (! done) {
        if (p->verbose)
	    rtx_message ("processing token (%d): %s", lstate, tok);
        switch (lstate) {
	    case LSTATE_START :
	        if (tok == NULL) {
		    done = 1;
		    continue;
		}
 	        if (strcmp (tok, "sample-time") == 0) {
		    lstate = LSTATE_DELAY;
		    item.type = DDXIIO_ITEMTYPE_DELAY;
		} else if (strcmp (tok, "input") == 0) {
		    lstate = LSTATE_INOUT;
		    item.type = DDXIIO_ITEMTYPE_INPUT;
		} else if (strcmp (tok, "output") == 0) {
		    lstate = LSTATE_INOUT;
		    item.type = DDXIIO_ITEMTYPE_OUTPUT;
		} else {
		    rtx_error_flush ("ddxiio_process_line(START): "
				     "unknown keyword <%s>:%s", tok, line);
		    return (NULL);
		}
	        break;
	    case LSTATE_INOUT :
	        if (tok == NULL) {
		    rtx_error_flush ("ddxiio_process_line(INOUT): "
				     "incomplete line:%s", line);
		    return (NULL);
		}
 	        if (strcmp (tok, "sample-time") == 0)
		    lstate = LSTATE_DELAY;
		else if (strcmp (tok, "char") == 0)
		    lstate = LSTATE_TYPE;
		else if (strcmp (tok, "short") == 0)
		    lstate = LSTATE_TYPE;
		else if (strcmp (tok, "int") == 0)
		    lstate = LSTATE_TYPE;
		else if (strcmp (tok, "long") == 0)
		    lstate = LSTATE_TYPE;
		else if (strcmp (tok, "float") == 0)
		    lstate = LSTATE_TYPE;
		else if (strcmp (tok, "double") == 0)
		    lstate = LSTATE_TYPE;
		else {
		    rtx_error_flush ("ddxiio_process_line(INOUT): "
				     "unknown keyword <%s>:%s", tok, line);
		    return (NULL);
		}
		item.storeType = tok;
	        break;
	    case LSTATE_DELAY :
	        if (tok == NULL) {
		    rtx_error_flush ("ddxiio_process_line(DELAY): "
				     "incomplete line:%s", line);
		    return (NULL);
		}
	        item.delay = atof (tok);
		lstate = LSTATE_DELAY_N;
	        break;
	    case LSTATE_DELAY_N :
	        if (tok == NULL) {
		    done = 1;
		    continue;
		}
		if (strcmp (tok, "char") == 0)
		    lstate = LSTATE_TYPE;
		else if (strcmp (tok, "short") == 0)
		    lstate = LSTATE_TYPE;
		else if (strcmp (tok, "int") == 0)
		    lstate = LSTATE_TYPE;
		else if (strcmp (tok, "long") == 0)
		    lstate = LSTATE_TYPE;
		else if (strcmp (tok, "float") == 0)
		    lstate = LSTATE_TYPE;
		else if (strcmp (tok, "double") == 0)
		    lstate = LSTATE_TYPE;
		else {
		    rtx_error_flush ("ddxiio_process_line(DELAY_N): "
				     "unknown keyword <%s>:%s", tok, line);
		    return (NULL);
		}
		item.storeType = tok;
	        break;
	    case LSTATE_TYPE :
	        if (tok == NULL) {
		    rtx_error_flush ("ddxiio_process_line(TYPE): "
				     "incomplete line:%s", line);
		    return (NULL);
		}
		item.storeName = tok;
	        lstate = LSTATE_NAME;
	        break;
	    case LSTATE_NAME :
	        if (tok == NULL) {
		    rtx_error_flush ("ddxiio_process_line(NAME): "
				     "incomplete line:%s", line);
		    return (NULL);
		}
		item.iioName = tok;
	        lstate = LSTATE_DONE;
	        break;
	    case LSTATE_DONE :
	        if (tok == NULL) {
		    done = 1;
		    continue;
		}
		rtx_error_flush ("ddxiio_process_line(DONE): "
				 "extra token <%s> on line:%s", tok, line);
		return (NULL);
	        break;
	    default :
	        rtx_error_flush ("ddxiio_process_line(%d): "
				 "unknown state - BAD; tok <%s>, line:%s",
				 lstate, tok, line);
	        return (NULL);
	        break;
	}
	tok = strtok (NULL, " \r\n\t=;");
    }
    if (p->verbose)
        rtx_message ("line processed, item type = %d", item.type);
    if (item.type == 0)
        return (NULL);
    if ((itemP = (DDXIIO_ITEM *) calloc (1, sizeof (DDXIIO_ITEM))) == NULL) {
        rtx_error_flush ("ddxiio_process_line: calloc() failed:%s", line);
        return (NULL);
    }
    itemP->delay = item.delay;
    itemP->type = item.type;
    if (itemP->type != DDXIIO_ITEMTYPE_DELAY) {
	if ((itemP->storeName = strdup (item.storeName)) == NULL) {
	    rtx_error_flush ("ddxiio_process_line: calloc() failed:%s", line);
	    return (NULL);
	}
	if ((itemP->storeType = strdup (item.storeType)) == NULL) {
	    rtx_error_flush ("ddxiio_process_line: calloc() failed:%s", line);
	    return (NULL);
	}
	if ((itemP->iioName = strdup (item.iioName)) == NULL) {
	    rtx_error_flush ("ddxiio_process_line: calloc() failed:%s", line);
	    return (NULL);
	}
    }
    return (itemP);
}

int
ddxiio_read_config_file (
			 DDXIIO * p
			 )
{
    FILE * fp = NULL;
    char line[BUFSIZ];
    DDXIIO_ITEM * itemP = NULL;
    DDXIIO_GROUP_ITEM * groupItemP = NULL;
    int numDefs = 0, i, found = 0;
    char b[BUFSIZ];

    /*
    numDefs = rtx_getopt_get_num_strings (logger->parms.defs);
    */
#ifdef sparc_solaris
    sprintf (b, "/usr/local/bin/cpp -P ");
#else
    sprintf (b, "/usr/bin/cpp -P ");
#endif
    for (i=0; i<numDefs; i++) {
        strcat (b, "-D");
	/*
	strcat (b, logger->parms.defs[i]);
	*/
	strcat (b, " ");
    }
    strcat (b, p->configFile);
    if ((fp = fopen (p->configFile, "r")) == NULL)
        return (rtx_error_errno ("ddxiio_read_config_file<%s>: no such file",
				 p->configFile));
    fclose (fp);
    if ((fp = popen (b, "r")) == NULL)
        return (rtx_error_errno ("ddxiio_read_config_file: popen(%s)"
				 " failed", p->configFile));
    /* process the config file */
    while (fgets (line, BUFSIZ, fp) != NULL) {
        if (line[0] == '%')
	    continue;
        if ((itemP = ddxiio_process_line (p, line)) == NULL)
	    continue;
	if (p->verbose)
	    rtx_message ("got valid item, type = %d, delay = %f", 
			 itemP->type, itemP->delay);
	if (itemP->type == DDXIIO_ITEMTYPE_DELAY) {
	    p->delay = itemP->delay;
	    free (itemP);
	    continue;
	}
	/* we have a valid node, put it in the list */
	/* nodes with delays 0f 0.0 go into a separate group */
	if (fabs (itemP->delay) < 0.0001) {
	    if ((groupItemP = (DDXIIO_GROUP_ITEM *) 
		 calloc (1, sizeof (DDXIIO_GROUP_ITEM))) == NULL)
	        return (rtx_error ("ddxiio_read_config_file: calloc() failed"));
	    groupItemP->delay = 0.0;
	    groupItemP->p = p;
	    if ((groupItemP->varList = rtx_list_init ()) == NULL)
	        return (rtx_error ("ddxiio_read_config_file: rtx_list_init() failed"));
	    if (rtx_list_enqueue (groupItemP->varList, itemP) == -1)
	        return (rtx_error ("ddxiio_read_config_file: rtx_list_enqueue() failed"));
	    if (rtx_list_enqueue (p->groupList, groupItemP) == -1)
	        return (rtx_error ("ddxiio_read_config_file: rtx_list_enqueue() failed"));
	    continue;
	}
	found = 0;
	while ((groupItemP = rtx_list_iterate (p->groupList)) != NULL) {
	    if (fabs (groupItemP->delay - itemP->delay) < 0.0001) {
	        if (rtx_list_enqueue (groupItemP->varList, itemP) == -1)
	            return (rtx_error ("ddxiio_read_config_file: "
				       "rtx_list_enqueue() failed"));
		if (p->verbose)
		    rtx_message ("found existing group");
		found = 1;
	    }
	}
	if (found)
	    continue;
	if (p->verbose)
	    rtx_message ("adding new group, delay = %f", itemP->delay);
	if ((groupItemP = (DDXIIO_GROUP_ITEM *) 
	     calloc (1, sizeof (DDXIIO_GROUP_ITEM))) == NULL)
	    return (rtx_error ("ddxiio_read_config_file: calloc() failed"));
	groupItemP->delay = itemP->delay;
	groupItemP->p = p;
	if ((groupItemP->varList = rtx_list_init ()) == NULL)
	    return (rtx_error ("ddxiio_read_config_file: rtx_list_init() failed"));
	if (rtx_list_enqueue (groupItemP->varList, itemP) == -1)
	    return (rtx_error ("ddxiio_read_config_file: rtx_list_enqueue() failed"));
	if (rtx_list_enqueue (p->groupList, groupItemP) == -1)
	    return (rtx_error ("ddxiio_read_config_file: rtx_list_enqueue() failed"));
    }
    pclose (fp);
    return (0);
}

void
ddxiio_print_config (
		     DDXIIO * p
		     )
{
    DDXIIO_ITEM * itemP = NULL;
    DDXIIO_GROUP_ITEM * groupItemP = NULL;
    int i = 0;

    rtx_message ("global delay = %f", p->delay);
    while ((groupItemP = rtx_list_iterate (p->groupList)) != NULL) {
        rtx_message ("group %d: delay = %f", ++i, groupItemP->delay);
	while ((itemP = rtx_list_iterate (groupItemP->varList)) != NULL)
	    rtx_message ("item: %s, %s, %s, %d", itemP->storeName,
		     itemP->storeType, itemP->iioName, itemP->type);
    }
}

int
ddxiio_done (
	     DDXIIO * p
	     )
{
    DDXIIO_GROUP_ITEM * groupItemP = NULL;
    DDXIIO_ITEM * itemP = NULL;
    int errs = 0;

    if (p->verbose)
        rtx_message ("ddxiio_done: deleting threads");
    /* delete all the threads */
    while ((groupItemP = rtx_list_iterate (p->groupList)) != NULL) {
        if (p->verbose)
	    rtx_message ("ddxiio_done: deleting thread for delay %f", 
			 groupItemP->delay);
        if (rtx_thread_destroy_sync (groupItemP->th) == -1) {
	    rtx_error ("ddxiio_done: rtx_thread_destroy_sync (%f) failed",
		       groupItemP->delay);
	    errs++;
	}
    }

    /* close all the IIO channels */
    if (p->verbose)
        rtx_message ("ddxiio_done: closing IIO channels");
    while ((groupItemP = rtx_list_iterate (p->groupList)) != NULL) {
        while ((itemP = rtx_list_iterate (groupItemP->varList)) != NULL) {
	    if (p->verbose)
	        rtx_message ("ddxiio_done: closing IIO channel for %s:%s", 
			     itemP->iioName, itemP->storeName);
	    if (iio_close (itemP->iioChannel)) {
	        rtx_error ("ddxiio_done: iio_close(%s) failed: %s",
			   itemP->iioName, iio_emessage_get ());
		errs++;
	    }
	}
    }

    /* close IIO */
    if (p->verbose)
        rtx_message ("ddxiio_done: closing IIO");
    if (iio_done ())
        return (rtx_error ("ddxiio_done: iio_done() failed: %s",
			   iio_emessage_get ()));

    /* notify store for all looked up variables */
    if (p->verbose)
        rtx_message ("ddxiio_done: notifying store");
    while ((groupItemP = rtx_list_iterate (p->groupList)) != NULL) {
        while ((itemP = rtx_list_iterate (groupItemP->varList)) != NULL) {
	    if (p->verbose)
	        rtx_message ("ddxiio_done: notifying store for %s:%s", 
			     itemP->iioName, itemP->storeName);
	    if (ddx_store_done_item(itemP->storeItem) == -1) {
	        rtx_error ("ddxiio_done: ddx_store_done_item(%s) failed",
			   itemP->storeName);
		errs++;
	    }
	}
    }
    
    if (p->verbose)
        rtx_message ("ddxiio_done: closing store");
    if (ddx_store_close (p->store) == -1) {
        rtx_error ("ddxiio_done: ddx_store_close() failed");
	errs++;
    }
    if (ddx_client_done () == -1) {
        rtx_error ("ddxiio_done: ddx_client_done() failed");
	errs++;
    }
    if (p->verbose)
        rtx_message ("ddxiio_done: finished");
    if (errs)
        return (-1);
    return (0);
}

